--
--  D_Bus/Ada - An Ada binding to D-Bus
--
--  Copyright (C) 2011  Reto Buerki <reet@codelabs.ch>
--
--  This program is free software; you can redistribute it and/or
--  modify it under the terms of the GNU General Public License
--  as published by the Free Software Foundation; either version 2
--  of the License, or (at your option) any later version.
--
--  This program is distributed in the hope that it will be useful,
--  but WITHOUT ANY WARRANTY; without even the implied warranty of
--  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--  GNU General Public License for more details.
--
--  You should have received a copy of the GNU General Public License
--  along with this program; if not, write to the Free Software
--  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
--  USA.
--
--  As a special exception, if other files instantiate generics from this
--  unit,  or  you  link  this  unit  with  other  files  to  produce  an
--  executable   this  unit  does  not  by  itself  cause  the  resulting
--  executable to  be  covered by the  GNU General  Public License.  This
--  exception does  not  however  invalidate  any  other reasons why  the
--  executable file might be covered by the GNU Public License.
--

with System;

with D_Bus.Arguments;
with D_Bus.Callbacks;
with D_Bus.Messages;

package D_Bus.Connection is

   type Connection_Type (<>) is limited private;
   --  D-Bus connection.

   function Connect (Bus : Bus_Type := Bus_Session) return Connection_Type;
   --  Connect to the given message bus type.

   function Connect (Address : String) return Connection_Type;
   --  Connect to the given remote address.

   function Call_Blocking
     (Connection  : Connection_Type;
      Destination : String;
      Path        : String;
      Iface       : String;
      Method      : String;
      Args        : Arguments.Argument_List_Type :=
        Arguments.Empty_Argument_List)
      return Arguments.Argument_List_Type;
   --  Synchronously call the given method.

   procedure Send_Signal
     (Connection  : Connection_Type;
      Object_Name : String;
      Iface       : String;
      Name        : String;
      Args        : Arguments.Argument_List_Type :=
        Arguments.Empty_Argument_List);
   --  Send a signal over the given connection.

   procedure Send
     (Connection : Connection_Type;
      Message    : Messages.Message_Type);
   --  Add given message to the outgoing message queue.

   procedure Request_Name
     (Connection : Connection_Type;
      Name       : String);
   --  Request name on the bus for given connection.

   procedure Add_Match
     (Connection : Connection_Type;
      Rule       : String);
   --  Add given match rule to match messages going through the message bus.

   procedure Dispatch
     (Connection : Connection_Type;
      Callback   : Callbacks.Message_Callback);
   --  Dispatch messages to the given callback procedure. Use the Add_Match
   --  procedure to add match rules.

private

   type Connection_Type is limited record
      Thin_Connection : System.Address := System.Null_Address;
   end record;

end D_Bus.Connection;
