/*
 * The MIT License
 *
 * Copyright (c) 2019 The Broad Institute
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */


package picard.sam.SamErrorMetric;

import htsjdk.samtools.CigarElement;
import htsjdk.samtools.reference.SamLocusAndReferenceIterator;
import htsjdk.samtools.util.AbstractRecordAndOffset;
import htsjdk.samtools.util.SamLocusIterator;

/**
 * A calculator that estimates the error rate of the bases it observes for indels only.
 */
public class IndelErrorCalculator extends BaseErrorCalculator {
    /**
     * Total number of inserts
     */
    protected long nInsertions;

    /**
     * Total number of inserted bases
     */
    protected long nInsertedBases;

    /**
     * Total number of deletions
     */
    protected long nDeletions;

    /**
     * Total number of deleted bases
     */
    protected long nDeletedBases;

    @Override
    public void addBase(final SamLocusIterator.RecordAndOffset recordAndOffset, final SamLocusAndReferenceIterator.SAMLocusAndReference locusAndRef) {
        super.addBase(recordAndOffset, locusAndRef);

        if (recordAndOffset.getAlignmentType() == AbstractRecordAndOffset.AlignmentType.Insertion) {
            nInsertions++;
            CigarElement cigarElement = ReadBaseStratification.getIndelElement(recordAndOffset);
            if (cigarElement != null) {
                nInsertedBases += cigarElement.getLength();
            }
        } else if (recordAndOffset.getAlignmentType() == AbstractRecordAndOffset.AlignmentType.Deletion) {
            nDeletions++;
            CigarElement cigarElement = ReadBaseStratification.getIndelElement(recordAndOffset);
            if (cigarElement != null) {
                nDeletedBases += cigarElement.getLength();
            }
        }
    }

    /**
     * The suffix that pertains to the implementation of aggregation
     **/
    @Override
    public String getSuffix() {
        return "indel_error";
    }

    /**
     * Returns the metric generated by the observed loci
     **/
    @Override
    public IndelErrorMetric getMetric() {
        return new IndelErrorMetric("", nBases, nInsertions, nInsertedBases, nDeletions, nDeletedBases);
    }
}
