# -*- coding: utf-8 -*-


# PyMeeus: Python module implementing astronomical algorithms.
# Copyright (C) 2018  Dagoberto Salazar
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.


from math import sin, cos, tan, acos, atan2, sqrt, radians, log10

from pymeeus.Angle import Angle
from pymeeus.Epoch import Epoch, JDE2000
from pymeeus.Interpolation import Interpolation
from pymeeus.Coordinates import (
    geometric_vsop_pos, apparent_vsop_pos, orbital_elements,
    nutation_longitude, true_obliquity, ecliptical2equatorial,
    passage_nodes_elliptic
)
from pymeeus.Earth import Earth
from pymeeus.Sun import Sun


"""
.. module:: Jupiter
   :synopsis: Class to model Jupiter planet
   :license: GNU Lesser General Public License v3 (LGPLv3)

.. moduleauthor:: Dagoberto Salazar
"""


VSOP87_L = [
    # L0
    [
        [59954691.495, 0.00000000000, 0.00000000000],
        [9695898.711, 5.06191793105, 529.69096509460],
        [573610.145, 1.44406205976, 7.11354700080],
        [306389.180, 5.41734729976, 1059.38193018920],
        [97178.280, 4.14264708819, 632.78373931320],
        [72903.096, 3.64042909255, 522.57741809380],
        [64263.986, 3.41145185203, 103.09277421860],
        [39806.051, 2.29376744855, 419.48464387520],
        [38857.780, 1.27231724860, 316.39186965660],
        [27964.622, 1.78454589485, 536.80451209540],
        [13589.738, 5.77481031590, 1589.07289528380],
        [8246.362, 3.58227961655, 206.18554843720],
        [8768.686, 3.63000324417, 949.17560896980],
        [7368.057, 5.08101125612, 735.87651353180],
        [6263.171, 0.02497643742, 213.29909543800],
        [6114.050, 4.51319531666, 1162.47470440780],
        [4905.419, 1.32084631684, 110.20632121940],
        [5305.283, 1.30671236848, 14.22709400160],
        [5305.457, 4.18625053495, 1052.26838318840],
        [4647.249, 4.69958109497, 3.93215326310],
        [3045.009, 4.31675960318, 426.59819087600],
        [2610.001, 1.56667594850, 846.08283475120],
        [2028.191, 1.06376547379, 3.18139373770],
        [1764.768, 2.14148077766, 1066.49547719000],
        [1722.983, 3.88036008872, 1265.56747862640],
        [1920.959, 0.97168928755, 639.89728631400],
        [1633.217, 3.58201089758, 515.46387109300],
        [1431.997, 4.29683690269, 625.67019231240],
        [973.278, 4.09764957065, 95.97922721780],
        [884.439, 2.43701426123, 412.37109687440],
        [732.875, 6.08534113239, 838.96928775040],
        [731.072, 3.80591233956, 1581.95934828300],
        [691.928, 6.13368222939, 2118.76386037840],
        [709.190, 1.29272573658, 742.99006053260],
        [614.464, 4.10853496756, 1478.86657406440],
        [495.224, 3.75567461379, 323.50541665740],
        [581.902, 4.53967717552, 309.27832265580],
        [375.657, 4.70299124833, 1368.66025284500],
        [389.864, 4.89716105852, 1692.16566950240],
        [341.006, 5.71452525783, 533.62311835770],
        [330.458, 4.74049819491, 0.04818410980],
        [440.854, 2.95818460943, 454.90936652730],
        [417.266, 1.03554430161, 2.44768055480],
        [244.170, 5.22020878900, 728.76296653100],
        [261.540, 1.87652461032, 0.96320784650],
        [256.568, 3.72410724159, 199.07200143640],
        [261.009, 0.82047246448, 380.12776796000],
        [220.382, 1.65115015995, 543.91805909620],
        [201.996, 1.80684574186, 1375.77379984580],
        [207.327, 1.85461666594, 525.75881183150],
        [197.046, 5.29252149016, 1155.36115740700],
        [235.141, 1.22693908124, 909.81873305460],
        [174.809, 5.90973505276, 956.28915597060],
        [149.368, 4.37745104275, 1685.05212250160],
        [175.184, 3.22634903433, 1898.35121793960],
        [175.191, 3.72966554761, 942.06206196900],
        [157.909, 4.36483921766, 1795.25844372100],
        [137.871, 1.31797920785, 1169.58825140860],
        [117.495, 2.50022140890, 1596.18644228460],
        [150.502, 3.90625022622, 74.78159856730],
        [116.757, 3.38920921041, 0.52126486180],
        [105.895, 4.55439798236, 526.50957135690],
        [130.531, 4.16867945489, 1045.15483618760],
        [141.445, 3.13568357861, 491.55792945680],
        [99.511, 1.42117395747, 532.87235883230],
        [96.137, 1.18156870005, 117.31986822020],
        [91.758, 0.85756633461, 1272.68102562720],
        [87.695, 1.21738140813, 453.42489381900],
        [68.507, 2.35242959478, 2.92076130680],
        [66.098, 5.34386149468, 1471.75302706360],
        [77.401, 4.42676337124, 39.35687591520],
        [72.006, 4.23834923691, 2111.65031337760],
        [63.406, 4.97665525033, 0.75075952540],
        [59.427, 4.11130498612, 2001.44399215820],
        [62.481, 0.51211384012, 220.41264243880],
        [66.532, 2.98864358135, 2214.74308759620],
        [60.194, 4.12628179571, 4.19278569400],
        [56.012, 1.15493222602, 21.34064100240],
        [52.854, 0.91207215543, 10.29494073850],
        [70.297, 5.14180555282, 835.03713448730],
        [51.916, 4.10048180020, 1258.45393162560],
        [46.442, 4.66531163524, 5.62907429250],
        [58.190, 5.86646380344, 5753.38488489680],
        [40.103, 4.68801114087, 0.16005869440],
        [46.654, 4.79394835282, 305.34616939270],
        [39.298, 4.25448423697, 853.19638175200],
        [46.042, 5.10983515150, 4.66586644600],
        [54.459, 1.57072704127, 983.11585891360],
        [38.920, 6.07592905580, 518.64526483070],
        [38.450, 2.43836870888, 433.71173787680],
        [46.800, 3.54640538283, 5.41662597140],
        [41.830, 4.67982493646, 302.16477565500],
        [35.920, 2.45088036239, 430.53034413910],
        [37.888, 0.21127448431, 2648.45482547300],
        [39.190, 1.71835571629, 11.04570026390],
        [37.567, 6.19481310233, 831.85574074960],
        [35.828, 4.61459907698, 2008.55753915900],
        [43.402, 0.14992289081, 528.20649238630],
        [31.598, 5.14073450755, 1788.14489672020],
        [29.849, 5.34441117167, 2221.85663459700],
        [32.811, 5.28907118836, 88.86568021700],
        [27.686, 1.85227036207, 0.21244832110],
        [25.820, 3.85920882494, 2317.83586181480],
        [24.705, 2.63495214991, 114.13847448250],
        [33.844, 1.00563073268, 9683.59458111640],
        [24.266, 3.82355417268, 1574.84580128220],
        [27.111, 2.80845435102, 18.15924726470],
        [26.837, 1.77586123775, 532.13864564940],
        [26.064, 2.74361318804, 2531.13495725280],
        [30.765, 0.42330537728, 1.48447270830],
        [30.476, 3.66677894407, 508.35032409220],
        [23.282, 3.24372142416, 984.60033162190],
        [19.445, 0.52370214471, 14.97785352700],
        [19.332, 4.86314494382, 1361.54670584420],
        [22.910, 3.84914895064, 2428.04218303420],
        [21.617, 6.01696940024, 1063.31408345230],
        [20.155, 5.59582008789, 527.24328453980],
        [23.732, 2.52766031921, 494.26624244250],
        [20.189, 1.01560227681, 628.85158605010],
        [15.994, 5.09003530653, 529.73914920440],
        [16.134, 5.27095037302, 142.44965013380],
        [20.697, 4.03443281612, 355.74874557180],
        [21.479, 1.28668134295, 35.42472265210],
        [14.964, 4.86039684390, 2104.53676637680],
        [17.242, 1.59187913206, 1439.50969814920],
        [15.994, 1.89222417794, 529.64278098480],
        [17.958, 4.30178016003, 6.15033915430],
        [13.279, 2.18943981644, 1055.44977692610],
        [14.148, 2.71597731671, 0.26063243090],
        [14.689, 0.87944553412, 99.16062095550],
        [14.202, 2.41335693735, 530.65417294110],
        [15.320, 6.07703092728, 149.56319713460],
        [15.832, 4.11682440678, 636.71589257630],
        [12.398, 2.61042299578, 405.25754987360],
        [16.199, 2.77035044582, 760.25553592000],
        [13.665, 3.56039678310, 217.23124870110],
        [15.261, 2.81824770887, 621.73803904930],
        [14.681, 6.26423732742, 569.04784100980],
        [12.529, 1.39077179081, 7.06536289100],
        [11.677, 3.60447374272, 2634.22773147140],
        [11.603, 4.60461756191, 7.16173111060],
        [12.152, 0.24540531919, 1485.98012106520],
        [11.347, 2.00818458261, 1073.60902419080],
        [11.242, 2.48000947870, 423.41679713830],
        [10.942, 5.03602448252, 458.84151979040],
        [11.117, 4.04973271023, 519.39602435610],
        [12.256, 4.30153222783, 604.47256366190],
        [13.149, 2.72189077702, 1364.72809958190],
        [10.604, 3.11518747072, 1.27202438720],
        [9.874, 1.70200068743, 1699.27921650320],
        [10.851, 5.08554552028, 2324.94940881560],
        [10.692, 2.51401681528, 2847.52682690940],
        [12.640, 4.75572797691, 528.72775724810],
        [10.084, 4.05599810206, 38.13303563780],
        [11.536, 2.35034215745, 643.82943957710],
        [10.247, 3.63479911496, 2744.43405269080],
        [10.105, 3.65845333837, 107.02492748170],
        [10.121, 1.31482648275, 1905.46476494040],
        [9.341, 5.92176693887, 1148.24761040620],
        [8.796, 2.77421822809, 6.59228213900],
        [8.420, 4.52537756809, 1677.93857550080],
        [10.128, 2.09034472544, 511.53171782990],
        [8.272, 2.98682673354, 540.73666535850],
        [9.753, 1.22438911827, 32.24332891440],
        [10.630, 2.07777800288, 92.04707395470],
        [7.850, 0.98996894618, 408.43894361130],
        [8.811, 3.46911754939, 1021.24889455140],
        [7.946, 2.86682926070, 2125.87740737920],
        [8.575, 5.29590411702, 415.55249061210],
        [7.841, 6.08025056721, 70.84944530420],
        [7.706, 1.69832954219, 8.07675484730],
        [7.265, 4.65503563919, 629.60234557550],
        [7.164, 4.93400217968, 1056.20053645150],
        [7.247, 4.61607677560, 2420.92863603340],
        [7.753, 2.12871653382, 33.94024994380],
        [6.645, 0.45647460873, 635.96513305090],
        [9.377, 4.03158388202, 2810.92146160520],
        [8.263, 1.23558676139, 1802.37199072180],
        [6.341, 0.07278001580, 202.25339517410],
        [6.383, 3.54310669809, 1891.23767093880],
        [7.902, 2.32510002614, 230.56457082540],
        [6.214, 4.54560345237, 2.70831298570],
        [7.347, 1.24457237337, 24.37902238820],
        [7.451, 3.02719199239, 330.61896365820],
        [6.220, 1.77687561489, 1062.56332392690],
        [5.674, 5.14132196367, 746.92221379570],
        [5.855, 5.42130172896, 28.31117565130],
        [5.629, 3.24348217277, 529.16970023280],
        [7.652, 0.52813391052, 672.14061522840],
        [5.456, 3.34716871364, 2950.61960112800],
        [7.127, 1.43497795005, 6.21977512350],
        [5.388, 4.90175095580, 69.15252427480],
        [5.618, 4.97903783721, 2641.34127847220],
        [5.844, 2.95364118152, 490.33408917940],
        [4.943, 5.37597740579, 721.64941953020],
        [5.062, 4.84282906467, 31.01948863700],
        [5.163, 5.07410777073, 67.66805156650],
        [4.739, 6.10248862834, 106.27416795630],
        [4.879, 0.07095292379, 78.71375183040],
        [4.854, 5.63875661096, 1.69692102940],
        [5.629, 3.73871604865, 530.21222995640],
        [4.471, 4.49152882547, 505.31194270640],
        [4.313, 4.79369370451, 535.10759106600],
        [4.280, 0.54783823710, 1.43628859850],
        [4.453, 0.50551854591, 524.06189080210],
        [4.936, 4.82992988255, 422.66603761290],
        [4.701, 3.41634869046, 3060.82592234740],
        [4.261, 2.67044686458, 561.93429400900],
        [4.156, 4.00660658688, 99.91138048090],
        [4.561, 2.29650164054, 3163.91869656600],
        [4.414, 5.67224020329, 1464.63948006280],
        [5.345, 0.31513851830, 1289.94650101460],
        [5.269, 3.89116469022, 191.95845443560],
        [3.855, 4.28942301453, 1994.33044515740],
        [4.210, 5.32763589447, 2538.24850425360],
        [3.949, 4.56507101172, 1382.88734684660],
        [3.885, 1.56778786810, 647.01083331480],
        [4.227, 5.51697599030, 5223.69391980220],
        [4.129, 2.81119457666, 416.30325013750],
        [3.663, 4.35187510477, 2737.32050569000],
        [3.566, 5.48243943375, 750.10360753340],
        [4.330, 0.84941756640, 531.17543780290],
        [4.093, 0.19980340452, 525.02509864860],
        [4.022, 1.92293311337, 1512.80682400820],
        [3.400, 6.00302355875, 1.22384027740],
        [3.496, 0.31252921473, 597.35901666110],
        [3.299, 4.27596694481, 526.77020378780],
        [3.226, 2.90455264496, 963.40270297140],
        [3.150, 3.81061764181, 280.96714700450],
        [4.129, 4.74946631331, 0.89377187730],
        [3.840, 1.91064405186, 378.64329525170],
        [3.057, 1.65589659685, 528.94020556920],
        [3.011, 1.59276337369, 224.34479570190],
        [3.196, 5.86588452873, 4.14460158420],
        [3.628, 0.07930225897, 558.00214074590],
        [2.932, 0.41424445089, 7.86430652620],
        [3.316, 2.70211697795, 532.61172640140],
        [2.925, 4.47580363425, 533.88375078860],
        [3.690, 0.39897023849, 685.47393735270],
        [3.223, 2.45833032883, 960.22130923370],
        [3.059, 5.32616140812, 530.44172462000],
        [3.383, 4.42170370028, 312.45971639350],
        [3.320, 2.71417812514, 495.75071515080],
        [2.697, 5.23146633437, 739.80866679490],
        [3.590, 2.30999595873, 908.33426034630],
        [3.677, 5.07337955976, 73.29712585900],
        [2.618, 3.09118499149, 3267.01147078460],
        [2.796, 2.98942316119, 483.22054217860],
        [3.398, 3.29598270278, 911.30320576290],
        [3.352, 1.44391979336, 593.42686339800],
        [2.563, 3.35080110279, 2207.62954059540],
        [2.553, 0.36892288645, 1048.33622992530],
        [2.620, 3.82769874340, 520.12973753900],
        [3.356, 1.08315053878, 46.47042291600],
        [3.035, 5.52230028113, 618.55664531160],
        [3.397, 3.83084746522, 210.11770170030],
        [2.497, 0.47917884538, 945.24345570670],
        [2.341, 5.87941292649, 2751.54759969160],
        [2.656, 0.49713061045, 1057.89745748090],
        [2.581, 0.03759881914, 1.64453140270],
        [2.900, 2.50019054587, 525.49817940060],
        [3.153, 2.30900986177, 457.61767951300],
        [2.201, 3.94367109739, 31.49256938900],
        [2.381, 6.19252134885, 327.43756992050],
        [2.458, 0.65614291954, 9153.90361602180],
        [2.111, 5.61905648764, 16.46232623530],
        [2.130, 3.75880734109, 724.83081326790],
        [2.406, 2.29315649755, 195.13984817330],
        [2.166, 5.43262641046, 534.35683154060],
        [2.057, 1.49875151278, 551.03160609700],
        [2.676, 5.06374981112, 456.39383923560],
        [2.078, 5.28920097886, 76.26607127560],
        [2.261, 5.38117230692, 1781.03134971940],
        [2.356, 0.67392574097, 227.52618943960],
        [2.240, 3.18006978517, 3377.21779200400],
        [2.183, 3.08384250950, 524.27433912320],
        [2.119, 2.70107659927, 387.24131496080],
        [2.056, 4.82779196994, 2957.73314812880],
        [2.116, 6.20263841494, 209.36694217490],
        [2.712, 3.18157754631, 1474.67378837040],
        [2.127, 1.24424012514, 539.98590583310],
        [2.424, 3.57595925853, 953.10776223290],
        [1.947, 1.94468082546, 529.53090640020],
        [1.896, 4.01406242800, 2310.72231481400],
        [1.935, 4.10051493950, 3053.71237534660],
        [2.056, 6.27074148550, 245.54242435240],
        [2.108, 3.22886474225, 252.65597135320],
        [2.596, 2.77467278614, 177.87437278590],
        [1.919, 3.14834694111, 381.61224066830],
        [2.217, 1.92368906925, 535.91074021810],
        [1.947, 5.03751780002, 529.85102378900],
        [2.025, 4.82814272957, 17.26547538740],
        [1.945, 2.10611582568, 3480.31056622260],
        [1.899, 0.05104263891, 560.71045373160],
        [2.221, 0.58365090630, 3178.14579056760],
        [2.271, 1.67360565619, 731.94436026870],
        [1.706, 5.40277333462, 20.44686912510],
        [2.295, 4.20863103004, 1038.04128918680],
        [2.218, 3.65982280555, 282.45161971280],
        [2.181, 4.87369503022, 535.32003938710],
        [1.745, 1.34021867874, 25.12978191360],
        [1.601, 3.92730015840, 17.52610781830],
        [1.651, 0.63598292839, 17.40848773930],
        [1.826, 0.31592311031, 124.43341522100],
        [2.041, 0.15617294873, 598.84348936940],
        [1.494, 3.81418025130, 319.57326339430],
        [1.551, 5.25201528605, 437.64389113990],
        [1.852, 2.36130812462, 37.87240320690],
        [1.466, 1.72926380881, 59.80374504030],
        [1.417, 5.82273267086, 81.75213321620],
        [1.430, 1.17528806260, 440.82528487760],
        [1.906, 4.06896022692, 1819.63746610920],
        [1.397, 0.26383366743, 50.40257617910],
        [1.756, 2.32977483716, 938.12990870590],
        [1.487, 2.24866746540, 10.03430830760],
        [1.368, 3.56691602771, 1514.29129671650],
        [1.400, 4.84502200703, 295.05122865420],
        [1.344, 2.20177702122, 529.90341341570],
        [1.464, 1.42648716568, 1158.54255114470],
        [1.341, 1.15693423225, 2435.15573003500],
        [1.786, 5.44716330146, 2854.64037391020],
        [1.677, 6.22875777048, 833.55266177900],
        [1.471, 4.80574535807, 696.51963761660],
        [1.436, 1.45810957330, 537.76771994190],
        [1.657, 0.02890651793, 138.51749687070],
        [1.300, 3.14074420421, 547.85021235930],
        [1.343, 6.14827138025, 988.53248488500],
        [1.344, 4.78042160426, 529.47851677350],
        [1.234, 2.83294330979, 3583.40334044120],
        [1.651, 2.12056447005, 1061.82961074400],
        [1.479, 0.24646493075, 1593.00504854690],
        [1.413, 3.07444632745, 6283.07584999140],
        [1.246, 5.94882321661, 1056.93424963440],
        [1.225, 1.95642397635, 1969.20066324380],
        [1.388, 2.87749576073, 1023.95720753710],
        [1.263, 3.46181945031, 40.84134862350],
        [1.325, 4.15429781246, 916.93228005540],
        [1.477, 5.26691818477, 810.65811209910],
        [1.165, 4.65528125418, 944.98282327580],
        [1.137, 2.48561382158, 2.00573757010],
        [1.118, 3.80747957482, 7.00167241620],
        [1.138, 5.11611532241, 885.43971066640],
        [1.131, 1.54599459004, 775.23338944700],
        [1.477, 4.69742954455, 630.33605875840],
        [1.252, 1.34316620527, 739.05790726950],
        [1.273, 5.19070939905, 2097.42321937600],
        [1.446, 5.54999644374, 43.28902917830],
        [1.344, 4.75897665313, 1166.40685767090],
        [1.101, 4.56997613488, 3274.12501778540],
        [1.376, 3.60998729004, 415.29185818120],
        [1.437, 6.22410093972, 155.78297225810],
        [1.167, 4.09497264272, 203.00415469950],
        [1.237, 4.41132627005, 292.01284726840],
        [1.077, 2.57045229823, 25.27279426550],
        [1.341, 0.49262296655, 635.23141986800],
        [1.209, 3.36289125536, 521.61421024730],
        [1.030, 1.81822316284, 465.95506679120],
        [1.002, 3.21720955284, 2524.02141025200],
        [1.338, 1.26054917773, 902.70518605380],
        [1.037, 3.87858871885, 3370.10424500320],
        [1.224, 0.09219976028, 824.74219374880],
        [1.255, 3.04675952762, 447.79581952650],
        [0.991, 4.16587903812, 632.83192342300],
        [0.975, 3.80216680539, 2627.11418447060],
        [1.061, 5.60184374277, 732.69511979410],
        [1.049, 2.94931080683, 3693.60966166060],
        [0.984, 0.98260254313, 632.73555520340],
        [1.050, 2.20935815967, 7.22542158540],
        [0.996, 5.41921062583, 1059.43011429900],
        [0.961, 0.87315283361, 544.66881862160],
        [1.175, 3.09093466406, 1894.41906467650],
        [1.049, 5.81616384906, 26.82670294300],
        [1.161, 0.01274801567, 850.01498801430],
        [1.109, 3.63294273717, 306.83064210100],
        [1.077, 0.95716576092, 608.40471692500],
        [1.288, 4.23019288942, 1215.16490244730],
        [1.060, 3.85856787901, 631.82053146670],
        [1.251, 6.15889818604, 462.02291352810],
        [1.165, 3.50653563773, 8.59801970910],
        [0.933, 4.62559759882, 1049.08698945070],
        [1.035, 1.30805283339, 633.74694715970],
        [1.238, 2.21195391602, 25558.21217647960],
        [1.240, 2.27960685992, 6.90109867970],
        [0.942, 4.14526324371, 945.99421523210],
        [0.927, 6.10893117637, 514.71311156760],
        [0.914, 6.17656044376, 952.35700270750],
        [0.893, 4.27448748055, 0.63313944640],
        [1.045, 1.64682770236, 565.11568774670],
        [0.903, 1.94250156640, 3796.70243587920],
        [1.162, 5.51229668479, 2.96894541660],
        [0.901, 3.03568112112, 460.53844081980],
        [0.903, 2.24012822393, 523.54062594030],
        [1.060, 5.28027224466, 3171.03224356680],
        [1.064, 0.99330150801, 320.32402291970],
        [0.970, 4.56607888439, 429.04587143080],
        [1.071, 4.33203090957, 610.69233878540],
        [0.865, 0.21831429230, 1098.73880610440],
        [0.865, 2.82123742108, 1060.34513803570],
        [0.882, 4.80076824948, 384.05992122310],
        [0.959, 5.45468005818, 451.94042111070],
        [1.042, 5.79270325150, 303.86169668440],
        [0.784, 1.85150700827, 313.21047591890],
        [1.083, 1.40526460812, 72.07328558160],
        [0.782, 3.03559242565, 5.84152261360],
        [0.854, 1.22236205478, 611.44309831080],
        [0.996, 2.22139794743, 1059.33374607940],
        [0.719, 4.92550252164, 421.93232443000],
        [0.953, 3.98347050083, 836.52160719560],
        [0.822, 4.49679856387, 10213.28554621100],
        [0.707, 2.16473400319, 2228.97018159780],
        [0.715, 4.62515255534, 385.54439393140],
        [0.737, 4.63776694324, 1134.16352875650],
        [0.730, 1.87179326186, 153.49535039770],
        [0.709, 2.93132115910, 417.03696332040],
        [0.926, 1.77006317007, 2332.06295581640],
        [0.864, 3.03246275970, 1041.22268292450],
        [0.708, 6.01601101389, 395.10562148700],
        [0.935, 6.01864676296, 173.94221952280],
        [0.695, 1.39408383356, 432.01481684740],
        [0.687, 3.06548397586, 529.95159752550],
        [0.677, 3.58357527210, 244.31858407500],
        [0.850, 5.46114025921, 41.05379694460],
        [0.817, 4.65315342412, 535.84130424890],
        [0.652, 0.44173759183, 1201.83158032300],
        [0.711, 0.96283289310, 373.01422095920],
        [0.665, 1.03244633471, 623.22251175760],
        [0.643, 5.05335060049, 522.62560220360],
        [0.639, 4.22718483639, 25.86349509650],
        [0.718, 5.07576900710, 1058.41872234270],
        [0.664, 2.43728454444, 1585.14074202070],
        [0.833, 1.49468440213, 563.63121503840],
        [0.760, 4.34849823663, 100.64509366380],
        [0.633, 4.31796718640, 3590.51688744200],
        [0.629, 6.23431126402, 679.25416222920],
        [0.617, 2.68075016456, 3899.79521009780],
        [0.646, 2.88581188015, 13.49338081870],
        [0.768, 3.18498076120, 1151.42900414390],
        [0.731, 5.86653168561, 501.37978944330],
        [0.652, 0.82865771780, 2015.67108615980],
        [0.796, 5.36663489938, 420.96911658350],
        [0.647, 4.74965662438, 567.82400073240],
        [0.845, 1.69406147722, 1744.85586754190],
        [0.802, 5.79824707751, 981.63138620530],
        [0.764, 5.05232933368, 827.92358748650],
        [0.604, 5.11265182908, 1159.29331067010],
        [0.682, 3.68248136835, 2281.23049651060],
        [0.740, 0.74512356954, 1261.63532536330],
        [0.666, 2.06624389616, 27.08733537390],
        [0.652, 4.92932795958, 2413.81508903260],
        [0.559, 0.17558868481, 63.73589830340],
        [0.577, 3.82752312276, 1550.93985964600],
        [0.727, 1.05835550856, 490.07345674850],
        [0.574, 3.61492119092, 3686.49611465980],
        [0.732, 5.93179840659, 42.53826965290],
        [0.606, 2.71411884300, 1173.52040467170],
        [0.633, 4.21720828607, 166.82867252200],
        [0.687, 3.91671464962, 529.43033266370],
        [0.570, 2.73551750122, 4010.00153131720],
        [0.552, 2.36967119362, 1603.29998928540],
        [0.600, 1.82659364395, 522.52923398400],
        [0.558, 5.09099246601, 1354.43315884340],
        [0.519, 6.11952999304, 366.79444583570],
        [0.719, 0.85722557905, 362.86229257260],
        [0.518, 2.03954064144, 418.52143602870],
        [0.515, 3.51750445111, 528.41894070740],
        [0.515, 3.47930063838, 103.14095832840],
        [0.550, 5.77676837730, 420.44785172170],
        [0.702, 3.67952126446, 1279.79457262800],
        [0.550, 0.61451088395, 104.05598206510],
        [0.495, 2.41738205536, 179.35884549420],
        [0.513, 0.29823688044, 103.04459010880],
        [0.537, 5.47946238724, 771.30123618390],
        [0.507, 3.08777345288, 1357.61455258110],
        [0.495, 4.95362659160, 536.85269620520],
        [0.681, 4.56294416261, 112.65400177420],
        [0.500, 3.15631977489, 1070.42763045310],
        [0.484, 0.79038835602, 28.45418800320],
        [0.529, 5.46978501034, 419.43645976540],
        [0.597, 4.98058295172, 1251.34038462480],
        [0.492, 3.96066546484, 1269.49963188950],
        [0.482, 3.60167662490, 2943.50605412720],
        [0.630, 6.16496640092, 105.54045477340],
        [0.480, 0.86786400621, 35.21227433100],
        [0.516, 5.97528782923, 3067.93946934820],
        [0.586, 5.48467997697, 56.62235130260],
        [0.502, 1.43671788959, 469.88722005430],
        [0.473, 2.28007170041, 2042.49778910280],
        [0.565, 1.90952569252, 107.28555991260],
        [0.452, 3.13938145287, 934.94851496820],
        [0.605, 1.65413715574, 761.74000862830],
        [0.443, 5.46282223686, 135.33610313300],
        [0.580, 2.06327501551, 493.04240216510],
        [0.540, 1.73777995970, 536.75632798560],
        [0.432, 0.27167052107, 93.53154666300],
        [0.515, 3.46469417437, 530.96298948180],
        [0.440, 5.28884782489, 497.44763618020],
        [0.487, 5.78767525063, 12036.46073488820],
        [0.452, 2.57855172248, 1254.52177836250],
        [0.427, 3.21032629463, 2840.41327990860],
        [0.414, 1.54298025443, 115.62294719080],
        [0.424, 0.12699448931, 1268.74887236410],
        [0.411, 3.12424023238, 536.28324723360],
        [0.452, 1.00194596383, 113.38771495710],
        [0.419, 0.81834479225, 1165.65609814550],
        [0.490, 4.72785081986, 277.03499374140],
        [0.434, 0.36146539146, 1304.92435454160],
        [0.401, 5.70326543719, 1127.04998175570],
        [0.461, 3.26462894820, 102.12956637210],
        [0.533, 2.54951615753, 141.22580985640],
        [0.413, 4.38801694479, 6151.53388830500],
        [0.415, 1.68861617902, 391.17346822390],
        [0.385, 1.69092319074, 4113.09430553580],
        [0.450, 5.49339192735, 602.98809095360],
        [0.499, 3.80738617353, 81.00137369080],
        [0.454, 0.10952919733, 600.54041039880],
        [0.377, 6.25375060718, 913.75088631770],
        [0.453, 3.86104865567, 758.77106321170],
        [0.401, 4.44475618337, 990.22940591440],
        [0.407, 5.13442416563, 3487.42411322340],
        [0.435, 3.76103358490, 523.09868295560],
        [0.425, 3.22287851959, 2655.56837247380],
        [0.365, 5.16456645463, 4694.00295470760],
        [0.454, 1.63325197950, 976.00231191280],
        [0.406, 2.72102389267, 1438.02522544090],
        [0.349, 3.59598366422, 1058.86066532740],
        [0.354, 0.62136331420, 498.67147645760],
        [0.383, 5.09229089574, 539.25219265020],
        [0.380, 3.92653231573, 561.18353448360],
        [0.339, 4.12175871949, 3906.90875709860],
        [0.458, 3.42556794767, 121.25202148330],
        [0.427, 3.61285264910, 860.30992875280],
        [0.424, 4.72757252331, 1366.21257229020],
        [0.328, 4.55286002816, 1696.09782276550],
        [0.324, 4.23685005210, 642.34496686880],
        [0.395, 3.26282558955, 484.44438245600],
        [0.330, 6.05223507989, 215.74677599280],
        [0.318, 2.02072800070, 2964.84669512960],
        [0.417, 0.20173093597, 842.90144101350],
        [0.408, 0.45800247268, 1578.02719501990],
        [0.342, 6.15347077985, 1371.84164658270],
        [0.310, 1.97259286255, 754.03576079650],
        [0.340, 2.77813018312, 3.52311834900],
        [0.333, 2.91352254678, 576.16138801060],
        [0.324, 0.32544817254, 586.31331639720],
        [0.302, 2.08708848849, 526.98265210890],
        [0.363, 4.70567113230, 2730.20695868920],
        [0.300, 0.94464473068, 1432.39615114840],
        [0.352, 5.75013621801, 806.72595883600],
        [0.296, 3.97807312133, 2043.98226181110],
        [0.295, 2.35257797599, 4216.18707975440],
        [0.309, 2.49768755925, 4326.39340097380],
        [0.306, 3.35876843257, 2424.11002977110],
        [0.300, 4.94288858368, 1379.70595310890],
        [0.336, 4.49193455535, 1585.89150154610],
        [0.402, 2.04684001796, 842.15068148810],
        [0.312, 4.59043534747, 188.92007304980],
        [0.346, 5.19792097706, 523.47118997110],
        [0.380, 1.67961600066, 36.64856292950],
        [0.338, 1.32014513725, 148.07872442630],
        [0.391, 4.82224015188, 1012.91150727320],
        [0.285, 3.43655052437, 1053.96530421780],
        [0.332, 2.02575636311, 1091.62525910360],
        [0.282, 5.78865321890, 1064.04779663520],
        [0.282, 0.39153852422, 207.67002114550],
        [0.280, 3.80196391678, 298.23262239190],
        [0.387, 6.26819309990, 1141.13406340540],
        [0.349, 4.09121908199, 1059.90319505100],
        [0.320, 0.39871942000, 2122.69601364150],
        [0.327, 4.76503823073, 134.58534360760],
        [0.283, 3.90409016441, 127.47179660680],
        [0.301, 4.30291951219, 299.12639426920],
        [0.322, 2.48251052680, 1065.60170531270],
        [0.297, 2.40814103509, 1591.52057583860],
        [0.286, 5.85849626574, 172.24529849340],
        [0.285, 4.55845472479, 1446.62324515000],
        [0.270, 4.08342186112, 1578.77795454530],
        [0.362, 1.06148806683, 181.80652604900],
        [0.335, 4.51094500655, 2349.32843120380],
        [0.347, 0.62281394535, 1542.60247236780],
        [0.275, 3.38473403113, 4002.88798431640],
        [0.255, 1.52357936497, 1688.23351623930],
        [0.276, 4.32192160071, 1912.57831194120],
        [0.253, 2.40482338279, 97.67614824720],
        [0.248, 4.45058246237, 1688.98427576470],
        [0.300, 3.07435583442, 1902.28337120270],
        [0.257, 4.79180478086, 1670.82502850000],
        [0.319, 1.34244222683, 1288.46202830630],
        [0.245, 4.01852686769, 1567.73225428140],
        [0.278, 0.25406312148, 874.39401040250],
        [0.324, 5.57824969423, 1670.07426897460],
        [0.300, 4.67161812947, 1329.30337692980],
        [0.241, 0.01789818312, 1586.62521472900],
        [0.295, 5.86996114913, 2804.23779773110],
        [0.317, 3.17967272487, 1020.02505427400],
        [0.238, 4.97765946754, 351.81659230870],
        [0.302, 1.20236375616, 232.04904353370],
        [0.301, 5.53432687957, 2274.54683263650],
        [0.286, 2.41008592059, 2545.36205125440],
        [0.294, 2.01783542485, 313.94418910180],
        [0.292, 2.12690999284, 1592.25428902150],
        [0.250, 2.31712163679, 632.26247445140],
        [0.238, 5.06557054569, 3803.81598288000],
        [0.226, 0.05916712753, 1518.22344997960],
        [0.235, 0.16574304942, 137.03302416240],
        [0.298, 2.99720233431, 1467.82087380050],
        [0.286, 5.08357076653, 774.00954916960],
        [0.246, 2.81685822336, 633.30500417500],
        [0.269, 4.93023426152, 151.04766984290],
        [0.228, 6.13118739321, 3281.23856478620],
        [0.228, 1.22066024988, 700.45179087970],
        [0.239, 0.71695698501, 1276.61317889030],
        [0.289, 6.08263862565, 3384.33133900480],
        [0.218, 2.90308501961, 85.82729883120],
        [0.283, 6.28058228271, 71.81265315070],
        [0.271, 6.01605074549, 170.76082578510],
        [0.221, 0.99914179141, 1053.75285589670],
        [0.218, 1.50681393471, 1087.69310584050],
        [0.223, 3.39126063354, 3259.89792378380],
        [0.229, 1.19373202707, 1060.86640289750],
        [0.264, 3.93467945263, 1363.24362687360],
        [0.228, 5.04188376116, 1064.79855616060],
        [0.295, 2.15253086390, 6386.16862421000],
        [0.214, 3.85961180377, 4223.30062675520],
        [0.218, 0.79681703388, 1909.39691820350],
        [0.212, 4.11706418218, 269.92144674060],
        [0.264, 5.81676406517, 77.96299230500],
        [0.256, 5.65978708108, 799.61241183520],
        [0.242, 6.25078283449, 1621.31622419820],
        [0.235, 2.20668997852, 1570.91364801910],
        [0.212, 2.88214546012, 1674.00642223770],
        [0.206, 1.59586787037, 4429.48617519240],
        [0.208, 2.31366614282, 878.32616366560],
        [0.213, 0.30373338388, 8624.21265092720],
        [0.223, 4.88419887133, 1035.00290780100],
        [0.279, 3.65173543621, 84.93352695390],
        [0.210, 4.08825553401, 203.73786788240],
        [0.214, 4.63498396475, 812.14258480740],
        [0.258, 1.73501688450, 1887.30551767570],
        [0.210, 4.51798082710, 1262.38608488870],
        [0.252, 5.69246905091, 104.57724692690],
        [0.205, 4.62946016431, 1056.46116888240],
        [0.263, 3.04951219565, 1493.09366806600],
        [0.222, 5.54424082649, 5216.58037280140],
        [0.244, 0.91026645686, 3707.83675566220],
        [0.204, 0.90117975859, 1408.01712876020],
        [0.225, 1.23997048012, 3340.61242669980],
        [0.258, 2.35906183505, 2861.75392091100],
        [0.267, 3.27705002283, 5120.60114558360],
        [0.214, 0.66988779149, 9146.79006902100],
        [0.235, 4.93761209111, 1443.44185141230],
        [0.194, 1.60798828275, 102.57150935680],
        [0.215, 0.97603524747, 479.28838891550],
        [0.205, 5.23642605904, 4649.89881763120],
        [0.257, 4.70227260707, 9050.81084180320],
        [0.228, 6.23410921116, 64.95973858080],
        [0.180, 4.21309134581, 143.93412284210],
        [0.180, 4.82870451226, 1063.57471588320],
        [0.180, 5.06126965624, 52.69019803950],
        [0.226, 0.55334952097, 554.06998748280],
        [0.209, 5.67975843693, 48.75804477640],
        [0.186, 3.66368928017, 108.72184851110],
        [0.190, 2.00852986549, 1058.63117066380],
        [0.183, 3.17358464220, 140.96517742550],
        [0.198, 5.49816579454, 4333.50694797460],
        [0.240, 6.06602357868, 1821.12193881750],
        [0.172, 3.04802064781, 54.33472944220],
        [0.170, 4.66520291204, 1372.59240610810],
        [0.173, 4.72884056307, 77204.32749453338],
        [0.174, 0.85370421252, 1587.58842257550],
        [0.215, 0.68219980704, 1054.71606374320],
        [0.170, 1.52204803308, 5591.96087960020],
        [0.200, 1.60275092073, 6681.22485339960],
        [0.193, 2.13003479280, 103.61403908040],
        [0.231, 4.69962389031, 1966.23171782720],
        [0.179, 5.57395905447, 1457.52593306200],
        [0.205, 3.65507571128, 906.84978763800],
        [0.181, 4.52272934666, 24498.83024629040],
        [0.223, 0.11650319998, 67.88049988760],
        [0.172, 5.68083885227, 1884.12412393800],
        [0.219, 0.60964963735, 2729.45619916380],
        [0.164, 1.06675279755, 594.65070367540],
        [0.176, 2.36848603898, 977.48678462110],
        [0.170, 2.43036684800, 4532.57894941100],
        [0.191, 3.64255924842, 1440.99417085750],
        [0.207, 0.49276008455, 71.60020482960],
        [0.157, 4.26888100582, 5069.38346150640],
        [0.157, 5.14847227422, 451.72797278960],
        [0.158, 5.00063628575, 650.94298657790],
        [0.159, 5.37530499642, 20426.57109242200],
        [0.218, 0.27875408082, 175.16605980020],
        [0.155, 0.83696849428, 1474.93442080130],
        [0.154, 2.62839957291, 683.18631549230],
        [0.171, 1.79511736017, 1123.11782849260],
        [0.188, 5.24747110812, 25565.32572348040],
        [0.168, 4.14907553818, 946.72792841500],
        [0.203, 2.83699715530, 1489.91227432830],
        [0.173, 4.34546063838, 3046.59882834580],
        [0.190, 5.67865607835, 1060.13268971460],
        [0.201, 2.38524182920, 419.53282798500],
        [0.152, 5.89088685790, 208.63322899200],
        [0.206, 4.46933127349, 2654.67460059650],
        [0.156, 2.37819796438, 2758.66114669240],
        [0.203, 0.70565514297, 498.19839570560],
        [0.205, 3.05468636546, 1062.30269149600],
        [0.174, 3.50824761708, 2004.36475346500],
        [0.148, 4.73961194393, 1799.19059698410],
        [0.188, 3.62315953725, 3156.80514956520],
        [0.183, 2.35011338194, 25551.09862947879],
        [0.162, 1.58053710589, 628.59095361920],
        [0.162, 3.99983876824, 1482.79872732750],
        [0.181, 2.85489861839, 1055.18914449520],
        [0.151, 3.43198157222, 629.86297800640],
        [0.157, 3.15195826490, 1025.44168024540],
        [0.194, 5.13049187783, 1818.15299340090],
        [0.193, 1.92287052164, 1140.38330388000],
        [0.137, 4.22335221970, 1049.82070263360],
        [0.167, 2.85163087563, 5746.27133789600],
        [0.167, 5.73970282991, 5760.49843189760],
        [0.138, 2.23519776527, 1176.70179840940],
        [0.151, 4.89507270899, 532.39927808030],
        [0.147, 2.65931838448, 987.30864460760],
        [0.135, 0.12836417770, 991.71387862270],
        [0.166, 3.12682515439, 580.09354127370],
        [0.118, 5.98810576300, 531.38788612400],
        [0.135, 5.26601313643, 1065.01100448170],
        [0.138, 3.18511244397, 707.56533788050],
        [0.122, 1.34377059565, 446.31134681820],
        [0.120, 2.29717714347, 1059.22187149480],
        [0.121, 0.58145552537, 5621.84292321040],
        [0.103, 4.75645235023, 1226.21060271120],
        [0.104, 6.08481630139, 528.25467649610],
        [0.119, 1.06475523307, 527.99404406520],
        [0.104, 0.89730746841, 531.12725369310],
        [0.120, 5.39001411803, 1059.54198888360],
        [0.104, 0.44849170648, 1128.53445446400],
        [0.117, 5.42449214711, 986.08480433020],
        [0.101, 5.09893554462, 530.58473697190],
        [0.102, 0.26948040239, 450.97721326420],
        [0.107, 1.58724086516, 1069.67687092770],
        [0.086, 2.28711702506, 2498.89162833840],
        [0.101, 1.88318822518, 528.79719321730],
        [0.086, 1.37568728263, 970.51624997220],
        [0.083, 0.06930748288, 530.91480537200],
        [0.085, 3.22094000094, 1553.64817263170],
        [0.083, 0.62963097974, 528.46712481720],
        [0.083, 4.16314675511, 849.26422848890],
        [0.079, 3.46688102340, 1077.54117745390],
        [0.097, 0.87886975916, 9690.70812811720],
        [0.097, 4.27398311206, 9676.48103411560],
        [0.101, 0.29639798579, 857.12853501510],
        [0.083, 2.55427333923, 1059.59437851030],
        [0.078, 0.06461496210, 521.82665856840],
        [0.078, 0.76677000862, 525.54636351040],
        [0.096, 0.33631035749, 1090.40141882620],
        [0.098, 1.42815294497, 757.21715453420],
        [0.077, 0.85066773729, 537.55527162080],
        [0.084, 5.04765104413, 1160.02702385300],
        [0.076, 3.62264327413, 782.34693644780],
        [0.085, 1.86831145784, 25028.52121138500],
        [0.079, 2.90602202890, 2114.83170711530],
    ],
    # L1
    [
        [52993480757.497, 0.00000000000, 0.00000000000],
        [489741.194, 4.22066689928, 529.69096509460],
        [228918.538, 6.02647464016, 7.11354700080],
        [27655.380, 4.57265956824, 1059.38193018920],
        [20720.943, 5.45938936295, 522.57741809380],
        [12105.732, 0.16985765041, 536.80451209540],
        [6068.051, 4.42419502005, 103.09277421860],
        [5433.924, 3.98478382565, 419.48464387520],
        [4237.795, 5.89009351271, 14.22709400160],
        [2211.854, 5.26771446618, 206.18554843720],
        [1295.769, 5.55132765087, 3.18139373770],
        [1745.919, 4.92669378486, 1589.07289528380],
        [1163.411, 0.51450895328, 3.93215326310],
        [1007.216, 0.46478398551, 735.87651353180],
        [1173.129, 5.85647304350, 1052.26838318840],
        [847.678, 5.75805850450, 110.20632121940],
        [827.329, 4.80312015734, 213.29909543800],
        [1003.574, 3.15040301822, 426.59819087600],
        [1098.735, 5.30704981594, 515.46387109300],
        [816.397, 0.58643054886, 1066.49547719000],
        [725.447, 5.51827471473, 639.89728631400],
        [567.845, 5.98867049451, 625.67019231240],
        [474.181, 4.13245269168, 412.37109687440],
        [412.930, 5.73652891261, 95.97922721780],
        [335.817, 3.73248749046, 1162.47470440780],
        [345.249, 4.24159565410, 632.78373931320],
        [234.066, 6.24302226646, 309.27832265580],
        [194.784, 2.21879010911, 323.50541665740],
        [234.340, 4.03469970332, 949.17560896980],
        [183.938, 6.27963588822, 543.91805909620],
        [198.525, 1.50458442825, 838.96928775040],
        [186.899, 6.08620565908, 742.99006053260],
        [171.380, 5.41655983845, 199.07200143640],
        [130.771, 0.62643377351, 728.76296653100],
        [107.575, 4.49282760117, 956.28915597060],
        [115.393, 0.68019050174, 846.08283475120],
        [115.047, 5.28641699144, 2118.76386037840],
        [66.824, 5.73365126533, 21.34064100240],
        [69.618, 5.97263450278, 532.87235883230],
        [64.850, 6.08803490288, 1581.95934828300],
        [79.686, 5.82412400273, 1045.15483618760],
        [57.939, 0.99453087342, 1596.18644228460],
        [65.635, 0.12924191430, 526.50957135690],
        [58.509, 0.58626971028, 1155.36115740700],
        [56.600, 1.41198438841, 533.62311835770],
        [71.643, 5.34162650321, 942.06206196900],
        [57.368, 5.96851304799, 1169.58825140860],
        [54.935, 5.42806383723, 10.29494073850],
        [52.016, 0.22981299129, 1368.66025284500],
        [52.309, 5.72661448388, 117.31986822020],
        [50.418, 6.08075147811, 525.75881183150],
        [47.418, 3.62611843241, 1478.86657406440],
        [39.888, 4.16158013600, 1692.16566950240],
        [46.678, 0.51144073175, 1265.56747862640],
        [32.827, 5.03596689455, 220.41264243880],
        [33.558, 0.09913904872, 302.16477565500],
        [29.379, 3.35927241533, 4.66586644600],
        [29.307, 0.75907909735, 88.86568021700],
        [32.449, 5.37492530697, 508.35032409220],
        [29.483, 5.42208897099, 1272.68102562720],
        [21.802, 6.15054054070, 1685.05212250160],
        [25.195, 1.60723063387, 831.85574074960],
        [21.133, 5.86346824200, 1258.45393162560],
        [19.747, 2.17205957814, 316.39186965660],
        [17.871, 0.82841413516, 433.71173787680],
        [17.703, 5.95527049039, 5.41662597140],
        [17.230, 2.76395560958, 853.19638175200],
        [17.453, 0.70749901224, 1471.75302706360],
        [17.508, 0.49799925173, 1375.77379984580],
        [14.368, 0.91459831140, 18.15924726470],
        [14.107, 0.63031082833, 2.92076130680],
        [11.559, 4.30379009964, 405.25754987360],
        [11.728, 1.76426582357, 380.12776796000],
        [11.054, 5.56735602213, 1574.84580128220],
        [10.425, 0.31355034390, 1361.54670584420],
        [9.804, 5.90363777277, 519.39602435610],
        [9.805, 0.38648727979, 1073.60902419080],
        [9.285, 3.21842287530, 1795.25844372100],
        [8.864, 0.53776257958, 1788.14489672020],
        [8.370, 5.88484552222, 2001.44399215820],
        [8.148, 5.10162311410, 1485.98012106520],
        [7.658, 5.64890060131, 2648.45482547300],
        [6.690, 2.41093459420, 4.19278569400],
        [5.840, 4.22347896053, 2008.55753915900],
        [7.256, 6.19384525651, 11.04570026390],
        [6.266, 1.36137786945, 1148.24761040620],
        [5.141, 5.23083932012, 628.85158605010],
        [5.140, 2.92955981951, 518.64526483070],
        [4.765, 0.16838181862, 629.60234557550],
        [4.603, 0.78529559911, 721.64941953020],
        [4.575, 6.24794935732, 1677.93857550080],
        [4.537, 4.95096707833, 635.96513305090],
        [4.518, 2.06523915453, 453.42489381900],
        [4.414, 0.15381186059, 1699.27921650320],
        [5.593, 5.57489981207, 191.95845443560],
        [5.403, 1.46004886198, 330.61896365820],
        [4.285, 0.23949868127, 2104.53676637680],
        [4.223, 1.44087555881, 2125.87740737920],
        [4.101, 6.19274358942, 636.71589257630],
        [4.432, 4.35811524051, 423.41679713830],
        [4.132, 0.50170694173, 1056.20053645150],
        [4.398, 4.14280286969, 511.53171782990],
        [5.406, 4.40429493698, 2221.85663459700],
        [4.467, 0.08534650684, 1062.56332392690],
        [3.569, 5.66540477010, 2317.83586181480],
        [4.007, 2.54845549248, 74.78159856730],
        [3.515, 0.25495124831, 1055.44977692610],
        [3.687, 2.93378008847, 32.24332891440],
        [2.883, 5.72793010505, 99.91138048090],
        [2.969, 5.50054720569, 107.02492748170],
        [2.720, 1.25222590925, 540.73666535850],
        [2.808, 3.30714813896, 0.75075952540],
        [2.768, 1.61339487804, 1063.31408345230],
        [2.666, 4.28662288102, 106.27416795630],
        [2.704, 3.03615556153, 422.66603761290],
        [3.290, 5.89081682150, 1802.37199072180],
        [2.578, 3.60390367979, 750.10360753340],
        [2.661, 0.35249312659, 1898.35121793960],
        [2.486, 5.28950877719, 1891.23767093880],
        [2.936, 1.09052029450, 1464.63948006280],
        [3.190, 4.60740643547, 416.30325013750],
        [2.390, 6.01779736611, 551.03160609700],
        [2.214, 5.24450923180, 621.73803904930],
        [2.319, 5.82920300130, 305.34616939270],
        [2.089, 5.99310370434, 1994.33044515740],
        [2.042, 0.75008788531, 142.44965013380],
        [2.121, 0.01537599023, 2420.92863603340],
        [2.114, 6.25308371567, 647.01083331480],
        [2.020, 4.17560390841, 569.04784100980],
        [2.109, 5.18682321403, 227.52618943960],
        [2.283, 5.80043809222, 539.98590583310],
        [1.977, 3.99197009651, 24.37902238820],
        [1.960, 1.35288793079, 963.40270297140],
        [1.903, 2.78349628184, 2428.04218303420],
        [1.915, 4.22134509685, 2324.94940881560],
        [1.971, 5.88715684267, 217.23124870110],
        [1.917, 3.03728154374, 1382.88734684660],
        [2.026, 3.08606488714, 408.43894361130],
        [1.834, 5.61474110217, 430.53034413910],
        [1.838, 1.25467410218, 81.75213321620],
        [2.460, 4.63268678998, 1905.46476494040],
        [1.820, 5.97497926120, 114.13847448250],
        [2.043, 4.34047514845, 70.84944530420],
        [1.959, 4.03116026306, 92.04707395470],
        [1.768, 0.33097462499, 35.42472265210],
        [2.334, 5.87042638470, 1038.04128918680],
        [1.835, 4.81326127892, 124.43341522100],
        [2.269, 1.02549350754, 618.55664531160],
        [1.919, 5.01297395549, 99.16062095550],
        [1.923, 0.28688549585, 31.01948863700],
        [1.878, 5.69299116574, 210.11770170030],
        [1.679, 0.25635730278, 295.05122865420],
        [1.656, 5.46039280732, 2634.22773147140],
        [1.675, 6.15609073315, 643.82943957710],
        [1.953, 5.09846435548, 17.40848773930],
        [1.539, 2.75316078346, 415.55249061210],
        [1.467, 0.54812675158, 458.84151979040],
        [1.482, 3.76736278426, 534.35683154060],
        [1.446, 3.15802770791, 25.12978191360],
        [1.667, 0.26406950755, 835.03713448730],
        [1.472, 0.83054329617, 28.31117565130],
        [1.655, 0.88908548504, 1781.03134971940],
        [1.294, 5.76241191046, 440.82528487760],
        [1.348, 2.49823510924, 984.60033162190],
        [1.352, 5.10869562455, 149.56319713460],
        [1.344, 0.01942249067, 2214.74308759620],
        [1.188, 2.24279457878, 31.49256938900],
        [1.166, 0.80686346228, 739.80866679490],
        [1.322, 4.25691184168, 2538.24850425360],
        [1.094, 6.02985819406, 2737.32050569000],
        [1.112, 4.38204360670, 561.93429400900],
        [1.346, 3.20575848870, 525.02509864860],
        [1.056, 5.76507115032, 2310.72231481400],
        [1.159, 0.46189564970, 67.66805156650],
        [1.027, 0.20709586018, 7.86430652620],
        [1.143, 5.56626418636, 46.47042291600],
        [1.012, 0.54293005597, 532.13864564940],
        [0.978, 5.13939194101, 2207.62954059540],
        [0.993, 2.03698185233, 319.57326339430],
        [1.035, 2.90231353535, 611.44309831080],
        [1.021, 4.75651217048, 527.24328453980],
        [1.308, 1.78809336431, 824.74219374880],
        [0.964, 2.82269601958, 2111.65031337760],
        [0.896, 2.54505998806, 2744.43405269080],
        [0.890, 5.41036782817, 28.45418800320],
        [0.906, 0.76565238554, 1439.50969814920],
        [0.985, 0.88687623770, 5760.49843189760],
        [0.983, 1.42102343372, 5746.27133789600],
        [0.892, 5.87250060663, 203.00415469950],
        [0.942, 2.31049430734, 9690.70812811720],
        [0.941, 2.84331157527, 9676.48103411560],
        [0.867, 0.81020362547, 524.27433912320],
        [0.829, 2.35178495412, 312.45971639350],
        [0.912, 2.80494184378, 6.21977512350],
        [0.809, 1.05148218513, 529.64278098480],
        [0.779, 4.80009242059, 945.24345570670],
        [0.878, 5.76532521399, 1.64453140270],
        [0.953, 4.30945738629, 209.36694217490],
        [0.772, 5.25607113566, 2950.61960112800],
        [0.745, 0.03810558502, 535.10759106600],
        [0.744, 0.58381523987, 25.27279426550],
        [0.734, 0.20800485100, 1049.08698945070],
        [0.747, 2.71772840871, 38.13303563780],
        [0.728, 5.97210358938, 945.99421523210],
        [0.769, 4.51394016967, 952.35700270750],
        [0.710, 0.38016353553, 69.15252427480],
        [0.760, 3.07033779824, 39.35687591520],
        [0.802, 1.14191463412, 532.61172640140],
        [0.704, 1.25447308120, 547.85021235930],
        [0.721, 0.73855379162, 2228.97018159780],
        [0.794, 4.25051539085, 2641.34127847220],
        [0.795, 3.20588363820, 604.47256366190],
        [0.818, 1.05229815343, 909.81873305460],
        [0.724, 5.68281830264, 953.10776223290],
        [0.836, 0.60410469174, 2097.42321937600],
        [0.669, 5.75757140051, 2015.67108615980],
        [0.682, 1.19994890339, 387.24131496080],
        [0.640, 3.91546675664, 528.72775724810],
        [0.809, 4.24929331276, 529.73914920440],
        [0.819, 4.91540072376, 2751.54759969160],
        [0.692, 2.51162384766, 916.93228005540],
        [0.784, 4.23651511312, 195.13984817330],
        [0.762, 1.12201139619, 732.69511979410],
        [0.617, 5.80920925081, 739.05790726950],
        [0.727, 4.24401822698, 760.25553592000],
        [0.591, 3.26075006572, 202.25339517410],
        [0.552, 5.83533550039, 526.77020378780],
        [0.640, 1.38530872949, 530.65417294110],
        [0.577, 6.09100925678, 2531.13495725280],
        [0.620, 3.01917904435, 902.70518605380],
        [0.722, 5.18171159557, 1.48447270830],
        [0.540, 3.78809230820, 2957.73314812880],
        [0.523, 3.63882376000, 437.64389113990],
        [0.527, 5.80796427555, 3053.71237534660],
        [0.488, 4.99103190309, 483.22054217860],
        [0.557, 4.11381202161, 2854.64037391020],
        [0.492, 0.76371083106, 1603.29998928540],
        [0.487, 5.55383951779, 2627.11418447060],
        [0.487, 5.86510858429, 724.83081326790],
        [0.453, 0.61375011101, 1159.29331067010],
        [0.450, 2.28121042355, 3060.82592234740],
        [0.515, 4.78126059280, 447.79581952650],
        [0.449, 4.70231576312, 934.94851496820],
        [0.450, 1.91049508739, 597.35901666110],
        [0.438, 6.01178917646, 3178.14579056760],
        [0.494, 0.53844942275, 1354.43315884340],
        [0.501, 5.51752195462, 2435.15573003500],
        [0.432, 3.64903264921, 313.21047591890],
        [0.435, 3.02449828967, 533.88375078860],
        [0.426, 5.07945534339, 2524.02141025200],
        [0.491, 3.59286364200, 230.56457082540],
        [0.547, 0.34432090949, 1251.34038462480],
        [0.503, 1.57454509207, 454.90936652730],
        [0.486, 4.39351469958, 462.02291352810],
        [0.524, 2.03003740296, 1279.79457262800],
        [0.388, 5.58318013074, 731.94436026870],
        [0.449, 1.11025492739, 56.62235130260],
        [0.398, 5.19943284273, 3267.01147078460],
        [0.416, 1.70821917336, 245.54242435240],
        [0.379, 1.80234948769, 2655.56837247380],
        [0.355, 1.65214516751, 78.71375183040],
        [0.404, 1.72647262603, 1141.13406340540],
        [0.335, 6.01254286794, 960.22130923370],
        [0.331, 1.74086938716, 490.33408917940],
        [0.401, 0.30034336462, 2332.06295581640],
        [0.336, 2.64385574909, 1021.24889455140],
        [0.389, 0.31259289221, 2413.81508903260],
        [0.314, 5.73833529708, 1158.54255114470],
        [0.313, 4.74363791106, 938.12990870590],
        [0.333, 0.80112437148, 1585.14074202070],
        [0.323, 3.52656245280, 3274.12501778540],
        [0.395, 1.73181407631, 1593.00504854690],
        [0.302, 4.64184749164, 1261.63532536330],
        [0.325, 0.54991590409, 43.28902917830],
        [0.293, 0.97977818746, 1585.89150154610],
        [0.341, 2.80833606944, 1514.29129671650],
        [0.304, 6.12522825214, 1262.38608488870],
        [0.286, 2.89800423081, 530.21222995640],
        [0.387, 0.46648572639, 1592.25428902150],
        [0.285, 4.56394598052, 1268.74887236410],
        [0.310, 4.69102289591, 76.26607127560],
        [0.278, 5.49867187248, 280.96714700450],
        [0.358, 5.45926487831, 113.38771495710],
        [0.283, 1.09230506350, 1061.82961074400],
        [0.326, 0.60265259639, 827.92358748650],
        [0.284, 5.36580034539, 1165.65609814550],
        [0.281, 5.54635461050, 3370.10424500320],
        [0.269, 3.92616563946, 42.53826965290],
        [0.275, 2.58465453365, 373.01422095920],
        [0.357, 1.39391983207, 1493.09366806600],
        [0.258, 5.96670694140, 1269.49963188950],
        [0.259, 2.56026216784, 9146.79006902100],
        [0.281, 2.74823090198, 4694.00295470760],
        [0.281, 3.01324655940, 320.32402291970],
        [0.272, 4.18504958920, 8624.21265092720],
        [0.245, 1.24462798353, 252.65597135320],
        [0.244, 2.02892764690, 3377.21779200400],
        [0.324, 1.84851618413, 1289.94650101460],
        [0.221, 6.22167997496, 3281.23856478620],
        [0.238, 3.93371505401, 3171.03224356680],
        [0.226, 5.94296271326, 224.34479570190],
        [0.213, 3.68264234750, 1048.33622992530],
        [0.216, 5.82941334164, 1567.73225428140],
        [0.295, 4.70194747095, 3067.93946934820],
        [0.206, 4.98184230959, 1357.61455258110],
        [0.202, 1.32439444045, 4326.39340097380],
        [0.227, 0.78540105705, 59.80374504030],
        [0.237, 5.56926897693, 2943.50605412720],
        [0.207, 0.07907015398, 5223.69391980220],
        [0.199, 3.30501818656, 4120.20785253660],
        [0.194, 5.95526916809, 84.93352695390],
        [0.266, 1.58032565718, 983.11585891360],
        [0.198, 4.31078641704, 4017.11507831800],
        [0.198, 0.30166351366, 1166.40685767090],
        [0.188, 0.90738705875, 135.33610313300],
        [0.186, 0.69289672485, 92.79783348010],
        [0.182, 1.18931462257, 1512.80682400820],
        [0.191, 1.04146023518, 1884.12412393800],
        [0.174, 6.13734594396, 3597.63043444280],
        [0.189, 0.35191512844, 1372.59240610810],
        [0.172, 4.35250972697, 1578.02719501990],
        [0.173, 2.30241719278, 1176.70179840940],
        [0.220, 1.06991056825, 2200.51599359460],
        [0.186, 4.90511103807, 3583.40334044120],
        [0.189, 0.24160744024, 1670.82502850000],
        [0.206, 0.01485146863, 2730.20695868920],
        [0.174, 1.83997277029, 746.92221379570],
        [0.225, 3.13108099660, 630.33605875840],
        [0.206, 5.22730929781, 3995.77443731560],
        [0.169, 2.57956682688, 9161.01716302260],
        [0.165, 1.51795928301, 4010.00153131720],
        [0.181, 2.05055200822, 842.90144101350],
        [0.181, 5.96554625357, 1578.77795454530],
        [0.166, 1.55114863100, 1070.42763045310],
        [0.157, 5.87839958880, 3914.02230409940],
        [0.160, 0.43729819176, 2545.36205125440],
        [0.168, 5.73975661792, 2847.52682690940],
        [0.157, 2.25764581068, 850.01498801430],
        [0.187, 0.64918748618, 842.15068148810],
        [0.180, 1.88055488803, 685.47393735270],
        [0.153, 4.15259684562, 4333.50694797460],
        [0.154, 3.65536637158, 77734.01845962799],
        [0.151, 3.17795437121, 3590.51688744200],
        [0.155, 3.87623547990, 327.43756992050],
        [0.171, 3.33647878498, 1912.57831194120],
        [0.188, 4.53005359421, 1041.22268292450],
        [0.134, 4.09921613445, 530.44172462000],
        [0.123, 4.79543460218, 1098.73880610440],
        [0.161, 2.02006564218, 860.30992875280],
        [0.143, 2.40197278329, 529.16970023280],
        [0.115, 1.55831212007, 9153.90361602180],
        [0.106, 5.94313244357, 1057.89745748090],
        [0.119, 5.10578428676, 1056.93424963440],
        [0.100, 5.74974781049, 501.23677709140],
        [0.094, 1.40134175492, 1059.33374607940],
        [0.098, 3.79115318281, 497.44763618020],
        [0.090, 4.09610113044, 1064.04779663520],
        [0.102, 1.10442899544, 1969.20066324380],
        [0.087, 0.58218477838, 1173.52040467170],
        [0.109, 3.83745968299, 525.49817940060],
        [0.094, 4.59915291355, 1059.43011429900],
        [0.118, 6.11701561559, 1069.67687092770],
        [0.107, 5.40509332689, 679.25416222920],
        [0.089, 5.90037690244, 757.21715453420],
        [0.078, 6.06217863109, 970.51624997220],
        [0.080, 5.45470236239, 3163.91869656600],
        [0.072, 5.65789862232, 1151.42900414390],
        [0.080, 0.04539720100, 1080.72257119160],
        [0.075, 4.26526686574, 1058.41872234270],
    ],
    # L2
    [
        [47233.598, 4.32148323554, 7.11354700080],
        [30629.053, 2.93021440216, 529.69096509460],
        [38965.550, 0.00000000000, 0.00000000000],
        [3189.317, 1.05504615595, 522.57741809380],
        [2723.358, 3.41411526638, 1059.38193018920],
        [2729.292, 4.84545481351, 536.80451209540],
        [1721.069, 4.18734385158, 14.22709400160],
        [383.258, 5.76790714387, 419.48464387520],
        [367.498, 6.05509120409, 103.09277421860],
        [377.524, 0.76048964872, 515.46387109300],
        [337.386, 3.78644384244, 3.18139373770],
        [308.200, 0.69356654052, 206.18554843720],
        [218.408, 3.81389191353, 1589.07289528380],
        [198.883, 5.33996443444, 1066.49547719000],
        [197.445, 2.48356402053, 3.93215326310],
        [146.230, 3.81373196838, 639.89728631400],
        [155.862, 1.40642426467, 1052.26838318840],
        [129.570, 5.83738872525, 412.37109687440],
        [141.932, 1.63435169016, 426.59819087600],
        [117.327, 1.41435462588, 625.67019231240],
        [96.733, 4.03383427887, 110.20632121940],
        [90.823, 1.10630629042, 95.97922721780],
        [78.769, 4.63726131329, 543.91805909620],
        [72.392, 2.21716670026, 735.87651353180],
        [87.292, 2.52235174825, 632.78373931320],
        [56.910, 3.12292059854, 213.29909543800],
        [48.622, 1.67283791618, 309.27832265580],
        [58.475, 0.83216317444, 199.07200143640],
        [40.150, 4.02485444740, 21.34064100240],
        [39.784, 0.62416945827, 323.50541665740],
        [35.718, 2.32581247002, 728.76296653100],
        [25.620, 2.51240623862, 1162.47470440780],
        [29.255, 3.60838327799, 10.29494073850],
        [23.591, 3.00532139306, 956.28915597060],
        [27.814, 3.23992013743, 838.96928775040],
        [25.993, 4.50118298290, 742.99006053260],
        [25.194, 1.21868110687, 1045.15483618760],
        [19.458, 4.29028644674, 532.87235883230],
        [17.660, 0.80953941560, 508.35032409220],
        [15.355, 5.81037986941, 1596.18644228460],
        [17.058, 4.20001977723, 2118.76386037840],
        [17.040, 1.83402146640, 526.50957135690],
        [14.661, 3.99989622586, 117.31986822020],
        [13.639, 1.80336677963, 302.16477565500],
        [13.230, 2.51856643603, 88.86568021700],
        [12.756, 4.36856232414, 1169.58825140860],
        [15.292, 0.68174165476, 942.06206196900],
        [10.986, 4.43586634639, 525.75881183150],
        [13.920, 5.95169568482, 316.39186965660],
        [9.437, 2.17684563456, 1155.36115740700],
        [8.812, 3.29452783338, 220.41264243880],
        [7.823, 5.75672228354, 846.08283475120],
        [7.549, 2.70955516779, 533.62311835770],
        [9.681, 1.71563161051, 1581.95934828300],
        [8.690, 3.31924493607, 831.85574074960],
        [6.285, 0.49939863541, 949.17560896980],
        [6.685, 2.17560093281, 1265.56747862640],
        [5.381, 6.00510875948, 405.25754987360],
        [4.676, 1.40846192799, 1258.45393162560],
        [4.421, 3.02360159274, 1692.16566950240],
        [4.403, 5.47737266160, 433.71173787680],
        [4.286, 5.07139951645, 1073.60902419080],
        [4.201, 5.28560721767, 18.15924726470],
        [3.933, 1.26665387164, 853.19638175200],
        [5.351, 3.65320121089, 1272.68102562720],
        [4.392, 2.27325303667, 1368.66025284500],
        [3.482, 1.53983001273, 519.39602435610],
        [2.745, 2.09685315627, 1478.86657406440],
        [2.737, 1.06017230524, 1574.84580128220],
        [2.897, 2.05128453665, 1361.54670584420],
        [3.075, 0.99085727534, 191.95845443560],
        [2.462, 2.37173605635, 1471.75302706360],
        [2.203, 2.47960567714, 721.64941953020],
        [2.096, 3.71482580504, 1485.98012106520],
        [1.984, 1.88475229557, 1685.05212250160],
        [2.274, 3.03360234351, 1148.24761040620],
        [2.041, 6.17114556019, 330.61896365820],
        [1.451, 4.72055072637, 32.24332891440],
        [1.454, 5.14703918585, 1375.77379984580],
        [1.447, 3.18833439444, 635.96513305090],
        [1.403, 4.26712075104, 551.03160609700],
        [1.420, 1.99288040133, 629.60234557550],
        [1.269, 0.03300387779, 2125.87740737920],
        [1.276, 2.26356919237, 1788.14489672020],
        [1.189, 1.70223550488, 1677.93857550080],
        [1.182, 2.18142313946, 1795.25844372100],
        [1.366, 1.27629917215, 1038.04128918680],
        [1.306, 4.76302079847, 1062.56332392690],
        [1.109, 2.97787130235, 81.75213321620],
        [1.027, 1.99236027398, 295.05122865420],
        [1.349, 4.01621534182, 539.98590583310],
        [1.025, 3.75336759986, 28.45418800320],
        [0.977, 3.01355125761, 124.43341522100],
        [1.290, 4.62594234857, 2648.45482547300],
        [1.065, 5.06153058155, 1699.27921650320],
        [0.965, 1.17716405513, 99.91138048090],
        [1.021, 1.90712102660, 750.10360753340],
        [0.923, 3.53450109212, 227.52618943960],
        [1.059, 0.13532061468, 416.30325013750],
        [0.836, 2.07492422755, 1056.20053645150],
        [0.889, 1.75177808106, 1898.35121793960],
        [0.772, 2.89217715561, 2008.55753915900],
        [1.014, 2.80847772922, 1464.63948006280],
        [0.820, 1.99735697577, 2111.65031337760],
        [0.787, 4.91912237671, 1055.44977692610],
        [0.743, 2.65209650690, 106.27416795630],
        [0.705, 0.08006443278, 963.40270297140],
        [0.724, 3.29664246938, 628.85158605010],
        [0.791, 1.64655202110, 2001.44399215820],
        [0.822, 2.74067639972, 618.55664531160],
        [0.761, 1.26393500358, 1382.88734684660],
        [0.650, 1.19590511216, 422.66603761290],
        [0.677, 1.88476058357, 2104.53676637680],
        [0.681, 5.47481665606, 5760.49843189760],
        [0.681, 3.11621209674, 5746.27133789600],
        [0.644, 4.68385640894, 611.44309831080],
        [0.752, 3.03497138894, 2221.85663459700],
        [0.641, 1.86274530783, 636.71589257630],
        [0.614, 3.07677356670, 380.12776796000],
        [0.635, 4.53916684689, 9676.48103411560],
        [0.635, 0.61458805483, 9690.70812811720],
        [0.822, 6.25170365084, 423.41679713830],
        [0.762, 4.32362906505, 1802.37199072180],
        [0.582, 0.84137872868, 1891.23767093880],
        [0.558, 3.96171840325, 440.82528487760],
        [0.624, 2.83657771014, 1905.46476494040],
        [0.711, 3.43538032357, 824.74219374880],
        [0.517, 1.10660016329, 107.02492748170],
        [0.535, 1.55761050176, 1994.33044515740],
        [0.501, 4.44389802599, 647.01083331480],
        [0.414, 5.37130370397, 2228.97018159780],
        [0.533, 2.54756313371, 1781.03134971940],
        [0.393, 1.26351262287, 210.11770170030],
        [0.433, 2.90103969634, 1063.31408345230],
        [0.384, 1.36194621083, 203.00415469950],
        [0.440, 1.46934545869, 2214.74308759620],
        [0.424, 4.98974282486, 3178.14579056760],
        [0.338, 2.72210106345, 2324.94940881560],
        [0.332, 0.37505564414, 2655.56837247380],
        [0.318, 6.11024720065, 934.94851496820],
        [0.405, 3.51005860013, 2751.54759969160],
        [0.388, 5.00609647265, 2015.67108615980],
        [0.424, 4.29668654117, 5753.38488489680],
        [0.328, 2.35571531981, 1251.34038462480],
        [0.316, 0.16949503062, 1279.79457262800],
        [0.345, 2.89328206121, 2957.73314812880],
        [0.303, 1.63964826684, 2428.04218303420],
        [0.328, 3.36132375845, 1141.13406340540],
        [0.294, 2.48947693371, 2641.34127847220],
        [0.350, 1.50537240918, 2317.83586181480],
        [0.287, 1.69638214958, 2420.92863603340],
        [0.272, 0.27466529753, 319.57326339430],
        [0.303, 2.43034117616, 70.84944530420],
        [0.251, 0.43544711316, 3259.89792378380],
        [0.224, 4.49752269293, 5223.69391980220],
        [0.272, 2.98590404673, 1457.52593306200],
        [0.228, 5.47896916415, 1603.29998928540],
        [0.288, 2.30146999217, 2854.64037391020],
        [0.207, 5.94297320087, 9153.90361602180],
        [0.243, 1.58604251447, 2744.43405269080],
        [0.228, 1.28182702946, 2310.72231481400],
        [0.224, 1.28623905132, 3060.82592234740],
        [0.222, 0.63265553397, 3163.91869656600],
        [0.242, 2.52382905368, 3274.12501778540],
        [0.188, 6.00513627145, 92.04707395470],
        [0.239, 1.93897157244, 2413.81508903260],
        [0.214, 1.14529237568, 2531.13495725280],
        [0.200, 3.42280996072, 99.16062095550],
        [0.179, 0.53892926207, 2207.62954059540],
        [0.177, 5.56545270243, 2332.06295581640],
        [0.172, 1.38604067808, 945.99421523210],
        [0.203, 0.41899069603, 2840.41327990860],
        [0.231, 2.26353330460, 2097.42321937600],
        [0.228, 3.82701076821, 113.38771495710],
        [0.165, 4.08776703733, 6283.07584999140],
        [0.202, 3.30429764992, 3067.93946934820],
        [0.224, 3.69285208525, 2435.15573003500],
        [0.214, 2.55756944911, 2538.24850425360],
        [0.203, 2.24205059922, 67.66805156650],
        [0.152, 5.48122906518, 10213.28554621100],
        [0.191, 2.68685722531, 1773.91780271860],
        [0.189, 2.95184620359, 732.69511979410],
        [0.149, 1.98737542735, 1049.08698945070],
        [0.163, 1.24084734609, 3053.71237534660],
        [0.171, 2.34210749987, 1354.43315884340],
        [0.112, 5.77407285790, 547.85021235930],
        [0.124, 0.14001204498, 860.30992875280],
        [0.086, 1.26924601636, 511.53171782990],
        [0.114, 5.15982838070, 1592.25428902150],
        [0.091, 1.48896790758, 1567.73225428140],
        [0.086, 4.34444949905, 1069.67687092770],
    ],
    # L3
    [
        [6501.665, 2.59862880482, 7.11354700080],
        [1356.524, 1.34635886411, 529.69096509460],
        [470.716, 2.47503977883, 14.22709400160],
        [416.960, 3.24451243214, 536.80451209540],
        [352.851, 2.97360159003, 522.57741809380],
        [154.880, 2.07565585817, 1059.38193018920],
        [86.771, 2.51431584316, 515.46387109300],
        [33.538, 3.82633794497, 1066.49547719000],
        [44.378, 0.00000000000, 0.00000000000],
        [22.644, 2.98231326774, 543.91805909620],
        [23.737, 1.27667172313, 412.37109687440],
        [28.457, 2.44754756058, 206.18554843720],
        [19.798, 2.10099934005, 639.89728631400],
        [19.740, 1.40255938973, 419.48464387520],
        [18.768, 1.59368403500, 103.09277421860],
        [17.033, 2.30214681202, 21.34064100240],
        [16.774, 2.59821460673, 1589.07289528380],
        [16.214, 3.14521117299, 625.67019231240],
        [16.055, 3.36030126297, 1052.26838318840],
        [13.392, 2.75973892202, 95.97922721780],
        [13.234, 2.53862244340, 199.07200143640],
        [12.611, 6.26578110400, 426.59819087600],
        [8.637, 2.26563256289, 110.20632121940],
        [6.725, 3.42566433316, 309.27832265580],
        [8.701, 1.76334960737, 10.29494073850],
        [6.527, 4.03869562907, 728.76296653100],
        [5.368, 5.25196153539, 323.50541665740],
        [5.675, 2.52096417685, 508.35032409220],
        [5.399, 2.91184687105, 1045.15483618760],
        [3.996, 4.30290261177, 88.86568021700],
        [3.857, 3.52381361552, 302.16477565500],
        [3.774, 4.09125315146, 735.87651353180],
        [3.269, 1.43175991274, 956.28915597060],
        [2.783, 4.35817507670, 1596.18644228460],
        [2.661, 1.25276590759, 213.29909543800],
        [2.553, 2.23785673285, 117.31986822020],
        [2.371, 2.89662409244, 742.99006053260],
        [2.656, 5.01505839848, 838.96928775040],
        [1.948, 2.77248294666, 1169.58825140860],
        [2.279, 2.35581871230, 942.06206196900],
        [1.474, 1.61011468581, 220.41264243880],
        [1.457, 3.09381959396, 2118.76386037840],
        [1.937, 5.01388256693, 831.85574074960],
        [1.585, 1.40097680805, 405.25754987360],
        [1.257, 3.97811260358, 1155.36115740700],
        [1.227, 3.45959919972, 1073.60902419080],
        [0.986, 3.39209446167, 532.87235883230],
        [0.942, 2.70200385825, 191.95845443560],
        [0.828, 1.48348768286, 632.78373931320],
        [0.797, 1.10706688850, 1162.47470440780],
        [0.822, 3.30295824153, 1258.45393162560],
        [0.710, 5.89798771980, 853.19638175200],
        [0.766, 3.66351539483, 1581.95934828300],
        [0.722, 3.74673245797, 433.71173787680],
        [0.663, 2.93063953915, 1574.84580128220],
        [0.658, 3.52797311863, 525.75881183150],
        [0.609, 4.14881313523, 721.64941953020],
        [0.598, 4.69454609357, 81.75213321620],
        [0.668, 1.96442971289, 1272.68102562720],
        [0.515, 1.57251270902, 949.17560896980],
        [0.658, 2.02329201466, 526.50957135690],
        [0.517, 4.35827478516, 1368.66025284500],
        [0.510, 4.95846155301, 1148.24761040620],
        [0.507, 4.31396370095, 330.61896365820],
        [0.567, 2.27813343743, 551.03160609700],
        [0.480, 3.86758235988, 1361.54670584420],
        [0.383, 0.24287136454, 611.44309831080],
        [0.434, 2.95461755540, 1038.04128918680],
        [0.377, 1.42957648215, 124.43341522100],
        [0.391, 4.07770324592, 1471.75302706360],
        [0.385, 4.70295179800, 519.39602435610],
        [0.428, 2.22472522305, 539.98590583310],
        [0.343, 4.83463725823, 2125.87740737920],
        [0.394, 4.52891996323, 1464.63948006280],
        [0.305, 2.02797683648, 1485.98012106520],
        [0.283, 0.97461612169, 1905.46476494040],
        [0.276, 3.83552772064, 1062.56332392690],
        [0.351, 2.06334334462, 533.62311835770],
        [0.304, 3.93228052293, 1685.05212250160],
        [0.322, 3.54763044791, 846.08283475120],
        [0.345, 4.18332148409, 1788.14489672020],
        [0.253, 3.12703531516, 1994.33044515740],
        [0.257, 1.05361498985, 1478.86657406440],
        [0.232, 1.69999081817, 1692.16566950240],
        [0.225, 2.51624149780, 1891.23767093880],
        [0.217, 4.58512911216, 963.40270297140],
        [0.277, 3.63353707701, 1677.93857550080],
        [0.242, 2.90163762388, 2310.72231481400],
        [0.211, 3.96419403991, 295.05122865420],
        [0.199, 5.17046500750, 618.55664531160],
        [0.256, 4.19052619061, 1781.03134971940],
        [0.192, 0.81556540966, 2221.85663459700],
        [0.187, 3.49895198981, 2648.45482547300],
        [0.208, 4.11838429822, 2097.42321937600],
        [0.183, 3.30680692414, 1699.27921650320],
        [0.231, 2.54516792766, 1375.77379984580],
        [0.189, 5.74277274755, 2627.11418447060],
        [0.214, 5.48031974537, 1354.43315884340],
        [0.220, 3.87471989410, 2104.53676637680],
        [0.171, 6.10827209399, 1382.88734684660],
        [0.184, 5.98415847544, 750.10360753340],
        [0.171, 5.25744961028, 824.74219374880],
        [0.151, 4.30799091626, 2001.44399215820],
        [0.140, 4.27089466070, 1265.56747862640],
        [0.097, 4.67188056608, 647.01083331480],
        [0.088, 2.43775210355, 440.82528487760],
        [0.075, 3.93105183253, 1055.44977692610],
        [0.079, 1.88533153220, 934.94851496820],
        [0.077, 3.80503143236, 1603.29998928540],
    ],
    # L4
    [
        [669.483, 0.85282421090, 7.11354700080],
        [99.961, 0.74258947751, 14.22709400160],
        [114.019, 3.14159265359, 0.00000000000],
        [50.024, 1.65346208248, 536.80451209540],
        [43.585, 5.82026386621, 529.69096509460],
        [31.813, 4.85829986650, 522.57741809380],
        [14.742, 4.29061635784, 515.46387109300],
        [8.899, 0.71478520741, 1059.38193018920],
        [4.957, 1.29502259434, 543.91805909620],
        [4.484, 2.31715516627, 1066.49547719000],
        [4.251, 0.48326797501, 21.34064100240],
        [3.100, 3.00245542678, 412.37109687440],
        [2.055, 0.39858940218, 639.89728631400],
        [1.762, 4.90536207307, 625.67019231240],
        [1.902, 4.25925620271, 199.07200143640],
        [1.695, 4.26147580803, 206.18554843720],
        [1.375, 5.25546955667, 1052.26838318840],
        [1.203, 4.71614633845, 95.97922721780],
        [1.086, 1.28604571172, 1589.07289528380],
        [0.982, 4.77990073662, 1045.15483618760],
        [0.935, 6.05847062188, 88.86568021700],
        [0.916, 5.77537499431, 728.76296653100],
        [0.890, 4.55299189579, 426.59819087600],
        [0.784, 3.40161567950, 419.48464387520],
        [0.768, 3.54672049322, 103.09277421860],
        [0.670, 0.52223307700, 110.20632121940],
        [0.415, 5.22809480633, 302.16477565500],
        [0.393, 6.24184621807, 956.28915597060],
        [0.381, 5.25466966040, 309.27832265580],
        [0.421, 0.59561318533, 117.31986822020],
        [0.346, 4.78348312106, 508.35032409220],
        [0.319, 3.47979828725, 323.50541665740],
        [0.331, 2.95893485883, 1596.18644228460],
        [0.295, 4.32713459459, 942.06206196900],
        [0.319, 0.47990052824, 831.85574074960],
        [0.251, 1.79898001222, 1073.60902419080],
        [0.212, 0.43917684084, 220.41264243880],
        [0.188, 1.12654974776, 1169.58825140860],
        [0.188, 2.16135407548, 1361.54670584420],
        [0.180, 3.43266428069, 1148.24761040620],
        [0.164, 1.92864127211, 2118.76386037840],
        [0.157, 3.02963907392, 1272.68102562720],
        [0.093, 5.60436000012, 1581.95934828300],
        [0.085, 5.02317256200, 1155.36115740700],
        [0.075, 3.13198879608, 632.78373931320],
    ],
    # L5
    [
        [49.577, 5.25658966184, 7.11354700080],
        [15.761, 5.25126837478, 14.22709400160],
        [4.343, 0.01461869263, 536.80451209540],
        [1.526, 1.09739911439, 522.57741809380],
        [0.728, 5.85949047619, 543.91805909620],
        [0.694, 0.87382487754, 515.46387109300],
        [0.845, 3.14159265359, 0.00000000000],
        [0.456, 0.81521692852, 1066.49547719000],
        [0.293, 5.62909357048, 1059.38193018920],
        [0.090, 0.21178119710, 529.69096509460],
    ],
]
"""This table contains Jupiter's periodic terms (all of them) from the
planetary theory VSOP87 for the heliocentric longitude at the equinox of date
(taken from the 'D' solution). In Meeus' book a shortened version can be found
in pages 426-430."""


VSOP87_B = [
    # B0
    [
        [2268615.703, 3.55852606718, 529.69096509460],
        [109971.634, 3.90809347389, 1059.38193018920],
        [110090.358, 0.00000000000, 0.00000000000],
        [8101.427, 3.60509573368, 522.57741809380],
        [6043.996, 4.25883108794, 1589.07289528380],
        [6437.782, 0.30627121409, 536.80451209540],
        [1106.880, 2.98534421928, 1162.47470440780],
        [941.651, 2.93619072405, 1052.26838318840],
        [894.088, 1.75447429921, 7.11354700080],
        [767.280, 2.15473594060, 632.78373931320],
        [944.328, 1.67522288396, 426.59819087600],
        [684.220, 3.67808770098, 213.29909543800],
        [629.223, 0.64343282328, 1066.49547719000],
        [835.861, 5.17881973234, 103.09277421860],
        [531.670, 2.70305954352, 110.20632121940],
        [558.524, 0.01354830508, 846.08283475120],
        [464.449, 1.17337249185, 949.17560896980],
        [431.072, 2.60825000494, 419.48464387520],
        [351.433, 4.61062990714, 2118.76386037840],
        [123.148, 3.34968181384, 1692.16566950240],
        [115.038, 5.04892295442, 316.39186965660],
        [132.160, 4.77816990670, 742.99006053260],
        [103.402, 2.31878999565, 1478.86657406440],
        [116.379, 1.38688232033, 323.50541665740],
        [102.420, 3.15293785436, 1581.95934828300],
        [103.762, 3.70103838110, 515.46387109300],
        [78.650, 3.98318653238, 1265.56747862640],
        [69.935, 2.56006216424, 956.28915597060],
        [55.597, 0.37500753017, 1375.77379984580],
        [51.986, 0.99007119033, 1596.18644228460],
        [55.194, 0.40176412035, 525.75881183150],
        [63.456, 4.50073574333, 735.87651353180],
        [49.691, 0.18649893085, 543.91805909620],
        [48.831, 3.57260550671, 533.62311835770],
        [28.353, 1.53532744749, 625.67019231240],
        [29.209, 5.43145863011, 206.18554843720],
        [23.255, 5.95197992848, 838.96928775040],
        [22.841, 6.19262787685, 532.87235883230],
        [23.202, 4.06473368575, 526.50957135690],
        [24.436, 6.10947656959, 1169.58825140860],
        [21.116, 4.96322972735, 2648.45482547300],
        [17.879, 3.08704395969, 1795.25844372100],
        [16.234, 4.83515727869, 1368.66025284500],
        [21.314, 2.69476951059, 1045.15483618760],
        [15.740, 1.15130330106, 942.06206196900],
        [17.325, 1.61550009206, 14.22709400160],
        [13.396, 2.30539585502, 853.19638175200],
        [11.904, 3.09811974536, 2111.65031337760],
        [11.734, 2.83006431723, 2008.55753915900],
        [11.291, 0.98957560201, 433.71173787680],
        [11.830, 4.76527836803, 309.27832265580],
        [10.702, 3.70181397065, 2221.85663459700],
        [10.815, 5.81958878617, 1272.68102562720],
        [13.505, 3.28126975760, 1155.36115740700],
        [10.179, 2.58691128827, 117.31986822020],
        [10.632, 5.23487936086, 95.97922721780],
        [8.771, 0.40456546655, 220.41264243880],
        [7.439, 2.94638292086, 412.37109687440],
        [6.151, 2.69100382247, 380.12776796000],
        [5.028, 0.72750312028, 1055.44977692610],
        [4.939, 0.73756716762, 1905.46476494040],
        [5.421, 4.08612438558, 1685.05212250160],
        [5.936, 4.32059910537, 1063.31408345230],
        [4.737, 4.09303016850, 527.24328453980],
        [4.010, 0.51530008355, 1073.60902419080],
        [4.709, 1.84067645204, 984.60033162190],
        [3.974, 1.33608029246, 2125.87740737920],
        [3.762, 3.58647039394, 529.73914920440],
        [4.731, 6.16377350841, 532.13864564940],
        [4.666, 5.88762905802, 639.89728631400],
        [3.763, 0.38865925413, 529.64278098480],
        [3.409, 4.05398247269, 1898.35121793960],
        [3.457, 3.43865563497, 1485.98012106520],
        [4.229, 2.23767157901, 74.78159856730],
        [3.091, 0.16470256025, 1699.27921650320],
        [2.975, 0.72268908074, 530.65417294110],
        [3.162, 1.25048416420, 330.61896365820],
        [2.727, 4.37679213321, 149.56319713460],
        [2.837, 0.05987107395, 1439.50969814920],
        [2.983, 3.25251207220, 528.72775724810],
        [2.232, 0.26149880534, 1062.56332392690],
        [2.464, 1.16913304420, 453.42489381900],
        [2.596, 3.30510149086, 2324.94940881560],
        [1.988, 2.85269577619, 1574.84580128220],
        [2.527, 5.94458202950, 909.81873305460],
        [2.269, 1.30379329597, 3.93215326310],
        [1.742, 4.49909767044, 1258.45393162560],
        [1.714, 4.12945878208, 2001.44399215820],
        [2.029, 3.97938086639, 1056.20053645150],
        [1.667, 0.36037092553, 10213.28554621100],
        [1.579, 6.11640144795, 1802.37199072180],
        [1.393, 3.69324470827, 2214.74308759620],
        [1.604, 1.98841031703, 38.13303563780],
        [1.325, 1.74025919863, 529.16970023280],
        [1.451, 2.39804501178, 2428.04218303420],
        [1.594, 2.07556780757, 1021.24889455140],
        [1.320, 1.33770977126, 618.55664531160],
        [1.346, 3.27591492540, 2641.34127847220],
        [1.230, 0.19552728220, 305.34616939270],
        [1.223, 2.86681556337, 1382.88734684660],
        [1.324, 2.23549334986, 530.21222995640],
        [1.056, 3.80579750957, 76.26607127560],
        [1.050, 4.68011652614, 1788.14489672020],
        [1.226, 5.34003255221, 3178.14579056760],
        [1.009, 3.19608028376, 2538.24850425360],
        [1.266, 3.04704446731, 604.47256366190],
        [0.954, 3.86932544808, 728.76296653100],
        [1.124, 1.59560367480, 3.18139373770],
        [0.978, 0.25223689838, 983.11585891360],
        [0.948, 0.21552742733, 750.10360753340],
        [0.946, 3.93927748120, 508.35032409220],
        [0.920, 1.14672086939, 963.40270297140],
        [0.817, 5.93809619876, 831.85574074960],
        [0.770, 2.96062737592, 526.77020378780],
        [1.017, 5.55711112145, 199.07200143640],
        [0.761, 1.38163787157, 532.61172640140],
        [0.726, 3.98337964395, 2317.83586181480],
        [0.862, 0.87975657414, 490.33408917940],
        [0.868, 3.44331872364, 569.04784100980],
        [0.711, 4.11107052823, 2751.54759969160],
        [0.708, 0.33555577415, 528.94020556920],
        [0.708, 4.00539820601, 530.44172462000],
        [0.656, 4.39568451439, 519.39602435610],
        [0.801, 4.03984430862, 1364.72809958190],
        [0.679, 1.18645749024, 525.49817940060],
        [0.645, 5.10510349996, 1361.54670584420],
        [0.668, 3.15607509055, 533.88375078860],
        [0.663, 0.73722024843, 5223.69391980220],
        [0.663, 1.57092786811, 6283.07584999140],
        [0.543, 0.26376529935, 227.52618943960],
        [0.525, 6.22318693939, 539.98590583310],
        [0.513, 4.98337900151, 302.16477565500],
        [0.544, 2.22227019273, 2744.43405269080],
        [0.532, 2.62425372687, 99.16062095550],
        [0.602, 1.56074089013, 454.90936652730],
        [0.518, 0.26343805959, 551.03160609700],
        [0.516, 1.09376390349, 934.94851496820],
        [0.659, 0.62560671589, 1512.80682400820],
        [0.524, 0.64710955846, 524.06189080210],
        [0.516, 3.69478866795, 535.32003938710],
        [0.491, 3.63039940597, 2531.13495725280],
        [0.570, 0.61976758791, 540.73666535850],
        [0.496, 2.19398015038, 1514.29129671650],
        [0.532, 0.20040217534, 525.02509864860],
        [0.493, 0.39160693598, 224.34479570190],
        [0.449, 0.62392433691, 529.53090640020],
        [0.449, 3.71676131146, 529.85102378900],
        [0.450, 5.02467015031, 1048.33622992530],
        [0.428, 5.44804660290, 11.04570026390],
        [0.499, 4.13924061941, 534.35683154060],
        [0.528, 1.76471074936, 524.27433912320],
        [0.454, 4.53321742354, 1056.93424963440],
        [0.520, 2.57406093768, 535.10759106600],
        [0.398, 1.40345870113, 960.22130923370],
        [0.457, 4.17708652827, 2104.53676637680],
        [0.505, 5.36536256321, 1057.89745748090],
        [0.535, 4.80455380313, 1593.00504854690],
        [0.415, 0.96548127237, 2435.15573003500],
        [0.519, 0.54543519483, 1061.82961074400],
        [0.359, 4.02704454075, 1059.43011429900],
        [0.356, 2.66818105522, 835.03713448730],
        [0.443, 5.27513700376, 1.48447270830],
        [0.358, 5.94423960514, 440.82528487760],
        [0.471, 6.05791940453, 1471.75302706360],
        [0.386, 2.15984900214, 9153.90361602180],
        [0.424, 2.70929670030, 1038.04128918680],
        [0.359, 0.82922836987, 1059.33374607940],
        [0.310, 0.88102053266, 529.90341341570],
        [0.310, 3.45966511571, 529.47851677350],
        [0.300, 3.70331799503, 2634.22773147140],
        [0.292, 2.63594456361, 415.55249061210],
        [0.279, 1.60669121578, 643.82943957710],
        [0.291, 5.83134071820, 1148.24761040620],
        [0.370, 5.71572992274, 531.17543780290],
        [0.268, 5.39275891813, 1891.23767093880],
        [0.275, 3.34108666036, 518.64526483070],
        [0.269, 1.06051406954, 1585.14074202070],
        [0.306, 2.50289017370, 511.53171782990],
        [0.295, 1.84394223501, 547.85021235930],
        [0.254, 2.98312992496, 1134.16352875650],
        [0.289, 1.86070918711, 21.34064100240],
        [0.265, 4.93075479744, 679.25416222920],
        [0.250, 0.42860925124, 1969.20066324380],
        [0.308, 2.67237933272, 2957.73314812880],
        [0.313, 4.88085697819, 528.20649238630],
        [0.222, 4.78828764413, 514.71311156760],
        [0.221, 4.32763468981, 1677.93857550080],
        [0.217, 3.46278526461, 2950.61960112800],
        [0.216, 0.52207667980, 2228.97018159780],
        [0.214, 5.83569926578, 544.66881862160],
        [0.283, 2.88709716090, 35.42472265210],
        [0.272, 1.65708415457, 3060.82592234740],
        [0.234, 1.68821537711, 2655.56837247380],
        [0.205, 3.36186888290, 2847.52682690940],
        [0.264, 3.62722625694, 2420.92863603340],
        [0.191, 4.26821147044, 430.53034413910],
        [0.179, 3.91470663005, 3340.61242669980],
        [0.180, 0.04531671003, 387.24131496080],
        [0.241, 4.03927631611, 494.26624244250],
        [0.176, 4.26298906325, 672.14061522840],
        [0.187, 2.72587420586, 299.12639426920],
        [0.234, 1.34474827450, 173.94221952280],
        [0.171, 0.85473611718, 1603.29998928540],
        [0.224, 0.33130232434, 565.11568774670],
        [0.200, 1.27632489123, 39.35687591520],
        [0.170, 4.96479470273, 1464.63948006280],
        [0.211, 1.00937080256, 523.54062594030],
        [0.210, 3.75793720248, 2854.64037391020],
        [0.162, 5.87784787295, 3480.31056622260],
        [0.163, 4.62850343495, 2015.67108615980],
        [0.191, 3.33159283750, 535.84130424890],
        [0.151, 1.17096741034, 1060.34513803570],
        [0.160, 1.81852636004, 312.45971639350],
        [0.158, 2.59595816107, 529.43033266370],
        [0.158, 1.74472748730, 529.95159752550],
        [0.173, 3.62399350412, 230.56457082540],
        [0.142, 0.70435921398, 522.52923398400],
        [0.144, 5.35763122430, 107.02492748170],
        [0.144, 6.13954848857, 1158.54255114470],
        [0.178, 0.27566275049, 3906.90875709860],
        [0.126, 5.14832919826, 2207.62954059540],
        [0.126, 3.41994798109, 2.44768055480],
        [0.127, 0.39825164051, 70.84944530420],
        [0.123, 4.77865550523, 2524.02141025200],
        [0.123, 0.46184813516, 647.01083331480],
        [0.144, 3.60261852727, 1058.41872234270],
        [0.158, 3.76231915252, 92.04707395470],
        [0.119, 4.08266911415, 1585.89150154610],
        [0.125, 2.35496721797, 3163.91869656600],
        [0.122, 3.21027426317, 3377.21779200400],
        [0.121, 3.39770381916, 18.15924726470],
        [0.131, 1.67926417552, 1289.94650101460],
        [0.115, 2.35735471566, 1550.93985964600],
        [0.126, 2.40833814513, 106.27416795630],
        [0.131, 1.37610474529, 1023.95720753710],
        [0.121, 1.60252617273, 10.29494073850],
        [0.121, 0.61420823557, 1592.25428902150],
        [0.135, 3.60177675518, 124.43341522100],
        [0.137, 2.41724947062, 3274.12501778540],
        [0.129, 0.09702914345, 2332.06295581640],
        [0.093, 4.88949890397, 1098.73880610440],
        [0.106, 5.18592950792, 2281.23049651060],
        [0.114, 2.96523316419, 1166.40685767090],
        [0.092, 1.65166124027, 860.30992875280],
        [0.102, 3.64093193142, 3171.03224356680],
        [0.103, 1.63066232967, 1894.41906467650],
        [0.080, 0.38766601876, 4694.00295470760],
        [0.074, 3.86865238736, 3067.93946934820],
        [0.095, 1.66362447044, 1151.42900414390],
    ],
    # B1
    [
        [177351.787, 5.70166488486, 529.69096509460],
        [3230.171, 5.77941619340, 1059.38193018920],
        [3081.364, 5.47464296527, 522.57741809380],
        [2211.914, 4.73477480209, 536.80451209540],
        [1694.232, 3.14159265359, 0.00000000000],
        [346.445, 4.74595174109, 1052.26838318840],
        [234.264, 5.18856099929, 1066.49547719000],
        [196.154, 6.18554286642, 7.11354700080],
        [150.468, 3.92721226087, 1589.07289528380],
        [114.128, 3.43897271830, 632.78373931320],
        [96.667, 2.91426304090, 949.17560896980],
        [76.599, 2.50522188662, 103.09277421860],
        [81.671, 5.07666097497, 1162.47470440780],
        [76.572, 0.61288981445, 419.48464387520],
        [73.875, 5.49958292155, 515.46387109300],
        [49.915, 3.94799616572, 735.87651353180],
        [60.544, 5.44740084359, 213.29909543800],
        [36.561, 4.69828392839, 543.91805909620],
        [46.032, 0.53850360901, 110.20632121940],
        [45.123, 1.89516645239, 846.08283475120],
        [36.019, 6.10952578764, 316.39186965660],
        [31.975, 4.92452714629, 1581.95934828300],
        [21.015, 5.62957731410, 1596.18644228460],
        [23.156, 5.84829490183, 323.50541665740],
        [24.719, 3.94107395247, 2118.76386037840],
        [17.274, 5.65310656429, 533.62311835770],
        [16.521, 5.89840100621, 526.50957135690],
        [16.698, 5.66663034948, 1265.56747862640],
        [15.815, 4.43314786393, 1045.15483618760],
        [13.398, 4.30179033605, 532.87235883230],
        [11.744, 1.80990486955, 956.28915597060],
        [11.925, 4.30094564154, 525.75881183150],
        [9.514, 2.02589667166, 206.18554843720],
        [10.542, 6.15533910933, 14.22709400160],
        [8.414, 3.92910450340, 1478.86657406440],
        [8.099, 4.20152809071, 1169.58825140860],
        [7.712, 2.99160389601, 942.06206196900],
        [8.825, 1.55897920307, 426.59819087600],
        [8.884, 4.87430124264, 1155.36115740700],
        [7.793, 3.84684930196, 625.67019231240],
        [5.646, 3.40915964493, 639.89728631400],
        [4.615, 0.83374662294, 117.31986822020],
        [4.020, 5.50502127885, 433.71173787680],
        [3.704, 0.90226777963, 95.97922721780],
        [3.859, 0.69640284662, 853.19638175200],
        [3.091, 5.09115860882, 1073.60902419080],
        [3.360, 5.10133284081, 1692.16566950240],
        [2.892, 4.90418916660, 220.41264243880],
        [2.772, 5.09066125724, 2111.65031337760],
        [2.425, 3.74438653232, 742.99006053260],
        [2.558, 5.46955948791, 1795.25844372100],
        [2.466, 4.22278355430, 2648.45482547300],
        [1.968, 0.57192251841, 309.27832265580],
        [1.794, 4.60765219417, 1272.68102562720],
        [1.822, 1.98842964323, 1375.77379984580],
        [1.703, 6.12660562937, 2125.87740737920],
        [2.011, 5.00936865256, 412.37109687440],
        [1.645, 0.08830372958, 1063.31408345230],
        [1.875, 5.81006158403, 330.61896365820],
        [1.741, 4.58650290431, 1574.84580128220],
        [1.529, 5.81660291389, 1258.45393162560],
        [1.916, 0.85150399517, 1368.66025284500],
        [1.614, 4.36839107221, 728.76296653100],
        [1.510, 2.79374165455, 1485.98012106520],
        [1.333, 4.84260898693, 1062.56332392690],
        [1.359, 5.16511980864, 838.96928775040],
        [1.165, 5.66275740881, 508.35032409220],
        [1.092, 4.68797557406, 1699.27921650320],
        [1.438, 5.78105679279, 1056.20053645150],
        [1.083, 3.99886917926, 1471.75302706360],
        [1.002, 4.79949608524, 1055.44977692610],
        [0.749, 6.14400862030, 519.39602435610],
        [0.657, 5.63765568876, 1898.35121793960],
        [0.702, 5.04126574492, 1685.05212250160],
        [0.607, 3.15707515246, 618.55664531160],
        [0.587, 1.37658820775, 199.07200143640],
        [0.552, 4.80657729450, 551.03160609700],
        [0.494, 4.43417307482, 539.98590583310],
        [0.517, 0.05161181997, 3.18139373770],
        [0.469, 3.81715950042, 2008.55753915900],
        [0.415, 1.34693184108, 1382.88734684660],
        [0.382, 4.86764073919, 227.52618943960],
        [0.473, 1.72405831407, 532.13864564940],
        [0.458, 4.44604993015, 1038.04128918680],
        [0.376, 2.23190744786, 529.64278098480],
        [0.451, 3.75869883836, 984.60033162190],
        [0.376, 5.42971857629, 529.73914920440],
        [0.389, 1.92698506631, 525.02509864860],
        [0.364, 3.35456685746, 2221.85663459700],
        [0.476, 5.93625415892, 527.24328453980],
        [0.383, 6.12255867339, 149.56319713460],
        [0.301, 4.09378934049, 440.82528487760],
        [0.310, 5.58150418981, 2428.04218303420],
        [0.282, 4.85996662231, 1788.14489672020],
        [0.298, 5.09589374634, 528.72775724810],
        [0.340, 4.56537070220, 750.10360753340],
        [0.272, 2.35346960340, 534.35683154060],
        [0.360, 3.91050161665, 74.78159856730],
        [0.299, 1.43093538841, 909.81873305460],
        [0.297, 2.56584512211, 530.65417294110],
        [0.235, 4.81644489422, 535.10759106600],
        [0.306, 0.68420442848, 380.12776796000],
        [0.236, 4.63162956792, 526.77020378780],
        [0.270, 0.18549916939, 21.34064100240],
        [0.288, 4.26655874393, 1802.37199072180],
        [0.196, 5.35950443033, 2214.74308759620],
        [0.190, 4.54615193260, 2104.53676637680],
        [0.193, 4.35426216497, 511.53171782990],
        [0.178, 4.51895208036, 3178.14579056760],
        [0.194, 0.57050756837, 1361.54670584420],
        [0.200, 1.48040474749, 302.16477565500],
        [0.168, 5.40141749419, 524.27433912320],
        [0.152, 0.68077486546, 1905.46476494040],
        [0.149, 1.06678990744, 831.85574074960],
        [0.182, 3.62401009613, 38.13303563780],
        [0.176, 5.64331384323, 963.40270297140],
        [0.184, 4.48850356629, 604.47256366190],
        [0.133, 5.45026366125, 2641.34127847220],
        [0.143, 2.21577268292, 1439.50969814920],
        [0.130, 4.88155705493, 2531.13495725280],
        [0.129, 6.15206333598, 547.85021235930],
        [0.133, 5.43193972385, 1603.29998928540],
        [0.133, 3.49297492409, 529.16970023280],
        [0.132, 3.98820790955, 530.21222995640],
        [0.118, 5.38352943814, 1891.23767093880],
        [0.133, 5.65694269884, 76.26607127560],
        [0.145, 2.94976686191, 454.90936652730],
        [0.115, 3.29206553804, 3.93215326310],
        [0.102, 4.48856749557, 2001.44399215820],
        [0.106, 6.08434275898, 10.29494073850],
        [0.093, 5.84737771840, 2324.94940881560],
        [0.101, 0.15815934254, 2655.56837247380],
        [0.115, 3.59221021604, 2015.67108615980],
        [0.103, 4.70399583323, 305.34616939270],
        [0.084, 0.44180206332, 1593.00504854690],
        [0.092, 2.44863388631, 490.33408917940],
        [0.087, 6.23817512863, 6283.07584999140],
        [0.095, 3.30154605532, 2317.83586181480],
        [0.072, 1.90578907085, 528.94020556920],
        [0.072, 5.57619428876, 530.44172462000],
        [0.078, 5.97323507836, 1585.89150154610],
    ],
    # B2
    [
        [8094.051, 1.46322843658, 529.69096509460],
        [742.415, 0.95691639003, 522.57741809380],
        [813.244, 3.14159265359, 0.00000000000],
        [398.951, 2.89888666447, 536.80451209540],
        [342.226, 1.44683789727, 1059.38193018920],
        [73.948, 0.40724675866, 1052.26838318840],
        [46.151, 3.48036895772, 1066.49547719000],
        [29.314, 0.99088831805, 515.46387109300],
        [29.717, 1.92504171329, 1589.07289528380],
        [22.753, 4.27124052435, 7.11354700080],
        [13.916, 2.92242387338, 543.91805909620],
        [12.067, 5.22168932482, 632.78373931320],
        [10.703, 4.88024222475, 949.17560896980],
        [6.078, 6.21089108431, 1045.15483618760],
        [5.935, 0.52977760072, 1581.95934828300],
        [5.037, 1.43444929374, 526.50957135690],
        [4.564, 0.91811732585, 1162.47470440780],
        [4.547, 4.01953745202, 1596.18644228460],
        [5.098, 6.03169795231, 735.87651353180],
        [3.593, 4.54080164408, 110.20632121940],
        [3.443, 1.38618954572, 533.62311835770],
        [3.277, 4.39650286553, 14.22709400160],
        [3.407, 0.42275631534, 419.48464387520],
        [2.904, 2.06041641723, 316.39186965660],
        [2.541, 3.98323842017, 323.50541665740],
        [3.113, 2.48079280193, 2118.76386037840],
        [3.061, 2.39880866911, 532.87235883230],
        [2.155, 4.77990637140, 942.06206196900],
        [2.143, 3.88727338786, 426.59819087600],
        [2.252, 0.37196434120, 1155.36115740700],
        [2.019, 3.89985000464, 846.08283475120],
        [1.857, 1.19658907851, 103.09277421860],
        [1.683, 1.42264195434, 1265.56747862640],
        [2.313, 0.87671613055, 213.29909543800],
        [1.443, 2.38565505909, 1169.58825140860],
        [1.823, 5.80106463776, 625.67019231240],
        [1.728, 2.24114678267, 525.75881183150],
        [1.198, 0.03252059731, 956.28915597060],
        [1.138, 3.46420904745, 1073.60902419080],
        [1.086, 5.35279146700, 117.31986822020],
        [0.840, 2.89946334223, 95.97922721780],
        [0.746, 5.53017890231, 1478.86657406440],
        [0.944, 4.05587053500, 206.18554843720],
        [0.758, 3.74770617289, 433.71173787680],
        [0.673, 1.26396626349, 508.35032409220],
        [0.889, 6.07878453176, 728.76296653100],
        [0.600, 1.82954494089, 639.89728631400],
        [0.589, 1.23625943417, 1258.45393162560],
        [0.619, 0.67923057477, 838.96928775040],
        [0.566, 5.36336098734, 742.99006053260],
        [0.648, 5.32990375008, 853.19638175200],
        [0.553, 3.15511946637, 220.41264243880],
        [0.432, 1.03719283016, 1692.16566950240],
        [0.435, 1.65056479007, 519.39602435610],
        [0.430, 1.41830384501, 412.37109687440],
        [0.431, 2.20986254651, 1368.66025284500],
        [0.415, 4.35372561905, 330.61896365820],
        [0.438, 0.16552277290, 1574.84580128220],
        [0.312, 4.50639455819, 2125.87740737920],
        [0.280, 3.01441283033, 551.03160609700],
        [0.309, 0.67399908949, 2111.65031337760],
        [0.301, 3.06868080871, 1062.56332392690],
        [0.236, 1.94696842200, 1485.98012106520],
        [0.235, 3.41850395941, 199.07200143640],
        [0.246, 2.61803442505, 309.27832265580],
        [0.238, 2.56643737684, 539.98590583310],
        [0.248, 2.96997778167, 2648.45482547300],
        [0.209, 5.82481690851, 1471.75302706360],
        [0.205, 1.20202002469, 1056.20053645150],
        [0.188, 0.97113663101, 1685.05212250160],
        [0.137, 2.91203499563, 1699.27921650320],
        [0.131, 1.79274504072, 1063.31408345230],
        [0.161, 1.05926568614, 1795.25844372100],
        [0.112, 2.62660288825, 440.82528487760],
        [0.110, 3.56263668146, 227.52618943960],
        [0.114, 6.13907482464, 1038.04128918680],
        [0.103, 4.64287101040, 3.18139373770],
        [0.123, 4.81268110532, 21.34064100240],
        [0.102, 4.27603827970, 1375.77379984580],
        [0.089, 1.22926014128, 1898.35121793960],
        [0.080, 0.62129648755, 831.85574074960],
    ],
    # B3
    [
        [251.624, 3.38087923084, 529.69096509460],
        [121.738, 2.73311837200, 522.57741809380],
        [48.694, 1.03689996685, 536.80451209540],
        [10.988, 2.31463561347, 1052.26838318840],
        [8.067, 2.76729757621, 515.46387109300],
        [6.205, 1.78115827370, 1066.49547719000],
        [7.287, 4.25268318975, 1059.38193018920],
        [3.627, 1.13028917221, 543.91805909620],
        [2.798, 3.14159265359, 0.00000000000],
        [1.898, 2.28934054087, 7.11354700080],
        [1.643, 1.77507208483, 1045.15483618760],
        [0.945, 0.45261136388, 632.78373931320],
        [0.758, 0.30577920142, 949.17560896980],
        [0.731, 2.63748223583, 14.22709400160],
        [0.876, 0.32927768725, 1589.07289528380],
        [0.678, 2.36909615348, 1581.95934828300],
        [0.623, 2.48056213600, 1596.18644228460],
        [0.736, 1.52532370632, 735.87651353180],
        [0.499, 3.67985494258, 419.48464387520],
        [0.454, 0.26977404624, 942.06206196900],
        [0.453, 3.18232334886, 526.50957135690],
        [0.409, 2.88147337106, 110.20632121940],
        [0.347, 5.76244285870, 103.09277421860],
        [0.310, 2.98017326384, 508.35032409220],
        [0.321, 4.40642025933, 532.87235883230],
        [0.300, 1.66936571536, 625.67019231240],
        [0.295, 1.75924202728, 1073.60902419080],
        [0.282, 3.11087801399, 533.62311835770],
        [0.263, 0.55255030187, 426.59819087600],
        [0.208, 2.17540496886, 1155.36115740700],
        [0.183, 4.34670868038, 525.75881183150],
        [0.180, 6.07777744541, 639.89728631400],
        [0.159, 2.60843864402, 1162.47470440780],
        [0.117, 4.70141431381, 95.97922721780],
        [0.107, 5.48942805114, 433.71173787680],
        [0.105, 3.75192101775, 316.39186965660],
        [0.130, 1.37897716939, 323.50541665740],
        [0.094, 3.05797832024, 1265.56747862640],
        [0.114, 3.75170981478, 117.31986822020],
        [0.095, 0.54905691533, 1169.58825140860],
        [0.088, 3.26874502411, 213.29909543800],
        [0.098, 2.00704668688, 1574.84580128220],
    ],
    # B4
    [
        [15.050, 4.52956999637, 522.57741809380],
        [5.370, 4.47427159142, 529.69096509460],
        [4.456, 5.43908581047, 536.80451209540],
        [3.422, 0.00000000000, 0.00000000000],
        [1.833, 4.51807036227, 515.46387109300],
        [1.322, 4.20117611581, 1052.26838318840],
        [0.755, 5.59451554966, 543.91805909620],
        [0.512, 0.05803177475, 1066.49547719000],
        [0.282, 3.66807771223, 1059.38193018920],
        [0.147, 3.56490986181, 1045.15483618760],
        [0.142, 5.69936472988, 7.11354700080],
        [0.112, 1.16718383135, 14.22709400160],
    ],
    # B5
    [
        [1.445, 0.09198554072, 522.57741809380],
        [0.368, 0.00874408003, 515.46387109300],
        [0.304, 3.27902945138, 536.80451209540],
        [0.129, 0.33959775247, 529.69096509460],
        [0.095, 1.29305954542, 543.91805909620],
    ],
]
"""This table contains Jupiter's periodic terms (all of them) from the
planetary theory VSOP87 for the heliocentric latitude at the equinox of date
(taken from the 'D' solution). In Meeus' book a shortened version can be found
in pages 430-432."""


VSOP87_R = [
    # R0
    [
        [520887429.471, 0.00000000000, 0.00000000000],
        [25209327.020, 3.49108640015, 529.69096509460],
        [610599.902, 3.84115365602, 1059.38193018920],
        [282029.465, 2.57419879933, 632.78373931320],
        [187647.391, 2.07590380082, 522.57741809380],
        [86792.941, 0.71001090609, 419.48464387520],
        [72062.869, 0.21465694745, 536.80451209540],
        [65517.227, 5.97995850843, 316.39186965660],
        [29134.620, 1.67759243710, 103.09277421860],
        [30135.275, 2.16132058449, 949.17560896980],
        [23453.209, 3.54023147303, 735.87651353180],
        [22283.710, 4.19362773546, 1589.07289528380],
        [23947.340, 0.27457854894, 7.11354700080],
        [13032.600, 2.96043055741, 1162.47470440780],
        [9703.346, 1.90669572402, 206.18554843720],
        [12749.004, 2.71550102862, 1052.26838318840],
        [9161.431, 4.41352618935, 213.29909543800],
        [7894.539, 2.47907551404, 426.59819087600],
        [7057.978, 2.18184753111, 1265.56747862640],
        [6137.755, 6.26417542514, 846.08283475120],
        [5477.093, 5.65729325169, 639.89728631400],
        [3502.519, 0.56531297394, 1066.49547719000],
        [4136.890, 2.72219979684, 625.67019231240],
        [4170.012, 2.01605033912, 515.46387109300],
        [2499.966, 4.55182055941, 838.96928775040],
        [2616.955, 2.00993967129, 1581.95934828300],
        [1911.876, 0.85621927419, 412.37109687440],
        [2127.644, 6.12751461750, 742.99006053260],
        [1610.549, 3.08867789275, 1368.66025284500],
        [1479.484, 2.68026191372, 1478.86657406440],
        [1230.708, 1.89042979701, 323.50541665740],
        [1216.810, 1.80171561024, 110.20632121940],
        [961.072, 4.54876989805, 2118.76386037840],
        [885.708, 4.14785948471, 533.62311835770],
        [776.700, 3.67696954690, 728.76296653100],
        [998.579, 2.87208940110, 309.27832265580],
        [1014.959, 1.38673237666, 454.90936652730],
        [727.162, 3.98824686402, 1155.36115740700],
        [655.289, 2.79065604219, 1685.05212250160],
        [821.465, 1.59342534396, 1898.35121793960],
        [620.798, 4.82284338962, 956.28915597060],
        [653.981, 3.38150775269, 1692.16566950240],
        [812.036, 5.94091899141, 909.81873305460],
        [562.120, 0.08095987241, 543.91805909620],
        [542.221, 0.28360266386, 525.75881183150],
        [457.859, 0.12722694510, 1375.77379984580],
        [614.784, 2.27624915604, 942.06206196900],
        [435.805, 2.60272129748, 95.97922721780],
        [496.066, 5.53005947761, 380.12776796000],
        [469.965, 2.81896276101, 1795.25844372100],
        [445.003, 0.14623567024, 14.22709400160],
        [290.869, 3.89339143564, 1471.75302706360],
        [276.627, 2.52238450687, 2001.44399215820],
        [275.084, 2.98863518924, 526.50957135690],
        [293.875, 2.04938438861, 199.07200143640],
        [290.985, 6.03131226226, 1169.58825140860],
        [338.342, 2.79873192583, 1045.15483618760],
        [257.482, 6.13395478303, 532.87235883230],
        [319.013, 1.34803130803, 2214.74308759620],
        [309.352, 5.36855804945, 1272.68102562720],
        [345.804, 1.56404293688, 491.55792945680],
        [303.364, 1.15407454372, 5753.38488489680],
        [192.325, 0.91996333387, 1596.18644228460],
        [215.398, 2.63572815848, 2111.65031337760],
        [200.738, 2.37259566683, 1258.45393162560],
        [239.036, 3.57397189838, 835.03713448730],
        [197.073, 5.92859096863, 453.42489381900],
        [139.440, 3.63960322318, 1788.14489672020],
        [191.373, 6.28251311870, 983.11585891360],
        [176.551, 2.57669991654, 9683.59458111640],
        [123.567, 2.26158186345, 2317.83586181480],
        [128.176, 4.66585907670, 831.85574074960],
        [112.430, 0.85604150812, 433.71173787680],
        [128.817, 1.10567106595, 2531.13495725280],
        [99.390, 4.50312054049, 518.64526483070],
        [93.870, 2.72553879990, 853.19638175200],
        [106.481, 5.81462222290, 220.41264243880],
        [120.188, 2.95156363556, 3.93215326310],
        [104.002, 2.22221906187, 74.78159856730],
        [81.655, 3.23481337678, 1361.54670584420],
        [112.513, 4.86216964016, 528.20649238630],
        [79.539, 0.88542246830, 430.53034413910],
        [85.801, 2.11458386763, 1574.84580128220],
        [85.685, 2.33823884827, 2428.04218303420],
        [68.311, 3.35727048905, 2104.53676637680],
        [69.570, 3.04164697156, 302.16477565500],
        [69.775, 3.22402404312, 305.34616939270],
        [69.570, 0.20494979941, 532.13864564940],
        [56.991, 2.00204191909, 2634.22773147140],
        [77.062, 2.09816000231, 508.35032409220],
        [56.716, 3.91743976711, 2221.85663459700],
        [58.325, 5.72360355252, 628.85158605010],
        [52.485, 4.02485010492, 527.24328453980],
        [63.645, 1.09973563964, 1364.72809958190],
        [53.607, 0.87425992614, 2847.52682690940],
        [59.598, 0.95822471775, 494.26624244250],
        [57.960, 3.45779497978, 2008.55753915900],
        [41.512, 3.51955526735, 529.73914920440],
        [44.666, 1.62313786651, 984.60033162190],
        [44.883, 4.90091959557, 2648.45482547300],
        [53.206, 1.19800364308, 760.25553592000],
        [44.393, 4.42623747662, 1063.31408345230],
        [37.566, 2.93021095213, 1677.93857550080],
        [41.516, 0.32174409278, 529.64278098480],
        [42.855, 0.03093594081, 1439.50969814920],
        [45.963, 2.54342106514, 636.71589257630],
        [40.181, 4.39381642864, 1148.24761040620],
        [38.770, 4.31675565025, 149.56319713460],
        [40.348, 2.10140891053, 2744.43405269080],
        [48.851, 5.60297777544, 2810.92146160520],
        [37.085, 5.07828164301, 1905.46476494040],
        [43.875, 1.24536971083, 621.73803904930],
        [34.005, 3.09360167248, 2420.92863603340],
        [36.782, 0.84232174637, 530.65417294110],
        [31.139, 5.35811251334, 1485.98012106520],
        [39.295, 4.70800489067, 569.04784100980],
        [39.700, 2.46163878814, 355.74874557180],
        [31.527, 6.19284070863, 3.18139373770],
        [28.399, 2.48456666067, 519.39602435610],
        [32.432, 2.73281750275, 604.47256366190],
        [27.119, 3.92341697086, 2324.94940881560],
        [26.753, 1.74975198417, 2950.61960112800],
        [28.986, 1.83535862643, 1891.23767093880],
        [26.493, 0.60380196895, 1055.44977692610],
        [33.525, 0.76068430639, 643.82943957710],
        [26.568, 1.03594610835, 405.25754987360],
        [25.534, 3.46320665375, 458.84151979040],
        [24.421, 0.88181836930, 423.41679713830],
        [32.949, 3.18597137308, 528.72775724810],
        [22.456, 0.43129919683, 1073.60902419080],
        [21.599, 1.41820425091, 540.73666535850],
        [25.673, 0.52358194760, 511.53171782990],
        [21.115, 3.08023522766, 629.60234557550],
        [22.713, 0.65234613144, 3163.91869656600],
        [19.189, 5.16589014963, 635.96513305090],
        [26.042, 1.33629471285, 330.61896365820],
        [18.263, 3.59973446951, 746.92221379570],
        [18.210, 2.66819439927, 1994.33044515740],
        [19.724, 4.13552133321, 1464.63948006280],
        [19.480, 1.85656428109, 3060.82592234740],
        [23.927, 4.99826361784, 1289.94650101460],
        [21.886, 5.91718683551, 1802.37199072180],
        [17.482, 2.82161612542, 2737.32050569000],
        [16.608, 5.67394889755, 408.43894361130],
        [22.892, 5.26731352093, 672.14061522840],
        [18.349, 1.89869734949, 1021.24889455140],
        [19.123, 3.65882402977, 415.55249061210],
        [15.735, 3.34772676006, 1056.20053645150],
        [16.373, 0.18094878053, 1699.27921650320],
        [18.899, 3.69120638874, 88.86568021700],
        [18.655, 1.97327300097, 38.13303563780],
        [15.542, 3.82204881010, 721.64941953020],
        [16.780, 1.90976657921, 217.23124870110],
        [15.313, 1.05907174619, 114.13847448250],
        [15.190, 1.32317039042, 117.31986822020],
        [15.080, 3.74469077216, 2641.34127847220],
        [19.836, 2.73184571324, 39.35687591520],
        [14.708, 1.67270454473, 529.16970023280],
        [14.036, 3.54305270022, 142.44965013380],
        [12.931, 1.48829749349, 3267.01147078460],
        [14.924, 1.32546085940, 490.33408917940],
        [14.753, 4.64530618027, 6283.07584999140],
        [14.672, 0.80451954754, 5223.69391980220],
        [12.085, 3.67072510553, 750.10360753340],
        [11.954, 2.97127390765, 505.31194270640],
        [14.650, 2.16792930250, 530.21222995640],
        [11.869, 1.66551754962, 2207.62954059540],
        [12.273, 0.20690014405, 1062.56332392690],
        [11.460, 1.11906683214, 561.93429400900],
        [11.083, 3.22049096074, 535.10759106600],
        [11.567, 5.22625628971, 524.06189080210],
        [11.161, 3.82945634036, 76.26607127560],
        [10.918, 1.27796962818, 2125.87740737920],
        [12.685, 3.96848605476, 2538.24850425360],
        [11.230, 3.23092119889, 422.66603761290],
        [12.645, 0.73670428580, 908.33426034630],
        [11.330, 5.56127247007, 531.17543780290],
        [9.509, 5.00507284204, 597.35901666110],
        [10.291, 3.84159025239, 1781.03134971940],
        [10.762, 4.91380719453, 525.02509864860],
        [11.786, 5.11863653538, 685.47393735270],
        [11.980, 1.72470898635, 911.30320576290],
        [8.937, 2.40338241992, 2310.72231481400],
        [9.253, 2.57670338148, 3053.71237534660],
        [9.488, 2.95089828501, 1382.88734684660],
        [9.889, 0.43758517388, 3480.31056622260],
        [8.781, 3.66562388594, 739.80866679490],
        [8.664, 2.70398612383, 526.77020378780],
        [9.505, 1.61249870019, 3377.21779200400],
        [11.540, 1.59520481029, 1474.67378837040],
        [9.533, 0.35468711552, 1512.80682400820],
        [9.980, 4.80984684596, 558.00214074590],
        [9.014, 1.21458362718, 416.30325013750],
        [7.969, 0.08480602718, 528.94020556920],
        [8.668, 5.29060005706, 945.24345570670],
        [7.851, 1.46751861875, 963.40270297140],
        [8.611, 1.13232641062, 532.61172640140],
        [7.838, 6.26933498027, 647.01083331480],
        [7.581, 2.90608705954, 533.88375078860],
        [8.583, 6.06634530166, 10213.28554621100],
        [10.198, 2.48743123636, 1819.63746610920],
        [8.536, 2.22700701790, 9153.90361602180],
        [9.759, 6.15593336218, 593.42686339800],
        [7.968, 3.75535355212, 530.44172462000],
        [7.142, 3.58836120327, 2957.73314812880],
        [7.122, 0.11970048938, 224.34479570190],
        [8.731, 0.75302913970, 960.22130923370],
        [7.063, 2.16793037690, 724.83081326790],
        [7.263, 2.29499675875, 520.12973753900],
        [6.418, 1.25058991868, 3583.40334044120],
        [8.270, 1.24806288317, 495.75071515080],
        [6.483, 4.74567772640, 202.25339517410],
        [7.197, 3.84169279666, 618.55664531160],
        [8.146, 0.73147060302, 230.56457082540],
        [6.165, 5.50124418381, 11.04570026390],
        [7.946, 2.07754951174, 953.10776223290],
        [7.675, 0.92400307662, 525.49817940060],
        [6.210, 1.45641362115, 483.22054217860],
        [7.359, 0.31355650764, 378.64329525170],
        [6.707, 2.92071167098, 1038.04128918680],
        [7.143, 0.18218134889, 731.94436026870],
        [7.309, 6.27084533477, 21.34064100240],
        [6.135, 2.67651237303, 312.45971639350],
        [5.558, 3.83419160288, 534.35683154060],
        [5.344, 5.25294750019, 1048.33622992530],
        [7.504, 0.74281415471, 457.61767951300],
        [5.335, 6.23059924424, 551.03160609700],
        [5.613, 1.51210605952, 524.27433912320],
        [5.284, 2.18579185671, 280.96714700450],
        [5.475, 5.95864753605, 539.98590583310],
        [5.056, 0.37387972537, 529.53090640020],
        [6.202, 5.53813122743, 2.44768055480],
        [5.490, 5.97692444199, 227.52618943960],
        [6.266, 0.76632858238, 938.12990870590],
        [5.750, 2.13496323512, 191.95845443560],
        [5.218, 4.69335266854, 560.71045373160],
        [5.480, 5.21157595558, 1057.89745748090],
        [5.738, 0.34249718209, 535.91074021810],
        [4.816, 1.51326236835, 2524.02141025200],
        [5.056, 3.46671669992, 529.85102378900],
        [4.710, 2.27813830550, 3370.10424500320],
        [5.228, 3.61776977584, 2097.42321937600],
        [4.878, 1.39829798223, 3693.60966166060],
        [5.727, 4.80120381106, 598.84348936940],
        [5.707, 3.94177950323, 2854.64037391020],
        [4.988, 4.87244187719, 1.48447270830],
        [5.424, 3.53268613904, 456.39383923560],
        [4.288, 4.84438067847, 70.84944530420],
        [5.944, 3.79180483544, 25558.21217647960],
        [4.195, 2.09136830994, 2627.11418447060],
        [4.582, 5.61707254513, 2435.15573003500],
        [4.268, 6.20250525415, 775.23338944700],
        [4.521, 0.20049967962, 92.04707395470],
        [5.405, 4.66492781581, 833.55266177900],
        [5.607, 3.30226645638, 535.32003938710],
        [4.171, 3.14873010832, 944.98282327580],
        [4.108, 5.84489743779, 440.82528487760],
        [4.367, 4.68363584557, 327.43756992050],
        [4.033, 3.30883782817, 3274.12501778540],
        [4.292, 0.20604269202, 3796.70243587920],
        [4.270, 0.98941708997, 387.24131496080],
        [4.259, 3.21120589971, 696.51963761660],
        [4.673, 1.96606729969, 107.02492748170],
        [4.031, 4.62854606236, 2751.54759969160],
        [5.115, 2.66416451377, 1215.16490244730],
        [4.181, 4.74527698816, 988.53248488500],
        [4.374, 1.50010561403, 1894.41906467650],
        [3.803, 3.59911687954, 437.64389113990],
        [3.761, 3.96903199782, 732.69511979410],
        [3.620, 1.57847427805, 381.61224066830],
        [3.490, 0.63097592112, 529.90341341570],
        [4.019, 2.57664165720, 916.93228005540],
        [4.133, 4.78417930217, 824.74219374880],
        [4.411, 3.13179382423, 630.33605875840],
        [4.099, 3.63702212253, 810.65811209910],
        [3.704, 6.17243801274, 537.76771994190],
        [4.124, 2.14248285449, 210.11770170030],
        [3.490, 3.20962050417, 529.47851677350],
        [3.281, 1.53106243317, 547.85021235930],
        [3.554, 6.03787799174, 739.05790726950],
        [4.101, 6.00406226999, 902.70518605380],
        [3.267, 3.49354065789, 1166.40685767090],
        [3.286, 2.55966870530, 945.99421523210],
        [4.041, 4.78735413707, 850.01498801430],
        [4.304, 0.11406117717, 1744.85586754190],
        [4.043, 5.20417093600, 635.23141986800],
        [3.115, 4.61986265585, 952.35700270750],
        [3.016, 0.95126220905, 3899.79521009780],
        [3.017, 2.59699501992, 632.83192342300],
        [3.219, 1.83594791142, 18.15924726470],
        [3.203, 6.12597544496, 10.29494073850],
        [3.220, 6.14213423140, 1158.54255114470],
        [3.000, 5.69509924353, 632.73555520340],
        [3.226, 5.59910267099, 608.40471692500],
        [3.118, 5.64998934505, 99.16062095550],
        [3.745, 2.08111521615, 282.45161971280],
        [2.837, 4.60175594220, 245.54242435240],
        [3.093, 6.02049413961, 633.74694715970],
        [3.120, 2.29047945342, 631.82053146670],
        [2.662, 3.69016679729, 885.43971066640],
        [3.150, 1.79784999553, 521.61421024730],
        [2.822, 3.14927418161, 295.05122865420],
        [2.615, 0.20732170653, 35.42472265210],
        [2.971, 1.28795094653, 1023.95720753710],
        [2.571, 2.01817133502, 1514.29129671650],
        [2.592, 0.48790221200, 195.13984817330],
        [3.263, 2.38820607343, 836.52160719560],
        [2.501, 0.21653750027, 465.95506679120],
        [2.451, 5.58559489768, 544.66881862160],
        [2.535, 1.44414086617, 460.53844081980],
        [2.666, 3.30350145485, 2413.81508903260],
        [2.412, 4.36756580310, 1056.93424963440],
        [2.452, 4.53818816565, 514.71311156760],
        [3.239, 1.17022488774, 177.87437278590],
        [3.218, 0.60551913257, 1061.82961074400],
        [2.408, 0.65423523810, 523.54062594030],
        [2.299, 2.15247752560, 319.57326339430],
        [2.791, 2.71505085086, 610.69233878540],
        [2.729, 1.77685979153, 252.65597135320],
        [2.666, 3.77750458842, 3171.03224356680],
        [2.303, 0.36676453766, 1969.20066324380],
        [2.664, 0.09674841214, 565.11568774670],
        [2.312, 2.07210502831, 3686.49611465980],
        [2.680, 4.94445888050, 1593.00504854690],
        [2.193, 0.55645982205, 2228.97018159780],
        [2.526, 1.07528597373, 12036.46073488820],
        [2.778, 1.48379350517, 447.79581952650],
        [2.235, 5.95475282699, 6151.53388830500],
        [2.759, 4.63976153480, 462.02291352810],
        [2.175, 4.53588570240, 501.37978944330],
        [2.323, 5.93670041006, 611.44309831080],
        [2.384, 2.81746622971, 3340.61242669980],
        [2.087, 3.10716079675, 1049.08698945070],
        [1.994, 2.02500860064, 1058.86066532740],
        [2.199, 2.20937490997, 1269.49963188950],
        [2.705, 1.97665276677, 415.29185818120],
        [2.787, 1.31053438756, 1041.22268292450],
        [2.003, 4.66904374443, 679.25416222920],
        [1.962, 1.82999730674, 2943.50605412720],
        [2.289, 2.96480800939, 69.15252427480],
        [2.192, 4.47837196209, 209.36694217490],
        [2.020, 0.04621364490, 4113.09430553580],
        [2.082, 1.11203059170, 4010.00153131720],
        [1.991, 3.20108648275, 3590.51688744200],
        [1.900, 3.32227077969, 421.93232443000],
        [2.193, 2.82218305362, 292.01284726840],
        [2.288, 1.94695631885, 1279.79457262800],
        [1.843, 5.23293634337, 14.97785352700],
        [1.932, 5.46684252030, 2281.23049651060],
        [2.177, 2.93031976617, 429.04587143080],
        [2.125, 0.06224847826, 24.37902238820],
        [2.464, 5.39581078430, 1261.63532536330],
        [1.938, 3.79908004671, 1059.43011429900],
        [2.029, 3.95461157815, 771.30123618390],
        [1.841, 4.74905354737, 78.71375183040],
        [1.922, 2.21862085389, 99.91138048090],
        [1.836, 5.75449805175, 623.22251175760],
        [2.145, 3.87052575546, 451.94042111070],
        [1.782, 0.40860352236, 754.03576079650],
        [1.784, 1.49468287576, 529.95159752550],
        [1.842, 3.49726261337, 1354.43315884340],
        [1.748, 3.48730020953, 522.62560220360],
        [1.816, 1.24334711210, 417.03696332040],
        [1.752, 1.15500390019, 1060.34513803570],
        [1.729, 2.69831073799, 642.34496686880],
        [1.985, 1.99916658759, 934.94851496820],
        [1.828, 5.44095029767, 1201.83158032300],
        [2.158, 3.45672748590, 827.92358748650],
        [1.959, 1.06033047373, 33.94024994380],
        [1.751, 3.13572498964, 384.05992122310],
        [1.781, 5.02895146997, 1098.73880610440],
        [2.074, 3.18582065441, 1366.21257229020],
        [1.757, 5.02778552877, 586.31331639720],
        [2.045, 3.08816627459, 535.84130424890],
        [2.273, 5.17998505813, 3178.14579056760],
        [1.617, 3.16674916201, 67.66805156650],
        [1.627, 6.10603469594, 432.01481684740],
        [1.930, 1.63968957659, 5.41662597140],
        [1.741, 0.99408274736, 1254.52177836250],
        [1.607, 5.65498642076, 1165.65609814550],
        [1.676, 3.06138410273, 1134.16352875650],
        [1.821, 3.05183555090, 567.82400073240],
        [1.677, 3.09175084930, 1251.34038462480],
        [1.994, 2.52023134712, 1059.90319505100],
        [2.204, 6.15376698510, 563.63121503840],
        [1.692, 4.19142612803, 106.27416795630],
        [1.906, 5.58417395051, 32.24332891440],
        [2.206, 1.75883974012, 1151.42900414390],
        [1.552, 3.04262360186, 385.54439393140],
        [1.508, 0.42002830727, 313.21047591890],
        [1.494, 1.43672345922, 2840.41327990860],
        [1.678, 2.17255433434, 306.83064210100],
        [1.511, 4.44377608685, 395.10562148700],
        [1.958, 0.05215107058, 761.74000862830],
        [1.760, 1.27045286501, 1173.52040467170],
        [1.463, 6.07810373103, 0.96320784650],
        [1.498, 2.79408561759, 277.03499374140],
        [1.636, 0.26199351490, 522.52923398400],
        [1.507, 0.48961801593, 4216.18707975440],
        [1.530, 3.42953827550, 1159.29331067010],
        [1.744, 2.39637837261, 203.00415469950],
        [1.569, 2.55719070621, 4.19278569400],
        [1.576, 3.45039607104, 1058.41872234270],
        [1.466, 2.24427539934, 1550.93985964600],
        [1.784, 2.34591354953, 529.43033266370],
        [1.939, 4.73685428610, 3067.93946934820],
        [1.938, 0.60126164334, 1059.33374607940],
        [1.523, 2.98744673443, 2730.20695868920],
        [1.834, 3.78099298791, 420.96911658350],
        [1.372, 3.53997115825, 5.62907429250],
        [1.361, 0.45533257707, 418.52143602870],
        [1.833, 5.12743628215, 1578.02719501990],
        [1.839, 4.24616044210, 981.63138620530],
        [1.567, 3.32429870195, 532.39927808030],
        [1.340, 1.94668282270, 528.41894070740],
        [1.422, 1.83191577465, 4002.88798431640],
        [1.745, 5.76913240451, 490.07345674850],
        [1.437, 4.19470227783, 420.44785172170],
        [1.419, 0.74849005330, 632.26247445140],
        [1.447, 5.65611888743, 373.01422095920],
        [1.578, 3.90273683089, 602.98809095360],
        [1.385, 3.88479835656, 419.43645976540],
        [1.352, 0.81697905853, 1585.14074202070],
        [1.399, 1.24785452243, 633.30500417500],
        [1.297, 5.57914023189, 1276.61317889030],
        [1.491, 1.66541781223, 2655.56837247380],
        [1.252, 0.72155670765, 173.94221952280],
        [1.658, 5.60924662850, 362.86229257260],
        [1.606, 3.95301396173, 2274.54683263650],
        [1.213, 4.55264289565, 366.79444583570],
        [1.521, 0.55773831071, 1592.25428902150],
        [1.220, 3.63029788040, 497.44763618020],
        [1.215, 4.42854185903, 531.38788612400],
        [1.549, 5.73765962068, 320.32402291970],
        [1.480, 4.29779032931, 303.86169668440],
        [1.507, 2.27998567874, 758.77106321170],
        [1.212, 3.38335836048, 536.85269620520],
        [1.245, 4.21639959154, 4.66586644600],
        [1.507, 3.52136655355, 774.00954916960],
        [1.481, 3.06156044618, 1585.89150154610],
        [1.462, 2.30628702634, 1363.24362687360],
        [1.180, 3.52708055024, 1064.79855616060],
        [1.193, 5.88284733845, 1060.86640289750],
        [1.398, 4.99456521692, 842.90144101350],
        [1.406, 1.53799746944, 1020.02505427400],
        [1.367, 4.10254739443, 799.61241183520],
        [1.336, 1.89387272380, 530.96298948180],
        [1.238, 3.62226383331, 3487.42411322340],
        [1.306, 3.39985119727, 539.25219265020],
        [1.156, 0.77127511567, 1603.29998928540],
        [1.482, 0.48451915093, 493.04240216510],
        [1.247, 5.64344659992, 479.28838891550],
        [1.195, 2.39909893341, 561.18353448360],
        [1.106, 0.89453807282, 2.92076130680],
        [1.227, 2.76231244946, 299.12639426920],
        [1.128, 4.72319873338, 124.43341522100],
        [1.086, 5.66180289525, 1053.75285589670],
        [1.329, 0.16664094530, 536.75632798560],
        [1.082, 4.51407359350, 528.25467649610],
        [1.105, 1.93890691771, 244.31858407500],
        [1.446, 0.65096230619, 1091.62525910360],
        [1.071, 4.67974963103, 521.82665856840],
        [1.413, 4.72936311016, 1141.13406340540],
        [1.086, 2.88721124443, 1262.38608488870],
        [1.254, 5.74156595137, 527.99404406520],
        [1.082, 5.60975006771, 531.12725369310],
        [1.148, 3.27410230525, 1035.00290780100],
        [1.224, 3.68807537150, 81.75213321620],
        [1.072, 0.48068438564, 1058.63117066380],
        [1.036, 1.68789163831, 1070.42763045310],
        [1.052, 4.72763208332, 913.75088631770],
        [1.166, 4.97812626679, 450.97721326420],
        [1.042, 2.90894542321, 3906.90875709860],
        [0.997, 1.65967703856, 3259.89792378380],
        [1.113, 3.06502453809, 1482.79872732750],
        [0.991, 0.91568114148, 576.16138801060],
        [0.987, 0.91349590742, 2332.06295581640],
        [1.003, 6.17381204883, 391.17346822390],
        [1.087, 3.19260020877, 151.04766984290],
        [0.987, 2.48065918834, 1912.57831194120],
        [0.975, 1.55458771092, 536.28324723360],
        [1.193, 2.19383228000, 523.09868295560],
        [0.979, 3.28693620660, 1379.70595310890],
        [0.963, 2.29845109892, 1467.82087380050],
        [1.279, 4.73978455573, 600.54041039880],
        [1.269, 1.77171706595, 5120.60114558360],
        [0.938, 3.13636271584, 1372.59240610810],
        [0.956, 0.94045126791, 429.77958461370],
        [1.130, 4.87259620358, 874.39401040250],
        [1.044, 3.52819283674, 530.58473697190],
        [1.244, 0.80634178279, 419.53282798500],
        [0.914, 4.34324212455, 1127.04998175570],
        [1.095, 3.17513475763, 6681.22485339960],
        [0.926, 5.53099018797, 537.55527162080],
        [1.025, 6.08315999637, 469.88722005430],
        [0.928, 2.64064849636, 31.01948863700],
        [0.887, 5.53922649066, 498.67147645760],
        [1.153, 5.20213407651, 554.06998748280],
        [0.976, 4.26047885490, 806.72595883600],
        [0.871, 5.79751110150, 594.65070367540],
        [1.044, 0.31244551729, 528.79719321730],
        [0.911, 0.94039205468, 337.73251065900],
        [1.197, 3.12884590029, 1966.23171782720],
        [0.930, 2.88178471518, 1056.46116888240],
        [1.052, 1.69484089706, 484.44438245600],
        [0.862, 0.67309397482, 20426.57109242200],
        [1.152, 1.16751621652, 1489.91227432830],
        [0.847, 3.25831322825, 1063.57471588320],
        [0.884, 0.71487680084, 2042.49778910280],
        [0.888, 5.38714907441, 5621.84292321040],
        [1.137, 4.02029739425, 1670.07426897460],
        [0.844, 3.31846798590, 812.14258480740],
        [0.860, 4.78175008217, 530.91480537200],
        [0.835, 3.63117401608, 451.72797278960],
        [0.931, 2.27352189963, 100.64509366380],
        [0.939, 3.51238251326, 523.47118997110],
        [0.860, 5.34207357904, 528.46712481720],
        [0.875, 0.87775537110, 4326.39340097380],
        [0.961, 5.69327275886, 498.19839570560],
        [0.966, 6.25512226434, 700.45179087970],
        [0.842, 3.20535945596, 1670.82502850000],
        [0.808, 1.09148925587, 683.18631549230],
        [0.810, 5.47935192896, 525.54636351040],
        [0.855, 6.06969867736, 446.31134681820],
        [0.989, 1.55623875216, 1493.09366806600],
        [0.837, 1.49510080792, 1025.44168024540],
        [0.974, 3.67667471757, 25565.32572348040],
        [0.788, 0.51622458293, 526.98265210890],
        [0.820, 1.86002542644, 629.86297800640],
        [0.813, 0.45441968195, 4694.00295470760],
        [0.953, 0.58786779132, 627.36711334180],
        [0.908, 2.82093327912, 3046.59882834580],
        [0.912, 2.69124310451, 946.72792841500],
        [0.820, 4.14947931572, 1884.12412393800],
        [0.948, 0.77931728039, 25551.09862947879],
        [0.844, 0.00976249584, 628.59095361920],
        [0.910, 0.99542530366, 5760.49843189760],
        [0.844, 0.22630964490, 1123.11782849260],
        [0.924, 4.41952345708, 5746.27133789600],
        [0.967, 3.20618313117, 9050.81084180320],
        [0.800, 0.10663079153, 4532.57894941100],
        [0.748, 3.01376405927, 5481.75455838080],
        [0.752, 5.82360472890, 701.93626358800],
        [0.771, 0.12101982692, 635.70450062000],
        [0.725, 2.81220410314, 3597.63043444280],
        [0.944, 0.40327408174, 1140.38330388000],
        [0.726, 5.28930472464, 1304.92435454160],
        [0.994, 5.16391370100, 10316.37832042960],
        [0.890, 4.10819809692, 1060.13268971460],
        [0.962, 1.48376004549, 1062.30269149600],
        [0.883, 5.26813169286, 1542.60247236780],
        [0.916, 6.02908368648, 7.86430652620],
        [0.725, 2.18773773010, 1176.70179840940],
        [0.808, 5.81725174908, 1087.69310584050],
        [0.757, 0.77440414330, 977.48678462110],
        [0.838, 3.81585420192, 986.08480433020],
        [0.888, 1.89634795578, 707.56533788050],
        [0.854, 5.47701506544, 2818.03500860600],
        [0.796, 1.08794807212, 987.30864460760],
        [0.856, 2.58042139486, 2803.80791460440],
        [0.708, 1.09492310353, 248.72381809010],
        [0.811, 3.23726191865, 121.25202148330],
        [0.727, 1.56150632966, 4319.27985397300],
        [0.687, 2.65457835371, 1567.73225428140],
        [0.675, 1.78690909614, 103.14095832840],
        [0.853, 4.74476428852, 951.62328952460],
        [0.832, 5.14362789810, 1054.71606374320],
        [0.846, 1.47557828604, 898.77303279070],
        [0.701, 1.72139817505, 5230.80746680300],
        [0.863, 3.98700238575, 686.95841006100],
        [0.703, 2.89202252444, 63.73589830340],
        [0.673, 6.11618580510, 738.32419408660],
        [0.806, 4.64475158248, 533.83556667880],
        [0.670, 2.67625974048, 1012.91150727320],
        [0.668, 4.93815253692, 5172.47623572500],
        [0.818, 1.41973280302, 580.09354127370],
        [0.652, 3.41422919445, 650.94298657790],
        [0.643, 2.46566726278, 1049.82070263360],
        [0.859, 2.50530106631, 782.34693644780],
        [0.662, 4.13533996643, 733.42883297700],
        [0.812, 1.30325352179, 1055.18914449520],
        [0.638, 4.21760246824, 1064.04779663520],
        [0.637, 6.13121700151, 4752.99159184980],
        [0.636, 0.83411828974, 711.49749114360],
        [0.642, 1.86741704507, 1053.96530421780],
        [0.795, 4.54081089118, 1457.52593306200],
        [0.783, 4.37652961667, 105.54045477340],
        [0.640, 5.44039474349, 632.03297978780],
        [0.651, 5.02431301146, 528.04643369190],
        [0.686, 0.27079898498, 11.77941344680],
        [0.644, 5.36935176134, 835.78789401270],
        [0.639, 1.86699974431, 6172.86952877200],
        [0.630, 2.86895754523, 633.53449883860],
        [0.826, 1.46026926041, 2199.76523406920],
        [0.687, 3.81221717134, 73.29712585900],
        [0.697, 4.18082589322, 1.69692102940],
        [0.788, 0.21278801649, 313.94418910180],
        [0.686, 2.51807576494, 638.41281360570],
        [0.847, 5.56263749391, 4429.48617519240],
        [0.673, 4.87494072856, 103.04459010880],
        [0.663, 4.80713895807, 991.71387862270],
        [0.614, 3.87231597482, 767.36908292080],
        [0.666, 5.71697262323, 661.09491496450],
        [0.681, 2.33844767741, 501.23677709140],
        [0.597, 3.03921014345, 6.95348830640],
        [0.777, 3.08786050361, 441.57604440300],
        [0.588, 0.08236113246, 4164.31198961300],
        [0.693, 4.66190836234, 3384.33133900480],
        [0.810, 1.97701084490, 860.30992875280],
        [0.602, 5.56403449542, 1587.58842257550],
        [0.622, 6.11554348965, 7.06536289100],
        [0.592, 3.29013906024, 10103.07922499160],
        [0.692, 6.10931942233, 12.74262129330],
        [0.597, 6.13204711801, 7.27360569520],
        [0.594, 2.58839673551, 849.26422848890],
        [0.728, 2.73732195088, 6.15033915430],
        [0.602, 5.28816527514, 949.12742486000],
        [0.568, 1.75508433865, 1077.54117745390],
        [0.575, 4.50676079721, 1230.14275597430],
        [0.588, 0.65827893998, 4642.78527063040],
        [0.561, 3.87565914360, 135.33610313300],
        [0.558, 3.36094471852, 24498.83024629040],
        [0.557, 3.45629457197, 19896.88012732740],
        [0.558, 1.17103892689, 3576.28979344040],
        [0.574, 5.19235074140, 104.05598206510],
        [0.560, 3.57141429379, 5333.90024102160],
        [0.555, 0.18349908409, 512.42548970720],
        [0.571, 0.83070148820, 1570.91364801910],
        [0.632, 3.67893818442, 1065.01100448170],
        [0.744, 2.33083237537, 620.25356634100],
        [0.540, 5.15775909675, 1751.53953141600],
        [0.592, 3.07238123875, 1446.62324515000],
        [0.537, 1.52803865425, 8094.52168583260],
        [0.550, 5.50701003577, 1432.39615114840],
        [0.546, 2.34388967045, 949.22379307960],
        [0.534, 3.04076654796, 7.16173111060],
        [0.619, 6.07865159203, 46.47042291600],
        [0.562, 0.96641974928, 1438.02522544090],
        [0.531, 1.06695547390, 100.17201291180],
        [0.599, 3.59295739143, 1144.31545714310],
        [0.526, 3.51641923371, 0.75075952540],
        [0.564, 0.72677136494, 1059.22187149480],
        [0.537, 5.72603965787, 513.22863885930],
        [0.630, 2.31183143900, 2729.45619916380],
        [0.530, 4.99510636441, 9264.10993724120],
        [0.649, 0.95666735852, 920.86443331850],
        [0.547, 1.18801926149, 11506.76976979360],
        [0.516, 3.28562070858, 734.91330568530],
        [0.567, 5.13926871155, 288.08069400530],
        [0.538, 0.28159637680, 153.49535039770],
        [0.718, 0.48326672359, 842.15068148810],
        [0.526, 4.39778401928, 546.15329132990],
        [0.695, 2.44235086902, 657.16276170140],
        [0.697, 4.99042365686, 12.53017297220],
        [0.519, 6.27847163164, 59.80374504030],
        [0.504, 2.58550284000, 5378.66178416220],
        [0.496, 2.43659402827, 990.22940591440],
        [0.617, 5.73284985700, 745.43774108740],
        [0.519, 3.10157097770, 9161.01716302260],
        [0.654, 1.31181453784, 878.32616366560],
        [0.619, 3.71554817226, 2090.30967237520],
        [0.500, 4.28937439066, 5216.58037280140],
        [0.621, 3.98893673383, 409.92341631960],
        [0.685, 1.95310431695, 3156.80514956520],
        [0.552, 2.81774132958, 344.70304530790],
        [0.551, 1.91969778405, 113.38771495710],
        [0.682, 0.87321578326, 6069.77675455340],
        [0.651, 5.09951064975, 531.33549649730],
        [0.537, 3.67357440226, 605.95703637020],
        [0.525, 0.74584814988, 736.83972137830],
        [0.505, 3.12494814307, 1475.68518032670],
        [0.622, 3.00013939606, 2349.32843120380],
        [0.644, 3.00156986335, 298.23262239190],
        [0.564, 3.81960833949, 1059.54198888360],
        [0.468, 3.50348554992, 4841.85727206680],
        [0.491, 1.28535573072, 247.23934538180],
        [0.458, 0.45056377876, 1065.60170531270],
        [0.543, 2.39704308320, 9690.70812811720],
        [0.459, 5.29870259698, 1474.93442080130],
        [0.483, 3.63649121244, 131.40394986990],
        [0.632, 2.75028345792, 334.55111692130],
        [0.483, 0.42979609421, 735.82832942200],
        [0.540, 0.54791737146, 51646.11531805379],
        [0.531, 0.30026207053, 912.78767847120],
        [0.449, 3.02583472996, 5901.23920225600],
        [0.544, 2.98747240952, 4223.30062675520],
        [0.557, 5.83542572008, 9676.48103411560],
        [0.501, 0.03408180117, 1080.72257119160],
        [0.517, 4.40400852026, 2545.36205125440],
        [0.481, 3.63292807076, 5584.84733259940],
        [0.557, 6.11443978190, 976.00231191280],
        [0.481, 3.41035583659, 3803.81598288000],
        [0.622, 2.29597570837, 9999.98645077300],
        [0.454, 2.88584538455, 1987.21689815660],
        [0.439, 4.83198101064, 50.40257617910],
        [0.475, 2.69994471394, 491.81856188770],
        [0.618, 0.72471290082, 1291.43097372290],
        [0.503, 0.13449993622, 2015.67108615980],
        [0.551, 2.13418546604, 1440.99417085750],
        [0.595, 3.78181802545, 6386.16862421000],
        [0.434, 2.64411689486, 748.40668650400],
        [0.592, 0.32587740408, 737.36098624010],
        [0.490, 2.37988828800, 2225.78878786010],
        [0.439, 1.33582802018, 995.64603188580],
        [0.543, 2.05067702505, 906.84978763800],
        [0.466, 2.43707405011, 3362.99069800240],
        [0.481, 2.32223226419, 1357.61455258110],
        [0.566, 0.59740900184, 350.33211960040],
        [0.429, 2.46287580628, 3914.02230409940],
        [0.429, 1.01299906509, 4333.50694797460],
        [0.425, 1.67255823369, 148.07872442630],
        [0.412, 3.29630633921, 7.32599532190],
        [0.508, 1.16158524676, 9.56122755560],
        [0.524, 5.02562926120, 1090.40141882620],
        [0.409, 5.80053072411, 9146.79006902100],
        [0.497, 0.01579913593, 1069.67687092770],
        [0.548, 6.03429743373, 9367.20271145980],
        [0.433, 5.93688350840, 1688.23351623930],
        [0.424, 4.18150111530, 550.13783421970],
        [0.401, 0.11519846139, 970.51624997220],
        [0.503, 5.28212300854, 668.20846196530],
        [0.555, 1.00328633255, 141.22580985640],
        [0.404, 2.48633976473, 519.65665678700],
        [0.441, 6.06185501734, 25.12978191360],
        [0.412, 5.87495245826, 6.90109867970],
        [0.478, 0.71264950607, 1094.80665284130],
        [0.446, 2.71248183031, 31.49256938900],
        [0.404, 5.49462012486, 447.93883187840],
        [0.391, 1.26105612700, 8.07675484730],
        [0.463, 1.93535321271, 6275.96230299060],
        [0.507, 3.61089992782, 546.95644048200],
        [0.402, 5.86200127054, 927.83496796740],
        [0.481, 6.21043578332, 683.98946464440],
        [0.483, 5.02142924458, 857.12853501510],
        [0.444, 0.84873092377, 1371.84164658270],
        [0.391, 2.81753436573, 5798.14642803740],
        [0.395, 0.22367886581, 51116.42435295920],
        [0.378, 6.03765733432, 1268.74887236410],
        [0.471, 6.24506463249, 946.46729598410],
        [0.405, 0.57785207581, 107.28555991260],
        [0.371, 6.15750793727, 509.24409596950],
        [0.370, 4.90330687618, 1436.54075273260],
        [0.448, 4.76565111029, 284.14854074220],
        [0.474, 0.71146352197, 2108.46891963990],
        [0.509, 5.53328407404, 1128.53445446400],
    ],
    # R1
    [
        [1271801.596, 2.64937511122, 529.69096509460],
        [61661.771, 3.00076251018, 1059.38193018920],
        [53443.592, 3.89717644226, 522.57741809380],
        [31185.167, 4.88276663526, 536.80451209540],
        [41390.257, 0.00000000000, 0.00000000000],
        [11847.190, 2.41329588176, 419.48464387520],
        [9166.360, 4.75979408587, 7.11354700080],
        [3175.763, 2.79297987071, 103.09277421860],
        [3203.446, 5.21083285476, 735.87651353180],
        [3403.605, 3.34688537997, 1589.07289528380],
        [2600.003, 3.63435101622, 206.18554843720],
        [2412.207, 1.46947308304, 426.59819087600],
        [2806.064, 3.74223693580, 515.46387109300],
        [2676.575, 4.33052878699, 1052.26838318840],
        [2100.507, 3.92762682306, 639.89728631400],
        [1646.182, 5.30953510947, 1066.49547719000],
        [1641.257, 4.41628669824, 625.67019231240],
        [1049.866, 3.16113622955, 213.29909543800],
        [1024.802, 2.55432643018, 412.37109687440],
        [740.996, 2.17094630558, 1162.47470440780],
        [806.404, 2.67750801380, 632.78373931320],
        [676.928, 6.24953479790, 838.96928775040],
        [468.895, 4.70973463481, 543.91805909620],
        [444.683, 0.40281181402, 323.50541665740],
        [567.076, 4.57655414712, 742.99006053260],
        [415.894, 5.36836018215, 728.76296653100],
        [484.689, 2.46882793186, 949.17560896980],
        [337.555, 3.16781951120, 956.28915597060],
        [401.738, 4.60528841541, 309.27832265580],
        [347.378, 4.68148808722, 14.22709400160],
        [260.753, 5.34290306101, 846.08283475120],
        [220.084, 4.84210964963, 1368.66025284500],
        [203.217, 5.59995425432, 1155.36115740700],
        [246.603, 3.92313823537, 942.06206196900],
        [183.504, 4.26526769703, 95.97922721780],
        [180.134, 4.40165491159, 532.87235883230],
        [197.134, 3.70551461394, 2118.76386037840],
        [196.005, 3.75877587139, 199.07200143640],
        [200.190, 4.43888814441, 1045.15483618760],
        [170.225, 4.84647488867, 526.50957135690],
        [146.335, 6.12958365535, 533.62311835770],
        [133.483, 1.32245735855, 110.20632121940],
        [132.076, 4.51187950811, 525.75881183150],
        [123.851, 2.04290370696, 1478.86657406440],
        [121.861, 4.40581788491, 1169.58825140860],
        [115.313, 4.46741278152, 1581.95934828300],
        [98.527, 5.72833991647, 1596.18644228460],
        [91.608, 4.52965592121, 1685.05212250160],
        [110.638, 3.62504147403, 1272.68102562720],
        [80.536, 4.11311699583, 1258.45393162560],
        [79.552, 2.71898473954, 1692.16566950240],
        [100.164, 5.24693885858, 1265.56747862640],
        [77.854, 5.56722651753, 1471.75302706360],
        [85.766, 0.07906707372, 831.85574074960],
        [82.132, 3.80763015979, 508.35032409220],
        [55.319, 0.35180851191, 316.39186965660],
        [52.338, 5.53074272117, 433.71173787680],
        [55.769, 4.75141241141, 302.16477565500],
        [50.597, 4.85603161770, 1375.77379984580],
        [43.554, 4.94441642712, 1361.54670584420],
        [42.172, 1.22404278447, 853.19638175200],
        [37.695, 4.26767539209, 2001.44399215820],
        [49.395, 4.01422828967, 220.41264243880],
        [38.263, 5.33025236797, 1788.14489672020],
        [35.611, 1.76205571128, 1795.25844372100],
        [36.296, 3.84995284393, 1574.84580128220],
        [29.332, 5.16619257786, 3.93215326310],
        [25.180, 4.33777727362, 519.39602435610],
        [24.778, 2.72907897410, 405.25754987360],
        [27.025, 6.09669947903, 1148.24761040620],
        [22.604, 0.19173890105, 380.12776796000],
        [20.499, 4.32881495378, 3.18139373770],
        [19.925, 4.62967500111, 1677.93857550080],
        [19.528, 5.10596326232, 1073.60902419080],
        [18.427, 3.76522178300, 1485.98012106520],
        [18.869, 5.05259402407, 2104.53676637680],
        [17.031, 4.01843356903, 2317.83586181480],
        [16.671, 5.42931676507, 88.86568021700],
        [15.337, 2.92700926091, 2008.55753915900],
        [14.499, 3.63339836845, 628.85158605010],
        [14.575, 5.50832843322, 721.64941953020],
        [13.728, 4.87623389735, 629.60234557550],
        [18.481, 6.03032762264, 330.61896365820],
        [13.499, 1.38539534821, 518.64526483070],
        [15.740, 2.93038271684, 1905.46476494040],
        [12.459, 1.58587053146, 2111.65031337760],
        [12.272, 3.37671053917, 635.96513305090],
        [11.836, 4.08486322993, 2648.45482547300],
        [11.166, 4.62623267608, 636.71589257630],
        [14.348, 2.74177797727, 2221.85663459700],
        [11.221, 3.55311861205, 1891.23767093880],
        [13.121, 5.83845065644, 1464.63948006280],
        [11.351, 2.57606886230, 511.53171782990],
        [10.487, 0.49850799841, 453.42489381900],
        [9.728, 4.38837468002, 1994.33044515740],
        [10.131, 2.76432756215, 423.41679713830],
        [8.620, 5.16374493158, 1056.20053645150],
        [8.952, 4.79407952752, 2420.92863603340],
        [8.126, 3.72977106954, 2634.22773147140],
        [8.078, 1.29246272894, 2428.04218303420],
        [8.867, 1.85684753622, 750.10360753340],
        [8.912, 4.80973516711, 1062.56332392690],
        [8.552, 4.53818617984, 21.34064100240],
        [9.468, 4.33472161983, 1802.37199072180],
        [6.904, 5.96616555709, 540.73666535850],
        [7.293, 4.97763580465, 1699.27921650320],
        [7.083, 4.99096728816, 1055.44977692610],
        [7.226, 4.97823884383, 1898.35121793960],
        [6.464, 1.39173466879, 422.66603761290],
        [6.214, 4.46490158256, 551.03160609700],
        [6.794, 2.90878831415, 2324.94940881560],
        [6.173, 3.65617162985, 621.73803904930],
        [6.243, 6.13691919694, 2125.87740737920],
        [5.936, 2.58312235120, 569.04784100980],
        [6.504, 4.56908431757, 1038.04128918680],
        [7.305, 3.02062127734, 416.30325013750],
        [6.598, 5.55348005731, 1781.03134971940],
        [5.133, 6.21646917980, 963.40270297140],
        [5.876, 4.23153077453, 539.98590583310],
        [5.119, 0.06942832171, 1063.31408345230],
        [5.460, 4.91084384602, 835.03713448730],
        [4.989, 1.35153694680, 1382.88734684660],
        [5.224, 0.18468411116, 117.31986822020],
        [6.187, 3.87193497099, 191.95845443560],
        [4.681, 4.61057119508, 643.82943957710],
        [4.627, 3.34644534691, 2207.62954059540],
        [4.526, 4.07729737127, 2310.72231481400],
        [4.718, 4.55578336947, 2737.32050569000],
        [4.471, 1.47603161897, 408.43894361130],
        [4.073, 1.13014903180, 415.55249061210],
        [5.476, 5.63198569698, 618.55664531160],
        [4.034, 4.09631702747, 430.53034413910],
        [4.304, 4.60536378943, 647.01083331480],
        [3.765, 3.42751259825, 2950.61960112800],
        [4.559, 4.23723998745, 227.52618943960],
        [3.695, 1.03127824978, 2744.43405269080],
        [3.667, 4.12268925541, 440.82528487760],
        [3.677, 2.19480200527, 534.35683154060],
        [3.818, 1.14800596289, 74.78159856730],
        [4.221, 2.37721579949, 2538.24850425360],
        [3.488, 5.33792561596, 458.84151979040],
        [3.437, 4.26164443643, 10.29494073850],
        [4.394, 0.18808423412, 824.74219374880],
        [3.339, 4.85708402591, 295.05122865420],
        [3.329, 5.50043586719, 739.80866679490],
        [3.623, 4.64011531952, 2214.74308759620],
        [3.185, 2.69708590442, 561.93429400900],
        [3.421, 3.38512615384, 149.56319713460],
        [3.442, 4.34217280083, 305.34616939270],
        [3.580, 5.29481665335, 2097.42321937600],
        [3.401, 2.74761862893, 2641.34127847220],
        [2.901, 0.91012525424, 984.60033162190],
        [3.566, 1.63400343968, 525.02509864860],
        [2.869, 1.31799241974, 611.44309831080],
        [2.635, 5.25517910535, 532.13864564940],
        [2.683, 4.24641945773, 3053.71237534660],
        [2.614, 3.17862099921, 527.24328453980],
        [2.251, 4.21598247360, 739.05790726950],
        [2.268, 5.52248110560, 524.27433912320],
        [2.372, 4.19741177512, 217.23124870110],
        [2.623, 5.82647427958, 732.69511979410],
        [2.666, 3.92538056951, 210.11770170030],
        [2.036, 4.84043420813, 1049.08698945070],
        [2.441, 2.63840901843, 760.25553592000],
        [2.095, 5.76269812349, 529.64278098480],
        [2.021, 3.81308146017, 2627.11418447060],
        [2.089, 4.18463193132, 945.99421523210],
        [2.305, 1.61220665690, 604.47256366190],
        [1.969, 5.37427735384, 142.44965013380],
        [1.923, 4.75088270631, 535.10759106600],
        [1.955, 5.49000238006, 1439.50969814920],
        [1.877, 3.26978877187, 3267.01147078460],
        [2.286, 2.93885172004, 76.26607127560],
        [2.074, 5.85386852879, 532.61172640140],
        [2.121, 3.92430797099, 2435.15573003500],
        [1.807, 3.17208959472, 2524.02141025200],
        [1.712, 4.02986641257, 731.94436026870],
        [2.119, 0.41049593984, 1279.79457262800],
        [1.660, 2.34370903423, 528.72775724810],
        [1.655, 0.78809717175, 3060.82592234740],
        [1.729, 4.26127896267, 724.83081326790],
        [2.060, 5.04785330873, 2413.81508903260],
        [2.095, 2.67732367556, 529.73914920440],
        [1.933, 2.49162437046, 2957.73314812880],
        [1.898, 2.71948262975, 952.35700270750],
        [1.634, 2.98113068812, 945.24345570670],
        [1.582, 5.84373095005, 547.85021235930],
        [1.662, 0.27359627181, 454.90936652730],
        [1.595, 1.18530167095, 38.13303563780],
        [1.550, 0.64264572959, 312.45971639350],
        [1.525, 4.08789824989, 1158.54255114470],
        [1.542, 1.12520322326, 1021.24889455140],
        [1.539, 0.37324921979, 319.57326339430],
        [1.628, 5.24285773388, 1354.43315884340],
        [1.897, 3.79973291113, 953.10776223290],
        [1.440, 4.37872256685, 3178.14579056760],
        [1.439, 4.26513521887, 526.77020378780],
        [1.557, 5.43779802371, 81.75213321620],
        [1.656, 6.09667089740, 530.65417294110],
        [1.548, 3.48799710267, 934.94851496820],
        [1.772, 5.82549274759, 909.81873305460],
        [1.615, 1.45018725033, 902.70518605380],
        [1.387, 2.52840497309, 530.44172462000],
        [1.574, 1.89565809136, 437.64389113990],
        [1.459, 3.32546061506, 1041.22268292450],
        [1.377, 0.10015418633, 490.33408917940],
        [1.460, 4.00706825185, 3370.10424500320],
        [1.605, 4.27993020192, 2531.13495725280],
        [1.707, 6.28253681644, 18.15924726470],
        [1.802, 2.23019296374, 2854.64037391020],
        [1.390, 3.76737324192, 1165.65609814550],
        [1.498, 0.17285954362, 1141.13406340540],
        [1.401, 4.81225317549, 1251.34038462480],
        [1.244, 2.83383980283, 124.43341522100],
        [1.320, 5.80675430384, 387.24131496080],
        [1.329, 0.88314574243, 916.93228005540],
        [1.558, 6.17808619637, 983.11585891360],
        [1.243, 0.29239666059, 597.35901666110],
        [1.541, 3.51095241498, 2751.54759969160],
        [1.482, 0.83066678204, 529.16970023280],
        [1.149, 3.91142023857, 99.91138048090],
        [1.114, 3.53339637290, 483.22054217860],
        [1.195, 4.16301075999, 203.00415469950],
        [1.100, 1.74769285223, 497.44763618020],
        [1.458, 5.19315120878, 1592.25428902150],
        [1.123, 1.45270581179, 533.88375078860],
        [1.078, 5.23991792940, 1159.29331067010],
        [1.083, 3.57026506855, 2943.50605412720],
        [1.072, 0.07132659992, 1070.42763045310],
        [1.037, 5.48955598976, 1585.89150154610],
        [1.343, 0.29600445633, 860.30992875280],
        [1.361, 3.46603373194, 107.02492748170],
        [1.061, 2.44580706826, 1048.33622992530],
        [1.002, 5.55216117410, 337.73251065900],
        [0.981, 3.15500987023, 70.84944530420],
        [1.007, 4.11504050436, 501.23677709140],
        [0.965, 5.63719524421, 1603.29998928540],
        [1.083, 4.88373909810, 1166.40685767090],
        [0.953, 2.83352026342, 3583.40334044120],
        [1.060, 3.18542176646, 447.79581952650],
        [1.136, 2.26568590950, 525.49817940060],
        [1.191, 2.25249961404, 106.27416795630],
        [0.884, 4.69777781327, 960.22130923370],
        [1.165, 1.56030440737, 630.33605875840],
        [0.947, 0.50856414717, 842.90144101350],
        [1.011, 0.30814674949, 1593.00504854690],
        [0.924, 2.31939900786, 327.43756992050],
        [0.896, 0.22222521202, 746.92221379570],
        [1.078, 4.78329116086, 2730.20695868920],
        [0.938, 5.42471506763, 1585.14074202070],
        [0.923, 4.44469169065, 9676.48103411560],
        [0.894, 0.26940821870, 2655.56837247380],
        [1.131, 5.46382510304, 224.34479570190],
        [0.808, 0.48295590141, 3377.21779200400],
        [0.809, 4.14122746067, 114.13847448250],
        [0.864, 1.83217006136, 4.66586644600],
        [1.106, 2.60444312553, 209.36694217490],
        [0.790, 0.11493626208, 460.53844081980],
        [0.799, 1.60426497590, 5223.69391980220],
        [0.933, 0.30976125598, 685.47393735270],
        [1.053, 5.23433104008, 842.15068148810],
        [0.846, 3.02878393490, 5746.27133789600],
        [0.799, 2.08457026425, 77734.01845962799],
        [0.820, 0.99821486743, 373.01422095920],
        [0.892, 5.36446426391, 827.92358748650],
        [0.821, 3.53889274951, 498.67147645760],
        [0.741, 1.32379374647, 530.21222995640],
        [0.790, 2.88034567513, 938.12990870590],
        [0.842, 3.39449778904, 484.44438245600],
        [0.785, 0.57841470897, 850.01498801430],
        [0.759, 3.82014112009, 6283.07584999140],
        [0.954, 2.94534072982, 462.02291352810],
        [0.767, 3.33725133157, 99.16062095550],
        [0.810, 4.69425300466, 2228.97018159780],
        [0.700, 1.72050221502, 775.23338944700],
        [0.764, 4.91747674296, 1670.82502850000],
        [0.724, 6.08692841992, 2281.23049651060],
        [0.711, 4.82250918143, 11.77941344680],
        [0.692, 2.63705354662, 6.59228213900],
        [0.771, 3.87410612014, 9690.70812811720],
        [0.906, 2.47189948442, 3274.12501778540],
        [0.781, 1.25357484582, 202.25339517410],
        [0.757, 3.78079814332, 2818.03500860600],
        [0.756, 4.28312053897, 2803.80791460440],
        [0.663, 5.27704405712, 4532.57894941100],
        [0.759, 5.45358686570, 9683.59458111640],
        [0.698, 5.43712520216, 565.11568774670],
        [0.709, 3.71117647887, 3686.49611465980],
        [0.677, 4.27891183416, 25028.52121138500],
        [0.643, 1.40239510103, 9161.01716302260],
        [0.656, 0.60909845504, 835.78789401270],
        [0.635, 5.75373871128, 429.77958461370],
        [0.702, 6.10412979847, 4635.67172362960],
        [0.627, 3.03666956129, 2840.41327990860],
        [0.802, 4.18688054701, 5753.38488489680],
        [0.838, 4.51386507097, 1069.67687092770],
        [0.633, 4.37183361444, 5.41662597140],
        [0.652, 5.79409889124, 1061.82961074400],
        [0.638, 2.18896270346, 313.21047591890],
        [0.827, 5.94231186039, 1457.52593306200],
        [0.678, 2.45013730979, 5760.49843189760],
        [0.814, 4.89578791170, 1567.73225428140],
        [0.624, 0.61631100566, 1176.70179840940],
        [0.600, 3.20918322285, 1098.73880610440],
        [0.717, 1.82349064490, 3171.03224356680],
        [0.651, 4.14419317491, 2847.52682690940],
        [0.629, 1.75272560843, 92.04707395470],
        [0.626, 3.53146082217, 3067.93946934820],
        [0.667, 4.22974611158, 4539.69249641180],
        [0.565, 0.99416346033, 1894.41906467650],
        [0.752, 0.46063700150, 635.23141986800],
        [0.622, 1.98136818407, 25565.32572348040],
        [0.614, 2.48275371627, 25551.09862947879],
        [0.560, 1.40733893388, 446.31134681820],
        [0.558, 4.37217796469, 1057.89745748090],
        [0.628, 4.65037810102, 6275.96230299060],
        [0.659, 2.41470950463, 195.13984817330],
        [0.616, 2.08837621877, 10.03430830760],
        [0.692, 3.13229025530, 7.63481186260],
        [0.685, 4.18539472904, 46.47042291600],
        [0.624, 0.02693303471, 1493.09366806600],
        [0.594, 2.13375704438, 121.25202148330],
        [0.508, 2.13584300710, 1.69692102940],
        [0.674, 1.47570122611, 4694.00295470760],
        [0.559, 4.48852017557, 531.17543780290],
        [0.640, 3.10239233469, 11.04570026390],
        [0.496, 1.29000001439, 927.83496796740],
        [0.587, 3.30651435298, 600.54041039880],
        [0.582, 0.44540948860, 113.38771495710],
        [0.492, 4.83275232000, 9492.14631500480],
        [0.549, 4.34579166146, 3046.59882834580],
        [0.576, 1.22846846364, 1514.29129671650],
        [0.593, 5.86079640612, 524.06189080210],
        [0.510, 2.62557031270, 529.85102378900],
        [0.489, 6.26855707323, 3693.60966166060],
        [0.480, 0.30754294369, 528.94020556920],
        [0.582, 3.51934668795, 1056.93424963440],
        [0.493, 5.52699906925, 512.28247735530],
        [0.481, 2.99681040149, 9153.90361602180],
        [0.562, 3.73437025868, 2015.67108615980],
        [0.458, 3.86646994292, 11.30633269480],
        [0.457, 1.80238019931, 3281.23856478620],
        [0.453, 6.17995938655, 1059.33374607940],
        [0.551, 0.13794958618, 1912.57831194120],
        [0.446, 5.53828660924, 2332.06295581640],
        [0.444, 5.06219342598, 7.86430652620],
        [0.461, 0.16951411708, 26087.90314157420],
        [0.439, 4.14986379679, 1151.42900414390],
        [0.614, 5.42289673768, 2090.30967237520],
        [0.488, 3.71681959056, 447.93883187840],
        [0.592, 2.91424148255, 8624.21265092720],
        [0.433, 2.55336268329, 1064.04779663520],
        [0.449, 5.24955106938, 10213.28554621100],
        [0.510, 5.81591864532, 529.53090640020],
        [0.435, 5.34355963629, 560.71045373160],
        [0.449, 0.72330388784, 2758.66114669240],
        [0.430, 0.94519103478, 6.36278747540],
        [0.563, 6.19175228344, 1884.12412393800],
        [0.443, 3.39246520261, 1152.17976366930],
        [0.430, 1.28652623263, 505.31194270640],
        [0.422, 5.12631540623, 944.98282327580],
        [0.464, 2.90444584145, 398.14400287280],
        [0.410, 1.24248975309, 5069.38346150640],
        [0.411, 2.95117124177, 4326.39340097380],
        [0.418, 5.15499986314, 1173.52040467170],
        [0.412, 2.98125446330, 554.06998748280],
        [0.403, 0.34381388674, 32.24332891440],
        [0.402, 5.88926765351, 1570.91364801910],
        [0.505, 1.49028912471, 3782.47534187760],
        [0.447, 0.03952029309, 245.54242435240],
        [0.453, 3.09458004153, 1059.43011429900],
        [0.411, 3.21727542472, 1475.68518032670],
        [0.426, 3.12237794195, 12566.15169998280],
        [0.434, 3.59362426939, 3259.89792378380],
        [0.398, 4.91510709622, 4120.20785253660],
        [0.399, 4.67075122011, 234.63973644040],
        [0.386, 4.81320787761, 970.51624997220],
        [0.427, 3.21176085113, 977.48678462110],
        [0.411, 4.31566962034, 757.21715453420],
        [0.392, 1.86527946688, 885.43971066640],
        [0.416, 3.81408093105, 3156.80514956520],
    ],
    # R2
    [
        [79644.833, 1.35865896596, 529.69096509460],
        [8251.618, 5.77773935444, 522.57741809380],
        [7029.864, 3.27476965833, 536.80451209540],
        [5314.006, 1.83835109712, 1059.38193018920],
        [1860.833, 2.97682139367, 7.11354700080],
        [836.267, 4.19889881718, 419.48464387520],
        [964.466, 5.48031822015, 515.46387109300],
        [406.453, 3.78250730354, 1066.49547719000],
        [426.570, 2.22753101795, 639.89728631400],
        [377.316, 2.24248352873, 1589.07289528380],
        [497.920, 3.14159265359, 0.00000000000],
        [339.043, 6.12690864038, 625.67019231240],
        [362.943, 5.36761847267, 206.18554843720],
        [342.048, 6.09922969324, 1052.26838318840],
        [279.920, 4.26162555827, 412.37109687440],
        [332.578, 0.00328961161, 426.59819087600],
        [229.777, 0.70530766213, 735.87651353180],
        [200.783, 3.06850623368, 543.91805909620],
        [199.807, 4.42884165317, 103.09277421860],
        [257.290, 0.96295364983, 632.78373931320],
        [138.606, 2.93235671606, 14.22709400160],
        [113.535, 0.78713911289, 728.76296653100],
        [86.025, 5.14434751994, 323.50541665740],
        [94.565, 1.70498041073, 838.96928775040],
        [83.469, 0.05834873484, 309.27832265580],
        [75.198, 1.60495195911, 956.28915597060],
        [70.451, 1.50988357484, 213.29909543800],
        [80.328, 2.98122361797, 742.99006053260],
        [56.203, 0.95534810533, 1162.47470440780],
        [61.649, 6.10137889854, 1045.15483618760],
        [66.572, 5.47307178077, 199.07200143640],
        [50.057, 2.72063162317, 532.87235883230],
        [51.904, 5.58435625607, 942.06206196900],
        [39.833, 5.94566506227, 95.97922721780],
        [44.548, 5.52445621411, 508.35032409220],
        [44.282, 0.27118152557, 526.50957135690],
        [29.944, 0.93641735919, 1155.36115740700],
        [28.412, 2.87835720211, 525.75881183150],
        [26.330, 4.26891877269, 1596.18644228460],
        [27.039, 2.80607741398, 1169.58825140860],
        [27.477, 2.64841266238, 2118.76386037840],
        [22.705, 0.17830004133, 302.16477565500],
        [29.347, 1.78589692350, 831.85574074960],
        [19.991, 0.04328951895, 949.17560896980],
        [19.906, 1.16072627347, 533.62311835770],
        [21.714, 1.88820231818, 1272.68102562720],
        [17.581, 4.14974757919, 846.08283475120],
        [17.085, 5.89188996975, 1258.45393162560],
        [21.407, 4.35468497204, 316.39186965660],
        [21.295, 0.54429472455, 1265.56747862640],
        [19.859, 0.06453825800, 1581.95934828300],
        [17.025, 0.53383755278, 1368.66025284500],
        [12.804, 3.90044242142, 433.71173787680],
        [13.072, 0.79468040717, 110.20632121940],
        [11.945, 0.40671403646, 1361.54670584420],
        [11.695, 4.44394618065, 405.25754987360],
        [11.979, 2.22872778682, 220.41264243880],
        [9.633, 6.01002272123, 853.19638175200],
        [10.163, 0.99504635158, 1471.75302706360],
        [8.977, 1.60328709409, 1692.16566950240],
        [8.701, 3.52167876799, 1073.60902419080],
        [8.314, 5.60169732564, 1574.84580128220],
        [8.958, 6.26708748901, 519.39602435610],
        [7.828, 0.65241611799, 1478.86657406440],
        [7.833, 0.17920601344, 1685.05212250160],
        [7.451, 0.88421084942, 88.86568021700],
        [7.320, 0.89341249264, 721.64941953020],
        [9.135, 1.51210840939, 1148.24761040620],
        [6.110, 2.50080005128, 3.18139373770],
        [7.037, 4.44127496638, 330.61896365820],
        [5.163, 2.79219166952, 21.34064100240],
        [5.079, 2.97991736844, 1375.77379984580],
        [4.930, 0.04683167622, 1677.93857550080],
        [4.664, 2.28007273876, 1485.98012106520],
        [4.692, 0.86220230505, 3.93215326310],
        [5.307, 0.85008578245, 1788.14489672020],
        [4.239, 0.40758287124, 629.60234557550],
        [4.230, 1.61046658091, 635.96513305090],
        [3.627, 2.71151441113, 551.03160609700],
        [3.314, 0.55067236587, 1795.25844372100],
        [4.409, 1.28127751050, 1464.63948006280],
        [3.270, 1.18744032691, 1905.46476494040],
        [3.226, 6.18716071251, 1038.04128918680],
        [3.103, 6.22971614425, 2001.44399215820],
        [3.410, 2.44624067925, 539.98590583310],
        [3.174, 5.54870592599, 191.95845443560],
        [2.590, 3.24430559059, 1062.56332392690],
        [2.614, 0.55149554149, 2104.53676637680],
        [2.174, 5.32613824409, 1891.23767093880],
        [2.659, 4.82459974220, 416.30325013750],
        [2.187, 1.71707514653, 628.85158605010],
        [2.263, 6.19233486371, 1994.33044515740],
        [2.328, 4.28236795066, 963.40270297140],
        [2.579, 0.03256542251, 1898.35121793960],
        [2.077, 3.32602157426, 1699.27921650320],
        [2.529, 2.39697505835, 227.52618943960],
        [2.468, 0.06551346218, 750.10360753340],
        [1.989, 0.29206371261, 636.71589257630],
        [1.927, 0.32286661566, 295.05122865420],
        [1.904, 3.43534792123, 647.01083331480],
        [1.940, 0.29170673525, 2111.65031337760],
        [1.880, 3.14403615586, 611.44309831080],
        [2.324, 1.94960720763, 824.74219374880],
        [1.854, 4.71794950485, 2125.87740737920],
        [2.547, 1.23908353090, 2221.85663459700],
        [1.814, 1.60250861074, 2008.55753915900],
        [1.611, 5.83466560322, 422.66603761290],
        [1.667, 2.32455940876, 440.82528487760],
        [1.622, 0.36650974375, 1056.20053645150],
        [1.624, 2.42139677881, 10.29494073850],
        [1.622, 3.51892791175, 1055.44977692610],
        [1.606, 5.76205763975, 117.31986822020],
        [1.646, 5.88662636573, 2317.83586181480],
        [2.026, 4.61781314145, 423.41679713830],
        [2.098, 1.04559231028, 1781.03134971940],
        [1.868, 1.12487729469, 618.55664531160],
        [1.885, 2.78775930564, 1802.37199072180],
        [1.445, 0.08308050305, 1382.88734684660],
        [1.797, 3.00776822706, 2648.45482547300],
        [1.422, 0.17649746278, 2420.92863603340],
        [1.129, 1.59030291320, 380.12776796000],
        [1.126, 4.19989673600, 547.85021235930],
        [1.186, 5.98943062173, 2310.72231481400],
        [1.108, 4.22655117757, 934.94851496820],
        [1.259, 1.19687222266, 1063.31408345230],
        [1.072, 3.86169004168, 1603.29998928540],
        [0.946, 5.59968097387, 99.91138048090],
        [0.937, 1.03083276760, 81.75213321620],
        [0.938, 6.18136092771, 945.99421523210],
        [0.908, 2.54355964041, 6283.07584999140],
        [0.874, 5.21903196047, 2207.62954059540],
        [0.874, 6.01240284465, 511.53171782990],
        [1.188, 0.75698357968, 2097.42321937600],
        [0.789, 3.91035208173, 10213.28554621100],
        [1.000, 1.34667100304, 732.69511979410],
        [0.952, 1.55355777420, 2324.94940881560],
        [0.811, 5.00475553271, 319.57326339430],
        [0.763, 3.98527559630, 337.73251065900],
        [0.880, 1.14789972199, 952.35700270750],
        [0.780, 4.69463316930, 5746.27133789600],
        [0.910, 0.08774541571, 2737.32050569000],
        [0.773, 0.77131695762, 5760.49843189760],
        [0.764, 6.11686539353, 9676.48103411560],
        [0.758, 2.19350719860, 9690.70812811720],
        [0.671, 1.19532387143, 124.43341522100],
        [0.661, 5.99578306627, 501.23677709140],
        [0.729, 0.65312263578, 2538.24850425360],
        [0.825, 2.70770030205, 3370.10424500320],
        [0.670, 5.44169923277, 107.02492748170],
        [0.739, 1.14609907817, 2641.34127847220],
        [0.866, 3.02831268213, 3046.59882834580],
        [0.718, 4.83684196454, 860.30992875280],
        [0.813, 6.01229270247, 2214.74308759620],
        [0.746, 1.12371143332, 739.80866679490],
        [0.741, 5.93171662010, 2634.22773147140],
        [0.667, 0.89885058003, 106.27416795630],
        [0.573, 2.42701822581, 739.05790726950],
        [0.734, 0.72837704619, 1354.43315884340],
        [0.662, 2.21768976390, 2015.67108615980],
        [0.782, 2.52401202862, 3679.38256765900],
        [0.779, 2.38608991574, 3267.01147078460],
        [0.553, 1.85211127676, 453.42489381900],
        [0.701, 4.23431087374, 9683.59458111640],
        [0.571, 2.98435419019, 1262.38608488870],
        [0.621, 1.24462887440, 3803.81598288000],
        [0.563, 5.99845316446, 1049.08698945070],
        [0.538, 4.92334194042, 447.79581952650],
        [0.534, 0.99911551571, 462.02291352810],
        [0.541, 6.19275150397, 1987.21689815660],
        [0.511, 3.28553278370, 4.66586644600],
        [0.539, 5.33214565622, 2751.54759969160],
        [0.651, 5.12199308959, 3156.80514956520],
        [0.483, 3.03782387056, 3281.23856478620],
        [0.476, 2.17592053936, 149.56319713460],
        [0.510, 5.35664230912, 9.56122755560],
        [0.490, 1.57324553106, 1251.34038462480],
        [0.467, 5.92343423840, 203.00415469950],
        [0.528, 5.81786945766, 2627.11418447060],
        [0.447, 3.51498961805, 18.15924726470],
        [0.429, 0.16627197188, 74.78159856730],
        [0.497, 0.30985248432, 2428.04218303420],
        [0.516, 3.89424540015, 2516.90786325120],
        [0.519, 2.43126348834, 3686.49611465980],
        [0.404, 2.77840802846, 7.16173111060],
        [0.533, 4.77083438961, 3473.19701922180],
        [0.515, 3.54549816613, 3178.14579056760],
        [0.533, 5.61415688189, 2524.02141025200],
        [0.458, 4.91616403047, 3067.93946934820],
        [0.400, 3.13887720912, 540.73666535850],
        [0.378, 0.86122450940, 525.02509864860],
    ],
    # R3
    [
        [3519.257, 6.05800633846, 529.69096509460],
        [1073.239, 1.67321345760, 536.80451209540],
        [915.666, 1.41329676116, 522.57741809380],
        [341.593, 0.52296542656, 1059.38193018920],
        [254.893, 1.19625473533, 7.11354700080],
        [221.512, 0.95225226237, 515.46387109300],
        [69.078, 2.26885282314, 1066.49547719000],
        [89.729, 3.14159265359, 0.00000000000],
        [57.827, 1.41389745339, 543.91805909620],
        [57.653, 0.52580117593, 639.89728631400],
        [51.079, 5.98016364677, 412.37109687440],
        [46.935, 1.57864237959, 625.67019231240],
        [42.824, 6.11689609099, 419.48464387520],
        [37.477, 1.18262762330, 14.22709400160],
        [33.816, 1.66671706951, 1052.26838318840],
        [31.195, 1.04290245896, 1589.07289528380],
        [30.023, 4.63236245032, 426.59819087600],
        [33.531, 0.84784977903, 206.18554843720],
        [20.804, 2.50071243814, 728.76296653100],
        [14.466, 0.96040197071, 508.35032409220],
        [12.969, 1.50233788550, 1045.15483618760],
        [11.654, 3.55513510121, 323.50541665740],
        [12.319, 2.60952614503, 735.87651353180],
        [15.023, 0.89136998434, 199.07200143640],
        [11.160, 1.79041437555, 309.27832265580],
        [10.554, 6.27845112678, 956.28915597060],
        [9.812, 6.26016859519, 103.09277421860],
        [9.301, 3.45126812476, 838.96928775040],
        [6.672, 1.87004905364, 302.16477565500],
        [7.442, 1.28047007623, 742.99006053260],
        [7.178, 0.92022189637, 942.06206196900],
        [5.577, 1.37980792905, 95.97922721780],
        [6.834, 3.45228722967, 831.85574074960],
        [4.632, 2.82934545414, 1596.18644228460],
        [3.969, 1.21290005054, 1169.58825140860],
        [3.869, 5.99495313698, 213.29909543800],
        [3.551, 6.10714791535, 405.25754987360],
        [2.943, 2.32831075458, 1155.36115740700],
        [2.442, 1.86965213405, 532.87235883230],
        [2.410, 0.42627205128, 220.41264243880],
        [2.289, 1.94941487274, 1073.60902419080],
        [2.274, 0.09211517505, 632.78373931320],
        [2.189, 1.58907745204, 2118.76386037840],
        [2.387, 5.97080671477, 1162.47470440780],
        [2.104, 1.06751462671, 21.34064100240],
        [2.128, 1.51119399925, 1258.45393162560],
        [2.491, 0.35125020737, 1272.68102562720],
        [2.006, 5.94487388360, 110.20632121940],
        [1.980, 2.54989377864, 88.86568021700],
        [2.040, 2.16463966964, 433.71173787680],
        [1.955, 2.70341589777, 721.64941953020],
        [1.670, 4.46255717328, 853.19638175200],
        [1.910, 2.25964760758, 1361.54670584420],
        [1.710, 1.98372066321, 525.75881183150],
        [1.520, 0.11641358425, 949.17560896980],
        [2.003, 3.16520599208, 1148.24761040620],
        [1.710, 2.70850417287, 330.61896365820],
        [1.629, 0.47376028854, 526.50957135690],
        [1.229, 3.01987279595, 963.40270297140],
        [1.671, 0.44352103086, 533.62311835770],
        [1.207, 1.15774089269, 1574.84580128220],
        [1.146, 2.54505851138, 846.08283475120],
        [1.355, 1.17462112647, 1038.04128918680],
        [1.001, 2.70272799283, 519.39602435610],
        [1.372, 0.67467128629, 551.03160609700],
        [0.983, 4.17198081351, 2627.11418447060],
        [1.084, 1.07011164067, 227.52618943960],
        [0.892, 2.92543286761, 1368.66025284500],
        [0.823, 4.86559196955, 611.44309831080],
        [1.136, 1.78981738432, 1581.95934828300],
        [0.897, 4.91073630270, 1670.82502850000],
        [0.908, 3.68804047330, 824.74219374880],
        [0.789, 3.23380893250, 2125.87740737920],
        [0.771, 2.39070707004, 2317.83586181480],
        [0.891, 0.59692950778, 539.98590583310],
        [0.876, 4.52127091462, 750.10360753340],
        [0.802, 0.20759322884, 1141.13406340540],
        [0.850, 0.94145487094, 191.95845443560],
        [0.762, 2.25149516048, 2538.24850425360],
        [0.694, 0.67080348659, 440.82528487760],
        [0.741, 5.79934203525, 1485.98012106520],
        [0.643, 2.48127580335, 1265.56747862640],
        [0.575, 6.13756590872, 1279.79457262800],
        [0.636, 5.51001645505, 2413.81508903260],
        [0.636, 4.40777238491, 1382.88734684660],
        [0.555, 2.18233983981, 1062.56332392690],
        [0.564, 1.92775967119, 2634.22773147140],
        [0.531, 2.04824376019, 295.05122865420],
        [0.541, 2.32424368689, 1471.75302706360],
        [0.697, 2.27179476322, 1699.27921650320],
        [0.546, 1.95774905730, 1677.93857550080],
        [0.465, 4.35550844067, 1692.16566950240],
        [0.508, 2.50298248836, 2207.62954059540],
        [0.496, 5.77087043616, 1478.86657406440],
        [0.440, 5.98661963879, 934.94851496820],
        [0.424, 2.80194129521, 81.75213321620],
        [0.406, 3.93940190897, 316.39186965660],
        [0.506, 0.18719982992, 10.29494073850],
    ],
    # R4
    [
        [128.628, 0.08419309557, 536.80451209540],
        [113.458, 4.24858855779, 529.69096509460],
        [82.650, 3.29754909408, 522.57741809380],
        [37.883, 2.73326611144, 515.46387109300],
        [26.694, 5.69142588558, 7.11354700080],
        [17.650, 5.40012536918, 1059.38193018920],
        [12.612, 6.01560416057, 543.91805909620],
        [9.287, 0.76813946494, 1066.49547719000],
        [8.107, 5.68228065707, 14.22709400160],
        [6.271, 5.12286932534, 639.89728631400],
        [6.978, 1.42751292055, 412.37109687440],
        [5.377, 3.33501947275, 625.67019231240],
        [2.911, 3.40334805052, 1052.26838318840],
        [2.593, 4.16090412984, 728.76296653100],
        [2.562, 2.89802035072, 426.59819087600],
        [2.268, 6.22195938856, 1589.07289528380],
        [2.114, 3.11758855774, 1045.15483618760],
        [1.673, 2.81399290364, 206.18554843720],
        [1.805, 2.60030006919, 199.07200143640],
        [1.823, 1.89432426038, 419.48464387520],
        [1.522, 1.33432648232, 1596.18644228460],
        [1.697, 0.00000000000, 0.00000000000],
        [1.039, 4.41904942302, 956.28915597060],
        [1.161, 5.16181311538, 831.85574074960],
        [0.916, 3.17245716108, 508.35032409220],
        [0.870, 5.79387813500, 1169.58825140860],
        [0.916, 1.87129662931, 1148.24761040620],
        [0.955, 0.66801367802, 1361.54670584420],
        [0.788, 1.47515450553, 1272.68102562720],
        [0.966, 5.47457968043, 220.41264243880],
        [0.788, 2.42252866885, 117.31986822020],
        [0.712, 0.49655897030, 1073.60902419080],
        [0.656, 3.53022740783, 302.16477565500],
        [0.681, 2.84507174340, 191.95845443560],
        [0.771, 2.19893222018, 942.06206196900],
        [0.765, 5.31147257700, 551.03160609700],
        [0.667, 3.72432305249, 88.86568021700],
        [0.534, 1.83172084748, 647.01083331480],
        [0.553, 0.85896003802, 330.61896365820],
        [0.543, 5.26057584439, 21.34064100240],
        [0.584, 3.82243061802, 618.55664531160],
        [0.512, 4.44485521707, 110.20632121940],
        [0.612, 1.59320941864, 3.18139373770],
        [0.631, 1.83863158533, 10.29494073850],
        [0.491, 1.52912023181, 405.25754987360],
        [0.521, 0.24011424451, 433.71173787680],
    ],
    # R5
    [
        [11.188, 4.75249399945, 536.80451209540],
        [4.255, 5.91516229170, 522.57741809380],
        [2.079, 5.56781555864, 515.46387109300],
        [1.908, 4.29659647286, 543.91805909620],
        [1.875, 3.69357495838, 7.11354700080],
        [1.590, 5.49312796166, 1066.49547719000],
        [1.612, 4.13222808529, 1059.38193018920],
        [1.240, 3.77981722506, 14.22709400160],
        [1.033, 4.50671820436, 529.69096509460],
    ],
]
"""This table contains Jupiter's periodic terms (all of them) from the
planetary theory VSOP87 for the radius vector at the equinox of date (taken
from the 'D' solution). In Meeus' book a shortened version can be found in
pages 432-434."""


ORBITAL_ELEM = [
    [34.351519, 3036.3027748, 0.0002233, 0.000000037],          # L
    [5.202603209, 0.0000001913, 0.0, 0.0],                      # a
    [0.04849793, 0.000163225, -0.0000004714, -0.00000000201],   # e
    [1.303267, -0.0054965, 0.00000466, -0.000000002],           # i
    [100.464407, 1.0209774, 0.00040315, 0.000000404],           # Omega
    [14.331207, 1.6126352, 0.00103042, -0.000004464]            # pie
]
"""This table contains the parameters to compute Jupiter's orbital elements for
the mean equinox of date. Based in Table 31.A, page 213"""


ORBITAL_ELEM_J2000 = [
    [34.351519, 3034.9056606, -0.00008501, 0.000000016],    # L
    [1.303267, -0.0019877, 0.0000332, 0.000000097],         # i
    [100.464407, 0.1767232, 0.000907, -0.000007272],        # Omega
    [14.331207, 0.2155209, 0.00072211, -0.000004485]        # pie
]
"""This table contains the parameters to compute Jupiter's orbital elements for
the standard equinox J2000.0. Based on Table 31.B, page 215"""


class Jupiter(object):
    """
    Class Jupiter models that planet.
    """

    @staticmethod
    def geometric_heliocentric_position(epoch, tofk5=True):
        """This method computes the geometric heliocentric position of planet
        Jupiter for a given epoch, using the VSOP87 theory.

        :param epoch: Epoch to compute Jupiter position, as an Epoch object
        :type epoch: :py:class:`Epoch`
        :param tofk5: Whether or not the small correction to convert to the FK5
            system will be applied or not
        :type tofk5: bool

        :returns: A tuple with the heliocentric longitude and latitude (as
            :py:class:`Angle` objects), and the radius vector (as a float,
            in astronomical units), in that order
        :rtype: tuple
        :raises: TypeError if input values are of wrong type.

        >>> epoch = Epoch(2018, 10, 27.0)
        >>> l, b, r = Jupiter.geometric_heliocentric_position(epoch)
        >>> print(round(l.to_positive(), 4))
        241.5873
        >>> print(round(b, 4))
        0.8216
        >>> print(round(r, 5))
        5.36848
        """

        return geometric_vsop_pos(epoch, VSOP87_L, VSOP87_B, VSOP87_R, tofk5)

    @staticmethod
    def apparent_heliocentric_position(epoch):
        """This method computes the apparent heliocentric position of planet
        Jupiter for a given epoch, using the VSOP87 theory.

        :param epoch: Epoch to compute Jupiter position, as an Epoch object
        :type epoch: :py:class:`Epoch`

        :returns: A tuple with the heliocentric longitude and latitude (as
            :py:class:`Angle` objects), and the radius vector (as a float,
            in astronomical units), in that order
        :rtype: tuple
        :raises: TypeError if input values are of wrong type.
        """

        return apparent_vsop_pos(epoch, VSOP87_L, VSOP87_B, VSOP87_R)

    @staticmethod
    def orbital_elements_mean_equinox(epoch):
        """This method computes the orbital elements of Jupiter for the mean
        equinox of the date for a given epoch.

        :param epoch: Epoch to compute orbital elements, as an Epoch object
        :type epoch: :py:class:`Epoch`

        :returns: A tuple containing the following six orbital elements:
            - Mean longitude of the planet (Angle)
            - Semimajor axis of the orbit (float, astronomical units)
            - eccentricity of the orbit (float)
            - inclination on the plane of the ecliptic (Angle)
            - longitude of the ascending node (Angle)
            - argument of the perihelion (Angle)
        :rtype: tuple
        :raises: TypeError if input values are of wrong type.

        >>> epoch = Epoch(2065, 6, 24.0)
        >>> l, a, e, i, ome, arg = Jupiter.orbital_elements_mean_equinox(epoch)
        >>> print(round(l, 6))
        222.433723
        >>> print(round(a, 8))
        5.20260333
        >>> print(round(e, 7))
        0.0486046
        >>> print(round(i, 6))
        1.29967
        >>> print(round(ome, 5))
        101.13309
        >>> print(round(arg, 6))
        -85.745532
        """

        return orbital_elements(epoch, ORBITAL_ELEM, ORBITAL_ELEM)

    @staticmethod
    def orbital_elements_j2000(epoch):
        """This method computes the orbital elements of Jupiter for the
        standard equinox J2000.0 for a given epoch.

        :param epoch: Epoch to compute orbital elements, as an Epoch object
        :type epoch: :py:class:`Epoch`

        :returns: A tuple containing the following six orbital elements:
            - Mean longitude of the planet (Angle)
            - Semimajor axis of the orbit (float, astronomical units)
            - eccentricity of the orbit (float)
            - inclination on the plane of the ecliptic (Angle)
            - longitude of the ascending node (Angle)
            - argument of the perihelion (Angle)
        :rtype: tuple
        :raises: TypeError if input values are of wrong type.

        >>> epoch = Epoch(2065, 6, 24.0)
        >>> l, a, e, i, ome, arg = Jupiter.orbital_elements_j2000(epoch)
        >>> print(round(l, 6))
        221.518802
        >>> print(round(a, 8))
        5.20260333
        >>> print(round(e, 7))
        0.0486046
        >>> print(round(i, 6))
        1.30198
        >>> print(round(ome, 5))
        100.58051
        >>> print(round(arg, 6))
        -86.107875
        """

        return orbital_elements(epoch, ORBITAL_ELEM, ORBITAL_ELEM_J2000)

    @staticmethod
    def geocentric_position(epoch):
        """This method computes the geocentric position of Jupiter (right
        ascension and declination) for the given epoch, as well as the
        elongation angle.

        :param epoch: Epoch to compute geocentric position, as an Epoch object
        :type epoch: :py:class:`Epoch`

        :returns: A tuple containing the right ascension, the declination and
            the elongation angle as Angle objects
        :rtype: tuple
        :raises: TypeError if input value is of wrong type.

        >>> epoch = Epoch(1992, 12, 20.0)
        >>> ra, dec, elon = Jupiter.geocentric_position(epoch)
        >>> print(ra.ra_str(n_dec=1))
        12h 47' 9.6''
        >>> print(dec.dms_str(n_dec=1))
        -3d 41' 55.3''
        >>> print(elon.dms_str(n_dec=1))
        76d 2' 26.0''
        """

        # First check that input value is of correct types
        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input type")
        # Compute the heliocentric position of Jupiter
        l, b, r = Jupiter.geometric_heliocentric_position(epoch, tofk5=False)
        # Compute the heliocentric position of the Earth
        l0, b0, r0 = Earth.geometric_heliocentric_position(epoch, tofk5=False)
        # Convert to radians
        lr = l.rad()
        br = b.rad()
        l0r = l0.rad()
        b0r = b0.rad()
        # Compute first iteration
        x = r * cos(br) * cos(lr) - r0 * cos(b0r) * cos(l0r)
        y = r * cos(br) * sin(lr) - r0 * cos(b0r) * sin(l0r)
        z = r * sin(br) - r0 * sin(b0r)
        delta = sqrt(x * x + y * y + z * z)
        tau = 0.0057755183 * delta
        # Adjust the epoch for light-time
        epoch -= tau
        # Compute again Jupiter coordinates with this correction
        l, b, r = Jupiter.geometric_heliocentric_position(epoch, tofk5=False)
        # Compute second iteration
        lr = l.rad()
        br = b.rad()
        x = r * cos(br) * cos(lr) - r0 * cos(b0r) * cos(l0r)
        y = r * cos(br) * sin(lr) - r0 * cos(b0r) * sin(l0r)
        z = r * sin(br) - r0 * sin(b0r)
        # Compute longitude and latitude
        lamb = atan2(y, x)
        beta = atan2(z, sqrt(x * x + y * y))
        # Now, let's compute the aberration effect
        t = (epoch - JDE2000) / 36525
        e = 0.016708634 + t * (-0.000042037 - t * 0.0000001267)
        pie = 102.93735 + t * (1.71946 + t * 0.00046)
        pie = radians(pie)
        lon = l0 + 180.0
        lon = lon.rad()
        k = 20.49552    # The constant of aberration
        deltal1 = k * (-cos(lon - lamb) + e * cos(pie - lamb)) / cos(beta)
        deltab1 = -k * sin(beta) * (sin(lon - lamb) - e * sin(pie - lamb))
        deltal1 = Angle(0, 0, deltal1)
        deltab1 = Angle(0, 0, deltab1)
        # Correction to FK5 system
        lamb = Angle(lamb, radians=True)
        lamb = lamb.to_positive()
        beta = Angle(beta, radians=True)
        l_prime = lamb - t * (1.397 + t * 0.00031)
        deltal2 = Angle(0, 0, -0.09033)
        a = 0.03916 * (cos(l_prime.rad()) + sin(l_prime.rad()))
        a = a * tan(b.rad())
        deltal2 += Angle(0, 0, a)
        deltab2 = 0.03916 * (cos(l_prime.rad()) - sin(l_prime.rad()))
        deltab2 = Angle(0, 0, deltab2)
        # Apply the corrections
        lamb = lamb + deltal1 + deltal2
        beta = beta + deltab1 + deltab2
        # Correction for nutation
        dpsi = nutation_longitude(epoch)
        lamb += dpsi
        e = true_obliquity(epoch)
        ra, dec = ecliptical2equatorial(lamb, beta, e)
        # Let's compute the elongation angle
        lons, lats, rs = Sun.apparent_geocentric_position(epoch)
        lambr = lamb.rad()
        lsr = lons.rad()
        betar = beta.rad()
        elon = acos(cos(betar) * cos(lambr - lsr))
        elon = Angle(elon, radians=True)
        return ra, dec, elon

    @staticmethod
    def conjunction(epoch):
        """This method computes the time of the conjunction closest to the
        given epoch.

        :param epoch: Epoch close to the desired conjunction
        :type epoch: :py:class:`Epoch`

        :returns: The time when the conjunction happens, as an Epoch
        :rtype: :py:class:`Epoch`
        :raises: TypeError if input value is of wrong type.
        :raises: ValueError if input epoch outside the -2000/4000 range.

        >>> epoch = Epoch(1993, 10, 1.0)
        >>> conj = Jupiter.conjunction(epoch)
        >>> y, m, d = conj.get_date()
        >>> print(y)
        1993
        >>> print(m)
        10
        >>> print(round(d, 4))
        18.3341
        """

        # First check that input value is of correct types
        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input type")
        # Check that the input epoch is within valid range
        y = epoch.year()
        if y < -2000.0 or y > 4000.0:
            raise ValueError("Epoch outside the -2000/4000 range")
        # Set some specific constants for Jupiter's conjunction
        a = 2451671.186
        b = 398.884046
        m0 = 121.898
        m1 = 33.140229
        k = round((365.2425 * y + 1721060.0 - a) / b)
        jde0 = a + k * b
        m = m0 + k * m1
        m = Angle(m).to_positive()
        m = m.rad()
        t = (jde0 - 2451545.0) / 36525.0
        # Compute an auxiliary angle
        aa = 82.74 + 40.76 * t
        aa = Angle(aa).rad()    # Convert to radians
        corr = (0.1027 + t * (0.0002 - t * 0.00009) +
                sin(m) * (-2.2637 + t * (0.0163 - t * 0.00003)) +
                cos(m) * (-6.154 + t * (-0.021 + t * 0.00008)) +
                sin(2.0 * m) * (-0.2021 + t * (-0.0017 + t * 0.00001)) +
                cos(2.0 * m) * (0.131 - t * 0.0008) +
                sin(3.0 * m) * (0.0086) +
                cos(3.0 * m) * (0.0087 + t * 0.0002) +
                sin(aa) * (0.0 + t * (0.0144 - t * 0.00008)) +
                cos(aa) * (0.3642 + t * (-0.0019 - t * 0.00029)))
        to_return = jde0 + corr
        return Epoch(to_return)

    @staticmethod
    def opposition(epoch):
        """This method computes the time of the opposition closest to the given
        epoch.

        :param epoch: Epoch close to the desired opposition
        :type epoch: :py:class:`Epoch`

        :returns: The time when the opposition happens, as an Epoch
        :rtype: :py:class:`Epoch`
        :raises: TypeError if input value is of wrong type.
        :raises: ValueError if input epoch outside the -2000/4000 range.

        >>> epoch = Epoch(-6, 9, 1.0)
        >>> oppo = Jupiter.opposition(epoch)
        >>> y, m, d = oppo.get_date()
        >>> print(y)
        -6
        >>> print(m)
        9
        >>> print(round(d, 4))
        15.2865
        """

        # First check that input value is of correct types
        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input type")
        # Check that the input epoch is within valid range
        y = epoch.year()
        if y < -2000.0 or y > 4000.0:
            raise ValueError("Epoch outside the -2000/4000 range")
        # Set some specific constants for Jupiter's opposition
        a = 2451870.628
        b = 398.884046
        m0 = 318.4681
        m1 = 33.140229
        k = round((365.2425 * y + 1721060.0 - a) / b)
        jde0 = a + k * b
        m = m0 + k * m1
        m = Angle(m).to_positive()
        m = m.rad()
        t = (jde0 - 2451545.0) / 36525.0
        # Compute an auxiliary angle
        aa = 82.74 + 40.76 * t
        aa = Angle(aa).rad()    # Convert to radians
        corr = (-0.1029 - t * t * 0.00009 +
                sin(m) * (-1.9658 + t * (-0.0056 + t * 0.00007)) +
                cos(m) * (6.1537 + t * (0.021 - t * 0.00006)) +
                sin(2.0 * m) * (-0.2081 - t * 0.0013) +
                cos(2.0 * m) * (-0.1116 - t * 0.001) +
                sin(3.0 * m) * (0.0074 + t * 0.0001) +
                cos(3.0 * m) * (-0.0097 - t * 0.0001) +
                sin(aa) * (0.0 + t * (0.0144 - t * 0.00008)) +
                cos(aa) * (0.3642 + t * (-0.0019 - t * 0.00029)))
        to_return = jde0 + corr
        return Epoch(to_return)

    @staticmethod
    def station_longitude_1(epoch):
        """This method computes the time of the 1st station in longitude
        (i.e. when the planet is stationary and begins to move westward -
        retrograde - among the starts) closest to the given epoch.

        :param epoch: Epoch close to the desired opposition
        :type epoch: :py:class:`Epoch`

        :returns: Time when the 1st station in longitude happens, as an Epoch
        :rtype: :py:class:`Epoch`
        :raises: TypeError if input value is of wrong type.
        :raises: ValueError if input epoch outside the -2000/4000 range.

        >>> epoch = Epoch(2018, 11, 1.0)
        >>> sta1 = Jupiter.station_longitude_1(epoch)
        >>> y, m, d = sta1.get_date()
        >>> print(y)
        2018
        >>> print(m)
        3
        >>> print(round(d, 4))
        9.1288
        """

        # First check that input value is of correct types
        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input type")
        # Check that the input epoch is within valid range
        y = epoch.year()
        if y < -2000.0 or y > 4000.0:
            raise ValueError("Epoch outside the -2000/4000 range")
        # Set some specific constants for Jupiter's opposition
        a = 2451870.628
        b = 398.884046
        m0 = 318.4681
        m1 = 33.140229
        k = round((365.2425 * y + 1721060.0 - a) / b)
        jde0 = a + k * b
        m = m0 + k * m1
        m = Angle(m).to_positive()
        m = m.rad()
        t = (jde0 - 2451545.0) / 36525.0
        # Compute an auxiliary angle
        aa = 82.74 + 40.76 * t
        aa = Angle(aa).rad()    # Convert to radians
        corr = (-60.367 + t * (-0.0001 - t * 0.00009) +
                sin(m) * (-2.3144 + t * (-0.0124 + t * 0.00007)) +
                cos(m) * (6.7439 + t * (0.0166 - t * 0.00006)) +
                sin(2.0 * m) * (-0.2259 - t * 0.001) +
                cos(2.0 * m) * (-0.1497 - t * 0.0014) +
                sin(3.0 * m) * (0.0105 + t * 0.0001) +
                cos(3.0 * m) * (-0.0098) +
                sin(aa) * (0.0 + t * (0.0144 - t * 0.00008)) +
                cos(aa) * (0.3642 + t * (-0.0019 - t * 0.00029)))
        to_return = jde0 + corr
        return Epoch(to_return)

    @staticmethod
    def station_longitude_2(epoch):
        """This method computes the time of the 2nd station in longitude
        (i.e. when the planet is stationary and begins to move eastward -
        prograde - among the starts) closest to the given epoch.

        :param epoch: Epoch close to the desired opposition
        :type epoch: :py:class:`Epoch`

        :returns: Time when the 1st station in longitude happens, as an Epoch
        :rtype: :py:class:`Epoch`
        :raises: TypeError if input value is of wrong type.
        :raises: ValueError if input epoch outside the -2000/4000 range.

        >>> epoch = Epoch(2018, 11, 1.0)
        >>> sta2 = Jupiter.station_longitude_2(epoch)
        >>> y, m, d = sta2.get_date()
        >>> print(y)
        2018
        >>> print(m)
        7
        >>> print(round(d, 4))
        10.6679
        """

        # First check that input value is of correct types
        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input type")
        # Check that the input epoch is within valid range
        y = epoch.year()
        if y < -2000.0 or y > 4000.0:
            raise ValueError("Epoch outside the -2000/4000 range")
        # Set some specific constants for Jupiter's opposition
        a = 2451870.628
        b = 398.884046
        m0 = 318.4681
        m1 = 33.140229
        k = round((365.2425 * y + 1721060.0 - a) / b)
        jde0 = a + k * b
        m = m0 + k * m1
        m = Angle(m).to_positive()
        m = m.rad()
        t = (jde0 - 2451545.0) / 36525.0
        # Compute an auxiliary angle
        aa = 82.74 + 40.76 * t
        aa = Angle(aa).rad()    # Convert to radians
        corr = (60.3023 + t * (0.0002 - t * 0.00009) +
                sin(m) * (0.3506 + t * (-0.0034 + t * 0.00004)) +
                cos(m) * (5.3635 + t * (0.0247 - t * 0.00007)) +
                sin(2.0 * m) * (-0.1872 - t * 0.0016) +
                cos(2.0 * m) * (-0.0037 - t * 0.0005) +
                sin(3.0 * m) * (0.0012 + t * 0.0001) +
                cos(3.0 * m) * (-0.0096 - t * 0.0001) +
                sin(aa) * (0.0 + t * (0.0144 - t * 0.00008)) +
                cos(aa) * (0.3642 + t * (-0.0019 - t * 0.00029)))
        to_return = jde0 + corr
        return Epoch(to_return)

    @staticmethod
    def perihelion_aphelion(epoch, perihelion=True):
        """This method computes the time of Perihelion (or Aphelion) closer to
        a given epoch.

        :param epoch: Epoch close to the desired Perihelion (or Aphelion)
        :type epoch: :py:class:`Epoch`
        :param peihelion: If True, the epoch of the closest Perihelion is
            computed, if False, the epoch of the closest Aphelion is found.
        :type bool:

        :returns: The epoch of the desired Perihelion (or Aphelion)
        :rtype: :py:class:`Epoch`
        :raises: TypeError if input values are of wrong type.

        >>> epoch = Epoch(2019, 2, 23.0)
        >>> e = Jupiter.perihelion_aphelion(epoch)
        >>> y, m, d, h, mi, s = e.get_full_date()
        >>> print(y)
        2023
        >>> print(m)
        1
        >>> print(d)
        20
        >>> print(h)
        11
        >>> epoch = Epoch(1981, 6, 1.0)
        >>> e = Jupiter.perihelion_aphelion(epoch, perihelion=False)
        >>> y, m, d, h, mi, s = e.get_full_date()
        >>> print(y)
        1981
        >>> print(m)
        7
        >>> print(d)
        28
        >>> print(h)
        6
        """

        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input value")
        # First approximation
        k = 0.0843 * (epoch.year() - 2011.2)
        if perihelion:
            k = round(k)
        else:
            k = round(k + 0.5) - 0.5
        jde = 2455636.936 + k * (4332.897065 - k * 0.0001367)
        # Compute the epochs a month before and after
        jde_before = jde - 30.0
        jde_after = jde + 30.0
        # Compute the Sun-Jupiter distance for each epoch
        l, b, r_b = Jupiter.geometric_heliocentric_position(Epoch(jde_before))
        l, b, r = Jupiter.geometric_heliocentric_position(Epoch(jde))
        l, b, r_a = Jupiter.geometric_heliocentric_position(Epoch(jde_after))
        # Call an interpolation object
        m = Interpolation([jde_before, jde, jde_after], [r_b, r, r_a])
        sol = m.minmax()
        return Epoch(sol)

    @staticmethod
    def passage_nodes(epoch, ascending=True):
        """This function computes the time of passage by the nodes (ascending
        or descending) of Jupiter, nearest to the given epoch.

        :param epoch: Epoch closest to the node passage
        :type epoch: :py:class:`Epoch`
        :param ascending: Whether the time of passage by the ascending (True)
            or descending (False) node will be computed
        :type ascending: bool

        :returns: Tuple containing:
            - Time of passage through the node (:py:class:`Epoch`)
            - Radius vector when passing through the node (in AU, float)
        :rtype: tuple
        :raises: TypeError if input values are of wrong type.

        >>> epoch = Epoch(2019, 1, 1)
        >>> time, r = Jupiter.passage_nodes(epoch)
        >>> year, month, day = time.get_date()
        >>> print(year)
        2025
        >>> print(month)
        9
        >>> print(round(day, 1))
        15.6
        >>> print(round(r, 4))
        5.1729
        """

        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input types")
        # Get the orbital parameters
        l, a, e, i, ome, arg = Jupiter.orbital_elements_mean_equinox(epoch)
        # Compute the time of passage through perihelion
        t = Jupiter.perihelion_aphelion(epoch)
        # Get the time of passage through the node
        time, r = passage_nodes_elliptic(arg, e, a, t, ascending)
        return time, r

    @staticmethod
    def magnitude(sun_dist, earth_dist):
        """This function computes the approximate magnitude of Jupiter.

        :param sun_dist: Distance from Jupiter to Sun, in Astronomical Units
        :type sun_dist: float
        :param earth_dist: Distance Jupiter to Earth, in Astronomical Units
        :type earth_dist: float

        :returns: Jupiter's magnitude
        :rtype: float
        :raises: TypeError if input values are of wrong type.
        """

        if not (isinstance(sun_dist, float) and isinstance(earth_dist, float)):
            raise TypeError("Invalid input types")
        m = -8.93 + 5.0 * log10(sun_dist * earth_dist)
        return round(m, 1)


def main():

    # Let's define a small helper function
    def print_me(msg, val):
        print("{}: {}".format(msg, val))

    # Let's show some uses of Jupiter class
    print("\n" + 35 * "*")
    print("*** Use of Jupiter class")
    print(35 * "*" + "\n")

    # Let's now compute the heliocentric position for a given epoch
    epoch = Epoch(2018, 10, 27.0)
    lon, lat, r = Jupiter.geometric_heliocentric_position(epoch)
    print_me("Geometric Heliocentric Longitude", lon.to_positive())
    print_me("Geometric Heliocentric Latitude", lat)
    print_me("Radius vector", r)

    print("")

    # Compute the geocentric position for 1992/12/20:
    epoch = Epoch(1992, 12, 20.0)
    ra, dec, elon = Jupiter.geocentric_position(epoch)
    print_me("Right ascension", ra.ra_str(n_dec=1))
    print_me("Declination", dec.dms_str(n_dec=1))
    print_me("Elongation", elon.dms_str(n_dec=1))

    print("")

    # Print mean orbital elements for Jupiter at 2065.6.24
    epoch = Epoch(2065, 6, 24.0)
    l, a, e, i, ome, arg = Jupiter.orbital_elements_mean_equinox(epoch)
    print_me("Mean longitude of the planet", round(l, 6))       # 222.433723
    print_me("Semimajor axis of the orbit (UA)", round(a, 8))   # 5.20260333
    print_me("Eccentricity of the orbit", round(e, 7))          # 0.0486046
    print_me("Inclination on plane of the ecliptic", round(i, 6))   # 1.29967
    print_me("Longitude of the ascending node", round(ome, 5))  # 101.13309
    print_me("Argument of the perihelion", round(arg, 6))       # -85.745532

    print("")

    # Compute the time of the conjunction close to 1993/10/1
    epoch = Epoch(1993, 10, 1.0)
    conj = Jupiter.conjunction(epoch)
    y, m, d = conj.get_date()
    d = round(d, 4)
    date = "{}/{}/{}".format(y, m, d)
    print_me("Conjunction date", date)

    # Compute the time of the opposition close to -6/9/1
    epoch = Epoch(-6, 9, 1.0)
    oppo = Jupiter.opposition(epoch)
    y, m, d = oppo.get_date()
    d = round(d, 4)
    date = "{}/{}/{}".format(y, m, d)
    print_me("Opposition date", date)

    print("")

    # Compute the time of the station in longitude #1 close to 2018/11/1
    epoch = Epoch(2018, 11, 1.0)
    sta1 = Jupiter.station_longitude_1(epoch)
    y, m, d = sta1.get_date()
    d = round(d, 4)
    date = "{}/{}/{}".format(y, m, d)
    print_me("Date of station in longitude #1", date)

    # Compute the time of the station in longitude #2 close to 2018/11/1
    epoch = Epoch(2018, 11, 1.0)
    sta2 = Jupiter.station_longitude_2(epoch)
    y, m, d = sta2.get_date()
    d = round(d, 4)
    date = "{}/{}/{}".format(y, m, d)
    print_me("Date of station in longitude #2", date)

    print("")

    # Find the epoch of the Aphelion closer to 1981/6/1
    epoch = Epoch(1981, 6, 1.0)
    e = Jupiter.perihelion_aphelion(epoch, perihelion=False)
    y, m, d, h, mi, s = e.get_full_date()
    peri = str(y) + '/' + str(m) + '/' + str(d) + ' at ' + str(h) + ' hours'
    print_me("The Aphelion closest to 1981/6/1 will happen on", peri)

    print("")

    # Compute the time of passage through an ascending node
    epoch = Epoch(2019, 1, 1)
    time, r = Jupiter.passage_nodes(epoch)
    y, m, d = time.get_date()
    d = round(d, 1)
    print("Time of passage through ascending node: {}/{}/{}".format(y, m, d))
    # 2025/9/15.6
    print("Radius vector at ascending node: {}".format(round(r, 4)))  # 5.1729


if __name__ == "__main__":

    main()
