# -*- coding: utf-8 -*-


# PyMeeus: Python module implementing astronomical algorithms.
# Copyright (C) 2018  Dagoberto Salazar
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.


from math import sin, cos, tan, acos, atan2, sqrt, radians, log10

from pymeeus.Angle import Angle
from pymeeus.Epoch import Epoch, JDE2000
from pymeeus.Coordinates import (
    geometric_vsop_pos, apparent_vsop_pos, orbital_elements,
    nutation_longitude, true_obliquity, ecliptical2equatorial
)
from pymeeus.Earth import Earth
from pymeeus.Sun import Sun


"""
.. module:: Neptune
   :synopsis: Class to model Neptune planet
   :license: GNU Lesser General Public License v3 (LGPLv3)

.. moduleauthor:: Dagoberto Salazar
"""


VSOP87_L = [
    # L0
    [
        [531188633.047, 0.00000000000, 0.00000000000],
        [1798475.509, 2.90101273050, 38.13303563780],
        [1019727.662, 0.48580923660, 1.48447270830],
        [124531.845, 4.83008090682, 36.64856292950],
        [42064.450, 5.41054991607, 2.96894541660],
        [37714.589, 6.09221834946, 35.16409022120],
        [33784.734, 1.24488865578, 76.26607127560],
        [16482.741, 0.00007729261, 491.55792945680],
        [9198.582, 4.93747059924, 39.61750834610],
        [8994.249, 0.27462142569, 175.16605980020],
        [4216.235, 1.98711914364, 73.29712585900],
        [3364.818, 1.03590121818, 33.67961751290],
        [2284.800, 4.20606932559, 4.45341812490],
        [1433.512, 2.78340432711, 74.78159856730],
        [900.240, 2.07606702418, 109.94568878850],
        [744.996, 3.19032530145, 71.81265315070],
        [506.206, 5.74785370252, 114.39910691340],
        [399.552, 0.34972342569, 1021.24889455140],
        [345.195, 3.46186210169, 41.10198105440],
        [306.338, 0.49684039897, 0.52126486180],
        [287.322, 4.50523446022, 0.04818410980],
        [323.004, 2.24815188609, 32.19514480460],
        [340.323, 3.30369900416, 77.75054398390],
        [266.605, 4.88932609483, 0.96320784650],
        [227.079, 1.79713054538, 453.42489381900],
        [244.722, 1.24693337933, 9.56122755560],
        [232.887, 2.50459795017, 137.03302416240],
        [282.170, 2.24565579693, 146.59425171800],
        [251.941, 5.78166597292, 388.46515523820],
        [150.180, 2.99706110414, 5.93789083320],
        [170.404, 3.32390630650, 108.46121608020],
        [151.401, 2.19153094280, 33.94024994380],
        [148.295, 0.85948986145, 111.43016149680],
        [118.672, 3.67706204305, 2.44768055480],
        [101.821, 5.70539236951, 0.11187458460],
        [97.873, 2.80518260528, 8.07675484730],
        [103.054, 4.40441222000, 70.32818044240],
        [103.305, 0.04078966679, 0.26063243090],
        [109.300, 2.41599378049, 183.24281464750],
        [73.938, 1.32805041516, 529.69096509460],
        [77.725, 4.16446516424, 4.19278569400],
        [86.379, 4.22834506045, 490.07345674850],
        [81.536, 5.19908046216, 493.04240216510],
        [71.503, 5.29530386579, 350.33211960040],
        [64.418, 3.54541016050, 168.05251279940],
        [62.570, 0.15028731465, 182.27960680100],
        [58.488, 3.50106873945, 145.10977900970],
        [48.276, 1.11259925628, 112.91463420510],
        [47.229, 4.57373229818, 46.20979048510],
        [39.124, 1.66569356050, 213.29909543800],
        [47.728, 0.12906212461, 484.44438245600],
        [46.858, 3.01699530327, 498.67147645760],
        [38.659, 2.38685706479, 2.92076130680],
        [47.046, 4.49844660400, 173.68158709190],
        [47.565, 2.58404814824, 219.89137757700],
        [44.714, 5.47302733614, 176.65053250850],
        [32.279, 3.45759151220, 30.71067209630],
        [28.249, 4.13282446716, 6.59228213900],
        [24.433, 4.55736848232, 106.97674337190],
        [24.661, 3.67822620786, 181.75834193920],
        [24.505, 1.55095867965, 7.11354700080],
        [21.848, 1.04366818343, 39.09624348430],
        [16.936, 6.10896452834, 44.72531777680],
        [22.169, 2.74932970271, 256.53994050650],
        [16.614, 4.98188930613, 37.61177077600],
        [17.728, 3.55049134167, 1.37259812370],
        [17.347, 2.14069234880, 42.58645376270],
        [14.953, 3.36405649131, 98.89998852460],
        [14.566, 0.69857991985, 1550.93985964600],
        [15.676, 6.22010212025, 454.90936652730],
        [13.243, 5.61712542227, 68.84370773410],
        [14.837, 3.52557245517, 25.60286266560],
        [12.757, 0.04509743861, 11.04570026390],
        [11.988, 4.81687553351, 24.11838995730],
        [11.060, 1.78958277553, 7.42236354150],
        [12.108, 1.87022663714, 79.23501669220],
        [11.698, 0.49005698002, 1.59634729290],
        [10.459, 2.38743199893, 381.35160823740],
        [11.681, 3.85151357766, 218.40690486870],
        [8.744, 0.14168568610, 148.07872442630],
        [9.196, 1.00274090619, 72.07328558160],
        [11.343, 0.81432278263, 525.49817940060],
        [10.097, 5.03383557061, 601.76425067620],
        [8.035, 1.77685723010, 0.21244832110],
        [8.382, 3.07534786987, 1.27202438720],
        [10.803, 2.92081211459, 293.18850343600],
        [7.666, 1.52223325105, 115.88357962170],
        [7.531, 5.37537256533, 5.10780943070],
        [8.691, 4.74352784364, 143.62530630140],
        [10.183, 1.15395455831, 6244.94281435360],
        [8.283, 0.35956716764, 138.51749687070],
        [9.544, 4.02452832984, 152.53214255120],
        [7.274, 4.10937535938, 251.43213107580],
        [7.465, 1.72131945843, 31.01948863700],
        [6.902, 4.62452068308, 2.70831298570],
        [7.094, 5.11528393609, 312.19908396260],
        [7.929, 2.10765101655, 27.08733537390],
        [6.156, 3.50746507109, 28.57180808220],
        [7.134, 2.05292376023, 278.25883401880],
        [8.193, 2.58588219154, 141.22580985640],
        [5.499, 2.09250039025, 1.69692102940],
        [5.279, 4.09390686798, 983.11585891360],
        [6.947, 3.48041784595, 415.29185818120],
        [5.916, 0.68957324226, 62.25142559510],
        [5.925, 4.02504592620, 255.05546779820],
        [4.606, 1.17779101436, 43.24084506850],
        [5.357, 3.63061058987, 5.41662597140],
        [5.918, 2.57693824084, 10175.15251057320],
        [5.482, 3.07979737280, 329.83706636550],
        [3.956, 5.00418696742, 184.72728735580],
        [5.408, 3.31313295602, 528.20649238630],
        [4.767, 4.91981150665, 456.39383923560],
        [3.770, 1.57277409442, 32.71640966640],
        [3.924, 4.92763242635, 180.27386923090],
        [3.707, 4.82965453201, 221.37585028530],
        [3.802, 4.96279204998, 594.65070367540],
        [4.014, 1.63905164030, 40.58071619260],
        [3.061, 0.39713858313, 1.43628859850],
        [3.261, 4.65478978469, 29.22619938800],
        [3.474, 5.65891305944, 395.57870223900],
        [2.918, 5.91079083895, 1.22384027740],
        [3.649, 3.88114678609, 494.52687487340],
        [3.225, 5.57423738665, 1014.13534755060],
        [2.845, 0.56009386585, 144.14657116320],
        [2.848, 0.55423029727, 567.82400073240],
        [3.440, 1.70887250883, 12.53017297220],
        [3.267, 5.63287799820, 488.58898404020],
        [3.107, 5.79335949207, 105.49227066360],
        [2.712, 2.43726364359, 60.76695288680],
        [3.202, 2.21483496593, 41.05379694460],
        [3.134, 4.69665220513, 82.85835341460],
        [3.590, 5.69939670162, 1124.34166877000],
        [2.967, 0.54448940101, 135.54855145410],
        [3.211, 4.19927605853, 291.70403072770],
        [2.899, 5.99669788291, 22.63391724900],
        [3.143, 2.93495725805, 31.23193695810],
        [2.729, 4.62707721219, 5.62907429250],
        [2.513, 5.60391563025, 19.12245511120],
        [2.690, 5.32070128202, 2.00573757010],
        [2.630, 6.00855841124, 37.16982779130],
        [2.296, 6.06934502789, 451.94042111070],
        [2.858, 4.88677262419, 258.02441321480],
        [2.879, 5.12239168488, 38.65430049960],
        [2.270, 2.08634524182, 30.05628079050],
        [2.301, 3.35951602914, 1028.36244155220],
        [3.001, 3.59143817947, 211.81462272970],
        [2.237, 0.38455553470, 3.62333672240],
        [2.901, 3.24755614136, 366.48562929500],
        [2.592, 1.36262641469, 35.42472265210],
        [2.418, 4.93467056526, 47.69426319340],
        [2.089, 5.79838063413, 4.66586644600],
        [2.586, 2.69392971321, 38.18121974760],
        [1.913, 5.53560681085, 149.56319713460],
        [1.971, 6.00790964671, 34.20088237470],
        [2.586, 6.24984047544, 38.08485152800],
        [2.098, 4.57819744766, 1019.76442184310],
        [1.869, 3.85907708723, 911.04257333200],
        [2.486, 5.21235809332, 140.00196957900],
        [1.795, 1.68012868451, 1059.38193018920],
        [2.326, 2.82664069146, 807.94979911340],
        [1.984, 5.54763522932, 1022.73336725970],
        [1.919, 5.10717766499, 216.92243216040],
        [2.004, 5.47811228948, 63.73589830340],
        [2.021, 4.15631916516, 178.13500521680],
        [1.760, 6.00927149342, 172.19711438360],
        [2.140, 2.65037925793, 700.66423920080],
        [1.988, 3.35850272780, 186.21176006410],
        [1.956, 5.01527508588, 294.67297614430],
        [1.966, 4.07957525462, 20.60692781950],
        [1.637, 0.53823942149, 67.35923502580],
        [1.540, 2.62327849119, 41.75637236020],
        [1.810, 5.81430038477, 129.91947716160],
        [1.776, 4.37047808449, 328.35259365720],
        [1.460, 2.63664516309, 2.85707083200],
        [1.388, 2.10598045632, 3.93215326310],
        [1.352, 0.55618245459, 0.65439130580],
        [1.668, 2.77543377384, 16.15350969460],
        [1.338, 0.37643611305, 14.01464568050],
        [1.218, 0.73456434750, 426.59819087600],
        [1.531, 4.54891769768, 526.72201967800],
        [1.610, 3.40993944436, 403.13419222450],
        [1.361, 4.48227243414, 17.63798240290],
        [1.589, 5.59323020112, 3302.47939106200],
        [1.132, 5.64520725360, 151.04766984290],
        [1.357, 4.06399031430, 26.82670294300],
        [1.494, 4.98692049495, 666.72398925700],
        [1.077, 4.30911470250, 0.63313944640],
        [1.042, 6.02756893581, 106.01353552540],
        [1.060, 0.74679491358, 487.36514376280],
        [1.310, 3.78526380930, 386.98068252990],
        [1.342, 4.52685061062, 563.63121503840],
        [0.986, 0.00600924269, 81.37388070630],
        [1.232, 5.17443930901, 331.32153907380],
        [0.929, 4.51267465978, 38.39366806870],
        [0.956, 3.50447791020, 64.95973858080],
        [0.929, 4.43109514438, 37.87240320690],
        [0.926, 6.09803297747, 4.14460158420],
        [0.972, 0.59038366513, 8.90683624980],
        [1.246, 4.69840351226, 389.94962794650],
        [1.009, 5.98451242784, 142.14083359310],
        [1.020, 0.83233892300, 39.35687591520],
        [1.013, 0.37845630298, 36.90919536040],
        [0.940, 2.42688145966, 343.21857259960],
        [0.974, 5.23958752786, 253.57099508990],
        [0.964, 5.09748190218, 357.44566660120],
        [0.835, 1.45568626670, 35.21227433100],
        [1.077, 0.71409061316, 44.07092647100],
        [1.083, 2.27578897621, 6.90109867970],
        [0.938, 5.03471583911, 69.36497259590],
        [1.078, 1.20253141912, 35.68535508300],
        [1.027, 0.18243183397, 84.34282612290],
        [0.764, 4.62720907712, 0.83008140250],
        [1.013, 0.42234855022, 32.45577723550],
        [0.939, 4.50445799766, 365.00115658670],
        [0.756, 0.82872484717, 17.52610781830],
        [0.916, 3.89409205418, 38.24491022240],
        [0.736, 4.78125743795, 5.36844186160],
        [0.762, 0.01897337130, 189.39315380180],
        [0.738, 2.31770478416, 42.32582133180],
        [0.860, 4.82440483506, 210.33015002140],
        [0.888, 3.20360339895, 348.84764689210],
        [0.916, 5.04967792934, 38.02116105320],
        [0.638, 0.63267396269, 244.31858407500],
        [0.636, 1.02615137352, 2080.63082474060],
        [0.774, 5.44432678139, 367.97010200330],
        [0.644, 1.94044989547, 446.31134681820],
        [0.631, 4.82928491724, 460.53844081980],
        [0.855, 3.57592750113, 439.78275515400],
        [0.678, 4.48687912809, 351.81659230870],
        [0.724, 4.89141609280, 119.50691634410],
        [0.594, 0.59315717529, 491.03666459500],
        [0.655, 1.99014093000, 19.01058052660],
        [0.580, 2.57189536188, 492.07919431860],
        [0.694, 0.08328521209, 5.67725840230],
        [0.733, 5.81485239057, 29.74746424980],
        [0.666, 3.42196897591, 179.09821306330],
        [0.678, 0.29428615814, 171.23390653710],
        [0.635, 2.13805182663, 164.12035953630],
        [0.623, 5.61454940380, 285.37238101960],
        [0.529, 1.88063108785, 416.77633088950],
        [0.529, 5.13250788030, 697.74347789400],
        [0.500, 1.49548514415, 704.85702489480],
        [0.487, 4.97772067947, 274.06604832480],
        [0.666, 6.26456825266, 1474.67378837040],
        [0.532, 0.25784352716, 477.33083545520],
        [0.557, 0.71378452161, 80.71948940050],
        [0.556, 2.60791360513, 418.26080359780],
        [0.584, 4.29064541383, 16.67477455640],
        [0.524, 5.42759392280, 290.21955801940],
        [0.524, 0.29054995359, 247.23934538180],
        [0.541, 4.36400580938, 815.06334611420],
        [0.526, 1.66512720297, 97.41551581630],
        [0.497, 4.72640318293, 401.64971951620],
        [0.432, 2.98481475894, 100.38446123290],
        [0.382, 0.28067758468, 8.38557138800],
        [0.424, 6.16774845481, 178.78939652260],
        [0.484, 0.01535318279, 738.79727483860],
        [0.518, 4.48916591410, 875.83029900100],
        [0.506, 5.38611121207, 404.61866493280],
        [0.396, 4.62747640832, 6.15033915430],
        [0.466, 0.23340415764, 120.99138905240],
        [0.409, 3.08849480895, 59.28248017850],
        [0.470, 5.01853200224, 313.68355667090],
        [0.442, 3.68919475089, 457.87831194390],
        [0.384, 3.69499925394, 160.93896579860],
        [0.364, 0.76192181046, 104.00779795530],
        [0.416, 0.26652109651, 103.09277421860],
        [0.401, 4.06530055968, 14.66903698630],
        [0.454, 3.72767803715, 476.43131808350],
        [0.434, 0.33533802200, 984.60033162190],
        [0.340, 0.99915726716, 31.54075349880],
        [0.420, 3.65147769268, 20.49505323490],
        [0.334, 0.35121412008, 1227.43444298860],
        [0.323, 5.45836731979, 918.15612033280],
        [0.407, 4.19457842203, 309.79958751760],
        [0.381, 0.01364856960, 495.49008271990],
        [0.334, 4.05924071124, 8.33738727820],
        [0.380, 3.17063415023, 487.62577619370],
        [0.309, 0.48352303405, 118.02244363580],
        [0.380, 2.70238752925, 134.11226285560],
        [0.362, 4.88985810610, 438.29828244570],
        [0.327, 2.91090790412, 505.78502345840],
        [0.308, 0.96082817124, 21.14944454070],
        [0.288, 1.48123872077, 220.41264243880],
        [0.293, 2.56582281789, 662.53120356300],
        [0.331, 4.37715965811, 180.79513409270],
        [0.326, 2.46104924164, 169.53698550770],
        [0.289, 2.63591886391, 55.77101804070],
        [0.288, 5.02487283285, 1440.73353842660],
        [0.344, 1.48930997270, 166.56804009110],
        [0.266, 0.63672427386, 79.18683258240],
        [0.268, 5.02354540478, 377.41945497430],
        [0.308, 1.50185265748, 77.22927912210],
        [0.324, 5.30240189273, 457.61767951300],
        [0.265, 1.08736632800, 450.45594840240],
        [0.264, 0.83337660655, 488.37653571910],
        [0.290, 1.80003152563, 101.86893394120],
        [0.262, 2.30390003360, 494.73932319450],
        [0.325, 5.52669889053, 441.26722786230],
        [0.254, 0.02963623277, 117.36805233000],
        [0.300, 0.17435705540, 252.91660378410],
        [0.315, 5.34885013040, 183.76407950930],
        [0.313, 5.45945846595, 13.49338081870],
        [0.306, 5.23085809622, 45.24658263860],
        [0.237, 0.32676889138, 208.84567731310],
        [0.263, 2.66670785888, 464.73122651380],
        [0.234, 1.82700149824, 52175.80628314840],
        [0.275, 5.04385701142, 156.15547927360],
        [0.265, 5.64967127743, 326.86812094890],
        [0.247, 1.74540930625, 65.87476231750],
        [0.269, 6.09827783249, 1654.03263386460],
        [0.229, 2.25832077914, 190.66517818900],
        [0.294, 5.45249564193, 206.18554843720],
        [0.238, 1.55647021369, 79.88940799800],
        [0.230, 6.13158632762, 178.34745353790],
        [0.274, 4.10829870815, 518.38463239980],
        [0.225, 3.86300359251, 171.98466606250],
        [0.228, 2.48511565618, 12566.15169998280],
        [0.272, 5.61149862463, 148.33935685720],
        [0.214, 1.45987216039, 522.57741809380],
        [0.211, 4.04791980901, 6205.32530600750],
        [0.266, 0.99036038827, 209.10630974400],
        [0.230, 0.54049951530, 532.61172640140],
        [0.226, 3.84152961620, 283.62727588040],
        [0.243, 5.32730346969, 485.92885516430],
        [0.209, 4.35051470487, 536.80451209540],
        [0.232, 3.01948719112, 10.93382567930],
        [0.264, 5.70536379124, 490.33408917940],
        [0.280, 3.99993658196, 674.80074410430],
        [0.246, 0.37698964335, 157.63995198190],
        [0.219, 5.67679857772, 52099.54021187280],
        [0.251, 1.52353965506, 6.85291456990],
        [0.203, 5.44328656642, 145.63104387150],
        [0.238, 0.96169723853, 497.18700374930],
        [0.219, 4.52300776062, 1615.89959822680],
        [0.275, 2.37619210741, 2118.76386037840],
        [0.258, 5.12448148780, 608.87779767700],
        [0.260, 3.88543008475, 513.07988101300],
        [0.191, 3.72574595369, 65.22037101170],
        [0.211, 0.06484535455, 215.43795945210],
        [0.236, 3.95835282821, 141.48644228730],
        [0.189, 5.28135043909, 377.15882254340],
        [0.243, 4.35559878377, 482.95990974770],
        [0.243, 6.06808644973, 154.01661525950],
        [0.249, 1.57215637373, 14.22709400160],
        [0.238, 1.93340192445, 500.15594916590],
        [0.209, 5.02893682321, 364.55921360200],
        [0.227, 5.72984298540, 1543.82631264520],
        [0.217, 2.45036922991, 187.17496791060],
        [0.181, 1.65699502247, 1627.20593092160],
        [0.214, 1.60213179145, 11.30633269480],
        [0.203, 0.74638490279, 14.55716240170],
        [0.192, 3.17719161639, 343.47920503050],
        [0.177, 1.50027795761, 9.44935297100],
        [0.177, 0.03038098292, 165.60483224460],
        [0.176, 4.64462444674, 315.16802937920],
        [0.208, 2.65835778368, 496.01134758170],
        [0.174, 2.76155855705, 49.17873590170],
        [0.196, 1.95549714182, 335.77495719870],
        [0.200, 4.16839394758, 285.11174858870],
        [0.199, 0.06168021293, 73.55775828990],
        [0.188, 6.17288913873, 535.32003938710],
        [0.215, 1.92414563346, 552.69738935910],
        [0.166, 5.49038139690, 10135.53500222710],
        [0.192, 0.96973434120, 304.23420369990],
        [0.209, 5.34065233845, 13.64213866500],
        [0.203, 5.11234865419, 324.72925693480],
        [0.177, 3.50680841790, 207.36120460480],
        [0.174, 1.95010708561, 319.31263096340],
        [0.187, 5.57685931698, 266.10116806210],
        [0.181, 1.43525075751, 279.74330672710],
        [0.165, 4.00537112057, 493.56366702690],
        [0.191, 1.68313683465, 563.37058260750],
        [0.173, 3.93200456456, 238.90195810360],
        [0.161, 5.96143146317, 36.12729806770],
        [0.194, 2.37664231450, 944.98282327580],
        [0.165, 0.97421918976, 556.51766803760],
        [0.189, 1.11279570541, 1127.26243007680],
        [0.172, 0.75085513952, 267.58564077040],
        [0.193, 2.12636756833, 20350.30502114640],
        [0.181, 2.10814562080, 113.87784205160],
        [0.194, 1.13504964219, 57.25549074900],
        [0.181, 6.23699820519, 355.96119389290],
        [0.198, 5.68125942959, 6280.10690457480],
        [0.173, 5.15083799917, 474.94684537520],
        [0.151, 1.66981962338, 116.53797092750],
        [0.150, 5.42593657173, 526.98265210890],
        [0.205, 4.16096717573, 711.44930703380],
        [0.177, 3.49360697678, 421.22974901440],
        [0.168, 0.52839230204, 487.10451133190],
        [0.160, 4.77712663799, 524.01370669230],
        [0.145, 2.81448128781, 1512.80682400820],
        [0.146, 4.99570112660, 142.66209845490],
        [0.188, 0.82104161550, 10210.31660079440],
        [0.145, 4.96888131586, 1189.30140735080],
        [0.181, 2.99704790590, 75.74480641380],
        [0.176, 0.41626373842, 222.86032299360],
        [0.137, 2.96534226337, 6206.80977871580],
        [0.138, 1.22260849471, 187.69623277240],
        [0.128, 2.53394068407, 276.77436131050],
        [0.130, 3.04810765699, 310.71461125430],
        [0.122, 3.01323006886, 70.84944530420],
        [0.111, 0.77449448649, 179.35884549420],
        [0.141, 0.18423889807, 131.40394986990],
        [0.126, 5.77648809669, 525.23754696970],
        [0.124, 2.93225731024, 179.61947792510],
        [0.111, 6.18471578216, 981.63138620530],
        [0.141, 2.63342951123, 381.61224066830],
        [0.110, 5.25053027081, 986.08480433020],
        [0.096, 3.86591534559, 240.12579838100],
        [0.120, 3.78755085035, 1057.89745748090],
        [0.093, 4.54014016637, 36.69674703930],
        [0.109, 1.53327585900, 419.74527630610],
        [0.094, 4.21870300178, 1024.21783996800],
        [0.109, 2.15905156247, 289.56516671360],
        [0.104, 0.20665642552, 564.85505531580],
        [0.081, 1.89134135215, 36.60037881970],
        [0.080, 4.38832594589, 10137.01947493540],
        [0.080, 1.73940577376, 39.50563376150],
        [0.084, 0.81316746605, 170.71264167530],
        [0.090, 0.60145818457, 36.76043751410],
        [0.074, 4.92511651321, 1549.45538693770],
        [0.072, 5.06852406179, 249.94765836750],
    ],
    # L1
    [
        [3837687716.731, 0.00000000000, 0.00000000000],
        [16604.187, 4.86319129565, 1.48447270830],
        [15807.148, 2.27923488532, 38.13303563780],
        [3334.701, 3.68199676020, 76.26607127560],
        [1305.840, 3.67320813491, 2.96894541660],
        [604.832, 1.50477747549, 35.16409022120],
        [178.623, 3.45318524147, 39.61750834610],
        [106.537, 2.45126138334, 4.45341812490],
        [105.747, 2.75479326550, 33.67961751290],
        [72.684, 5.48724732699, 36.64856292950],
        [57.069, 5.21649804970, 0.52126486180],
        [57.355, 1.85767603384, 114.39910691340],
        [35.368, 4.51676827545, 74.78159856730],
        [32.216, 5.90411489680, 77.75054398390],
        [29.871, 3.67043294114, 388.46515523820],
        [28.866, 5.16877529164, 9.56122755560],
        [28.742, 5.16732589024, 2.44768055480],
        [25.507, 5.24526281928, 168.05251279940],
        [24.869, 4.73193067810, 182.27960680100],
        [20.205, 5.78945415677, 1021.24889455140],
        [19.022, 1.82981144269, 484.44438245600],
        [18.661, 1.31606255521, 498.67147645760],
        [15.063, 4.95003893760, 137.03302416240],
        [15.094, 3.98705254940, 32.19514480460],
        [10.720, 2.44148149225, 4.19278569400],
        [11.725, 4.89255650674, 71.81265315070],
        [9.581, 1.23188039594, 5.93789083320],
        [9.606, 1.88534821556, 41.10198105440],
        [8.968, 0.01758559103, 8.07675484730],
        [9.882, 6.08165628679, 7.11354700080],
        [7.632, 5.51307048241, 73.29712585900],
        [6.992, 0.61688864282, 2.92076130680],
        [5.543, 2.24141557794, 46.20979048510],
        [4.845, 3.71055823750, 112.91463420510],
        [3.700, 5.25713252333, 111.43016149680],
        [3.233, 6.10303038418, 70.32818044240],
        [2.939, 4.86520586648, 98.89998852460],
        [2.403, 2.90637675099, 601.76425067620],
        [2.398, 1.04343654629, 6.59228213900],
        [2.784, 4.95821114677, 108.46121608020],
        [2.894, 4.20148844767, 381.35160823740],
        [2.111, 5.93089610785, 25.60286266560],
        [2.075, 5.20632201951, 30.71067209630],
        [2.126, 0.54976393136, 41.05379694460],
        [2.235, 2.38045158073, 453.42489381900],
        [1.859, 0.89409373259, 24.11838995730],
        [2.018, 3.42245274178, 31.01948863700],
        [1.700, 3.91715254287, 11.04570026390],
        [1.776, 3.86571077241, 395.57870223900],
        [1.644, 0.15855999051, 152.53214255120],
        [1.646, 3.34591387314, 44.72531777680],
        [1.876, 2.59784179105, 33.94024994380],
        [1.614, 0.42137145545, 175.16605980020],
        [1.468, 6.12983933526, 1550.93985964600],
        [1.408, 6.13722948564, 490.07345674850],
        [1.207, 0.59525736062, 312.19908396260],
        [1.336, 3.28611928206, 493.04240216510],
        [1.176, 5.87266726996, 5.41662597140],
        [1.517, 3.12967210501, 491.55792945680],
        [1.053, 4.60375516830, 79.23501669220],
        [1.037, 4.89007314395, 1.27202438720],
        [1.034, 5.93741289103, 32.71640966640],
        [1.038, 1.13470380744, 1014.13534755060],
        [1.002, 1.85850922283, 5.10780943070],
        [0.983, 0.05345050384, 7.42236354150],
        [0.998, 1.73689827444, 1028.36244155220],
        [1.193, 4.63176675581, 60.76695288680],
        [0.940, 3.09103721222, 62.25142559510],
        [0.994, 4.11489180313, 4.66586644600],
        [0.890, 0.87049255398, 31.23193695810],
        [0.852, 5.35508394316, 144.14657116320],
        [0.922, 5.12373360511, 145.10977900970],
        [0.789, 0.37496785039, 26.82670294300],
        [0.828, 4.06035194600, 115.88357962170],
        [0.711, 3.14189997439, 278.25883401880],
        [0.727, 1.39718382835, 213.29909543800],
        [0.781, 0.10946327923, 173.68158709190],
        [0.793, 6.13086312116, 567.82400073240],
        [0.669, 4.50554989443, 27.08733537390],
        [0.825, 1.35568908148, 129.91947716160],
        [0.738, 3.56766018960, 176.65053250850],
        [0.714, 6.24797992301, 106.97674337190],
        [0.654, 1.13177751192, 68.84370773410],
        [0.624, 0.01567750666, 28.57180808220],
        [0.608, 4.60180625368, 189.39315380180],
        [0.595, 0.00857468445, 42.58645376270],
        [0.530, 5.61201247153, 12.53017297220],
        [0.521, 1.02371768017, 415.29185818120],
        [0.639, 0.68930265745, 529.69096509460],
        [0.526, 3.02138731705, 5.62907429250],
        [0.456, 4.44331571392, 43.24084506850],
        [0.524, 3.43316448349, 38.65430049960],
        [0.436, 2.41630174435, 82.85835341460],
        [0.424, 1.95736011325, 477.33083545520],
        [0.443, 3.39350946329, 357.44566660120],
        [0.383, 1.90232196422, 22.63391724900],
        [0.479, 5.55141744216, 37.61177077600],
        [0.462, 3.80436154644, 343.21857259960],
        [0.384, 5.60377408953, 594.65070367540],
        [0.369, 4.45577410338, 6.90109867970],
        [0.358, 3.69126616347, 3.93215326310],
        [0.352, 3.10952926034, 135.54855145410],
        [0.368, 3.53577440355, 40.58071619260],
        [0.424, 5.27159202779, 181.75834193920],
        [0.361, 0.29018303419, 72.07328558160],
        [0.390, 5.49512204296, 350.33211960040],
        [0.378, 2.74122401337, 488.37653571910],
        [0.372, 0.39980033572, 494.73932319450],
        [0.353, 1.10614174053, 20.60692781950],
        [0.296, 0.86351261285, 149.56319713460],
        [0.307, 5.39420288683, 160.93896579860],
        [0.395, 1.93577214824, 10137.01947493540],
        [0.288, 2.28755739359, 47.69426319340],
        [0.295, 2.48737537240, 19.12245511120],
        [0.290, 0.18636083306, 143.62530630140],
        [0.266, 3.09977370364, 69.36497259590],
        [0.266, 1.21002824826, 505.78502345840],
        [0.252, 3.12745026026, 460.53844081980],
        [0.328, 0.50849285663, 6206.80977871580],
        [0.257, 3.64119914774, 446.31134681820],
        [0.239, 5.54080102299, 911.04257333200],
        [0.265, 0.62702473701, 253.57099508990],
        [0.287, 2.44403568436, 16.67477455640],
        [0.231, 2.47026250085, 454.90936652730],
        [0.230, 3.24571542922, 1066.49547719000],
        [0.282, 1.48595620175, 983.11585891360],
        [0.212, 5.41931177641, 64.95973858080],
        [0.213, 1.64175339637, 1089.12939443900],
        [0.238, 2.69801319489, 882.94384600180],
        [0.210, 4.53976756699, 1093.32218013300],
        [0.220, 2.30038816175, 1052.26838318840],
        [0.256, 0.42073598460, 23.90594163620],
        [0.216, 5.44225918870, 39.09624348430],
        [0.201, 2.58746514605, 119.50691634410],
        [0.224, 4.43751392203, 639.89728631400],
        [0.186, 2.50651218075, 487.36514376280],
        [0.189, 4.05785534221, 120.99138905240],
        [0.184, 2.24245977278, 815.06334611420],
        [0.202, 3.43517732411, 45.24658263860],
        [0.175, 4.49165234532, 171.23390653710],
        [0.171, 5.50633466316, 179.09821306330],
        [0.200, 6.12663205401, 14.22709400160],
        [0.173, 2.61090344107, 389.94962794650],
        [0.167, 3.94754384833, 77.22927912210],
        [0.166, 3.41009128748, 81.37388070630],
        [0.163, 3.88198848446, 556.51766803760],
        [0.164, 1.49614763046, 63.73589830340],
        [0.176, 3.86129425367, 148.33935685720],
        [0.161, 2.22215642318, 574.93754773320],
        [0.171, 0.66899426684, 179.31066138440],
        [0.161, 1.21480182441, 1024.43028828910],
        [0.155, 3.25842414799, 10251.41858184880],
        [0.183, 5.45168150656, 218.40690486870],
        [0.152, 3.35145509017, 285.37238101960],
        [0.152, 0.42398786475, 274.06604832480],
        [0.146, 5.70714579127, 419.48464387520],
        [0.156, 0.64321524870, 1029.84691426050],
        [0.147, 4.30958930740, 157.63995198190],
        [0.147, 1.80689177510, 377.41945497430],
        [0.140, 1.49826604627, 386.98068252990],
        [0.137, 2.14480243915, 563.63121503840],
        [0.127, 3.98726599710, 84.34282612290],
        [0.134, 4.16039455079, 169.53698550770],
        [0.121, 0.29300927469, 206.18554843720],
        [0.129, 2.67625057010, 180.79513409270],
        [0.134, 3.18868986487, 166.56804009110],
        [0.135, 5.07517561780, 426.59819087600],
        [0.136, 1.81672451740, 151.04766984290],
        [0.129, 3.64795525602, 183.76407950930],
        [0.116, 6.06435563172, 220.41264243880],
        [0.123, 4.46641157829, 1022.73336725970],
        [0.112, 4.34485256988, 138.51749687070],
        [0.116, 5.58946529961, 35.68535508300],
        [0.108, 1.03796693383, 488.58898404020],
        [0.108, 2.10378485880, 494.52687487340],
        [0.106, 0.87068583107, 1059.38193018920],
        [0.097, 0.74486741478, 485.92885516430],
        [0.095, 5.54259914856, 497.18700374930],
        [0.085, 3.16062141266, 522.57741809380],
        [0.097, 6.05634803604, 482.95990974770],
        [0.095, 0.23111852730, 500.15594916590],
        [0.084, 2.64687252518, 536.80451209540],
        [0.074, 3.90678924318, 1019.76442184310],
    ],
    # L2
    [
        [53892.649, 0.00000000000, 0.00000000000],
        [281.251, 1.19084538887, 38.13303563780],
        [295.693, 1.85520292248, 1.48447270830],
        [270.190, 5.72143228148, 76.26607127560],
        [23.023, 1.21035596452, 2.96894541660],
        [7.333, 0.54033306830, 2.44768055480],
        [9.057, 4.42544992035, 35.16409022120],
        [5.223, 0.67427930044, 168.05251279940],
        [5.201, 3.02338671812, 182.27960680100],
        [4.288, 3.84351844003, 114.39910691340],
        [3.925, 3.53214557374, 484.44438245600],
        [3.741, 5.90238217874, 498.67147645760],
        [2.966, 0.31002477611, 4.45341812490],
        [3.415, 0.55971639038, 74.78159856730],
        [3.255, 1.84921884906, 175.16605980020],
        [2.157, 1.89135758747, 388.46515523820],
        [2.211, 4.37997092240, 7.11354700080],
        [1.847, 3.48574435762, 9.56122755560],
        [2.451, 4.68586840176, 491.55792945680],
        [1.844, 5.12281562096, 33.67961751290],
        [2.204, 1.69321574906, 77.75054398390],
        [1.652, 2.55859494053, 36.64856292950],
        [1.309, 4.52400192922, 1021.24889455140],
        [1.124, 0.38710602242, 137.03302416240],
        [0.664, 0.88101734307, 4.19278569400],
        [0.497, 2.24615784762, 395.57870223900],
        [0.512, 6.22609200672, 381.35160823740],
        [0.582, 5.25716719826, 31.01948863700],
        [0.446, 0.36647221351, 98.89998852460],
        [0.383, 5.48585528762, 5.93789083320],
        [0.375, 4.61250246774, 8.07675484730],
        [0.354, 1.30783918287, 601.76425067620],
        [0.259, 5.66033623678, 112.91463420510],
        [0.247, 2.89695614593, 189.39315380180],
        [0.245, 4.26572913391, 220.41264243880],
        [0.200, 0.52604535784, 64.95973858080],
        [0.191, 4.88786653062, 39.61750834610],
        [0.233, 3.16423779113, 41.10198105440],
        [0.248, 5.85877831382, 1059.38193018920],
        [0.194, 2.37949641473, 73.29712585900],
        [0.227, 0.20028518978, 60.76695288680],
        [0.184, 3.01962045713, 1014.13534755060],
        [0.190, 5.57500985081, 343.21857259960],
        [0.172, 3.66036463613, 477.33083545520],
        [0.172, 0.59550457102, 46.20979048510],
        [0.182, 1.92429384025, 183.76407950930],
        [0.171, 1.61368476689, 357.44566660120],
        [0.173, 6.23717119485, 493.04240216510],
        [0.217, 1.46218158211, 71.81265315070],
        [0.178, 0.34928799031, 1028.36244155220],
        [0.169, 4.91086673212, 166.56804009110],
        [0.157, 5.89200571154, 169.53698550770],
        [0.182, 2.33457064554, 152.53214255120],
        [0.151, 3.81621340568, 146.59425171800],
        [0.136, 2.75150881988, 144.14657116320],
        [0.104, 6.03262825314, 529.69096509460],
        [0.076, 0.20932812381, 453.42489381900],
    ],
    # L3
    [
        [31.254, 0.00000000000, 0.00000000000],
        [12.461, 6.04431418812, 1.48447270830],
        [14.541, 1.35337075856, 76.26607127560],
        [11.547, 6.11257808366, 38.13303563780],
        [1.351, 4.93951495175, 2.96894541660],
        [0.741, 2.35936954597, 168.05251279940],
        [0.715, 1.27409542804, 182.27960680100],
        [0.537, 5.23632185196, 484.44438245600],
        [0.523, 4.16769839601, 498.67147645760],
        [0.664, 0.55871435877, 31.01948863700],
        [0.301, 2.69253200796, 7.11354700080],
        [0.194, 2.05904114139, 137.03302416240],
        [0.206, 2.51012178002, 74.78159856730],
        [0.160, 5.63111039032, 114.39910691340],
        [0.149, 3.09327713923, 35.16409022120],
    ],
    # L4
    [
        [113.998, 3.14159265359, 0.00000000000],
        [0.605, 3.18211885677, 76.26607127560]
    ],
    # L5
    [
        [0.874, 3.14159265359, 0.00000000000]
    ],
]
"""This table contains Neptune's periodic terms (all of them) from the
planetary theory VSOP87 for the heliocentric longitude at the equinox of date
(taken from the 'D' solution). In Meeus' book a shortened version can be found
in pages 452-453."""


VSOP87_B = [
    # B0
    [
        [3088622.933, 1.44104372626, 38.13303563780],
        [27780.087, 5.91271882843, 76.26607127560],
        [27623.609, 0.00000000000, 0.00000000000],
        [15355.490, 2.52123799481, 36.64856292950],
        [15448.133, 3.50877080888, 39.61750834610],
        [1999.919, 1.50998669505, 74.78159856730],
        [1967.540, 4.37778195768, 1.48447270830],
        [1015.137, 3.21561035875, 35.16409022120],
        [605.767, 2.80246601405, 73.29712585900],
        [594.878, 2.12892708114, 41.10198105440],
        [588.805, 3.18655882497, 2.96894541660],
        [401.830, 4.16883287237, 114.39910691340],
        [254.333, 3.27120499438, 453.42489381900],
        [261.647, 3.76722704749, 213.29909543800],
        [279.964, 1.68165309699, 77.75054398390],
        [205.590, 4.25652348864, 529.69096509460],
        [140.455, 3.52969556376, 137.03302416240],
        [98.530, 4.16774829927, 33.67961751290],
        [51.257, 1.95121181203, 4.45341812490],
        [67.971, 4.66970781659, 71.81265315070],
        [41.931, 5.41783694467, 111.43016149680],
        [41.822, 5.94832001477, 112.91463420510],
        [30.637, 0.93620571932, 42.58645376270],
        [11.084, 5.88898793049, 108.46121608020],
        [9.620, 0.03944255108, 70.32818044240],
        [9.664, 0.22455797403, 79.23501669220],
        [9.728, 5.30069593532, 32.19514480460],
        [7.386, 3.00684933642, 426.59819087600],
        [7.087, 0.12535040656, 109.94568878850],
        [6.021, 6.20514068152, 115.88357962170],
        [6.169, 3.62098109648, 983.11585891360],
        [4.777, 0.75210194972, 5.93789083320],
        [6.391, 5.84646101060, 148.07872442630],
        [6.251, 2.41678769385, 152.53214255120],
        [4.539, 5.58182098700, 175.16605980020],
        [5.006, 4.60815664851, 1059.38193018920],
        [4.289, 4.19647392821, 47.69426319340],
        [5.795, 5.07516716087, 415.29185818120],
        [4.749, 2.51605725604, 37.61177077600],
        [4.119, 1.72779509865, 28.57180808220],
        [4.076, 6.00252170354, 145.10977900970],
        [4.429, 5.65995321659, 98.89998852460],
        [3.950, 2.74104636753, 350.33211960040],
        [4.091, 1.61787956945, 39.09624348430],
        [4.131, 4.40682554313, 37.16982779130],
        [4.710, 3.50929350767, 38.65430049960],
        [4.440, 4.78977105547, 38.08485152800],
        [4.433, 1.23386935925, 38.18121974760],
        [3.762, 4.83940791709, 491.55792945680],
        [2.606, 1.20956732792, 451.94042111070],
        [2.537, 2.18628045751, 454.90936652730],
        [2.328, 5.19779918719, 72.07328558160],
        [2.502, 0.85987904350, 106.97674337190],
        [2.342, 0.81387240947, 4.19278569400],
        [1.981, 0.46617960831, 184.72728735580],
        [1.963, 6.01909114576, 44.07092647100],
        [2.180, 0.70099749844, 206.18554843720],
        [1.811, 0.40456996647, 40.58071619260],
        [1.814, 3.64699555185, 220.41264243880],
        [1.705, 6.13551142362, 181.75834193920],
        [1.855, 5.61635630213, 35.68535508300],
        [1.595, 2.97147156093, 37.87240320690],
        [1.785, 2.42154818096, 388.46515523820],
        [1.595, 3.05266110075, 38.39366806870],
        [1.437, 1.48678704605, 135.54855145410],
        [1.387, 2.46149266117, 138.51749687070],
        [1.366, 1.52026779665, 68.84370773410],
        [1.575, 3.58964541604, 38.02116105320],
        [1.297, 5.06156596196, 33.94024994380],
        [1.487, 0.20211121607, 30.05628079050],
        [1.504, 5.80298577327, 46.20979048510],
        [1.192, 0.87275514483, 42.32582133180],
        [1.569, 2.43405967107, 38.24491022240],
        [1.207, 1.84658687853, 251.43213107580],
        [1.015, 0.53439848924, 129.91947716160],
        [0.999, 2.47463873948, 312.19908396260],
        [0.990, 3.41514319052, 144.14657116320],
        [0.963, 4.31733242907, 151.04766984290],
        [1.020, 0.98226686775, 143.62530630140],
        [0.941, 1.02993053785, 221.37585028530],
        [0.938, 2.43648356625, 567.82400073240],
        [1.111, 0.65175024456, 146.59425171800],
        [0.777, 0.00175975222, 218.40690486870],
        [0.895, 0.25123869620, 30.71067209630],
        [0.795, 5.80519741659, 149.56319713460],
        [0.737, 3.40060492866, 446.31134681820],
        [0.719, 1.43795191278, 8.07675484730],
        [0.720, 0.00651007550, 460.53844081980],
        [0.766, 4.03399506246, 522.57741809380],
        [0.666, 1.39457824982, 84.34282612290],
        [0.584, 1.01405548136, 536.80451209540],
        [0.596, 0.62390100715, 35.21227433100],
        [0.598, 5.39946724188, 41.05379694460],
        [0.475, 5.80072248338, 7.42236354150],
        [0.510, 1.34478579740, 258.02441321480],
        [0.458, 5.25325523118, 80.71948940050],
        [0.421, 3.24496387889, 416.77633088950],
        [0.446, 1.19167306357, 180.27386923090],
        [0.471, 0.92632922375, 44.72531777680],
        [0.387, 1.68488418788, 183.24281464750],
        [0.375, 0.15223869165, 255.05546779820],
        [0.354, 4.21526988674, 0.96320784650],
        [0.379, 2.16947487177, 105.49227066360],
        [0.341, 4.79194051680, 110.20632121940],
        [0.427, 5.15774894584, 31.54075349880],
        [0.302, 3.45706306280, 100.38446123290],
        [0.298, 2.26790695187, 639.89728631400],
        [0.279, 0.25689162963, 39.50563376150],
        [0.320, 3.58085653166, 45.24658263860],
        [0.269, 5.72024180826, 36.76043751410],
        [0.247, 0.61040148804, 186.21176006410],
        [0.245, 0.64173616273, 419.48464387520],
        [0.235, 0.73189197665, 10213.28554621100],
        [0.232, 0.37399822852, 490.07345674850],
        [0.230, 5.76570492457, 12.53017297220],
        [0.240, 4.13447692727, 0.52126486180],
        [0.279, 1.62614865256, 294.67297614430],
        [0.238, 2.18528916550, 219.89137757700],
        [0.262, 3.08384135298, 6.59228213900],
        [0.217, 2.93214905312, 27.08733537390],
        [0.217, 4.69210602828, 406.10313764110],
        [0.219, 1.35212712560, 216.92243216040],
        [0.200, 2.35215465744, 605.95703637020],
        [0.232, 3.92583619589, 1512.80682400820],
        [0.223, 5.52392277606, 187.69623277240],
        [0.190, 0.29169556516, 291.70403072770],
        [0.236, 3.12464145036, 563.63121503840],
        [0.193, 0.53675942386, 60.76695288680],
        [0.215, 3.78391259001, 103.09277421860],
        [0.172, 5.63262770743, 7.11354700080],
        [0.164, 4.14700645532, 77.22927912210],
        [0.162, 0.72021213236, 11.04570026390],
        [0.160, 4.23490438166, 487.36514376280],
        [0.191, 0.37651439206, 31.01948863700],
        [0.157, 1.02419759383, 6283.07584999140],
        [0.157, 4.42530429545, 6206.80977871580],
        [0.178, 6.24797160202, 316.39186965660],
        [0.161, 5.65988283675, 343.21857259960],
        [0.153, 5.58405022784, 252.08652238160],
        [0.189, 4.80791039970, 641.12112659140],
        [0.166, 5.50438043692, 662.53120356300],
        [0.146, 5.08949604858, 286.59622129700],
        [0.145, 2.13015521881, 2042.49778910280],
        [0.156, 2.19452173251, 274.06604832480],
        [0.148, 4.85696640135, 442.75170057060],
        [0.187, 4.96121139073, 1589.07289528380],
        [0.155, 2.28260574227, 142.14083359310],
        [0.134, 1.29277093566, 456.39383923560],
        [0.126, 5.59769497652, 179.35884549420],
        [0.146, 2.53359213478, 256.53994050650],
        [0.140, 1.57962199954, 75.74480641380],
        [0.123, 0.05442220184, 944.98282327580],
        [0.122, 1.90676379802, 418.26080359780],
        [0.154, 1.86865302773, 331.32153907380],
        [0.144, 5.52229258454, 14.01464568050],
        [0.138, 2.80728175526, 82.85835341460],
        [0.107, 0.66995358132, 190.66517818900],
        [0.114, 1.48894980280, 253.57099508990],
        [0.110, 5.32587573069, 240.12579838100],
        [0.105, 0.65548440578, 173.68158709190],
        [0.102, 2.58735617801, 450.45594840240],
        [0.098, 0.44044795266, 328.35259365720],
        [0.101, 4.71267656829, 117.36805233000],
        [0.094, 0.54938580474, 293.18850343600],
        [0.095, 2.17636214523, 101.86893394120],
        [0.093, 0.63687810471, 377.15882254340],
        [0.091, 5.84828809934, 10137.01947493540],
        [0.089, 1.02830167997, 1021.24889455140],
        [0.094, 1.79320597168, 493.04240216510],
        [0.080, 1.58140274465, 69.15252427480],
        [0.075, 0.23453373368, 63.73589830340],
        [0.071, 1.51961989690, 488.58898404020],
    ],
    # B1
    [
        [227279.214, 3.80793089870, 38.13303563780],
        [1803.120, 1.97576485377, 76.26607127560],
        [1385.733, 4.82555548018, 36.64856292950],
        [1433.300, 3.14159265359, 0.00000000000],
        [1073.298, 6.08054240712, 39.61750834610],
        [147.903, 3.85766231348, 74.78159856730],
        [136.448, 0.47764957338, 1.48447270830],
        [70.285, 6.18782052139, 35.16409022120],
        [51.899, 5.05221791891, 73.29712585900],
        [37.273, 4.89476629246, 41.10198105440],
        [42.568, 0.30721737205, 114.39910691340],
        [37.104, 5.75999349109, 2.96894541660],
        [26.399, 5.21566335936, 213.29909543800],
        [16.949, 4.26463671859, 77.75054398390],
        [18.747, 0.90426522185, 453.42489381900],
        [12.951, 6.17709713139, 529.69096509460],
        [10.502, 1.20336443465, 137.03302416240],
        [4.416, 1.25478204684, 111.43016149680],
        [4.383, 6.14147099615, 71.81265315070],
        [3.694, 0.94837702528, 33.67961751290],
        [2.957, 4.77532871210, 4.45341812490],
        [2.698, 1.92435531119, 112.91463420510],
        [1.989, 3.96637567224, 42.58645376270],
        [1.150, 4.30568700024, 37.61177077600],
        [0.871, 4.81775882249, 152.53214255120],
        [0.944, 2.21777772050, 109.94568878850],
        [0.936, 1.17054983940, 148.07872442630],
        [0.925, 2.40329074000, 206.18554843720],
        [0.690, 1.57381082857, 38.65430049960],
        [0.624, 2.79466003645, 79.23501669220],
        [0.726, 4.13829519132, 28.57180808220],
        [0.640, 2.46161252327, 115.88357962170],
        [0.531, 2.96991530500, 98.89998852460],
        [0.537, 1.95986772922, 220.41264243880],
        [0.539, 2.06690307827, 40.58071619260],
        [0.716, 0.55781847010, 350.33211960040],
        [0.563, 1.84072805158, 983.11585891360],
        [0.533, 1.34787677940, 47.69426319340],
        [0.566, 1.80111775954, 175.16605980020],
        [0.449, 1.62191691011, 144.14657116320],
        [0.371, 2.74239666472, 415.29185818120],
        [0.381, 6.11910193382, 426.59819087600],
        [0.366, 2.39752585360, 129.91947716160],
        [0.456, 3.19611413854, 108.46121608020],
        [0.327, 3.62341506247, 38.18121974760],
        [0.328, 0.89613145346, 38.08485152800],
        [0.341, 3.87265469070, 35.68535508300],
        [0.331, 4.48858774501, 460.53844081980],
        [0.414, 1.03543720726, 70.32818044240],
        [0.310, 0.51297445145, 37.16982779130],
        [0.287, 2.18351651800, 491.55792945680],
        [0.274, 6.11504724934, 522.57741809380],
        [0.281, 3.81657117512, 5.93789083320],
        [0.298, 4.00532631258, 39.09624348430],
        [0.265, 5.26569823181, 446.31134681820],
        [0.319, 1.34097217817, 184.72728735580],
        [0.203, 6.02944475303, 149.56319713460],
        [0.205, 5.53935732020, 536.80451209540],
        [0.226, 6.17710997862, 454.90936652730],
        [0.186, 3.24302117645, 4.19278569400],
        [0.179, 4.91458426239, 451.94042111070],
        [0.198, 2.30775852880, 146.59425171800],
        [0.166, 1.16793600058, 72.07328558160],
        [0.147, 2.10574339673, 44.07092647100],
        [0.123, 1.98250467171, 46.20979048510],
        [0.159, 3.46955908364, 145.10977900970],
        [0.116, 5.88971113590, 38.02116105320],
        [0.115, 4.73412534395, 38.24491022240],
        [0.125, 3.42713474801, 251.43213107580],
        [0.128, 1.51108932026, 221.37585028530],
        [0.127, 0.17176461812, 138.51749687070],
        [0.124, 5.85160407534, 1059.38193018920],
        [0.091, 2.38273591235, 30.05628079050],
        [0.118, 5.27114846878, 37.87240320690],
        [0.117, 5.35267669439, 38.39366806870],
        [0.099, 5.19920708255, 135.54855145410],
        [0.114, 4.37452353441, 388.46515523820],
        [0.093, 4.64183693718, 106.97674337190],
        [0.084, 1.35269684746, 33.94024994380],
        [0.111, 3.56226463770, 181.75834193920],
        [0.082, 3.18401661435, 42.32582133180],
        [0.084, 5.51669920239, 8.07675484730],
    ],
    # B2
    [
        [9690.766, 5.57123750291, 38.13303563780],
        [78.815, 3.62705474219, 76.26607127560],
        [71.523, 0.45476688580, 36.64856292950],
        [58.646, 3.14159265359, 0.00000000000],
        [29.915, 1.60671721861, 39.61750834610],
        [6.472, 5.60736756575, 74.78159856730],
        [5.800, 2.25341847151, 1.48447270830],
        [4.309, 1.68126737666, 35.16409022120],
        [3.502, 2.39142672984, 114.39910691340],
        [2.649, 0.65061457644, 73.29712585900],
        [1.518, 0.37600329684, 213.29909543800],
        [1.223, 1.23116043030, 2.96894541660],
        [0.766, 5.45279753249, 453.42489381900],
        [0.779, 2.07081431472, 529.69096509460],
        [0.496, 0.26552533921, 41.10198105440],
        [0.469, 5.87866293959, 77.75054398390],
        [0.482, 5.63056237954, 137.03302416240],
        [0.345, 1.80085651594, 71.81265315070],
        [0.274, 2.86650141006, 33.67961751290],
        [0.158, 4.63868656467, 206.18554843720],
        [0.166, 1.24877330835, 220.41264243880],
        [0.153, 2.87376446497, 111.43016149680],
        [0.116, 3.63838544843, 112.91463420510],
        [0.085, 0.43712705655, 4.45341812490],
        [0.104, 6.12597614674, 144.14657116320],
    ],
    # B3
    [
        [273.423, 1.01688979072, 38.13303563780],
        [2.274, 2.36805657126, 36.64856292950],
        [2.029, 5.33364321342, 76.26607127560],
        [2.393, 0.00000000000, 0.00000000000],
        [0.538, 3.21934211365, 39.61750834610],
        [0.242, 4.52650721578, 114.39910691340],
        [0.185, 1.04913770083, 74.78159856730],
        [0.155, 3.62376309338, 35.16409022120],
        [0.157, 3.94195369610, 1.48447270830],
    ],
    # B4
    [[5.728, 2.66872693322, 38.13303563780]],
    # B5
    [[0.113, 4.70646877989, 38.13303563780]],
]
"""This table contains Neptune's periodic terms (all of them) from the
planetary theory VSOP87 for the heliocentric latitude at the equinox of date
(taken from the 'D' solution). In Meeus' book a shortened version can be found
in page 453."""


VSOP87_R = [
    # R0
    [
        [3007013206.102, 0.00000000000, 0.00000000000],
        [27062259.490, 1.32999458930, 38.13303563780],
        [1691764.281, 3.25186138896, 36.64856292950],
        [807830.737, 5.18592836167, 1.48447270830],
        [537760.613, 4.52113902845, 35.16409022120],
        [495725.642, 1.57105654815, 491.55792945680],
        [274571.970, 1.84552256801, 175.16605980020],
        [135134.095, 3.37220607384, 39.61750834610],
        [121801.825, 5.79754444303, 76.26607127560],
        [100895.397, 0.37702748681, 73.29712585900],
        [69791.722, 3.79617226928, 2.96894541660],
        [46687.838, 5.74937810094, 33.67961751290],
        [24593.778, 0.50801728204, 109.94568878850],
        [16939.242, 1.59422166991, 71.81265315070],
        [14229.686, 1.07786112902, 74.78159856730],
        [12011.825, 1.92062131635, 1021.24889455140],
        [8394.731, 0.67816895547, 146.59425171800],
        [7571.800, 1.07149263431, 388.46515523820],
        [5720.852, 2.59059512267, 4.45341812490],
        [4839.672, 1.90685991070, 41.10198105440],
        [4483.492, 2.90573457534, 529.69096509460],
        [4270.202, 3.41343865825, 453.42489381900],
        [4353.790, 0.67985662370, 32.19514480460],
        [4420.804, 1.74993796503, 108.46121608020],
        [2881.063, 1.98600105123, 137.03302416240],
        [2635.535, 3.09755943422, 213.29909543800],
        [3380.930, 0.84810683275, 183.24281464750],
        [2878.942, 3.67415901855, 350.33211960040],
        [2306.293, 2.80962935724, 70.32818044240],
        [2530.149, 5.79839567009, 490.07345674850],
        [2523.132, 0.48630800015, 493.04240216510],
        [2087.303, 0.61858378281, 33.94024994380],
        [1976.522, 5.11703044560, 168.05251279940],
        [1905.254, 1.72186472126, 182.27960680100],
        [1654.039, 1.92782545887, 145.10977900970],
        [1435.072, 1.70005157785, 484.44438245600],
        [1403.029, 4.58914203187, 498.67147645760],
        [1499.193, 1.01623299513, 219.89137757700],
        [1398.860, 0.76220317620, 176.65053250850],
        [1403.377, 6.07659416908, 173.68158709190],
        [1128.560, 5.96661179805, 9.56122755560],
        [1228.304, 1.59881465324, 77.75054398390],
        [835.414, 3.97066884218, 114.39910691340],
        [811.186, 3.00258880870, 46.20979048510],
        [731.925, 2.10447054189, 181.75834193920],
        [615.781, 2.97874625677, 106.97674337190],
        [704.778, 1.18738210880, 256.53994050650],
        [502.040, 1.38657803368, 5.93789083320],
        [530.357, 4.24059166485, 111.43016149680],
        [437.096, 2.27029212923, 1550.93985964600],
        [400.250, 1.25609325435, 8.07675484730],
        [421.011, 1.89084929506, 30.71067209630],
        [382.457, 3.29965259685, 983.11585891360],
        [422.485, 5.53186169605, 525.49817940060],
        [355.389, 2.27847846648, 218.40690486870],
        [280.062, 1.54129714238, 98.89998852460],
        [314.499, 3.95932948594, 381.35160823740],
        [280.556, 4.54238271682, 44.72531777680],
        [267.738, 5.13323364247, 112.91463420510],
        [333.311, 5.75067616021, 39.09624348430],
        [291.625, 4.02398326341, 68.84370773410],
        [321.429, 1.50625025822, 454.90936652730],
        [309.196, 2.85452752153, 72.07328558160],
        [345.094, 1.35905860594, 293.18850343600],
        [307.439, 0.31964571332, 601.76425067620],
        [251.356, 3.53992782846, 312.19908396260],
        [248.152, 3.41078346726, 37.61177077600],
        [306.000, 2.72475094464, 6244.94281435360],
        [293.532, 4.89079857814, 528.20649238630],
        [234.479, 0.59231043427, 42.58645376270],
        [239.628, 3.16441455173, 143.62530630140],
        [214.523, 3.62480283040, 278.25883401880],
        [246.198, 1.01506302015, 141.22580985640],
        [174.089, 5.55011789988, 567.82400073240],
        [163.934, 2.10166491786, 2.44768055480],
        [162.897, 2.48946521653, 4.19278569400],
        [193.455, 1.58425287580, 138.51749687070],
        [155.323, 3.28425127954, 31.01948863700],
        [182.469, 2.45244890571, 255.05546779820],
        [177.846, 4.14773474853, 10175.15251057320],
        [174.413, 1.53042999914, 329.83706636550],
        [137.649, 3.34900537767, 0.96320784650],
        [161.011, 5.16655038482, 211.81462272970],
        [113.473, 4.96286007991, 148.07872442630],
        [128.823, 3.25521535448, 24.11838995730],
        [107.363, 3.26457701792, 1059.38193018920],
        [122.732, 5.39399536941, 62.25142559510],
        [120.529, 3.08050145518, 184.72728735580],
        [99.356, 1.92888554099, 28.57180808220],
        [97.713, 2.59474415429, 6.59228213900],
        [124.095, 3.11516750340, 221.37585028530],
        [124.693, 2.97042405451, 251.43213107580],
        [114.252, 0.25039919123, 594.65070367540],
        [111.006, 3.34276426767, 180.27386923090],
        [120.939, 1.92914010593, 25.60286266560],
        [104.667, 0.94883561775, 395.57870223900],
        [109.779, 5.43147520571, 494.52687487340],
        [96.919, 0.86184760695, 1014.13534755060],
        [98.685, 0.89577952710, 488.58898404020],
        [88.968, 4.78109764779, 144.14657116320],
        [107.888, 0.98700578434, 1124.34166877000],
        [97.067, 2.62667400276, 291.70403072770],
        [75.131, 5.88936524779, 43.24084506850],
        [93.718, 6.09873565184, 526.72201967800],
        [94.822, 0.20662943940, 456.39383923560],
        [70.036, 2.39683345663, 426.59819087600],
        [77.187, 4.21076753240, 105.49227066360],
        [89.874, 3.25100749923, 258.02441321480],
        [69.133, 4.93031154435, 1028.36244155220],
        [90.657, 1.69466970587, 366.48562929500],
        [74.242, 3.14479101276, 82.85835341460],
        [57.995, 0.86159785905, 60.76695288680],
        [78.695, 1.09307575550, 700.66423920080],
        [57.230, 0.81331949225, 2.92076130680],
        [63.443, 4.39590123005, 149.56319713460],
        [55.698, 3.89047249911, 47.69426319340],
        [56.430, 5.15003563302, 0.52126486180],
        [56.174, 5.42986960794, 911.04257333200],
        [61.746, 6.16453667559, 1019.76442184310],
        [70.503, 0.08077330612, 40.58071619260],
        [74.677, 4.85904499980, 186.21176006410],
        [61.861, 4.78702599861, 11.04570026390],
        [61.135, 0.83712253227, 1022.73336725970],
        [61.268, 5.70228826765, 178.13500521680],
        [52.887, 0.37458943972, 27.08733537390],
        [56.722, 3.52318112447, 216.92243216040],
        [48.819, 5.10789123481, 64.95973858080],
        [63.290, 4.39424910030, 807.94979911340],
        [64.062, 6.28297531806, 7.11354700080],
        [46.356, 1.34735469284, 451.94042111070],
        [60.540, 3.40316162416, 294.67297614430],
        [46.900, 0.17048203552, 7.42236354150],
        [56.766, 0.45048868231, 140.00196957900],
        [55.887, 1.06815733757, 172.19711438360],
        [53.761, 2.79644687008, 328.35259365720],
        [43.828, 6.04655696644, 135.54855145410],
        [49.549, 0.64106656292, 41.05379694460],
        [53.960, 2.91774494436, 563.63121503840],
        [42.961, 5.40175361431, 487.36514376280],
        [51.508, 0.09105540708, 210.33015002140],
        [41.889, 3.12343223889, 29.22619938800],
        [47.655, 3.90701760087, 63.73589830340],
        [41.639, 6.26847783513, 32.71640966640],
        [41.429, 4.45464156759, 37.16982779130],
        [40.745, 0.16043648294, 79.23501669220],
        [48.205, 1.84198373010, 403.13419222450],
        [36.912, 0.44771386183, 30.05628079050],
        [47.762, 0.88083849566, 3302.47939106200],
        [39.465, 3.50565484069, 357.44566660120],
        [42.139, 0.63375113663, 343.21857259960],
        [41.275, 1.36370496322, 31.23193695810],
        [42.612, 3.55270845713, 38.65430049960],
        [38.931, 5.26691753270, 415.29185818120],
        [38.967, 5.25866056502, 386.98068252990],
        [33.734, 5.24400184426, 67.35923502580],
        [40.879, 3.55292279438, 331.32153907380],
        [38.768, 1.12288359393, 38.18121974760],
        [37.500, 6.08687972441, 35.42472265210],
        [38.831, 4.67876780698, 38.08485152800],
        [38.231, 6.26491054328, 389.94962794650],
        [29.976, 4.45759985804, 22.63391724900],
        [31.356, 0.07746010366, 12.53017297220],
        [26.341, 4.59559782754, 106.01353552540],
        [27.465, 5.99541587890, 206.18554843720],
        [25.152, 4.49867760320, 34.20088237470],
        [24.122, 5.17089441917, 129.91947716160],
        [28.997, 3.64927210210, 253.57099508990],
        [27.173, 4.37944546475, 142.14083359310],
        [30.634, 1.59348806560, 348.84764689210],
        [31.464, 1.05065113524, 100.38446123290],
        [24.056, 1.02801635413, 41.75637236020],
        [22.632, 4.72511111292, 81.37388070630],
        [21.942, 3.48416607882, 69.15252427480],
        [26.333, 3.01556008632, 365.00115658670],
        [22.355, 3.92220883921, 5.10780943070],
        [22.498, 4.03487494425, 19.12245511120],
        [22.885, 1.58977064672, 189.39315380180],
        [26.520, 3.61427038042, 367.97010200330],
        [25.496, 2.43810518614, 351.81659230870],
        [19.111, 2.59694457001, 2080.63082474060],
        [19.640, 6.15701741238, 35.21227433100],
        [25.688, 2.00512719767, 439.78275515400],
        [21.613, 3.32354204724, 119.50691634410],
        [25.389, 4.74025836522, 1474.67378837040],
        [18.107, 5.35129342595, 244.31858407500],
        [23.295, 5.93767742799, 316.39186965660],
        [22.087, 4.81594755148, 84.34282612290],
        [16.972, 3.05105149940, 220.41264243880],
        [20.022, 4.99276451168, 179.09821306330],
        [20.370, 1.86508317889, 171.23390653710],
        [19.426, 2.04829970231, 5.41662597140],
        [22.628, 0.27205783433, 666.72398925700],
        [19.072, 3.70882976684, 164.12035953630],
        [17.969, 3.40425338171, 69.36497259590],
        [18.716, 0.90215956591, 285.37238101960],
        [15.889, 0.42011285882, 697.74347789400],
        [14.988, 3.08544843665, 704.85702489480],
        [14.774, 3.36129613309, 274.06604832480],
        [15.972, 1.82864185268, 477.33083545520],
        [13.892, 2.94161501165, 38.39366806870],
        [13.922, 2.85574364078, 37.87240320690],
        [15.481, 4.94982954853, 101.86893394120],
        [17.571, 5.82317632469, 35.68535508300],
        [15.856, 5.04973561582, 36.90919536040],
        [16.414, 3.63049397028, 45.24658263860],
        [17.158, 2.51251149482, 20.60692781950],
        [12.941, 3.03041555329, 522.57741809380],
        [15.752, 5.00292909214, 247.23934538180],
        [12.679, 0.20331109568, 460.53844081980],
        [16.260, 5.93480347217, 815.06334611420],
        [12.903, 3.51141502996, 446.31134681820],
        [13.891, 5.51064697670, 31.54075349880],
        [13.668, 5.45576135320, 39.35687591520],
        [13.418, 3.95805150079, 290.21955801940],
        [15.368, 2.45783892707, 26.82670294300],
        [14.246, 3.18588280921, 401.64971951620],
        [12.222, 4.94370170146, 14.01464568050],
        [15.484, 3.79703715637, 404.61866493280],
        [13.427, 3.79527836573, 151.04766984290],
        [14.450, 4.93940408761, 120.99138905240],
        [14.331, 4.71117327722, 738.79727483860],
        [11.566, 5.91003539239, 536.80451209540],
        [15.578, 2.91836788254, 875.83029900100],
        [13.124, 2.16056013419, 152.53214255120],
        [11.744, 2.94770244071, 2.70831298570],
        [12.793, 1.97868575679, 1.37259812370],
        [12.969, 0.00535826017, 97.41551581630],
        [13.891, 4.76435441820, 0.26063243090],
        [13.729, 2.32306473850, 38.24491022240],
        [10.714, 6.18129683877, 115.88357962170],
        [11.610, 4.61712859898, 178.78939652260],
        [11.257, 0.79300245838, 42.32582133180],
        [14.500, 5.44690193314, 44.07092647100],
        [11.534, 5.26580538005, 160.93896579860],
        [13.355, 5.20849186729, 32.45577723550],
        [13.658, 2.15687632802, 476.43131808350],
        [13.782, 3.47865209163, 38.02116105320],
        [12.714, 2.09462988855, 20.49505323490],
        [13.257, 5.15138524813, 103.09277421860],
        [9.715, 0.74597883480, 918.15612033280],
        [10.340, 5.38977407079, 222.86032299360],
        [13.357, 5.89635739027, 748.09786996330],
        [12.632, 1.20306997433, 16.15350969460],
        [11.437, 1.58444114292, 495.49008271990],
        [11.424, 4.74142930795, 487.62577619370],
        [9.098, 5.19932138822, 118.02244363580],
        [9.336, 0.97313630925, 662.53120356300],
        [9.827, 4.48170250645, 505.78502345840],
        [8.585, 0.20375451897, 944.98282327580],
        [8.875, 5.53111742265, 17.52610781830],
        [9.957, 4.03258125243, 169.53698550770],
        [11.506, 3.11649121817, 17.63798240290],
        [9.818, 5.20376439002, 1.59634729290],
        [10.160, 3.74441320429, 457.61767951300],
        [8.661, 0.31247523804, 1440.73353842660],
        [8.496, 1.06445636872, 55.77101804070],
        [11.162, 1.92907800408, 564.85505531580],
        [8.057, 0.31116345866, 377.41945497430],
        [9.851, 4.23328578127, 418.26080359780],
        [7.938, 2.40417397694, 488.37653571910],
        [9.894, 0.63707319139, 183.76407950930],
        [9.913, 3.94049519088, 441.26722786230],
        [7.867, 3.87469522964, 494.73932319450],
        [7.589, 3.15909316566, 416.77633088950],
        [8.496, 5.38968698704, 104.00779795530],
        [9.716, 3.06038536864, 166.56804009110],
        [9.377, 0.56416645296, 673.31627139600],
        [8.771, 5.24534141981, 1057.89745748090],
        [7.990, 1.55726966638, 59.28248017850],
        [9.090, 4.32953439022, 29.74746424980],
        [9.667, 5.89033222679, 358.40887444770],
        [7.209, 2.29464803358, 79.18683258240],
        [8.062, 0.44458003524, 19.01058052660],
        [8.254, 3.47304582051, 156.15547927360],
        [9.804, 6.06393995615, 784.74643289280],
        [8.516, 5.99060386955, 180.79513409270],
        [8.090, 1.38588221442, 1654.03263386460],
        [9.074, 4.03971490460, 1017.05610885740],
        [6.908, 1.41919832926, 178.34745353790],
        [8.230, 2.53750470473, 518.38463239980],
        [8.594, 5.29104206063, 457.87831194390],
        [6.769, 5.43380191356, 171.98466606250],
        [8.571, 0.35876828441, 636.66770846650],
        [8.995, 1.36992508507, 6209.77872413240],
        [6.641, 2.92327140872, 0.04818410980],
        [9.278, 3.80308677009, 25558.21217647960],
        [6.567, 4.01934954352, 0.11187458460],
        [6.441, 4.28250687347, 36.12729806770],
        [7.257, 4.09776235307, 326.86812094890],
        [8.384, 5.49363770202, 532.61172640140],
        [7.471, 4.62144262894, 526.98265210890],
        [7.500, 0.61545750834, 485.92885516430],
        [7.716, 1.04880632264, 525.23754696970],
        [8.504, 2.79350586429, 10139.98842035200],
        [7.466, 5.07942174095, 157.63995198190],
        [7.186, 6.22833818429, 77.22927912210],
        [7.784, 1.89308880453, 984.60033162190],
        [6.513, 0.07498932215, 79.88940799800],
        [6.077, 2.96673519667, 36.69674703930],
        [7.706, 5.70632580790, 209.10630974400],
        [7.265, 4.94483532589, 131.40394986990],
        [6.984, 2.53239305821, 497.18700374930],
        [7.824, 2.31462643851, 513.07988101300],
        [7.175, 3.69203633127, 524.01370669230],
        [6.855, 0.14076801572, 283.62727588040],
        [6.922, 3.36515011915, 438.29828244570],
        [7.349, 3.50406958122, 500.15594916590],
        [6.301, 0.14776691217, 608.87779767700],
        [5.892, 4.24403528888, 4.66586644600],
        [7.613, 5.14905171677, 259.50888592310],
        [7.128, 5.92696788834, 482.95990974770],
        [6.829, 1.01745137848, 1543.82631264520],
        [5.981, 4.79954091087, 215.43795945210],
        [5.526, 2.34003154732, 65.22037101170],
        [6.817, 6.12162829690, 395.05743737720],
        [5.369, 3.76855960849, 52099.54021187280],
        [5.776, 5.61434462641, 987.56927703850],
        [7.523, 5.60432148128, 2810.92146160520],
        [7.329, 3.76815551582, 1512.80682400820],
        [5.616, 2.13872867116, 145.63104387150],
        [5.258, 0.30850836910, 36.60037881970],
        [5.688, 1.82274388581, 1227.43444298860],
        [5.658, 2.35049199704, 5.62907429250],
        [6.135, 4.23390561816, 496.01134758170],
        [5.128, 2.89050864873, 313.68355667090],
        [6.472, 3.49494191669, 552.69738935910],
        [4.983, 3.91958511552, 10135.53500222710],
        [5.217, 0.40052635702, 319.31263096340],
        [4.952, 1.42482088612, 49.17873590170],
        [5.964, 5.70758449643, 309.79958751760],
        [5.091, 6.00974510144, 1409.71404978960],
        [5.205, 5.50271334510, 238.90195810360],
        [4.800, 1.13450310670, 134.06407874580],
        [4.943, 1.43051344597, 422.40540518200],
        [5.604, 2.05669305961, 207.36120460480],
        [6.310, 5.22966882627, 139.74133714810],
        [4.772, 3.06668713747, 464.73122651380],
        [4.919, 3.57280542629, 52175.80628314840],
        [4.762, 5.90654311203, 838.96928775040],
        [4.848, 0.77467099227, 1.69692102940],
        [5.694, 0.77313415569, 709.96483432550],
        [5.455, 0.90289242792, 208.84567731310],
        [4.901, 3.79986913631, 15.49911838880],
        [4.772, 0.15755140037, 39.50563376150],
        [5.673, 2.68359159067, 1127.26243007680],
        [5.477, 0.53123497431, 113.87784205160],
        [5.077, 1.59268428609, 1547.97091422940],
        [4.981, 1.44584050478, 1.27202438720],
        [5.813, 5.85024085408, 57.25549074900],
        [5.520, 5.06396698257, 421.22974901440],
        [5.938, 0.96886308551, 6280.10690457480],
        [5.206, 3.58003819370, 474.94684537520],
        [5.256, 0.61005270999, 95.97922721780],
        [5.531, 5.28764137194, 36.76043751410],
        [6.158, 5.73176703797, 711.44930703380],
        [5.003, 2.19048397989, 501.64042187420],
        [5.150, 5.58407480282, 26049.77010593640],
        [5.138, 4.55234158942, 670.91677495100],
        [5.609, 4.37272759780, 52.80207262410],
        [5.636, 2.39183054397, 10210.31660079440],
        [4.512, 2.59978208967, 1234.54798998940],
        [5.412, 4.58813638089, 179.61947792510],
        [4.314, 3.38846714337, 142.66209845490],
        [4.708, 5.23537414423, 3.62333672240],
        [4.471, 3.94378336812, 12566.15169998280],
        [5.296, 1.12249063176, 134.11226285560],
        [4.188, 2.52490407427, 6205.32530600750],
        [4.645, 1.90644271528, 13324.31667116140],
        [4.502, 2.01956920977, 315.16802937920],
        [5.346, 2.94804816223, 353.04043258610],
        [4.177, 2.09489065926, 803.75701341940],
        [5.296, 3.88249567974, 2118.76386037840],
        [5.325, 4.28221258353, 477.91579079180],
        [5.519, 0.09960891963, 600.01914553700],
        [5.169, 0.59948596687, 6.90109867970],
        [4.179, 0.14619703083, 6644.57629047010],
        [4.490, 1.07042724999, 52139.15772021889],
        [3.970, 6.13227798578, 1553.90880506260],
        [3.970, 4.69887237362, 91.78644152380],
        [4.234, 0.14478458924, 65.87476231750],
        [5.183, 3.52837189306, 110.20632121940],
        [5.259, 6.20809827528, 142.71028256470],
        [3.869, 5.25125030487, 1558.05340664680],
        [4.457, 2.10248126544, 487.10451133190],
        [4.890, 1.83606790269, 46.51860702580],
        [3.875, 5.60269278935, 385.49620982160],
        [3.826, 1.30946706974, 2176.61005195840],
        [4.591, 4.84657580441, 1337.64076420800],
        [5.111, 1.18808079775, 981.63138620530],
        [4.709, 1.40878215308, 52213.93931878620],
        [3.891, 5.43661875415, 154.67100656530],
        [4.145, 4.32505910718, 363.51668387840],
        [4.441, 3.50158424570, 187.69623277240],
        [3.703, 2.48768949613, 67.88049988760],
        [4.094, 1.42347047260, 310.71461125430],
        [3.681, 5.70552661143, 491.66980404140],
        [4.787, 3.65822147476, 589.34595228860],
        [4.020, 5.45643059988, 6641.60734505350],
        [3.656, 0.57790726599, 491.44605487220],
        [4.288, 3.35265955957, 203.21660302060],
        [3.843, 4.61508898119, 1025.70231267630],
        [3.767, 0.05292047125, 320.27583880990],
        [4.632, 0.82011276589, 3265.83082813250],
        [4.609, 5.25443775917, 296.15744885260],
        [4.555, 5.30391170376, 26013.12154300690],
        [3.556, 4.80267245336, 224.34479570190],
        [4.859, 5.52756242256, 487.41332787260],
        [3.626, 1.44624342082, 70.84944530420],
        [4.302, 1.60914544159, 12529.50313705330],
        [3.493, 4.75315651083, 12489.88562870720],
        [3.722, 0.27433061822, 949.43624140070],
        [4.234, 5.25112033465, 194.28851491140],
        [3.451, 2.97409317928, 499.63468430410],
        [4.796, 6.21059766333, 491.81856188770],
        [3.639, 1.25605018211, 2603.20824283440],
        [4.646, 5.71392540144, 321.76031151820],
        [3.702, 2.08952561657, 491.03666459500],
        [3.672, 2.87489628704, 497.49582029000],
        [3.965, 1.05484988240, 75.74480641380],
        [3.416, 0.68584132933, 305.08553696180],
        [4.513, 4.38927002490, 425.11371816770],
        [3.853, 0.61321572401, 12526.53419163670],
        [3.788, 3.32221995840, 3140.01275492980],
        [3.781, 5.58125317044, 1652.54816115630],
        [3.903, 5.31609723466, 408.17831118040],
        [3.945, 3.60558877407, 1589.07289528380],
        [4.084, 0.83813879869, 52.36012963940],
        [4.084, 3.50290269471, 23.90594163620],
        [3.694, 1.03218855688, 481.47543703940],
        [3.636, 5.31068934607, 141.48644228730],
        [3.345, 3.94392179077, 20389.92252949249],
        [4.639, 6.24618220184, 821.39499582230],
        [3.934, 0.26992234338, 1655.51710657290],
        [4.431, 2.48647437800, 549.72844394250],
        [4.168, 5.39993754642, 236.50246165860],
        [4.020, 0.07393243012, 52136.18877480229],
        [4.055, 1.34004288978, 1054.92851206430],
        [3.275, 0.98533127454, 1344.75431120880],
        [3.213, 2.97105590703, 20386.95358407589],
        [4.428, 0.06728869735, 491.29729702590],
        [4.063, 0.06192838570, 6168.67674307800],
        [3.804, 5.34897033476, 523.75307426140],
        [3.917, 5.67905809516, 1131.19458333990],
        [3.833, 0.87811168267, 52.69019803950],
        [4.020, 2.69209723289, 1439.46151403940],
        [4.373, 1.86209663434, 73.55775828990],
        [3.159, 1.04693380342, 703.37255218650],
        [3.116, 5.20159166840, 449.23210812500],
        [3.258, 4.65131076542, 696.25900518570],
        [3.427, 0.27003884843, 2389.90914739640],
        [4.349, 0.07531141761, 20426.57109242200],
        [3.383, 5.61838426864, 699.22795060230],
        [3.305, 1.41666877290, 562.14674233010],
        [3.297, 5.46677712589, 1442.21801113490],
        [3.277, 2.71815883511, 980.14691349700],
        [3.171, 4.49510885866, 1439.24906571830],
        [4.175, 4.24327707038, 381.61224066830],
        [3.155, 3.40776789576, 39.72938293070],
        [4.112, 0.90309319273, 1087.69310584050],
        [3.350, 5.27474671017, 80.71948940050],
        [3.725, 1.52448613082, 1058.10990580200],
        [3.650, 3.59798316565, 192.80404220310],
        [3.837, 1.48519528444, 10098.88643929760],
        [2.959, 1.23012121982, 2500.11546861580],
        [3.330, 6.12470287875, 10172.18356515660],
        [3.361, 4.31837298696, 492.07919431860],
        [3.288, 3.14692435376, 347.36317418380],
        [2.992, 5.01304660316, 175.21424391000],
        [3.294, 2.52694043155, 1692.16566950240],
        [2.984, 1.81780659890, 175.11787569040],
        [3.013, 0.92957285991, 1515.77576942480],
        [3.863, 5.46044928570, 332.80601178210],
        [3.403, 1.10932483984, 987.30864460760],
        [3.312, 0.67710158807, 977.48678462110],
        [3.030, 1.77996261146, 156489.28581380739],
        [3.605, 4.89955108152, 1043.88281180040],
        [2.937, 0.60469671230, 990.22940591440],
        [3.276, 4.26765608367, 1189.30140735080],
        [2.966, 5.29808076929, 31.98269648350],
        [2.994, 2.58599359402, 178.08682110700],
        [3.905, 1.87748122254, 1158.28191871380],
        [3.110, 3.09203517638, 235.93301268700],
        [3.313, 2.70308129756, 604.47256366190],
        [3.276, 1.24440460327, 874.65464283340],
        [3.276, 5.58544609667, 950.92071410900],
        [3.746, 0.33859914037, 913.96333463880],
        [3.552, 3.07180917863, 240.38643081190],
        [2.885, 6.01130634957, 1097.51496582700],
        [3.643, 5.11977873355, 452.20105354160],
        [2.768, 4.38396269009, 391.43410065480],
        [2.776, 5.01821594830, 8.90683624980],
        [2.990, 5.62911695857, 140.65636088480],
        [2.761, 4.05534163807, 6283.07584999140],
        [3.226, 4.76711354367, 6241.97386893700],
        [3.748, 4.84009347869, 341.73409989130],
        [2.752, 4.53621078796, 6206.80977871580],
        [3.847, 2.40982343643, 26086.41866886590],
        [2.727, 3.28234198801, 483.48117460950],
        [2.884, 4.05452029151, 1.22384027740],
        [2.702, 3.72061244391, 946.46729598410],
        [2.723, 4.37517047024, 15.19030184810],
        [2.847, 5.22951186538, 661.04673085470],
        [2.680, 4.19379121323, 13.18456427800],
        [3.269, 0.43119778520, 496.97455542820],
        [3.489, 3.82213189319, 625.99451521810],
        [3.757, 3.88223872147, 495.70253104100],
        [2.872, 5.00345974886, 252.08652238160],
        [3.742, 2.03372773652, 8.59801970910],
        [3.172, 1.11135762382, 260.99335863140],
        [3.341, 2.91360557418, 304.23420369990],
        [2.915, 2.63627684599, 6681.22485339960],
        [2.915, 1.43773625890, 6604.95878212400],
        [2.629, 2.09824407450, 2713.41456405380],
        [2.901, 3.33924800230, 515.46387109300],
        [2.803, 1.23584865903, 6643.09181776180],
        [3.045, 3.33515866438, 921.07688163960],
        [2.699, 5.42597794650, 925.26966733360],
        [2.808, 5.77870303237, 1024.21783996800],
        [3.028, 3.75501312393, 511.59540830470],
        [3.090, 2.49453093252, 14.66903698630],
        [2.913, 4.83296711477, 515.93695184500],
        [3.139, 5.99134254710, 570.74476203920],
        [2.752, 3.08268180744, 853.19638175200],
        [2.779, 3.74527347899, 494.00561001160],
        [2.643, 1.99093797444, 470.21728845440],
        [2.763, 4.01095972177, 448.97147569410],
        [2.643, 5.24970673655, 249.94765836750],
        [3.426, 4.73955481174, 1050.99635880120],
        [2.573, 2.01267457287, 1514.29129671650],
        [2.633, 1.63640090603, 170.71264167530],
        [3.034, 4.48979734509, 560.71045373160],
        [3.025, 5.51446170055, 369.45457471160],
        [3.095, 4.01459691667, 1615.89959822680],
        [2.490, 0.15301603966, 78187.44335344699],
        [2.589, 0.79196093766, 1228.91891569690],
        [3.143, 5.33170343283, 1542.34183993690],
        [3.138, 4.50785484172, 461.76228109720],
        [2.812, 3.74246594120, 2.00573757010],
        [3.062, 4.88018345098, 227.96813242430],
        [2.553, 4.85437812287, 488.84961647110],
        [2.971, 1.27359129352, 530.91480537200],
        [2.646, 3.64828423565, 335.77495719870],
        [3.329, 2.71693827722, 171.02145821600],
        [2.648, 0.60243117586, 70.58881287330],
        [3.061, 5.05044834864, 378.64329525170],
        [2.738, 4.75405645015, 151.26011816400],
        [2.728, 5.89052930055, 213.95348674380],
        [3.411, 2.24137878065, 734.45573129830],
        [2.623, 0.54340876464, 1586.10394986720],
        [3.169, 5.84871429991, 1049.51188609290],
        [2.430, 2.34595493263, 450.45594840240],
        [2.907, 5.58085498481, 597.57146498220],
        [3.300, 0.94221473935, 58.17051448570],
        [2.543, 5.30426930256, 419.48464387520],
        [3.175, 2.32600231924, 339.28641933650],
        [2.858, 2.36621678719, 32.50396134530],
        [2.712, 5.79983621237, 1587.58842257550],
        [3.340, 1.36950315448, 384.27236954420],
        [3.301, 5.83023910521, 51.77517430280],
        [2.415, 0.69446923670, 489.55219188670],
        [2.736, 5.74320864965, 1167.84314626940],
        [2.956, 5.22962139507, 199.85389872910],
        [3.262, 0.01501002027, 1545.31078535350],
        [2.506, 4.84043333582, 943.49835056750],
        [3.240, 2.46676155925, 1016.79547642650],
        [3.148, 4.62079057738, 233.53351624200],
        [2.327, 4.10421417326, 70.11573212130],
        [2.371, 4.79963943424, 271.14528701800],
        [3.006, 3.66877796077, 1476.15826107870],
        [2.537, 5.66681769885, 21.14944454070],
        [3.006, 0.93048909480, 21.97952594320],
        [3.033, 0.67157488690, 292.48592802040],
        [2.344, 1.83547256266, 492.30868898220],
        [3.117, 2.76268894894, 1473.18931566210],
        [2.323, 2.88799980853, 533.62311835770],
        [2.340, 4.44862573253, 490.80716993140],
        [2.511, 0.99467349084, 266.10116806210],
        [2.919, 4.75889516601, 1511.32235129990],
        [2.493, 6.10541658597, 1225.94997028030],
        [2.798, 3.06162629894, 419.74527630610],
        [2.691, 3.20679023131, 463.50738623640],
        [2.291, 5.81534758547, 246.97871295090],
        [2.319, 6.05514281470, 525.75881183150],
        [3.112, 0.89712836583, 314.90739694830],
        [3.085, 5.84605938859, 1192.22216865760],
        [2.897, 0.54747024257, 20350.30502114640],
        [3.067, 2.22206306288, 248.46318565920],
        [2.252, 0.87483094907, 61.02758531770],
        [2.392, 3.62837597194, 439.19779981740],
        [2.817, 2.73562306571, 16.67477455640],
        [2.379, 6.17876088396, 467.65198782060],
        [2.598, 4.82643304253, 384.58118608490],
        [2.718, 1.01823841209, 215.95922431390],
        [2.998, 1.09755715300, 1964.74724511890],
        [2.884, 2.97813466834, 383.09671337660],
        [2.231, 4.48841493844, 4.14460158420],
        [2.203, 2.23336308907, 481.26298871830],
        [2.260, 2.35404913660, 659.61044225620],
        [2.491, 1.70236357070, 445.34813897170],
        [3.041, 5.55577674116, 674.80074410430],
        [2.289, 1.18497528002, 1552.42433235430],
        [2.975, 0.48272389481, 1052.48083150950],
        [2.339, 0.75318738767, 478.81530816350],
        [3.011, 0.16359500858, 54.28654533240],
        [2.820, 6.18522693724, 556.51766803760],
        [2.266, 5.91286000054, 3.49021027840],
        [2.231, 1.45038594906, 196.50670080260],
    ],
    # R1
    [
        [236338.502, 0.70498011235, 38.13303563780],
        [13220.279, 3.32015499895, 1.48447270830],
        [8621.863, 6.21628951630, 35.16409022120],
        [2701.740, 1.88140666779, 39.61750834610],
        [2153.150, 5.16873840979, 76.26607127560],
        [2154.735, 2.09431198086, 2.96894541660],
        [1463.924, 1.18417031047, 33.67961751290],
        [1603.165, 0.00000000000, 0.00000000000],
        [1135.773, 3.91891199655, 36.64856292950],
        [897.650, 5.24122933533, 388.46515523820],
        [789.908, 0.53315484580, 168.05251279940],
        [760.030, 0.02051033644, 182.27960680100],
        [607.183, 1.07706500350, 1021.24889455140],
        [571.622, 3.40060785432, 484.44438245600],
        [560.790, 2.88685815667, 498.67147645760],
        [490.190, 3.46830928696, 137.03302416240],
        [264.093, 0.86220057976, 4.45341812490],
        [270.526, 3.27355867939, 71.81265315070],
        [203.524, 2.41820674409, 32.19514480460],
        [155.438, 0.36537064534, 41.10198105440],
        [132.766, 3.60157672619, 9.56122755560],
        [93.626, 0.66670888163, 46.20979048510],
        [83.317, 3.25992461673, 98.89998852460],
        [72.205, 4.47717435693, 601.76425067620],
        [68.983, 1.46326969479, 74.78159856730],
        [86.953, 5.77228651853, 381.35160823740],
        [68.717, 4.52563942435, 70.32818044240],
        [64.724, 3.85477388838, 73.29712585900],
        [68.377, 3.39509945953, 108.46121608020],
        [53.375, 5.43650770516, 395.57870223900],
        [44.453, 3.61409723545, 2.44768055480],
        [41.243, 4.73866592865, 8.07675484730],
        [48.331, 1.98568593981, 175.16605980020],
        [41.744, 4.94257598763, 31.01948863700],
        [44.102, 1.41744904844, 1550.93985964600],
        [41.170, 1.41999374753, 490.07345674850],
        [41.099, 4.86312637841, 493.04240216510],
        [36.267, 5.30764043577, 312.19908396260],
        [36.284, 0.38187812797, 77.75054398390],
        [40.619, 2.27237172464, 529.69096509460],
        [32.360, 5.91123007786, 5.93789083320],
        [31.197, 2.70549944134, 1014.13534755060],
        [32.730, 5.22147683115, 41.05379694460],
        [36.079, 4.87817494829, 491.55792945680],
        [30.181, 3.63273193845, 30.71067209630],
        [29.991, 3.30769367603, 1028.36244155220],
        [27.048, 1.77647060739, 44.72531777680],
        [27.756, 4.55583165091, 7.11354700080],
        [27.475, 0.97228280623, 33.94024994380],
        [24.944, 3.10083391185, 144.14657116320],
        [25.958, 2.99724758632, 60.76695288680],
        [21.369, 4.71270048898, 278.25883401880],
        [21.283, 0.68957829113, 251.43213107580],
        [23.727, 5.12044184469, 176.65053250850],
        [21.392, 0.86286397645, 4.19278569400],
        [23.373, 1.64955088447, 173.68158709190],
        [24.163, 3.56602004577, 145.10977900970],
        [20.238, 5.61479765982, 24.11838995730],
        [26.958, 4.14294870704, 453.42489381900],
        [24.048, 1.00718363213, 213.29909543800],
        [18.322, 1.98028683488, 72.07328558160],
        [18.266, 6.17260374467, 189.39315380180],
        [19.201, 4.65162168927, 106.97674337190],
        [17.606, 1.60307551767, 62.25142559510],
        [16.545, 1.69931816587, 357.44566660120],
        [20.132, 3.29520553529, 114.39910691340],
        [15.425, 4.38812302799, 25.60286266560],
        [19.173, 2.20014267311, 343.21857259960],
        [15.077, 3.66802659382, 0.52126486180],
        [14.029, 0.55336333290, 129.91947716160],
        [13.361, 5.85751083720, 68.84370773410],
        [15.357, 4.20731277007, 567.82400073240],
        [12.746, 3.52815836608, 477.33083545520],
        [11.724, 5.57647263460, 31.23193695810],
        [11.533, 0.89138506506, 594.65070367540],
        [10.508, 4.35552732772, 32.71640966640],
        [10.826, 5.21826226871, 26.82670294300],
        [10.085, 1.98102855874, 40.58071619260],
        [10.518, 5.27281360238, 2.92076130680],
        [9.207, 0.50092534158, 64.95973858080],
        [9.231, 0.68180977710, 160.93896579860],
        [8.735, 5.80657503476, 6.59228213900],
        [10.114, 4.51164596694, 28.57180808220],
        [10.392, 5.18877536013, 42.58645376270],
        [9.873, 3.76512158080, 181.75834193920],
        [8.350, 2.82449631025, 43.24084506850],
        [9.838, 1.49438763600, 47.69426319340],
        [7.645, 4.07503370297, 389.94962794650],
        [8.004, 2.78082277326, 505.78502345840],
        [7.440, 2.35731983047, 11.04570026390],
        [7.342, 1.62279119952, 135.54855145410],
        [9.450, 0.27241261915, 426.59819087600],
        [7.192, 0.82841201068, 911.04257333200],
        [6.979, 1.86753914872, 206.18554843720],
        [6.874, 0.83802906828, 82.85835341460],
        [7.897, 1.86554246391, 38.65430049960],
        [6.729, 3.98338053636, 12.53017297220],
        [6.357, 0.90093123522, 487.36514376280],
        [6.720, 1.33936040700, 220.41264243880],
        [7.695, 5.13312500855, 23.90594163620],
        [7.059, 5.99832463494, 639.89728631400],
        [8.302, 3.85960902325, 37.61177077600],
        [6.412, 2.41743702679, 1059.38193018920],
        [6.751, 1.96860894470, 45.24658263860],
        [6.431, 4.07813226506, 35.68535508300],
        [5.517, 3.81325790890, 815.06334611420],
        [5.562, 0.41619602150, 563.63121503840],
        [6.115, 2.10934525342, 697.74347789400],
        [6.216, 4.79301628209, 143.62530630140],
        [5.346, 3.13071964722, 386.98068252990],
        [5.245, 6.06245070403, 171.23390653710],
        [5.129, 0.79394555531, 179.09821306330],
        [5.168, 4.73765992885, 522.57741809380],
        [6.422, 0.64684316894, 350.33211960040],
        [5.006, 2.37645082899, 77.22927912210],
        [5.005, 4.70632786971, 460.53844081980],
        [5.167, 5.20246616570, 446.31134681820],
        [5.119, 2.17338058771, 494.73932319450],
        [5.025, 4.21265519856, 536.80451209540],
        [4.722, 6.22814313946, 63.73589830340],
        [5.125, 5.38138329172, 179.31066138440],
        [4.918, 4.09031782903, 488.37653571910],
        [4.652, 5.10765073368, 274.06604832480],
        [4.711, 5.56542374115, 42.32582133180],
        [4.459, 1.30784829830, 69.36497259590],
        [5.485, 3.88088464259, 218.40690486870],
        [4.416, 3.05353893868, 27.08733537390],
        [4.559, 4.92224120952, 285.37238101960],
        [4.393, 4.18047835584, 5.41662597140],
        [4.687, 2.21401153210, 1029.84691426050],
        [4.644, 1.87902594973, 1433.61999142580],
        [5.639, 3.05596737234, 983.11585891360],
        [6.045, 5.68817982786, 351.81659230870],
        [4.430, 3.37768805833, 377.41945497430],
        [4.683, 2.14346624864, 97.41551581630],
        [5.845, 4.62301099402, 1024.21783996800],
        [4.536, 2.45860473853, 496.01134758170],
        [4.398, 5.65312496227, 3.93215326310],
        [4.287, 0.66340266603, 1012.65087484230],
        [4.086, 0.14551174994, 385.28376150050],
        [4.029, 5.98399329775, 178.34745353790],
        [4.276, 3.68205082970, 348.84764689210],
        [5.257, 3.75263242432, 379.86713552910],
        [4.012, 0.42559540783, 104313.47953065898],
        [4.025, 2.40645188238, 84.34282612290],
        [3.957, 0.86846121055, 171.98466606250],
        [3.961, 3.04953080906, 1017.31674128830],
        [5.559, 0.77714806229, 1447.84708542740],
        [5.071, 2.61075526868, 1536.71276564440],
        [4.052, 5.00014006312, 391.64654897590],
        [5.182, 4.73444634983, 382.83608094570],
        [3.763, 4.29449373755, 313.68355667090],
        [4.038, 2.82857942788, 1661.14618086540],
        [4.067, 5.73169928960, 169.53698550770],
        [3.841, 1.62580928420, 0.96320784650],
        [3.901, 2.70874386576, 14.01464568050],
        [3.721, 1.20062375429, 1026.87796884390],
        [3.911, 3.01809123569, 100.38446123290],
        [3.489, 4.28865448963, 1025.18104781450],
        [3.714, 5.05021268365, 292.48592802040],
        [3.816, 3.93084933114, 39.09624348430],
        [3.988, 2.82832650224, 134.11226285560],
        [3.745, 4.24728135115, 180.79513409270],
        [3.836, 1.02685786071, 1018.27994913480],
        [3.941, 5.21895739331, 183.76407950930],
        [4.669, 4.38080962573, 1066.49547719000],
        [3.780, 6.03723468132, 1022.73336725970],
        [3.647, 3.98130320367, 608.87779767700],
        [3.456, 5.54052355058, 846.08283475120],
        [4.047, 3.71041480907, 1018.06750081370],
        [3.865, 4.76002199091, 166.56804009110],
        [3.629, 3.29053233846, 447.79581952650],
        [3.564, 4.36703678321, 397.06317494730],
        [3.304, 1.49289552229, 1505.69327700740],
        [3.976, 2.42476188945, 106.01353552540],
        [4.217, 4.21677652639, 1052.26838318840],
        [3.294, 0.42088065654, 22.63391724900],
        [3.615, 3.68096122231, 494.52687487340],
        [3.230, 5.10786091356, 69.15252427480],
        [3.280, 3.62226152032, 531.17543780290],
        [3.337, 2.72502876320, 481.47543703940],
        [3.187, 0.08677634706, 399.51085550210],
        [3.389, 1.79454271219, 1519.92037100900],
        [3.179, 3.40418030121, 423.62924545940],
        [3.154, 3.69356460843, 470.21728845440],
        [3.706, 2.79048710497, 462.02291352810],
        [3.136, 4.38015969606, 385.49620982160],
        [3.122, 0.48346644637, 79.18683258240],
        [3.392, 0.48037804731, 521.09294538550],
        [3.465, 0.93152295589, 2183.72359895920],
        [3.735, 0.98809808606, 487.41332787260],
        [3.998, 3.38773325131, 6283.07584999140],
        [2.998, 2.61728063127, 487.62577619370],
        [3.295, 2.53821501556, 4.66586644600],
        [2.964, 3.66274645375, 495.49008271990],
        [3.901, 1.65463523144, 210.33015002140],
        [2.950, 1.99904237956, 872.90953769420],
        [2.948, 2.90769224206, 391.43410065480],
        [2.971, 0.31626092637, 5.10780943070],
        [3.085, 0.95725590904, 109.94568878850],
        [2.995, 3.34433305798, 394.09422953070],
        [3.126, 5.89472116854, 105.49227066360],
        [3.904, 3.01022809543, 556.51766803760],
        [3.388, 6.24936444215, 535.32003938710],
        [2.930, 6.15005257333, 164.12035953630],
        [3.267, 4.19718045293, 518.38463239980],
        [3.946, 2.88842759670, 151.26011816400],
        [3.076, 6.04134449219, 142.14083359310],
        [2.823, 0.60712626756, 214.78356814630],
        [2.917, 2.74502617182, 138.51749687070],
        [3.347, 6.09373507569, 6246.42728706190],
        [3.659, 5.12211619716, 79.23501669220],
        [3.010, 0.24656411754, 91.78644152380],
        [2.861, 6.17465663902, 422.40540518200],
        [2.989, 2.31620917965, 485.92885516430],
        [3.088, 2.29186342974, 110.20632121940],
        [3.030, 3.69866149100, 532.61172640140],
        [3.020, 2.36422658177, 290.21955801940],
        [3.170, 1.23078934548, 10176.63698328150],
        [2.652, 3.35836234807, 148.07872442630],
        [2.673, 6.03366372927, 196.50670080260],
        [2.630, 0.46957619348, 1970.42450352120],
        [2.599, 4.86022081674, 439.19779981740],
        [2.878, 2.61946597178, 488.58898404020],
        [2.720, 1.71836225398, 364.55921360200],
        [3.333, 3.25126857354, 30.05628079050],
        [3.053, 2.49346960035, 6243.45834164530],
        [3.062, 6.23776299963, 419.48464387520],
        [2.786, 0.83078219939, 497.18700374930],
        [2.834, 3.52926079424, 457.87831194390],
        [2.932, 1.80245810977, 500.15594916590],
        [3.030, 5.10152500393, 367.97010200330],
        [2.956, 5.76230870725, 986.08480433020],
        [3.116, 2.20042242739, 495.70253104100],
        [2.554, 0.65945973992, 67.35923502580],
        [2.901, 3.91891656185, 10173.66803786490],
        [2.840, 1.34453183591, 482.95990974770],
        [2.458, 1.20012815574, 489.11024890200],
        [2.556, 3.86921927085, 487.10451133190],
        [2.614, 1.51881085312, 463.50738623640],
        [2.386, 4.58400538443, 615.99134467780],
        [2.438, 5.19827220476, 501.11915701240],
        [2.537, 1.64802783144, 519.60847267720],
        [2.444, 3.87859489652, 185.24855221760],
        [2.795, 4.04265752580, 255.05546779820],
        [2.895, 3.26202698812, 1646.91908686380],
        [2.225, 5.75197574692, 605.95703637020],
        [2.324, 3.99503920129, 481.26298871830],
        [2.962, 1.74151265966, 2080.63082474060],
        [2.621, 1.74442251671, 35.21227433100],
    ],
    # R2
    [
        [4247.412, 5.89910679117, 38.13303563780],
        [217.570, 0.34581829080, 1.48447270830],
        [163.025, 2.23872947130, 168.05251279940],
        [156.285, 4.59414467342, 182.27960680100],
        [117.940, 5.10295026024, 484.44438245600],
        [112.429, 1.19000583596, 498.67147645760],
        [127.141, 2.84786298079, 35.16409022120],
        [99.467, 3.41578558739, 175.16605980020],
        [64.814, 3.46214064840, 388.46515523820],
        [77.286, 0.01659281785, 491.55792945680],
        [49.509, 4.06995509133, 76.26607127560],
        [39.330, 6.09521855958, 1021.24889455140],
        [36.450, 5.17130059988, 137.03302416240],
        [37.080, 5.97288967681, 2.96894541660],
        [30.484, 3.58259801313, 33.67961751290],
        [21.099, 0.76843555176, 36.64856292950],
        [13.886, 3.59248623971, 395.57870223900],
        [13.117, 5.09263515697, 98.89998852460],
        [11.379, 1.18060018898, 381.35160823740],
        [9.132, 2.34787658568, 601.76425067620],
        [8.527, 5.25134685897, 2.44768055480],
        [8.136, 4.96270726986, 4.45341812490],
        [7.417, 4.46775409796, 189.39315380180],
        [7.225, 1.92287508629, 9.56122755560],
        [7.289, 1.65519525780, 1028.36244155220],
        [8.076, 5.84268048311, 220.41264243880],
        [9.654, 0.00000000000, 0.00000000000],
        [6.554, 0.69397520733, 144.14657116320],
        [7.782, 1.14341656235, 1059.38193018920],
        [5.665, 6.25378258571, 74.78159856730],
        [5.628, 5.23383764266, 46.20979048510],
        [5.523, 4.59041448911, 1014.13534755060],
        [5.177, 5.23116646157, 477.33083545520],
        [5.503, 3.49522319102, 183.76407950930],
        [4.878, 3.52934357721, 39.61750834610],
        [4.787, 2.08260524745, 41.10198105440],
        [5.055, 0.19949888617, 166.56804009110],
        [4.751, 1.18054948270, 169.53698550770],
        [4.747, 1.50608965076, 73.29712585900],
        [6.113, 6.18326155595, 71.81265315070],
        [4.606, 3.91970908886, 587.53715667460],
        [5.756, 2.23667359233, 176.65053250850],
        [4.536, 2.84337336954, 7.11354700080],
        [4.338, 0.51553847388, 446.31134681820],
        [3.891, 0.26338839265, 1550.93985964600],
        [4.465, 3.01487041298, 129.91947716160],
        [3.727, 2.37977930658, 160.93896579860],
        [3.840, 3.79290381880, 111.43016149680],
        [4.142, 1.70293820961, 983.11585891360],
        [3.296, 1.07748822909, 505.78502345840],
        [4.008, 0.30663868827, 494.73932319450],
        [3.974, 5.97351783840, 488.37653571910],
        [3.925, 4.85736421123, 60.76695288680],
        [2.966, 2.01608546009, 822.17689311500],
        [3.972, 1.07780371834, 374.23806123660],
        [3.843, 5.23002047199, 350.33211960040],
        [2.848, 6.17799253802, 704.85702489480],
        [3.527, 0.79317138165, 274.06604832480],
        [2.828, 1.32275775835, 386.98068252990],
        [2.773, 5.37132330836, 251.43213107580],
        [3.113, 5.12622288690, 426.59819087600],
        [3.344, 5.61433537548, 1124.34166877000],
        [2.597, 0.67759426519, 312.19908396260],
        [2.581, 3.55847612121, 567.82400073240],
        [2.578, 1.45603792456, 1035.47598855300],
        [2.541, 5.19427579702, 1227.43444298860],
        [2.510, 4.12148891512, 171.23390653710],
        [2.511, 2.71606957319, 179.09821306330],
        [2.342, 0.96469916587, 1019.76442184310],
        [2.500, 0.70282276030, 707.77778620160],
        [2.480, 4.59623030219, 693.55069220000],
        [2.253, 0.74334306011, 976.00231191280],
    ],
    # R3
    [
        [166.297, 4.55243893489, 38.13303563780],
        [22.380, 3.94830879358, 168.05251279940],
        [21.348, 2.86296778794, 182.27960680100],
        [16.233, 0.54226725872, 484.44438245600],
        [15.623, 5.75702251906, 498.67147645760],
        [11.867, 4.40280192710, 1.48447270830],
        [6.448, 5.19003066847, 31.01948863700],
        [3.655, 5.91335292846, 1007.02180054980],
        [3.681, 1.62865545676, 388.46515523820],
        [3.198, 0.70197118575, 1558.05340664680],
        [3.243, 1.88035665980, 522.57741809380],
        [3.269, 2.94301808574, 76.26607127560],
        [2.688, 1.87062743473, 402.69224923980],
        [3.246, 0.79381356193, 536.80451209540],
        [2.650, 5.76858449026, 343.21857259960],
        [2.644, 4.64542905401, 500.15594916590],
        [2.541, 4.79217120822, 482.95990974770],
        [2.523, 1.72869889780, 395.57870223900],
        [2.690, 2.21096415618, 446.31134681820],
        [2.355, 5.77381398401, 485.92885516430],
        [2.874, 6.19643340540, 815.06334611420],
        [2.278, 3.66579603119, 497.18700374930],
    ],
    # R4
    [
        [4.227, 2.40375758563, 477.33083545520],
        [4.333, 0.10459484545, 395.57870223900],
        [3.545, 4.78431259422, 1028.36244155220],
        [3.154, 3.88192942366, 505.78502345840],
        [3.016, 1.03609346831, 189.39315380180],
        [2.294, 1.10879658603, 182.27960680100],
        [2.295, 5.67776133184, 168.05251279940],
    ],
]
"""This table contains Neptune's periodic terms (all of them) from the
planetary theory VSOP87 for the radius vector at the equinox of date (taken
from the 'D' solution). In Meeus' book a shortened version can be found in
page 454."""


ORBITAL_ELEM = [
    [304.348665, 219.8833092, 0.00030882, 0.000000018],     # L
    [30.110386869, -0.0000001663, 0.00000000069, 0.0],      # a
    [0.00945575, 0.000006033, 0.0, -0.00000000005],         # e
    [1.769953, -0.0093082, -0.00000708, 0.000000027],       # i
    [131.748057, 1.1022039, 0.00025952, -0.000000637],      # Omega
    [48.120276, 1.4262957, 0.00038434, 0.00000002]          # pie
]
"""This table contains the parameters to compute Neptune's orbital elements for
the mean equinox of date. Based in Table 31.A, page 213"""


ORBITAL_ELEM_J2000 = [
    [304.348665, 218.4862002, 0.00000059, -0.000000002],    # L
    [1.769953, 0.0002256, 0.00000023, 0.0],                 # i
    [131.748057, -0.0061651, -0.00000219, -0.000000078],    # Omega
    [48.120276, 0.0291866, 0.0000761, 0.0]                  # pie
]
"""This table contains the parameters to compute Neptune's orbital elements for
the standard equinox J2000.0. Based on Table 31.B, page 215"""


class Neptune(object):
    """
    Class Neptune models that planet.
    """

    @staticmethod
    def geometric_heliocentric_position(epoch, tofk5=True):
        """This method computes the geometric heliocentric position of planet
        Neptune for a given epoch, using the VSOP87 theory.

        :param epoch: Epoch to compute Neptune position, as an Epoch object
        :type epoch: :py:class:`Epoch`
        :param tofk5: Whether or not the small correction to convert to the FK5
            system will be applied or not
        :type tofk5: bool

        :returns: A tuple with the heliocentric longitude and latitude (as
            :py:class:`Angle` objects), and the radius vector (as a float,
            in astronomical units), in that order
        :rtype: tuple
        :raises: TypeError if input values are of wrong type.

        >>> epoch = Epoch(2018, 10, 27.0)
        >>> l, b, r = Neptune.geometric_heliocentric_position(epoch)
        >>> print(round(l.to_positive(), 4))
        345.3776
        >>> print(round(b, 4))
        -0.9735
        >>> print(round(r, 5))
        29.93966
        """

        return geometric_vsop_pos(epoch, VSOP87_L, VSOP87_B, VSOP87_R, tofk5)

    @staticmethod
    def apparent_heliocentric_position(epoch):
        """This method computes the apparent heliocentric position of planet
        Neptune for a given epoch, using the VSOP87 theory.

        :param epoch: Epoch to compute Neptune position, as an Epoch object
        :type epoch: :py:class:`Epoch`

        :returns: A tuple with the heliocentric longitude and latitude (as
            :py:class:`Angle` objects), and the radius vector (as a float,
            in astronomical units), in that order
        :rtype: tuple
        :raises: TypeError if input values are of wrong type.
        """

        return apparent_vsop_pos(epoch, VSOP87_L, VSOP87_B, VSOP87_R)

    @staticmethod
    def orbital_elements_mean_equinox(epoch):
        """This method computes the orbital elements of Neptune for the mean
        equinox of the date for a given epoch.

        :param epoch: Epoch to compute orbital elements, as an Epoch object
        :type epoch: :py:class:`Epoch`

        :returns: A tuple containing the following six orbital elements:
            - Mean longitude of the planet (Angle)
            - Semimajor axis of the orbit (float, astronomical units)
            - eccentricity of the orbit (float)
            - inclination on the plane of the ecliptic (Angle)
            - longitude of the ascending node (Angle)
            - argument of the perihelion (Angle)
        :rtype: tuple
        :raises: TypeError if input values are of wrong type.

        >>> epoch = Epoch(2065, 6, 24.0)
        >>> l, a, e, i, ome, arg = Neptune.orbital_elements_mean_equinox(epoch)
        >>> print(round(l, 6))
        88.321947
        >>> print(round(a, 8))
        30.11038676
        >>> print(round(e, 7))
        0.0094597
        >>> print(round(i, 6))
        1.763855
        >>> print(round(ome, 5))
        132.46986
        >>> print(round(arg, 6))
        -83.415521
        """

        return orbital_elements(epoch, ORBITAL_ELEM, ORBITAL_ELEM)

    @staticmethod
    def orbital_elements_j2000(epoch):
        """This method computes the orbital elements of Neptune for the
        standard equinox J2000.0 for a given epoch.

        :param epoch: Epoch to compute orbital elements, as an Epoch object
        :type epoch: :py:class:`Epoch`

        :returns: A tuple containing the following six orbital elements:
            - Mean longitude of the planet (Angle)
            - Semimajor axis of the orbit (float, astronomical units)
            - eccentricity of the orbit (float)
            - inclination on the plane of the ecliptic (Angle)
            - longitude of the ascending node (Angle)
            - argument of the perihelion (Angle)
        :rtype: tuple
        :raises: TypeError if input values are of wrong type.

        >>> epoch = Epoch(2065, 6, 24.0)
        >>> l, a, e, i, ome, arg = Neptune.orbital_elements_j2000(epoch)
        >>> print(round(l, 6))
        87.407029
        >>> print(round(a, 8))
        30.11038676
        >>> print(round(e, 7))
        0.0094597
        >>> print(round(i, 6))
        1.770101
        >>> print(round(ome, 5))
        131.74402
        >>> print(round(arg, 6))
        -83.6046
        """

        return orbital_elements(epoch, ORBITAL_ELEM, ORBITAL_ELEM_J2000)

    @staticmethod
    def geocentric_position(epoch):
        """This method computes the geocentric position of Neptune (right
        ascension and declination) for the given epoch, as well as the
        elongation angle.

        :param epoch: Epoch to compute geocentric position, as an Epoch object
        :type epoch: :py:class:`Epoch`

        :returns: A tuple containing the right ascension, the declination and
            the elongation angle as Angle objects
        :rtype: tuple
        :raises: TypeError if input value is of wrong type.

        >>> epoch = Epoch(1992, 12, 20.0)
        >>> ra, dec, elon = Neptune.geocentric_position(epoch)
        >>> print(ra.ra_str(n_dec=1))
        19h 17' 14.5''
        >>> print(dec.dms_str(n_dec=1))
        -21d 34' 15.1''
        >>> print(elon.dms_str(n_dec=1))
        19d 44' 59.6''
        """

        # First check that input value is of correct types
        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input type")
        # Compute the heliocentric position of Neptune
        l, b, r = Neptune.geometric_heliocentric_position(epoch, tofk5=False)
        # Compute the heliocentric position of the Earth
        l0, b0, r0 = Earth.geometric_heliocentric_position(epoch, tofk5=False)
        # Convert to radians
        lr = l.rad()
        br = b.rad()
        l0r = l0.rad()
        b0r = b0.rad()
        # Compute first iteration
        x = r * cos(br) * cos(lr) - r0 * cos(b0r) * cos(l0r)
        y = r * cos(br) * sin(lr) - r0 * cos(b0r) * sin(l0r)
        z = r * sin(br) - r0 * sin(b0r)
        delta = sqrt(x * x + y * y + z * z)
        tau = 0.0057755183 * delta
        # Adjust the epoch for light-time
        epoch -= tau
        # Compute again Neptune coordinates with this correction
        l, b, r = Neptune.geometric_heliocentric_position(epoch, tofk5=False)
        # Compute second iteration
        lr = l.rad()
        br = b.rad()
        x = r * cos(br) * cos(lr) - r0 * cos(b0r) * cos(l0r)
        y = r * cos(br) * sin(lr) - r0 * cos(b0r) * sin(l0r)
        z = r * sin(br) - r0 * sin(b0r)
        # Compute longitude and latitude
        lamb = atan2(y, x)
        beta = atan2(z, sqrt(x * x + y * y))
        # Now, let's compute the aberration effect
        t = (epoch - JDE2000) / 36525
        e = 0.016708634 + t * (-0.000042037 - t * 0.0000001267)
        pie = 102.93735 + t * (1.71946 + t * 0.00046)
        pie = radians(pie)
        lon = l0 + 180.0
        lon = lon.rad()
        k = 20.49552    # The constant of aberration
        deltal1 = k * (-cos(lon - lamb) + e * cos(pie - lamb)) / cos(beta)
        deltab1 = -k * sin(beta) * (sin(lon - lamb) - e * sin(pie - lamb))
        deltal1 = Angle(0, 0, deltal1)
        deltab1 = Angle(0, 0, deltab1)
        # Correction to FK5 system
        lamb = Angle(lamb, radians=True)
        lamb = lamb.to_positive()
        beta = Angle(beta, radians=True)
        l_prime = lamb - t * (1.397 + t * 0.00031)
        deltal2 = Angle(0, 0, -0.09033)
        a = 0.03916 * (cos(l_prime.rad()) + sin(l_prime.rad()))
        a = a * tan(b.rad())
        deltal2 += Angle(0, 0, a)
        deltab2 = 0.03916 * (cos(l_prime.rad()) - sin(l_prime.rad()))
        deltab2 = Angle(0, 0, deltab2)
        # Apply the corrections
        lamb = lamb + deltal1 + deltal2
        beta = beta + deltab1 + deltab2
        # Correction for nutation
        dpsi = nutation_longitude(epoch)
        lamb += dpsi
        e = true_obliquity(epoch)
        ra, dec = ecliptical2equatorial(lamb, beta, e)
        # Let's compute the elongation angle
        lons, lats, rs = Sun.apparent_geocentric_position(epoch)
        lambr = lamb.rad()
        lsr = lons.rad()
        betar = beta.rad()
        elon = acos(cos(betar) * cos(lambr - lsr))
        elon = Angle(elon, radians=True)
        return ra, dec, elon

    @staticmethod
    def conjunction(epoch):
        """This method computes the time of the conjunction closest to the
        given epoch.

        :param epoch: Epoch close to the desired conjunction
        :type epoch: :py:class:`Epoch`

        :returns: The time when the conjunction happens, as an Epoch
        :rtype: :py:class:`Epoch`
        :raises: TypeError if input value is of wrong type.
        :raises: ValueError if input epoch outside the -2000/4000 range.

        >>> epoch = Epoch(1993, 10, 1.0)
        >>> conj = Neptune.conjunction(epoch)
        >>> y, m, d = conj.get_date()
        >>> print(y)
        1994
        >>> print(m)
        1
        >>> print(round(d, 4))
        11.3057
        """

        # First check that input value is of correct types
        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input type")
        # Check that the input epoch is within valid range
        y = epoch.year()
        if y < -2000.0 or y > 4000.0:
            raise ValueError("Epoch outside the -2000/4000 range")
        # Set some specific constants for Neptune's conjunction
        a = 2451569.379
        b = 367.486703
        m0 = 21.5569
        m1 = 2.194998
        k = round((365.2425 * y + 1721060.0 - a) / b)
        jde0 = a + k * b
        m = m0 + k * m1
        m = Angle(m).to_positive()
        m = m.rad()
        t = (jde0 - 2451545.0) / 36525.0
        # Compute a couple auxiliary angles
        ee = 207.83 + 8.51 * t
        gg = 276.74 + 209.98 * t
        # Convert to radians
        ee = Angle(ee).rad()
        gg = Angle(gg).rad()
        corr = (0.0168 +
                sin(m) * (-2.5606 + t * (0.0088 + t * 0.00002)) +
                cos(m) * (-0.8611 + t * (-0.0037 + t * 0.00002)) +
                sin(2.0 * m) * (0.0118 + t * (-0.0004 + t * 0.00001)) +
                cos(2.0 * m) * (0.0307 - t * 0.0003) +
                cos(ee) * (-0.5964) +
                cos(gg) * (0.0728))
        to_return = jde0 + corr
        return Epoch(to_return)

    @staticmethod
    def opposition(epoch):
        """This method computes the time of the opposition closest to the given
        epoch.

        :param epoch: Epoch close to the desired opposition
        :type epoch: :py:class:`Epoch`

        :returns: The time when the opposition happens, as an Epoch
        :rtype: :py:class:`Epoch`
        :raises: TypeError if input value is of wrong type.
        :raises: ValueError if input epoch outside the -2000/4000 range.

        >>> epoch = Epoch(1846, 8, 1)
        >>> oppo = Neptune.opposition(epoch)
        >>> y, m, d = oppo.get_date()
        >>> print(y)
        1846
        >>> print(m)
        8
        >>> print(round(d, 4))
        20.1623
        """

        # First check that input value is of correct types
        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input type")
        # Check that the input epoch is within valid range
        y = epoch.year()
        if y < -2000.0 or y > 4000.0:
            raise ValueError("Epoch outside the -2000/4000 range")
        # Set some specific constants for Neptune's opposition
        a = 2451753.122
        b = 367.486703
        m0 = 202.6544
        m1 = 2.194998
        k = round((365.2425 * y + 1721060.0 - a) / b)
        jde0 = a + k * b
        m = m0 + k * m1
        m = Angle(m).to_positive()
        m = m.rad()
        t = (jde0 - 2451545.0) / 36525.0
        # Compute a couple auxiliary angles
        ee = 207.83 + 8.51 * t
        gg = 276.74 + 209.98 * t
        # Convert to radians
        ee = Angle(ee).rad()
        gg = Angle(gg).rad()
        corr = (-0.014 + t * t * 0.00001 +
                sin(m) * (-1.3486 + t * (0.001 + t * 0.00001)) +
                cos(m) * (0.8597 + t * 0.0037) +
                sin(2.0 * m) * (-0.0082 + t * (-0.0002 + t * 0.00001)) +
                cos(2.0 * m) * (0.0037 - t * 0.0003) +
                cos(ee) * (-0.5964) +
                cos(gg) * (0.0728))
        to_return = jde0 + corr
        return Epoch(to_return)

    @staticmethod
    def magnitude(sun_dist, earth_dist):
        """This function computes the approximate magnitude of Neptune.

        :param sun_dist: Distance from Neptune to Sun, in Astronomical Units
        :type sun_dist: float
        :param earth_dist: Distance Neptune to Earth, in Astronomical Units
        :type earth_dist: float

        :returns: Neptune's magnitude
        :rtype: float
        :raises: TypeError if input values are of wrong type.
        """

        if not (isinstance(sun_dist, float) and isinstance(earth_dist, float)):
            raise TypeError("Invalid input types")
        m = -7.05 + 5.0 * log10(sun_dist * earth_dist)
        return round(m, 1)


def main():

    # Let's define a small helper function
    def print_me(msg, val):
        print("{}: {}".format(msg, val))

    # Let's show some uses of Neptune class
    print("\n" + 35 * "*")
    print("*** Use of Neptune class")
    print(35 * "*" + "\n")

    # Let's now compute the heliocentric position for a given epoch
    epoch = Epoch(2018, 10, 27.0)
    lon, lat, r = Neptune.geometric_heliocentric_position(epoch)
    print_me("Geometric Heliocentric Longitude", lon.to_positive())
    print_me("Geometric Heliocentric Latitude", lat)
    print_me("Radius vector", r)

    print("")

    # Compute the geocentric position for 1992/12/20:
    epoch = Epoch(1992, 12, 20.0)
    ra, dec, elon = Neptune.geocentric_position(epoch)
    print_me("Right ascension", ra.ra_str(n_dec=1))
    print_me("Declination", dec.dms_str(n_dec=1))
    print_me("Elongation", elon.dms_str(n_dec=1))

    print("")

    # Print mean orbital elements for Neptune at 2065.6.24
    epoch = Epoch(2065, 6, 24.0)
    l, a, e, i, ome, arg = Neptune.orbital_elements_mean_equinox(epoch)
    print_me("Mean longitude of the planet", round(l, 6))  # 88.321947
    print_me("Semimajor axis of the orbit (UA)", round(a, 8))  # 30.11038676
    print_me("Eccentricity of the orbit", round(e, 7))  # 0.0094597
    print_me("Inclination on plane of the ecliptic", round(i, 6))  # 1.763855
    print_me("Longitude of the ascending node", round(ome, 5))  # 132.46986
    print_me("Argument of the perihelion", round(arg, 6))  # -83.415521

    print("")

    # Compute the time of the conjunction close to 1993/10/1
    epoch = Epoch(1993, 10, 1.0)
    conj = Neptune.conjunction(epoch)
    y, m, d = conj.get_date()
    d = round(d, 4)
    date = "{}/{}/{}".format(y, m, d)
    print_me("Conjunction date", date)

    # Compute the time of the opposition close to 1846/8/1
    epoch = Epoch(1846, 8, 1)
    oppo = Neptune.opposition(epoch)
    y, m, d = oppo.get_date()
    d = round(d, 4)
    date = "{}/{}/{}".format(y, m, d)
    print_me("Opposition date", date)


if __name__ == "__main__":

    main()
