/*
 * Stellarium
 * Copyright (C) 2016 Alexander Wolf
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Suite 500, Boston, MA  02110-1335, USA.
 */

#include "tests/testEphemeris.hpp"

#include <QDebug>
#include <QVariantList>
#include <QString>
#include <QtGlobal>

#include "StelFileMgr.hpp"
#include "EphemWrapper.hpp"
#include "vsop87.h"
#include "de430.hpp"
#include "de431.hpp"

QTEST_GUILESS_MAIN(TestEphemeris)

#define CENTRAL_BODY_ID 11  //ID of sun in JPL enumeration

//
// IMPORTANT NOTE: de430.cpp and de431.cpp files has hooks for unit test!
// Please update them if you changed StelCore::matJ2000ToVsop87 variable
//

void TestEphemeris::initTestCase()
{
	StelFileMgr::init();

	de430FilePath = StelFileMgr::findFile("ephem/" + QString(DE430_FILENAME), StelFileMgr::File);
	de431FilePath = StelFileMgr::findFile("ephem/" + QString(DE431_FILENAME), StelFileMgr::File);	

	if (!de430FilePath.isEmpty())
		qWarning() << "Use DE430 ephemeris file" << de430FilePath;

	if (!de431FilePath.isEmpty())
		qWarning() << "Use DE431 ephemeris file" << de431FilePath;

	// test data was obtained from http://ssd.jpl.nasa.gov/horizons.cgi#results

	// Mercury:
	// Start time      : B.C. 2000-Jan-01 00:00:00.0000 TDB ( 990923.500000000)
	// Stop  time      : A.D. 5976-Jan-01 00:00:00.0000 TDB (3903748.500000000)
	// Step-size       : 25 calendar years
	mercury <<   990923.5 <<  2.737051028862592E-01 << -3.032073077875030E-01 << -5.092260457237367E-02;
	mercury << 1000054.5 << -9.780316779199563E-02 << -4.544434119612545E-01 << -2.464407428526148E-02;
	mercury << 1009185.5 << -3.865026420790985E-01 << -1.970679266469892E-01 <<  2.427978457791408E-02;
	mercury << 1018316.5 << -2.398548294638118E-01 <<  2.378009289067007E-01 <<  4.248505015109032E-02;
	mercury << 1027448.5 <<  2.622380775378322E-01 <<  1.805336817325848E-01 << -1.286222543411248E-02;
	mercury << 1036579.5 <<  2.733264799187031E-01 << -3.041802467758867E-01 << -5.093338546957000E-02;
	mercury << 1045710.5 << -9.827896710154957E-02 << -4.544157944483015E-01 << -2.470845553759636E-02;
	mercury << 1054841.5 << -3.864960875013629E-01 << -1.962901060861803E-01 <<  2.418559402237671E-02;
	mercury << 1063973.5 << -2.628203491271739E-01 <<  2.195826208687079E-01 <<  4.339157403304508E-02;
	mercury << 1073104.5 <<  2.635758842689509E-01 <<  1.791315420303220E-01 << -1.299153362811812E-02;
	mercury << 1082235.5 <<  2.729503081333722E-01 << -3.051622535850105E-01 << -5.094438557096226E-02;
	mercury << 1091366.5 << -9.873830593941985E-02 << -4.543843126874692E-01 << -2.477397374192528E-02;
	mercury << 1100498.5 << -3.787480245726275E-01 << -2.184345677364048E-01 <<  2.155375002780093E-02;
	mercury << 1109629.5 << -2.614252625327254E-01 <<  2.204893434717379E-01 <<  4.328145687075907E-02;
	mercury << 1118760.5 <<  2.649073315642321E-01 <<  1.777061444756086E-01 << -1.312221106698514E-02;
	mercury << 1127891.5 <<  2.725711130938482E-01 << -3.061400416712122E-01 << -5.095520152489726E-02;
	mercury << 1137023.5 << -7.706956975058839E-02 << -4.577150407529333E-01 << -2.732775230171822E-02;
	mercury << 1146154.5 << -3.787852249102771E-01 << -2.176681376861202E-01 <<  2.146084833424147E-02;
	mercury << 1155285.5 << -2.600306657420036E-01 <<  2.213724054669923E-01 <<  4.317038916063113E-02;
	mercury << 1164416.5 <<  2.662497997354412E-01 <<  1.762765950585826E-01 << -1.325340163132772E-02;
	mercury << 1173548.5 <<  2.872242380974203E-01 << -2.858927875843607E-01 << -5.084708073545745E-02;
	mercury << 1182679.5 << -7.754885633321013E-02 << -4.577231925829736E-01 << -2.738839209711987E-02;
	mercury << 1191810.5 << -3.788289846202301E-01 << -2.168958009792178E-01 <<  2.136904870313526E-02;
	mercury << 1200941.5 << -2.586174635002441E-01 <<  2.222714520159141E-01 <<  4.305964217579610E-02;
	mercury << 1210073.5 <<  2.456706140624845E-01 <<  1.993344983744882E-01 << -9.301032036011127E-03;
	mercury << 1219204.5 <<  2.868821316605634E-01 << -2.869311500869164E-01 << -5.085813548081081E-02;
	mercury << 1228335.5 << -7.800668142583425E-02 << -4.577323887568406E-01 << -2.745067340509591E-02;
	mercury << 1237466.5 << -3.788625354869887E-01 << -2.161198655996524E-01 <<  2.127661748195891E-02;
	mercury << 1246598.5 << -2.802294306030420E-01 <<  2.027707506184460E-01 <<  4.368051270346329E-02;
	mercury << 1255729.5 <<  2.471379844509656E-01 <<  1.979687967892358E-01 << -9.441311133774987E-03;
	mercury << 1264860.5 <<  2.865535746180239E-01 << -2.879415225007288E-01 << -5.086810011398478E-02;
	mercury << 1273991.5 << -7.848754167264854E-02 << -4.577334157353907E-01 << -2.751081897710979E-02;
	mercury << 1283123.5 << -3.700002434387374E-01 << -2.376862254453935E-01 <<  1.857025351750154E-02;
	mercury << 1292254.5 << -2.789253295051547E-01 <<  2.037098582488567E-01 <<  4.357479897592964E-02;
	mercury << 1301385.5 <<  2.485938067495333E-01 <<  1.966040648411215E-01 << -9.580444143328761E-03;
	mercury << 1310516.5 <<  2.862179243019339E-01 << -2.889547554219582E-01 << -5.087810689575950E-02;
	mercury << 1319648.5 << -5.663970875784442E-02 << -4.599366986344122E-01 << -2.996440713695726E-02;
	mercury << 1328779.5 << -3.700730041854626E-01 << -2.369549363813905E-01 <<  1.847739720934309E-02;
	mercury << 1337910.5 << -2.775989818783621E-01 <<  2.046707424518093E-01 <<  4.346904439840237E-02;
	mercury << 1347041.5 <<  2.500267982738206E-01 <<  1.952484144066706E-01 << -9.716049302253034E-03;
	mercury << 1356173.5 <<  2.998303503841757E-01 << -2.677866395757525E-01 << -5.055306055325508E-02;
	mercury << 1365304.5 << -5.712759703570702E-02 << -4.599801601672042E-01 << -3.002056116886270E-02;
	mercury << 1374435.5 << -3.701539462032152E-01 << -2.362074184820628E-01 <<  1.838691870495597E-02;
	mercury << 1383566.5 << -2.762914623544597E-01 <<  2.056025045256869E-01 <<  4.336397176099169E-02;
	mercury << 1392698.5 <<  2.278031259814657E-01 <<  2.168794880774455E-01 << -5.721187392124692E-03;
	mercury << 1401829.5 <<  2.995490072665788E-01 << -2.688352721373500E-01 << -5.056294469067186E-02;
	mercury << 1410960.5 << -5.761292073198137E-02 << -4.600222375906051E-01 << -3.007667139837940E-02;
	mercury << 1420091.5 << -3.702295225283487E-01 << -2.354659767558345E-01 <<  1.829559373319016E-02;
	mercury << 1429223.5 << -2.964395355994465E-01 <<  1.849540851739298E-01 <<  4.370009421701471E-02;
	mercury << 1438354.5 <<  2.293514334370283E-01 <<  2.156174562569323E-01 << -5.863057531363128E-03;
	mercury << 1447485.5 <<  2.992521214362295E-01 << -2.698998621929891E-01 << -5.057286883453450E-02;
	mercury << 1456616.5 << -5.810093940516215E-02 << -4.600631971871462E-01 << -3.013221718937301E-02;
	mercury << 1465748.5 << -3.602721546787576E-01 << -2.564241573434918E-01 <<  1.552318117421811E-02;
	mercury << 1474879.5 << -2.952018943262640E-01 <<  1.859622933178150E-01 <<  4.359834201460911E-02;
	mercury << 1484010.5 <<  2.308964029430602E-01 <<  2.143355320307500E-01 << -6.006038208514579E-03;
	mercury << 1493141.5 <<  2.989660138003004E-01 << -2.709505511717439E-01 << -5.058258421897301E-02;
	mercury << 1502273.5 << -3.613575046801310E-02 << -4.611352194734688E-01 << -3.247689195810841E-02;
	mercury << 1511404.5 << -3.603878030035313E-01 << -2.557092789894917E-01 <<  1.543320503579240E-02;
	mercury << 1520535.5 << -2.939660168883322E-01 <<  1.869621664337711E-01 <<  4.349729716619295E-02;
	mercury << 1529666.5 <<  2.324450558015971E-01 <<  2.130494593154304E-01 << -6.148626428273308E-03;
	mercury << 1538798.5 <<  3.114738390878276E-01 << -2.488821583908569E-01 << -5.003839940036199E-02;
	mercury << 1547929.5 << -3.663403503988327E-02 << -4.612212512891983E-01 << -3.252820001293159E-02;
	mercury << 1557060.5 << -3.604916045888671E-01 << -2.550169262015450E-01 <<  1.534103469286398E-02;
	mercury << 1566191.5 << -2.927024156345379E-01 <<  1.879775066078546E-01 <<  4.339501733666705E-02;
	mercury << 1575323.5 <<  2.085949189447009E-01 <<  2.331824133865024E-01 << -2.126112184390725E-03;
	mercury << 1584454.5 <<  3.112238834093826E-01 << -2.500024333897477E-01 << -5.004944176652381E-02;
	mercury << 1593585.5 << -3.712140948710826E-02 << -4.612992182309762E-01 << -3.258010680423583E-02;
	mercury << 1602716.5 << -3.606026999338984E-01 << -2.542955641509775E-01 <<  1.525121728981779E-02;
	mercury << 1611848.5 << -3.113758053950736E-01 <<  1.662865485114378E-01 <<  4.345608818903222E-02;
	mercury << 1620979.5 <<  2.102336282581354E-01 <<  2.320020148416276E-01 << -2.272503244824944E-03;
	mercury << 1630110.5 <<  3.109771055558618E-01 << -2.511110406366078E-01 << -5.005999966555324E-02;
	mercury << 1639241.5 << -3.760703420088277E-02 << -4.613809187342739E-01 << -3.263211029027529E-02;
	mercury << 1648373.5 << -3.496025234134354E-01 << -2.745664729496690E-01 <<  1.242779166074641E-02;
	mercury << 1657504.5 << -3.102177281214844E-01 <<  1.673342271440410E-01 <<  4.335804555165156E-02;
	mercury << 1666635.5 <<  2.118670297888322E-01 <<  2.308271775389457E-01 << -2.417410319542437E-03;
	mercury << 1675766.5 <<  3.107217736409378E-01 << -2.522153281167170E-01 << -5.007010019110648E-02;
	mercury << 1684898.5 << -1.557277365252971E-02 << -4.613201627851550E-01 << -3.485996982067014E-02;
	mercury << 1694029.5 << -3.497468411270910E-01 << -2.738905738879937E-01 <<  1.233798175229267E-02;
	mercury << 1703160.5 << -3.090493755404146E-01 <<  1.683959792883534E-01 <<  4.326107766411438E-02;
	mercury << 1712291.5 <<  2.134715540262541E-01 <<  2.296516322883956E-01 << -2.559169623387791E-03;
	mercury << 1721423.5 <<  3.220630864420501E-01 << -2.293389393945036E-01 << -4.930890838763726E-02;
	mercury << 1730554.5 << -1.604762956922305E-02 << -4.614401349880796E-01 << -3.490888448330858E-02;
	mercury << 1739685.5 << -3.498932755045842E-01 << -2.731967347911381E-01 <<  1.224956431478299E-02;
	mercury << 1748816.5 << -3.078944000687556E-01 <<  1.694401524885940E-01 <<  4.316494006527646E-02;
	mercury << 1757948.5 <<  1.881634697509129E-01 <<  2.480858306432696E-01 <<  1.451470431608927E-03;
	mercury << 1767079.5 <<  3.218615088159129E-01 << -2.304934062971647E-01 << -4.932047783447383E-02;
	mercury << 1776210.5 << -1.653923014072170E-02 << -4.615583283973529E-01 << -3.495640474211341E-02;
	mercury << 1785341.5 << -3.500390904156545E-01 << -2.725082051584892E-01 <<  1.216112880508396E-02;
	mercury << 1794473.5 << -3.250413557707567E-01 <<  1.468132267091496E-01 <<  4.295703616756799E-02;
	mercury << 1803604.5 <<  1.898840678436862E-01 <<  2.470161481184188E-01 <<  1.303073483531581E-03;
	mercury << 1812735.5 <<  3.216583871841214E-01 << -2.316572423548924E-01 << -4.933240319359065E-02;
	mercury << 1821866.5 << -1.701901409662130E-02 << -4.616715296253830E-01 << -3.500444642134225E-02;
	mercury << 1830998.5 << -3.380420690629860E-01 << -2.920512408206175E-01 <<  9.298954624751105E-03;
	mercury << 1840129.5 << -3.239731249228009E-01 <<  1.479041685060108E-01 <<  4.286459047713662E-02;
	mercury << 1849260.5 <<  1.915974321169100E-01 <<  2.459301101692789E-01 <<  1.154546306661688E-03;
	mercury << 1858391.5 <<  3.214538287900236E-01 << -2.328191318050446E-01 << -4.934406440291714E-02;
	mercury << 1867523.5 <<  5.048040262854453E-03 << -4.604656122943123E-01 << -3.710899602182544E-02;
	mercury << 1876654.5 << -3.382251460245997E-01 << -2.913888166040822E-01 <<  9.212220567108985E-03;
	mercury << 1885785.5 << -3.229174950688269E-01 <<  1.489632243148048E-01 <<  4.277201323506895E-02;
	mercury << 1894916.5 <<  1.933320528900883E-01 <<  2.448306465290376E-01 <<  1.003660701323520E-03;
	mercury << 1904048.5 <<  3.315769584847365E-01 << -2.091299745531117E-01 << -4.836350078083840E-02;
	mercury << 1913179.5 <<  4.552752562969175E-03 << -4.606234884526168E-01 << -3.715210810633857E-02;
	mercury << 1922310.5 << -3.384057841687666E-01 << -2.907312076751666E-01 <<  9.125259547765344E-03;
	mercury << 1931441.5 << -3.218331074056913E-01 <<  1.500529386541980E-01 <<  4.267946058521499E-02;
	mercury << 1940573.5 <<  1.666509919717487E-01 <<  2.614747185162561E-01 <<  4.979062163886007E-03;
	mercury << 1949704.5 <<  3.314280945528729E-01 << -2.103391536416393E-01 << -4.837705944854103E-02;
	mercury << 1958835.5 <<  4.070218052673457E-03 << -4.607759395675578E-01 << -3.719581679649971E-02;
	mercury << 1967966.5 << -3.385831818656365E-01 << -2.900720305910719E-01 <<  9.038315933257341E-03;
	mercury << 1977098.5 << -3.374265060781010E-01 <<  1.266171373213934E-01 <<  4.221287650887897E-02;
	mercury << 1986229.5 <<  1.684722870594091E-01 <<  2.604934169519595E-01 <<  4.824724842937942E-03;
	mercury << 1995360.5 <<  3.312860262406485E-01 << -2.115217401116779E-01 << -4.838943147635759E-02;
	mercury << 2004491.5 <<  3.572851072104403E-03 << -4.609273570695350E-01 << -3.723848150995565E-02;
	mercury << 2013623.5 << -3.256368685405265E-01 << -3.088529940335629E-01 <<  6.149654675919482E-03;
	mercury << 2022754.5 << -3.364479343223331E-01 <<  1.277151263469052E-01 <<  4.212324205333130E-02;
	mercury << 2031885.5 <<  1.702968145240221E-01 <<  2.595039456005878E-01 <<  4.670311651668654E-03;
	mercury << 2041016.5 <<  3.311379118760124E-01 << -2.126985080034752E-01 << -4.840088799820974E-02;
	mercury << 2050148.5 <<  2.559590734843783E-02 << -4.585795382895859E-01 << -3.921067341927995E-02;
	mercury << 2059279.5 << -3.258374171393479E-01 << -3.082458682163584E-01 <<  6.062258293287233E-03;
	mercury << 2068410.5 << -3.354501703615874E-01 <<  1.288411371277513E-01 <<  4.203440703821169E-02;
	mercury << 2077541.5 <<  1.720764335142290E-01 <<  2.585295068543828E-01 <<  4.521839469440776E-03;
	mercury << 2086673.5 <<  3.399651302800188E-01 << -1.883104592695030E-01 << -4.720488762379448E-02;
	mercury << 2095804.5 <<  2.509794561941977E-02 << -4.587741451625025E-01 << -3.924916980279095E-02;
	mercury << 2104935.5 << -3.260469741693910E-01 << -3.076171744716837E-01 <<  5.976940257343994E-03;
	mercury << 2114066.5 << -3.344604771199884E-01 <<  1.299461386357742E-01 <<  4.194551819587693E-02;
	mercury << 2123198.5 <<  1.441998965315592E-01 <<  2.732497579886705E-01 <<  8.427933502183776E-03;
	mercury << 2132329.5 <<  3.398819021256583E-01 << -1.895264627872768E-01 << -4.721858632819514E-02;
	mercury << 2141460.5 <<  2.460058963607812E-02 << -4.589712417870845E-01 << -3.928761962022347E-02;
	mercury << 2150591.5 << -3.262508240700931E-01 << -3.069943602165459E-01 <<  5.891174598566313E-03;
	mercury << 2159723.5 << -3.484563624526565E-01 <<  1.058302008654663E-01 <<  4.122965500174101E-02;
	mercury << 2168854.5 <<  1.460680998499337E-01 <<  2.724021629416454E-01 <<  8.276120873955314E-03;
	mercury << 2177985.5 <<  3.397844327909025E-01 << -1.907648284880493E-01 << -4.723341658796309E-02;
	mercury << 2187116.5 <<  2.409902069043151E-02 << -4.591650287101612E-01 << -3.932556458531047E-02;
	mercury << 2196248.5 << -3.123790305681762E-01 << -3.249912036924762E-01 <<  2.984285589890865E-03;
	mercury << 2205379.5 << -3.475407045099009E-01 <<  1.069845487386884E-01 <<  4.114335303061954E-02;
	mercury << 2214510.5 <<  1.479207951851750E-01 <<  2.715442235634132E-01 <<  8.125987300180472E-03;
	mercury << 2223641.5 <<  3.396931130008133E-01 << -1.919942066746157E-01 << -4.724780143840161E-02;
	mercury << 2232773.5 <<  4.604640618075894E-02 << -4.556728167247664E-01 << -4.116013811573386E-02;
	mercury << 2241904.5 << -3.126172877295646E-01 << -3.243968159428095E-01 <<  2.900776946037332E-03;
	mercury << 2251035.5 << -3.466401889679923E-01 <<  1.081173033693826E-01 <<  4.105837623171322E-02;
	mercury << 2260166.5 <<  1.497943049704086E-01 <<  2.706706822234462E-01 <<  7.973380573162743E-03;
	mercury << 2269298.5 <<  3.471684307013990E-01 << -1.669381242067945E-01 << -4.583706832248360E-02;
	mercury << 2278429.5 <<  4.554776858729980E-02 << -4.559129059044027E-01 << -4.119441182900516E-02;
	mercury << 2287560.5 << -3.128425660435719E-01 << -3.238123827805834E-01 <<  2.815321505533709E-03;
	mercury << 2296691.5 << -3.457133553297054E-01 <<  1.092720093273959E-01 <<  4.097261167514837E-02;
	mercury << 2305813.5 <<  3.383262505635368E-01 <<  5.561807490638349E-02 << -2.688569081097927E-02;
	mercury << 2314944.5 <<  2.095688685302954E-01 << -3.806639501632584E-01 << -5.034584259600975E-02;
	mercury << 2324075.5 << -1.743448817699843E-01 << -4.309451626297167E-01 << -1.884665881950166E-02;
	mercury << 2333206.5 << -3.956544811891337E-01 << -9.650406711204389E-02 <<  2.880307890760519E-02;
	mercury << 2342337.5 << -1.024931061158045E-01 <<  2.968450558368274E-01 <<  3.361818086773331E-02;
	mercury << 2351468.5 <<  3.477025122644549E-01 <<  2.483222388290689E-02 << -3.013929129252293E-02;
	mercury << 2360599.5 <<  1.897203320300490E-01 << -3.956240863669067E-01 << -4.972399976979599E-02;
	mercury << 2369730.5 << -1.948955557207552E-01 << -4.209177772959865E-01 << -1.625940115542677E-02;
	mercury << 2378861.5 << -3.953100366026887E-01 << -6.916671178325895E-02 <<  3.085358097586205E-02;
	mercury << 2387992.5 << -6.784497059696484E-02 <<  3.037299859630596E-01 <<  3.099947277623389E-02;
	mercury << 2397123.5 <<  3.544363869644120E-01 << -6.092480299337812E-03 << -3.316452604811528E-02;
	mercury << 2406254.5 <<  1.691964390143365E-01 << -4.090935445178625E-01 << -4.893605634244735E-02;
	mercury << 2415385.5 << -2.148460794142748E-01 << -4.095445500288651E-01 << -1.362713843481346E-02;
	mercury << 2424516.5 << -3.930286881202809E-01 << -4.147270264963129E-02 <<  3.276542931015233E-02;
	mercury << 2433647.5 << -3.242795363115776E-02 <<  3.073661195034714E-01 <<  2.806923625667302E-02;
	mercury << 2442778.5 <<  3.586092826759569E-01 << -3.693190351776186E-02 << -3.595338180092523E-02;
	mercury << 2451910.5 <<  1.687264923576984E-01 << -4.096395147790790E-01 << -4.894981934538924E-02;
	mercury << 2461041.5 << -2.152013034823970E-01 << -4.092076160531841E-01 << -1.370327054572951E-02;
	mercury << 2470172.5 << -3.926325562160355E-01 << -4.035676956559111E-02 <<  3.268637408526332E-02;
	mercury << 2479303.5 << -3.042003956719055E-02 <<  3.074073607098288E-01 <<  2.792861934347989E-02;
	mercury << 2488434.5 <<  3.589892672608620E-01 << -3.842148935716652E-02 << -3.600059864576769E-02;
	mercury << 2497565.5 <<  1.476096423086783E-01 << -4.215584165202493E-01 << -4.800647441264983E-02;
	mercury << 2506696.5 << -2.344535807321704E-01 << -3.964692444644853E-01 << -1.103665950541597E-02;
	mercury << 2515827.5 << -3.882382894539212E-01 << -1.237121122034975E-02 <<  3.444474864905755E-02;
	mercury << 2524958.5 <<  5.368548717680148E-03 <<  3.074880968402448E-01 <<  2.470923184034722E-02;
	mercury << 2534089.5 <<  3.606214962840913E-01 << -6.894170574477418E-02 << -3.854160535130462E-02;
	mercury << 2543220.5 <<  1.259942420714331E-01 << -4.319939015810874E-01 << -4.691293421208570E-02;
	mercury << 2552351.5 << -2.529545216983266E-01 << -3.824308181921832E-01 << -8.340670905905652E-03;
	mercury << 2561482.5 << -3.817263608793871E-01 <<  1.572149111262677E-02 <<  3.603036843909992E-02;
	mercury << 2570613.5 <<  4.110895508715084E-02 <<  3.041388583360768E-01 <<  2.123536298575333E-02;
	mercury << 2579744.5 <<  3.599241238639727E-01 << -9.896574653945564E-02 << -4.083968663250280E-02;
	mercury << 2588875.5 <<  1.039588719630410E-01 << -4.409372131396845E-01 << -4.567581072659298E-02;
	mercury << 2598007.5 << -2.532804670467978E-01 << -3.819925025111593E-01 << -8.418416182228211E-03;
	mercury << 2607138.5 << -3.811489856771850E-01 <<  1.688218512074241E-02 <<  3.595248756699286E-02;
	mercury << 2616269.5 <<  4.313134140776279E-02 <<  3.038146869751351E-01 <<  2.108782453652205E-02;
	mercury << 2625400.5 <<  3.601034209936050E-01 << -1.003749123783049E-01 << -4.087066840949781E-02;
	mercury << 2634531.5 <<  1.034947070646389E-01 << -4.412843241187916E-01 << -4.569979384269483E-02;
	mercury << 2643662.5 << -2.709463517529512E-01 << -3.666313570173987E-01 << -5.699939335403062E-03;
	mercury << 2652793.5 << -3.723630128678217E-01 <<  4.493196455839566E-02 <<  3.734955121225661E-02;
	mercury << 2661924.5 <<  7.839714640143251E-02 <<  2.968504398531864E-01 <<  1.739208627595921E-02;
	mercury << 2671055.5 <<  3.571288214104045E-01 << -1.296931157847927E-01 << -4.291973854179588E-02;
	mercury << 2680186.5 <<  8.112551989632696E-02 << -4.486804896546118E-01 << -4.433035360008230E-02;
	mercury << 2689317.5 << -2.877091936735237E-01 << -3.500440485898667E-01 << -2.967022915970827E-03;
	mercury << 2698448.5 << -3.613122646229334E-01 <<  7.275145527378188E-02 <<  3.853791785234680E-02;
	mercury << 2707579.5 <<  1.127760484584739E-01 <<  2.865379758075249E-01 <<  1.352023507795889E-02;
	mercury << 2716710.5 <<  3.521060009464552E-01 << -1.582361416216322E-01 << -4.473067916989865E-02;
	mercury << 2725841.5 <<  5.851192827687036E-02 << -4.545805031558795E-01 << -4.283214110865017E-02;
	mercury << 2734972.5 << -3.035088310028080E-01 << -3.322386612995246E-01 << -2.259845936167225E-04;
	mercury << 2744104.5 << -3.605318815215233E-01 <<  7.393154954247054E-02 <<  3.845987456038102E-02;
	mercury << 2753235.5 <<  1.147172385468134E-01 <<  2.858594887331621E-01 <<  1.337216246025493E-02;
	mercury << 2762366.5 <<  3.521090113034246E-01 << -1.595199836650391E-01 << -4.474786024405885E-02;
	mercury << 2771497.5 <<  5.803541843073561E-02 << -4.548228707368177E-01 << -4.286287527141996E-02;
	mercury << 2780628.5 << -3.037633636929519E-01 << -3.316366376621592E-01 << -3.050600062251767E-04;
	mercury << 2789759.5 << -3.470707357042012E-01 <<  1.013118467435400E-01 <<  3.942036611300432E-02;
	mercury << 2798890.5 <<  1.477755297350234E-01 <<  2.721635035717487E-01 <<  9.366824104478038E-03;
	mercury << 2808021.5 <<  3.450958731803364E-01 << -1.870602179464416E-01 << -4.631667596953053E-02;
	mercury << 2817152.5 <<  3.523158954992874E-02 << -4.591915051220523E-01 << -4.124521119435772E-02;
	mercury << 2826283.5 << -3.184860415307735E-01 << -3.126129751569939E-01 <<  2.434982252996733E-03;
	mercury << 2835414.5 << -3.312455101903568E-01 <<  1.280726155356172E-01 <<  4.013608992754421E-02;
	mercury << 2844545.5 <<  1.791477380948414E-01 <<  2.554781129398273E-01 <<  5.279804801324012E-03;
	mercury << 2853676.5 <<  3.362917479567212E-01 << -2.136187942850424E-01 << -4.765786580781691E-02;
	mercury << 2862807.5 <<  1.234453658028811E-02 << -4.620826007406987E-01 << -3.951257381029039E-02;
	mercury << 2871938.5 << -3.320768509461525E-01 << -2.924534453403262E-01 <<  5.165091648672878E-03;
	mercury << 2881069.5 << -3.130485108349361E-01 <<  1.539630880182379E-01 <<  4.058696107468417E-02;
	mercury << 2890201.5 <<  1.809397008020666E-01 <<  2.544825308772549E-01 <<  5.137334966195893E-03;
	mercury << 2899332.5 <<  3.361659358752108E-01 << -2.147480745045914E-01 << -4.766455597588982E-02;
	mercury << 2908463.5 <<  1.185624458728356E-02 << -4.622174236605332E-01 << -3.954955835420373E-02;
	mercury << 2917594.5 << -3.322681102850591E-01 << -2.917481573458238E-01 <<  5.086487445283988E-03;
	mercury << 2926725.5 << -3.119234847349476E-01 <<  1.550817941002076E-01 <<  4.050271169236701E-02;
	mercury << 2935856.5 <<  2.102830407608310E-01 <<  2.348998906689543E-01 <<  1.011421036089389E-03;
	mercury << 2944987.5 <<  3.256548395213145E-01 << -2.401250809051328E-01 << -4.877736718632119E-02;
	mercury << 2954118.5 << -1.106643119416792E-02 << -4.635879617404472E-01 << -3.771036483674039E-02;
	mercury << 2963249.5 << -3.446101739632270E-01 << -2.704550223206592E-01 <<  7.797379130261749E-03;
	mercury << 2972380.5 << -2.912176705201666E-01 <<  1.798289700195935E-01 <<  4.066715593566805E-02;
	mercury << 2981511.5 <<  2.373774851402853E-01 <<  2.128978243391225E-01 << -3.105368257931400E-03;
	mercury << 2990642.5 <<  3.136124564916437E-01 << -2.643445012782537E-01 << -4.967344378155936E-02;
	mercury << 2999773.5 << -3.395261593631582E-02 << -4.634947526332983E-01 << -3.576721410435729E-02;
	mercury << 3008904.5 << -3.556738662052735E-01 << -2.480969711217014E-01 <<  1.048373594629727E-02;
	mercury << 3018035.5 << -2.682016675635888E-01 <<  2.031414822532684E-01 <<  4.053059920399319E-02;
	mercury << 3027166.5 <<  2.620859889429915E-01 <<  1.887281959730359E-01 << -7.183739111329713E-03;
	mercury << 3036298.5 <<  3.133890245001938E-01 << -2.653381617341028E-01 << -4.967500490068829E-02;
	mercury << 3045429.5 << -3.441844207044942E-02 << -4.635257804183924E-01 << -3.581203090778166E-02;
	mercury << 3054560.5 << -3.557786985048465E-01 << -2.472911593921394E-01 <<  1.040451378475499E-02;
	mercury << 3063691.5 << -2.668020254771547E-01 <<  2.041943005742460E-01 <<  4.043961278704458E-02;
	mercury << 3072822.5 <<  2.634702104035644E-01 <<  1.874262451928790E-01 << -7.298391370215523E-03;
	mercury << 3081953.5 <<  2.998969986495794E-01 << -2.882583683779159E-01 << -5.035843803852583E-02;
	mercury << 3091084.5 << -5.718870248480047E-02 << -4.619299488233062E-01 << -3.377482221651154E-02;
	mercury << 3100215.5 << -3.654170129955417E-01 << -2.239035573279698E-01 <<  1.305274141397735E-02;
	mercury << 3109346.5 << -2.413623639464049E-01 <<  2.257816854758777E-01 <<  3.998033225956574E-02;
	mercury << 3118477.5 <<  2.854591491412605E-01 <<  1.613938796856157E-01 << -1.128322123635908E-02;
	mercury << 3127608.5 <<  2.851327305983694E-01 << -3.098897893814214E-01 << -5.083773080699112E-02;
	mercury << 3136739.5 << -7.975831749429595E-02 << -4.588900068826032E-01 << -3.164816348234212E-02;
	mercury << 3145870.5 << -3.735983836708644E-01 << -1.995505392839899E-01 <<  1.565551173954515E-02;
	mercury << 3155001.5 << -2.137903257042624E-01 <<  2.453796600054135E-01 <<  3.919262374803493E-02;
	mercury << 3164132.5 <<  3.048024420566261E-01 <<  1.338816675096175E-01 << -1.515681757668096E-02;
	mercury << 3173263.5 <<  2.691883624040570E-01 << -3.301902981813777E-01 << -5.111943537790391E-02;
	mercury << 3182395.5 << -8.021356676416674E-02 << -4.588166159121524E-01 << -3.169926479322656E-02;
	mercury << 3191526.5 << -3.736039015909727E-01 << -1.986585622036333E-01 <<  1.557741255311674E-02;
	mercury << 3200657.5 << -2.121422175392442E-01 <<  2.462536320963858E-01 <<  3.909064582519712E-02;
	mercury << 3209788.5 <<  3.059019295578717E-01 <<  1.324664696902996E-01 << -1.524976724730418E-02;
	mercury << 3218919.5 <<  2.688449828317830E-01 << -3.310081832892638E-01 << -5.111984529624870E-02;
	mercury << 3228050.5 << -1.025319315845493E-01 << -4.542919367491993E-01 << -2.948994003475426E-02;
	mercury << 3237181.5 << -3.801769435302697E-01 << -1.733894002939385E-01 <<  1.812226001446750E-02;
	mercury << 3246312.5 << -1.824437771122683E-01 <<  2.634880635166234E-01 <<  3.795672484827239E-02;
	mercury << 3255443.5 <<  3.223965511055165E-01 <<  1.037728105569817E-01 << -1.897155223170741E-02;
	mercury << 3264574.5 <<  2.518075507995428E-01 << -3.498936932302935E-01 << -5.121216990469221E-02;
	mercury << 3273705.5 << -1.245305309944745E-01 << -4.483565889057268E-01 << -2.720163067999170E-02;
	mercury << 3282836.5 << -3.850913206603608E-01 << -1.473344199488020E-01 <<  2.059443883009871E-02;
	mercury << 3291967.5 << -1.509377899376681E-01 <<  2.781617498165889E-01 <<  3.647538135128205E-02;
	mercury << 3301098.5 <<  3.361947123462953E-01 <<  7.425054737831417E-02 << -2.252929955260407E-02;
	mercury << 3310229.5 <<  2.338169384748007E-01 << -3.673762565775330E-01 << -5.112107558161511E-02;
	mercury << 3319360.5 << -1.461592564417169E-01 << -4.410041811185284E-01 << -2.483823579291143E-02;
	mercury << 3328492.5 << -3.849903403356526E-01 << -1.463294622497228E-01 <<  2.052139315236979E-02;
	mercury << 3337623.5 << -1.490877468539843E-01 <<  2.787896137183948E-01 <<  3.636269965643206E-02;
	mercury << 3346754.5 <<  3.370157851682131E-01 <<  7.278063121590315E-02 << -2.260154243514315E-02;
	mercury << 3355885.5 <<  2.334201669035380E-01 << -3.680662662213979E-01 << -5.112351124443010E-02;
	mercury << 3365016.5 << -1.465625491212621E-01 << -4.407829872606063E-01 << -2.490021091037515E-02;
	mercury << 3374147.5 << -3.880921906504728E-01 << -1.195191738511817E-01 <<  2.291028022993870E-02;
	mercury << 3383278.5 << -1.159538852671656E-01 <<  2.905183014942316E-01 <<  3.453106362149427E-02;
	mercury << 3392409.5 <<  3.480107894184405E-01 <<  4.269326576141692E-02 << -2.596797028941395E-02;
	mercury << 3401540.5 <<  2.145479856794963E-01 << -3.840751200505548E-01 << -5.085798700707501E-02;
	mercury << 3410671.5 << -1.677269704689618E-01 << -4.319914143247839E-01 << -2.247173868645211E-02;
	mercury << 3419802.5 << -3.893601637426935E-01 << -9.211657219602443E-02 <<  2.520106592000565E-02;
	mercury << 3428933.5 << -8.157047013167944E-02 <<  2.991916804090513E-01 <<  3.235903186868309E-02;
	mercury << 3438064.5 <<  3.563882666323882E-01 <<  1.233486280169623E-02 << -2.913773259620836E-02;
	mercury << 3447195.5 <<  1.949079873346155E-01 << -3.986706525360477E-01 << -5.042398075982989E-02;
	mercury << 3456326.5 << -1.883557212761354E-01 << -4.218236484747193E-01 << -1.998261006038711E-02;
	mercury << 3465457.5 << -3.886814977617046E-01 << -6.421547738270579E-02 <<  2.737596491473770E-02;
	mercury << 3474589.5 << -7.956292015530105E-02 <<  2.995178728339111E-01 <<  3.223483148385383E-02;
	mercury << 3483720.5 <<  3.569577384267695E-01 <<  1.088014742930704E-02 << -2.918898034830334E-02;
	mercury << 3492851.5 <<  1.944760120018814E-01 << -3.992260647450150E-01 << -5.042994086497089E-02;
	mercury << 3501982.5 << -1.887489508523147E-01 << -4.214983958297556E-01 << -2.004729978876067E-02;
	mercury << 3511113.5 << -3.883518840096635E-01 << -6.307293351807415E-02 <<  2.730667073572889E-02;
	mercury << 3520244.5 << -4.425278629933399E-02 <<  3.047882140132953E-01 <<  2.973300960927650E-02;
	mercury << 3529375.5 <<  3.626911995477702E-01 << -1.951573606182543E-02 << -3.214464150288147E-02;
	mercury << 3538506.5 <<  1.741615736978859E-01 << -4.123099907273539E-01 << -4.983615539292918E-02;
	mercury << 3547637.5 << -2.087744378816144E-01 << -4.099170841164376E-01 << -1.750396283873282E-02;
	mercury << 3556768.5 << -3.855660042534493E-01 << -3.476518074616249E-02 <<  2.935298783776738E-02;
	mercury << 3565899.5 << -8.387329418081556E-03 <<  3.066921419638676E-01 <<  2.692201256280359E-02;
	mercury << 3575030.5 <<  3.659834193528766E-01 << -4.970536738154795E-02 << -3.488380801546578E-02;
	mercury << 3584161.5 <<  1.532344194469206E-01 << -4.239616855466047E-01 << -4.908616885802324E-02;
	mercury << 3593292.5 << -2.281278032401330E-01 << -3.970048263074872E-01 << -1.491157644858341E-02;
	mercury << 3602423.5 << -3.806534736812634E-01 << -6.234065273470547E-03 <<  3.125142051391373E-02;
	mercury << 3611554.5 <<  2.756224480414678E-02 <<  3.051468265482731E-01 <<  2.383287938020075E-02;
	mercury << 3620686.5 <<  3.663346157051767E-01 << -5.111310323391727E-02 << -3.491822829965663E-02;
	mercury << 3629817.5 <<  1.527952323970611E-01 << -4.243907180114589E-01 << -4.909786334384854E-02;
	mercury << 3638948.5 << -2.284865732936779E-01 << -3.965669273112832E-01 << -1.497967338813053E-02;
	mercury << 3648079.5 << -3.801415230397792E-01 << -5.015606237387476E-03 <<  3.118434433197192E-02;
	mercury << 3657210.5 <<  2.961561596685850E-02 <<  3.049343063099303E-01 <<  2.369852134998059E-02;
	mercury << 3666341.5 <<  3.672127778060165E-01 << -8.092731768452446E-02 << -3.743463352678698E-02;
	mercury << 3675472.5 <<  1.313665166077268E-01 << -4.345263496680464E-01 << -4.820150489514111E-02;
	mercury << 3684603.5 << -2.470808322835331E-01 << -3.822769096090372E-01 << -1.234582245479128E-02;
	mercury << 3693734.5 << -3.729302374347065E-01 <<  2.359923046441259E-02 <<  3.291716526691550E-02;
	mercury << 3702865.5 <<  6.526023746371348E-02 <<  2.997494143601170E-01 <<  2.035779570693985E-02;
	mercury << 3711996.5 <<  3.658983769405982E-01 << -1.101823181509009E-01 << -3.972945626969709E-02;
	mercury << 3721127.5 <<  1.095081596212821E-01 << -4.432052703909096E-01 << -4.716150986679156E-02;
	mercury << 3730258.5 << -2.648579881462747E-01 << -3.666880884191842E-01 << -9.676439007370209E-03;
	mercury << 3739389.5 << -3.634178078492635E-01 <<  5.215347868863000E-02 <<  3.446719163477657E-02;
	mercury << 3748520.5 <<  1.001191964977642E-01 <<  2.911810371514115E-01 <<  1.681295079076819E-02;
	mercury << 3757651.5 <<  3.625042638653243E-01 << -1.388010241916226E-01 << -4.180784865558364E-02;
	mercury << 3766783.5 <<  1.090585488903449E-01 << -4.435231331439095E-01 << -4.717945353222269E-02;
	mercury << 3775914.5 << -2.651670128876171E-01 << -3.661526626514777E-01 << -9.747302640572804E-03;
	mercury << 3785045.5 << -3.627009946281409E-01 <<  5.339383372390413E-02 <<  3.440041613868584E-02;
	mercury << 3794176.5 <<  1.021229838333939E-01 <<  2.906041310442242E-01 <<  1.667510388120971E-02;
	mercury << 3803307.5 <<  3.625882237407977E-01 << -1.400436761837303E-01 << -4.182004167656379E-02;
	mercury << 3812438.5 <<  8.685002927826418E-02 << -4.506837344588013E-01 << -4.600524089789045E-02;
	mercury << 3821569.5 << -2.820322613865006E-01 << -3.492310642876981E-01 << -7.048864051416905E-03;
	mercury << 3830700.5 << -3.507306383255034E-01 <<  8.163177065116714E-02 <<  3.574484427960072E-02;
	mercury << 3839831.5 <<  1.358162970186548E-01 <<  2.785990363822873E-01 <<  1.296033912183404E-02;
	mercury << 3848962.5 <<  3.571716117758634E-01 << -1.677955046982100E-01 << -4.367383405107755E-02;
	mercury << 3858093.5 <<  6.438085146684389E-02 << -4.563816070912226E-01 << -4.469963558425931E-02;
	mercury << 3867224.5 << -2.979020135826103E-01 << -3.310930395306438E-01 << -4.332015850273150E-03;
	mercury << 3876355.5 << -3.363322054344982E-01 <<  1.093756870155628E-01 <<  3.686441417076627E-02;
	mercury << 3885486.5 <<  1.679757906920229E-01 <<  2.635498733003201E-01 <<  9.126382905024773E-03;
	mercury << 3894617.5 <<  3.499342076348369E-01 << -1.946620192872525E-01 << -4.531377031389142E-02;
	mercury << 3903748.5 <<  4.171250658073675E-02 << -4.606116445923554E-01 << -4.326777374033656E-02;

	// Venus
	// Start time      : B.C. 2000-Jan-01 00:00:00.0000 TDB ( 990923.500000000)
	// Stop  time      : A.D. 5976-Jan-01 00:00:00.0000 TDB (3903748.500000000)
	// Step-size       : 25 calendar years
	venus <<  990923.5 <<  1.866478619003115E-01 <<  6.949152956030098E-01 << -9.268819876847656E-03;
	venus << 1000054.5 <<  4.167230520638602E-01 << -5.984288264859788E-01 << -2.610276002373031E-02;
	venus << 1009185.5 << -7.143672300316518E-01 <<  6.793034934447066E-02 <<  4.229850075571882E-02;
	venus << 1018316.5 <<  5.370770548239501E-01 <<  4.830591586148009E-01 << -3.036409669753048E-02;
	venus << 1027448.5 <<  5.274033852957472E-02 << -7.265340660689875E-01 << -5.121956604007454E-03;
	venus << 1036579.5 << -5.706793620605761E-01 <<  4.329617742266930E-01 <<  3.488238886750299E-02;
	venus << 1045710.5 <<  7.122125325208469E-01 <<  1.361699296104901E-01 << -4.160686618836160E-02;
	venus << 1054841.5 << -3.438599962938342E-01 << -6.387200124764235E-01 <<  1.837814953872071E-02;
	venus << 1063973.5 << -2.575032837087740E-01 <<  6.693265693713554E-01 <<  1.722582559887945E-02;
	venus << 1073104.5 <<  6.833378246405678E-01 << -2.499273509175198E-01 << -4.108507444449801E-02;
	venus << 1082235.5 << -6.214370215928356E-01 << -3.663524846485927E-01 <<  3.548953716125805E-02;
	venus << 1091366.5 <<  1.521237411642423E-01 <<  7.031465746051396E-01 << -6.684391177562478E-03;
	venus << 1100498.5 <<  4.604630998377339E-01 << -5.652069567623660E-01 << -2.904558558817399E-02;
	venus << 1109629.5 << -7.178817453495385E-01 <<  1.244906465096749E-02 <<  4.239740635127778E-02;
	venus << 1118760.5 <<  4.980563972736134E-01 <<  5.227423740283591E-01 << -2.757616311188020E-02;
	venus << 1127891.5 <<  8.743388924482960E-02 << -7.232333710563972E-01 << -7.714197428560740E-03;
	venus << 1137023.5 << -6.027574490537060E-01 <<  3.872810993932872E-01 <<  3.696191719054298E-02;
	venus << 1146154.5 <<  6.993402224891748E-01 <<  1.901213489078656E-01 << -4.057031111200849E-02;
	venus << 1155285.5 << -2.946640849030827E-01 << -6.633451473882556E-01 <<  1.490731297066839E-02;
	venus << 1164416.5 << -2.902601766432712E-01 <<  6.557481117881114E-01 <<  1.962588271988082E-02;
	venus << 1173548.5 <<  6.999426940592273E-01 << -1.972767523435573E-01 << -4.207152797141780E-02;
	venus << 1182679.5 << -5.920517645772814E-01 << -4.131458391714346E-01 <<  3.330798148396034E-02;
	venus << 1191810.5 <<  9.716846205306266E-02 <<  7.125722460559210E-01 << -2.871022314327168E-03;
	venus << 1200941.5 <<  4.868877279914612E-01 << -5.423584699088534E-01 << -3.095332784365754E-02;
	venus << 1210073.5 << -7.171254454233644E-01 << -4.312490533846129E-02 <<  4.214636833887482E-02;
	venus << 1219204.5 <<  4.560867659585763E-01 <<  5.593211213492104E-01 << -2.455435105482394E-02;
	venus << 1228335.5 <<  1.416534595817380E-01 << -7.146939952193087E-01 << -1.143224836837325E-02;
	venus << 1237466.5 << -6.212466873864877E-01 <<  3.571075264175088E-01 <<  3.822724526404001E-02;
	venus << 1246598.5 <<  6.823725423287851E-01 <<  2.429791342342943E-01 << -3.919823935225154E-02;
	venus << 1255729.5 << -2.437631403795200E-01 << -6.841185458542745E-01 <<  1.130465347690942E-02;
	venus << 1264860.5 << -3.403421013782446E-01 <<  6.310898570193882E-01 <<  2.297492064908597E-02;
	venus << 1273991.5 <<  7.083805342501041E-01 << -1.631495482693093E-01 << -4.256879586417677E-02;
	venus << 1283123.5 << -5.591806767650408E-01 << -4.575416745770062E-01 <<  3.084578409142883E-02;
	venus << 1292254.5 <<  4.160588719299285E-02 <<  7.177209889242555E-01 <<  9.788765856721116E-04;
	venus << 1301385.5 <<  5.262283481361141E-01 << -5.039293146688593E-01 << -3.350728486772473E-02;
	venus << 1310516.5 << -7.144500779146634E-01 << -7.825064018152614E-02 <<  4.178480375760463E-02;
	venus << 1319648.5 <<  4.114349376055968E-01 <<  5.925797719331821E-01 << -2.132431446939026E-02;
	venus << 1328779.5 <<  1.950778793346975E-01 << -7.020465664766171E-01 << -1.506980318137491E-02;
	venus << 1337910.5 << -6.472783648634360E-01 <<  3.078443332190084E-01 <<  3.978399637133155E-02;
	venus << 1347041.5 <<  6.694903859831413E-01 <<  2.758961099558075E-01 << -3.807798354086480E-02;
	venus << 1356173.5 << -1.914750856610874E-01 << -7.009330053923755E-01 <<  7.604341667117391E-03;
	venus << 1365304.5 << -3.883723628584996E-01 <<  6.026256320792751E-01 <<  2.613906610972569E-02;
	venus << 1374435.5 <<  7.183420960060645E-01 << -1.087839774972666E-01 << -4.298070676993634E-02;
	venus << 1383566.5 << -5.366207413847898E-01 << -4.843123445150023E-01 <<  2.900986181805978E-02;
	venus << 1392698.5 << -1.418883751134881E-02 <<  7.185524564980931E-01 <<  4.827650580166063E-03;
	venus << 1401829.5 <<  5.625667459492093E-01 << -4.625807026465775E-01 << -3.579819021703932E-02;
	venus << 1410960.5 << -7.067368069412768E-01 << -1.332315269220689E-01 <<  4.094359021079735E-02;
	venus << 1420091.5 <<  3.817432455665628E-01 <<  6.119485857242213E-01 << -1.899619178225605E-02;
	venus << 1429223.5 <<  2.473623369198535E-01 << -6.853608007998743E-01 << -1.859502212639851E-02;
	venus << 1438354.5 << -6.694491118368838E-01 <<  2.567169203942536E-01 <<  4.101406582384649E-02;
	venus << 1447485.5 <<  6.460582995164554E-01 <<  3.262464887906826E-01 << -3.616735909038164E-02;
	venus << 1456616.5 << -1.576935305322741E-01 << -7.094900277743058E-01 <<  5.015279076510797E-03;
	venus << 1465748.5 << -4.340571113067396E-01 <<  5.705415636445849E-01 <<  2.909252874995661E-02;
	venus << 1474879.5 <<  7.241557216597806E-01 << -5.377482707603485E-02 << -4.304817352823660E-02;
	venus << 1484010.5 << -4.984835344080033E-01 << -5.242242990160213E-01 <<  2.612322049659065E-02;
	venus << 1493141.5 << -4.961231883684730E-02 <<  7.168733322081910E-01 <<  7.458071303924104E-03;
	venus << 1502273.5 <<  5.956730523410191E-01 << -4.185705614798712E-01 << -3.780593815576726E-02;
	venus << 1511404.5 << -6.948538667038434E-01 << -1.873872466683545E-01 <<  3.976486980862848E-02;
	venus << 1520535.5 <<  3.332434824860065E-01 <<  6.393299552559909E-01 << -1.547216456601407E-02;
	venus << 1529666.5 <<  2.798208284962598E-01 << -6.726849866095189E-01 << -2.094290428281716E-02;
	venus << 1538798.5 << -6.875942915950023E-01 <<  2.040768469243477E-01 <<  4.190270152593172E-02;
	venus << 1547929.5 <<  6.188317973301027E-01 <<  3.746789153803567E-01 << -3.395625888823002E-02;
	venus << 1557060.5 << -1.036930832971739E-01 << -7.196629918149237E-01 <<  1.208220362106397E-03;
	venus << 1566191.5 << -4.617537305798601E-01 <<  5.483743041249791E-01 <<  3.099314950738940E-02;
	venus << 1575323.5 <<  7.257818410692201E-01 <<  1.561965565549466E-03 << -4.276653673525034E-02;
	venus << 1584454.5 << -4.574414798963615E-01 << -5.610515102588653E-01 <<  2.301641464921545E-02;
	venus << 1593585.5 << -1.050762358443277E-01 <<  7.106641860873735E-01 <<  1.123373992444259E-02;
	venus << 1602716.5 <<  6.149240000536410E-01 << -3.892888198989757E-01 << -3.902296854335156E-02;
	venus << 1611848.5 << -6.788697457483964E-01 << -2.404407482628094E-01 <<  3.825695120545775E-02;
	venus << 1620979.5 <<  2.827752885533613E-01 <<  6.629045125375410E-01 << -1.181492100792609E-02;
	venus << 1630110.5 <<  3.296533896321892E-01 << -6.496351731698636E-01 << -2.421155977772414E-02;
	venus << 1639241.5 << -6.970160935675546E-01 <<  1.700064008572748E-01 <<  4.232160378569094E-02;
	venus << 1648373.5 <<  5.879915833807193E-01 <<  4.208903158611372E-01 << -3.146618809934097E-02;
	venus << 1657504.5 << -4.909488088461288E-02 << -7.256737919934262E-01 << -2.613931161196429E-03;
	venus << 1666635.5 << -5.030071487474190E-01 <<  5.107987037542864E-01 <<  3.352788162436226E-02;
	venus << 1675766.5 <<  7.245984249522678E-01 <<  3.681935801027734E-02 << -4.237503407289224E-02;
	venus << 1684898.5 << -4.137448487240203E-01 << -5.945926262783501E-01 <<  1.971878971101253E-02;
	venus << 1694029.5 << -1.598703615680907E-01 <<  7.001787799241511E-01 <<  1.491703246781734E-02;
	venus << 1703160.5 <<  6.423331692907897E-01 << -3.414728799787632E-01 << -4.051227615540889E-02;
	venus << 1712291.5 << -6.666541988190754E-01 << -2.733886760847901E-01 <<  3.703781375087836E-02;
	venus << 1721423.5 <<  2.306238652180610E-01 <<  6.825115579049237E-01 << -8.054776747453021E-03;
	venus << 1730554.5 <<  3.776193887779704E-01 << -6.228408264955222E-01 << -2.729191776969902E-02;
	venus << 1739685.5 << -7.084069074498472E-01 <<  1.155140519968420E-01 <<  4.262070159841490E-02;
	venus << 1748816.5 <<  5.665379937334321E-01 <<  4.490684891974397E-01 << -2.959825115389207E-02;
	venus << 1757948.5 <<  5.806916108558458E-03 << -7.274997255290080E-01 << -6.423424878696162E-03;
	venus << 1767079.5 << -5.412318130489661E-01 <<  4.701780296035654E-01 <<  3.578701638315243E-02;
	venus << 1776210.5 <<  7.193257974343851E-01 <<  9.197721676377506E-02 << -4.150734579745005E-02;
	venus << 1785341.5 << -3.847776707788043E-01 << -6.140784954876901E-01 <<  1.735505287833703E-02;
	venus << 1794473.5 << -2.137122010681128E-01 <<  6.854828377673878E-01 <<  1.848076029561439E-02;
	venus << 1803604.5 <<  6.660608211626722E-01 << -2.916909620283313E-01 << -4.167869037939494E-02;
	venus << 1812735.5 << -6.441744796667571E-01 << -3.239821294421554E-01 <<  3.500092420682051E-02;
	venus << 1821866.5 <<  1.967156911266926E-01 <<  6.928949947884424E-01 << -5.423477369821728E-03;
	venus << 1830998.5 <<  4.234374980975897E-01 << -5.924539431965723E-01 << -3.015549045438846E-02;
	venus << 1840129.5 << -7.155627742449325E-01 <<  6.036057227092623E-02 <<  4.256396949546541E-02;
	venus << 1849260.5 <<  5.301243911838834E-01 <<  4.910947676562282E-01 << -2.667536233842282E-02;
	venus << 1858391.5 <<  4.062927147130475E-02 << -7.264413215916616E-01 << -9.019836262909409E-03;
	venus << 1867523.5 << -5.762139857174757E-01 <<  4.267102384295630E-01 <<  3.774761743259468E-02;
	venus << 1876654.5 <<  7.098649731787159E-01 <<  1.465857017407530E-01 << -4.029940439868924E-02;
	venus << 1885785.5 << -3.372777859070921E-01 << -6.418866302701207E-01 <<  1.379271855011318E-02;
	venus << 1894916.5 << -2.472969718386428E-01 <<  6.740173110738975E-01 <<  2.085014841415187E-02;
	venus << 1904048.5 <<  6.859679177149763E-01 << -2.401915845861585E-01 << -4.251115040175336E-02;
	venus << 1913179.5 << -6.178892011310116E-01 << -3.726846347328125E-01 <<  3.267301926244396E-02;
	venus << 1922310.5 <<  1.425104799267615E-01 <<  7.057896855064172E-01 << -1.564406199898853E-03;
	venus << 1931441.5 <<  4.512800711836006E-01 << -5.713245214451586E-01 << -3.199200362281619E-02;
	venus << 1940573.5 << -7.184796639724965E-01 <<  4.847426635541374E-03 <<  4.215680341105257E-02;
	venus << 1949704.5 <<  4.905867318323140E-01 <<  5.302066507513543E-01 << -2.353277301271070E-02;
	venus << 1958835.5 <<  9.527895074973453E-02 << -7.214210023510186E-01 << -1.272702333714638E-02;
	venus << 1967966.5 << -5.966577097125253E-01 <<  3.978032461089085E-01 <<  3.891622154343131E-02;
	venus << 1977098.5 <<  6.962666827397406E-01 <<  2.003436638003502E-01 << -3.876281184181353E-02;
	venus << 1986229.5 << -2.877887824225432E-01 << -6.659780482087710E-01 <<  1.011497065767937E-02;
	venus << 1995360.5 << -2.988341503846257E-01 <<  6.526415088580413E-01 <<  2.413751492714859E-02;
	venus << 2004491.5 <<  6.965505999522458E-01 << -2.066606490107878E-01 << -4.288307106580818E-02;
	venus << 2013623.5 << -5.879759180518740E-01 << -4.191760907672349E-01 <<  3.007387263693532E-02;
	venus << 2022754.5 <<  8.743310289240615E-02 <<  7.144570709684958E-01 <<  2.310925460410796E-03;
	venus << 2031885.5 <<  4.929888765142476E-01 << -5.354570591706020E-01 << -3.443615267633952E-02;
	venus << 2041016.5 << -7.181362705934828E-01 << -3.039980787073767E-02 <<  4.167069082289574E-02;
	venus << 2050148.5 <<  4.481447600731479E-01 <<  5.661743409163060E-01 << -2.019347366379489E-02;
	venus << 2059279.5 <<  1.494213800195405E-01 << -7.122477256253453E-01 << -1.633730508935640E-02;
	venus << 2068410.5 << -6.258749149621567E-01 <<  3.503135831384404E-01 <<  4.033702066844937E-02;
	venus << 2077541.5 <<  6.854920745682218E-01 <<  2.339760701102178E-01 << -3.751698604383186E-02;
	venus << 2086673.5 << -2.366294661981288E-01 << -6.861990717040730E-01 <<  6.352808592071481E-03;
	venus << 2095804.5 << -3.485837836286786E-01 <<  6.273557697292598E-01 <<  2.722531825819188E-02;
	venus << 2104935.5 <<  7.099321897615314E-01 << -1.530217454555283E-01 << -4.313025491599169E-02;
	venus << 2114066.5 << -5.672349054839196E-01 << -4.473794844635534E-01 <<  2.814136618072305E-02;
	venus << 2123198.5 <<  3.184702583318554E-02 <<  7.188427504863800E-01 <<  6.165124493658999E-03;
	venus << 2132329.5 <<  5.318664582727376E-01 << -4.965156872840587E-01 << -3.660216312700471E-02;
	venus << 2141460.5 << -7.140708058284845E-01 << -8.579854177370318E-02 <<  4.067228371607492E-02;
	venus << 2150591.5 <<  4.197599151346668E-01 <<  5.873312006115085E-01 << -1.779244917371197E-02;
	venus << 2159723.5 <<  2.027124611206822E-01 << -6.989624302931458E-01 << -1.981535890929211E-02;
	venus << 2168854.5 << -6.513430847014580E-01 <<  3.007180312171365E-01 <<  4.141909090516467E-02;
	venus << 2177985.5 <<  6.652928259088483E-01 <<  2.856750509472740E-01 << -3.544970087366552E-02;
	venus << 2187116.5 << -2.034413845985094E-01 << -6.969425284743663E-01 <<  3.726937576045195E-03;
	venus << 2196248.5 << -3.962382058749377E-01 <<  5.983077875045826E-01 <<  3.008539010641890E-02;
	venus << 2205379.5 <<  7.191969840227675E-01 << -9.849170611979982E-02 << -4.302643424418333E-02;
	venus << 2214510.5 << -5.317899866153248E-01 << -4.897332492565973E-01 <<  2.513719851340332E-02;
	venus << 2223641.5 << -3.609818886194022E-03 <<  7.194151474841591E-01 <<  8.792439858655894E-03;
	venus << 2232773.5 <<  5.677059142178278E-01 << -4.547085077488871E-01 << -3.847530961961632E-02;
	venus << 2241904.5 << -7.057762707252668E-01 << -1.406848957114513E-01 <<  3.933833648078856E-02;
	venus << 2251035.5 <<  3.731360498157023E-01 <<  6.176900809392578E-01 << -1.419042956446096E-02;
	venus << 2260166.5 <<  2.359586438680421E-01 << -6.883963841834834E-01 << -2.211796995855509E-02;
	venus << 2269298.5 << -6.729007746349005E-01 <<  2.493775516811641E-01 <<  4.215520121784476E-02;
	venus << 2278429.5 <<  6.411835104626888E-01 <<  3.357010315059299E-01 << -3.309257400972444E-02;
	venus << 2287560.5 << -1.502059547776423E-01 << -7.106057209801130E-01 << -1.044878039457488E-04;
	venus << 2296691.5 << -4.253118765809886E-01 <<  5.779866279165740E-01 <<  3.190422424130417E-02;
	venus << 2305813.5 <<  6.855913846697244E-01 << -2.401472585499848E-01 << -4.279685994036907E-02;
	venus << 2314944.5 << -6.186831240349306E-01 << -3.715545930586169E-01 <<  3.119512758193190E-02;
	venus << 2324075.5 <<  1.426432279768899E-01 <<  7.060243604123931E-01 <<  6.914160686068782E-04;
	venus << 2333206.5 <<  4.504777900584181E-01 << -5.713601567897317E-01 << -3.340830530123336E-02;
	venus << 2342337.5 << -7.181438611761387E-01 <<  2.632709805836748E-02 <<  4.195470821068387E-02;
	venus << 2351468.5 <<  5.053194350570493E-01 <<  5.165354087370932E-01 << -2.260759382689824E-02;
	venus << 2360599.5 <<  7.421732976720992E-02 << -7.234683334414471E-01 << -1.368744096508502E-02;
	venus << 2369730.5 << -5.848916129538702E-01 <<  4.154537864934569E-01 <<  3.928492057710568E-02;
	venus << 2378861.5 <<  7.064068995040862E-01 <<  1.615324335016247E-01 << -3.876710979641652E-02;
	venus << 2387992.5 << -3.253200604685140E-01 << -6.479818610475355E-01 <<  1.027930722907702E-02;
	venus << 2397123.5 << -2.611656654760223E-01 <<  6.691306864584577E-01 <<  2.397436778715530E-02;
	venus << 2406254.5 <<  6.838420252992348E-01 << -2.449002647980024E-01 << -4.280346408981966E-02;
	venus << 2415385.5 << -6.214286195192411E-01 << -3.669187708917689E-01 <<  3.100628228893334E-02;
	venus << 2424516.5 <<  1.476798514078249E-01 <<  7.050811981117042E-01 <<  9.554118483428889E-04;
	venus << 2433647.5 <<  4.463368533400343E-01 << -5.744590444205737E-01 << -3.355203998553722E-02;
	venus << 2442778.5 << -7.179678720105449E-01 <<  3.169381861336069E-02 <<  4.188584791808980E-02;
	venus << 2451910.5 <<  4.943094048764643E-01 <<  5.270080068294356E-01 << -2.132867292854978E-02;
	venus << 2461041.5 <<  8.887725078813637E-02 << -7.217623821863223E-01 << -1.504405625499716E-02;
	venus << 2470172.5 << -5.935671173915368E-01 <<  4.031147268338327E-01 <<  3.980342258188939E-02;
	venus << 2479303.5 <<  7.027774042935037E-01 <<  1.762382756752681E-01 << -3.807568235298864E-02;
	venus << 2488434.5 << -3.120894226678602E-01 << -6.545490515710356E-01 <<  8.859866665921489E-03;
	venus << 2497565.5 << -2.752061659271566E-01 <<  6.634916277896514E-01 <<  2.515571167268969E-02;
	venus << 2506696.5 <<  6.886399821842997E-01 << -2.306338940717201E-01 << -4.291219871585593E-02;
	venus << 2515827.5 << -6.138754859285870E-01 << -3.797150959647988E-01 <<  2.997090882044128E-02;
	venus << 2524958.5 <<  1.327827750985057E-01 <<  7.080184760754378E-01 <<  2.425285256209942E-03;
	venus << 2534089.5 <<  4.579262243451537E-01 << -5.650857367696638E-01 << -3.442022149715909E-02;
	venus << 2543220.5 << -7.186359742208782E-01 <<  1.673832197907852E-02 <<  4.156871417406039E-02;
	venus << 2552351.5 <<  4.979878150045808E-01 <<  5.236015125607497E-01 << -2.107484098249819E-02;
	venus << 2561482.5 <<  8.360054416128983E-02 << -7.222989692334904E-01 << -1.527537020389505E-02;
	venus << 2570613.5 << -5.906490718648320E-01 <<  4.075154028023226E-01 <<  3.986615438015603E-02;
	venus << 2579744.5 <<  7.039853319658190E-01 <<  1.713544162366755E-01 << -3.791682222916563E-02;
	venus << 2588875.5 << -3.169057977450944E-01 << -6.521673699404609E-01 <<  8.601774933662671E-03;
	venus << 2598007.5 << -2.891211948987798E-01 <<  6.575618805399366E-01 <<  2.630233702620709E-02;
	venus << 2607138.5 <<  6.931446919842260E-01 << -2.162665541195759E-01 << -4.296484103167308E-02;
	venus << 2616269.5 << -6.060845145677548E-01 << -3.923106550480651E-01 <<  2.889928641764362E-02;
	venus << 2625400.5 <<  1.178131684322711E-01 <<  7.106310426067327E-01 <<  3.889955843249011E-03;
	venus << 2634531.5 <<  4.693196925122469E-01 << -5.554745054316703E-01 << -3.524135009177460E-02;
	venus << 2643662.5 << -7.189801749819990E-01 <<  1.762307451860847E-03 <<  4.119478290521167E-02;
	venus << 2652793.5 <<  4.868451999335603E-01 <<  5.339016033375760E-01 << -1.976019530464588E-02;
	venus << 2661924.5 <<  9.825772253511819E-02 << -7.204024437799688E-01 << -1.660916419528670E-02;
	venus << 2671055.5 << -5.991576507609510E-01 <<  3.950487925131456E-01 <<  4.031865882362958E-02;
	venus << 2680186.5 <<  7.001742961652198E-01 <<  1.860031583155384E-01 << -3.716335219137145E-02;
	venus << 2689317.5 << -3.036005190644890E-01 << -6.585448625971041E-01 <<  7.168504047648497E-03;
	venus << 2698448.5 << -2.844056214684701E-01 <<  6.597310241473661E-01 <<  2.648895357686434E-02;
	venus << 2707579.5 <<  6.915633493095438E-01 << -2.210656278015113E-01 << -4.293229427998210E-02;
	venus << 2716710.5 << -6.089903454010995E-01 << -3.877889553885357E-01 <<  2.868196607492407E-02;
	venus << 2725841.5 <<  1.228545099300104E-01 <<  7.098588983278733E-01 <<  4.156371172051886E-03;
	venus << 2734972.5 <<  4.652796935947587E-01 << -5.587306995900480E-01 << -3.535576855373317E-02;
	venus << 2744104.5 << -7.190224700761558E-01 << -1.321596243053001E-02 <<  4.076945504428326E-02;
	venus << 2753235.5 <<  4.755005501426900E-01 <<  5.439577831555246E-01 << -1.842317351566227E-02;
	venus << 2762366.5 <<  1.128824830466794E-01 << -7.181992972847627E-01 << -1.791928607390888E-02;
	venus << 2771497.5 << -6.073804130435778E-01 <<  3.824480338983712E-01 <<  4.071550012415479E-02;
	venus << 2780628.5 <<  6.960440026769382E-01 <<  2.005797821375992E-01 << -3.636098358547588E-02;
	venus << 2789759.5 << -2.901721391810358E-01 << -6.646434773132696E-01 <<  5.728615446499628E-03;
	venus << 2798890.5 << -2.982618504099618E-01 <<  6.535980639323449E-01 <<  2.759096705682392E-02;
	venus << 2808021.5 <<  6.958671233751045E-01 << -2.066430217146768E-01 << -4.291182131114408E-02;
	venus << 2817152.5 << -6.010190325605247E-01 << -4.003224784481459E-01 <<  2.756109242207749E-02;
	venus << 2826283.5 <<  1.078509030077710E-01 <<  7.122540555589354E-01 <<  5.613859144015308E-03;
	venus << 2835414.5 <<  4.765646831277831E-01 << -5.489720145836290E-01 << -3.611991679484770E-02;
	venus << 2844545.5 << -7.191391557001601E-01 << -7.868913626668287E-03 <<  4.064951780052017E-02;
	venus << 2853676.5 <<  4.793234173125633E-01 <<  5.406661916571244E-01 << -1.815147330211110E-02;
	venus << 2862807.5 <<  1.076311844556196E-01 << -7.189048758991571E-01 << -1.813708664305596E-02;
	venus << 2871938.5 << -6.046277590279040E-01 <<  3.869418299418159E-01 <<  4.074395155300376E-02;
	venus << 2881069.5 <<  6.974187981852598E-01 <<  1.957488407437485E-01 << -3.616756167732563E-02;
	venus << 2890201.5 << -2.766235109965075E-01 << -6.704734341941097E-01 <<  4.283374658620078E-03;
	venus << 2899332.5 << -3.120053932728630E-01 <<  6.471687348189337E-01 <<  2.865641169197410E-02;
	venus << 2908463.5 <<  6.998829476367153E-01 << -1.921466534758086E-01 << -4.283722830083025E-02;
	venus << 2917594.5 << -5.927951640586133E-01 << -4.126770669740802E-01 <<  2.640725551623522E-02;
	venus << 2926725.5 <<  9.283971634099764E-02 <<  7.143323650019531E-01 <<  7.057962258172728E-03;
	venus << 2935856.5 <<  4.876464386775476E-01 << -5.389745478748686E-01 << -3.683463100020721E-02;
	venus << 2944987.5 << -7.189701177525316E-01 << -2.282822733935639E-02 <<  4.015446595979628E-02;
	venus << 2954118.5 <<  4.678358873045045E-01 <<  5.505667681927520E-01 << -1.678340490492057E-02;
	venus << 2963249.5 <<  1.222140446726362E-01 << -7.164980303967976E-01 << -1.941564246684239E-02;
	venus << 2972380.5 << -6.127152353075047E-01 <<  3.741860555895510E-01 <<  4.107316098693041E-02;
	venus << 2981511.5 <<  6.930865481971814E-01 <<  2.102627398451934E-01 << -3.530480865857037E-02;
	venus << 2990642.5 << -2.815557600448605E-01 << -6.683588584548419E-01 <<  4.015218699115584E-03;
	venus << 2999773.5 << -3.073721433848189E-01 <<  6.494759326197824E-01 <<  2.881732270568284E-02;
	venus << 3008904.5 <<  6.984751154178859E-01 << -1.970184716459250E-01 << -4.276511642371764E-02;
	venus << 3018035.5 << -5.958483172401841E-01 << -4.082581463839635E-01 <<  2.616467713063362E-02;
	venus << 3027166.5 <<  9.792473157187528E-02 <<  7.137352662854839E-01 <<  7.319050598843475E-03;
	venus << 3036298.5 <<  4.985177922488104E-01 << -5.287583927772195E-01 << -3.750186435302465E-02;
	venus << 3045429.5 << -7.185119251699571E-01 << -3.776632275188453E-02 <<  3.961064278638447E-02;
	venus << 3054560.5 <<  4.561307708398903E-01 <<  5.602307739611886E-01 << -1.539580543887291E-02;
	venus << 3063691.5 <<  1.367655101974759E-01 << -7.138029546776109E-01 << -2.066765633957779E-02;
	venus << 3072822.5 << -6.205243578092871E-01 <<  3.612761013606323E-01 <<  4.134741880336015E-02;
	venus << 3081953.5 <<  6.884658776566902E-01 <<  2.246627305870286E-01 << -3.439969647453496E-02;
	venus << 3091084.5 << -2.679109640943308E-01 << -6.740111838774416E-01 <<  2.565513430734976E-03;
	venus << 3100215.5 << -3.209976646255690E-01 <<  6.428597769799828E-01 <<  2.982849461697926E-02;
	venus << 3109346.5 <<  7.022978022641999E-01 << -1.824660656344419E-01 << -4.261712427973630E-02;
	venus << 3118477.5 << -5.874718936091489E-01 << -4.204746601829173E-01 <<  2.496863467820698E-02;
	venus << 3127608.5 <<  8.285454529639873E-02 <<  7.156104684815595E-01 <<  8.750879959311895E-03;
	venus << 3136739.5 <<  4.946787624890081E-01 << -5.322173032538619E-01 << -3.756914651750981E-02;
	venus << 3145870.5 << -7.188181332931405E-01 << -3.245247693163136E-02 <<  3.945380257225797E-02;
	venus << 3155001.5 <<  4.600536322781013E-01 <<  5.570613316045459E-01 << -1.510758486609187E-02;
	venus << 3164132.5 <<  1.315554421951594E-01 << -7.146886936872521E-01 << -2.086815983929913E-02;
	venus << 3173263.5 << -6.179398157422559E-01 <<  3.658220854032098E-01 <<  4.133869149352780E-02;
	venus << 3182395.5 <<  6.835587940705292E-01 <<  2.389667647384741E-01 << -3.345141854331341E-02;
	venus << 3191526.5 << -2.541655751429541E-01 << -6.793688569716663E-01 <<  1.115608345537707E-03;
	venus << 3200657.5 << -3.344808737195320E-01 <<  6.359770233786449E-01 <<  3.079929840893888E-02;
	venus << 3209788.5 <<  7.058207333249200E-01 << -1.678369635403852E-01 << -4.241539552421450E-02;
	venus << 3218919.5 << -5.788550327211498E-01 << -4.325229874846200E-01 <<  2.374449415058490E-02;
	venus << 3228050.5 <<  6.776095389303843E-02 <<  7.171665743017461E-01 <<  1.016562468134012E-02;
	venus << 3237181.5 <<  5.054170592163427E-01 << -5.218557499667921E-01 << -3.817016424771593E-02;
	venus << 3246312.5 << -7.181498190263217E-01 << -4.743532550173559E-02 <<  3.884179566753974E-02;
	venus << 3255443.5 <<  4.482275094584682E-01 <<  5.665407281588354E-01 << -1.369840454369508E-02;
	venus << 3264574.5 <<  1.460842217066147E-01 << -7.117943381844026E-01 << -2.208244177851299E-02;
	venus << 3273705.5 << -6.255653897559640E-01 <<  3.528119530228384E-01 <<  4.154102083086472E-02;
	venus << 3282836.5 <<  6.852048254308710E-01 <<  2.342011283539597E-01 << -3.321292905389896E-02;
	venus << 3291967.5 << -2.591703114492408E-01 << -6.774082693509800E-01 <<  8.470992467233321E-04;
	venus << 3301098.5 << -3.299308710608546E-01 <<  6.384456107620322E-01 <<  3.093153878885669E-02;
	venus << 3310229.5 <<  7.045733285430213E-01 << -1.727608486914104E-01 << -4.230281650385745E-02;
	venus << 3319360.5 << -5.820612002988577E-01 << -4.282184220202623E-01 <<  2.347975523403632E-02;
	venus << 3328492.5 <<  5.261506685177558E-02 <<  7.184003350660916E-01 <<  1.156606644011580E-02;
	venus << 3337623.5 <<  5.159612845008616E-01 << -5.112711933734295E-01 << -3.872164771373812E-02;
	venus << 3346754.5 << -7.171762350105555E-01 << -6.236607506258311E-02 <<  3.818110295952811E-02;
	venus << 3355885.5 <<  4.362116013544780E-01 <<  5.757749426338330E-01 << -1.227492171020631E-02;
	venus << 3365016.5 <<  1.605582782134001E-01 << -7.086024142189501E-01 << -2.326584219799592E-02;
	venus << 3374147.5 << -6.329108631763272E-01 <<  3.396704206168852E-01 <<  4.168854177603198E-02;
	venus << 3383278.5 <<  6.800979644109675E-01 <<  2.484238305842488E-01 << -3.221164950606028E-02;
	venus << 3392409.5 << -2.453708172312347E-01 << -6.825763254238062E-01 << -5.977132673813609E-04;
	venus << 3401540.5 << -3.433483909674314E-01 <<  6.313482759160544E-01 <<  3.184548931462308E-02;
	venus << 3410671.5 <<  7.078977482879167E-01 << -1.580891753386351E-01 << -4.202879322028734E-02;
	venus << 3419802.5 << -5.732660532735726E-01 << -4.401658716001570E-01 <<  2.221797160867678E-02;
	venus << 3428933.5 <<  5.769409830037332E-02 <<  7.180711481313066E-01 <<  1.181253846032900E-02;
	venus << 3438064.5 <<  5.122578607889228E-01 << -5.148693027595131E-01 << -3.875287332520629E-02;
	venus << 3447195.5 << -7.176585947383681E-01 << -5.707521419654339E-02 <<  3.798903084285896E-02;
	venus << 3456326.5 <<  4.402702433717043E-01 <<  5.727296869273611E-01 << -1.197715092912850E-02;
	venus << 3465457.5 <<  1.553695863593489E-01 << -7.096580536181736E-01 << -2.344392843814310E-02;
	venus << 3474589.5 << -6.400097094014956E-01 <<  3.263502273484309E-01 <<  4.178057989236711E-02;
	venus << 3483720.5 <<  6.746891476723232E-01 <<  2.625470304414993E-01 << -3.116803891984495E-02;
	venus << 3492851.5 << -2.314338912552829E-01 << -6.874736568844498E-01 << -2.041633017379318E-03;
	venus << 3501982.5 << -3.566071089194717E-01 <<  6.239812540079044E-01 <<  3.271537244579643E-02;
	venus << 3511113.5 <<  7.109258592111302E-01 << -1.433468218414591E-01 << -4.170178684321901E-02;
	venus << 3520244.5 << -5.642235824035069E-01 << -4.519385526958534E-01 <<  2.093085984740411E-02;
	venus << 3529375.5 <<  4.257205042805134E-02 <<  7.190925423371562E-01 <<  1.318278360986509E-02;
	venus << 3538506.5 <<  5.226530185115036E-01 << -5.041452317223351E-01 << -3.923480940594080E-02;
	venus << 3547637.5 << -7.164879753491498E-01 << -7.196670343331195E-02 <<  3.726600597714094E-02;
	venus << 3556768.5 <<  4.281173182990304E-01 <<  5.817920060323425E-01 << -1.053854938306488E-02;
	venus << 3565899.5 <<  1.697860312823248E-01 << -7.062744147890273E-01 << -2.458136624881312E-02;
	venus << 3575030.5 << -6.376736586627496E-01 <<  3.310420379770823E-01 <<  4.171655988102901E-02;
	venus << 3584161.5 <<  6.764947577435938E-01 <<  2.578237703034890E-01 << -3.090197671813309E-02;
	venus << 3593292.5 << -2.364751258379947E-01 << -6.857020437304542E-01 << -2.307272013680986E-03;
	venus << 3602423.5 << -3.521642160142131E-01 <<  6.265758851883306E-01 <<  3.281623719014399E-02;
	venus << 3611554.5 <<  7.098642417179812E-01 << -1.483282653039265E-01 << -4.155172502075884E-02;
	venus << 3620686.5 << -5.549451581620239E-01 << -4.635012543244415E-01 <<  1.961866586962576E-02;
	venus << 3629817.5 <<  2.740941041750606E-02 <<  7.198024809201211E-01 <<  1.453394369198983E-02;
	venus << 3638948.5 <<  5.328300401422168E-01 << -4.932095681439785E-01 << -3.966419029300846E-02;
	venus << 3648079.5 << -7.150184774746086E-01 << -8.683788083231146E-02 <<  3.649530759244790E-02;
	venus << 3657210.5 <<  4.157774201823722E-01 <<  5.906036711769228E-01 << -9.089100537457144E-03;
	venus << 3666341.5 <<  1.841593657550847E-01 << -7.025978227002042E-01 << -2.568613674687685E-02;
	venus << 3675472.5 << -6.445907349720537E-01 <<  3.176055298505308E-01 <<  4.173644193629357E-02;
	venus << 3684603.5 <<  6.709054548728599E-01 <<  2.718490912523395E-01 << -2.981263971424573E-02;
	venus << 3693734.5 << -2.224703210126262E-01 << -6.904069910952187E-01 << -3.739288456330946E-03;
	venus << 3702865.5 << -3.653017719903053E-01 <<  6.190287653439379E-01 <<  3.362281497873799E-02;
	venus << 3711996.5 <<  7.126926355540434E-01 << -1.335516263638473E-01 << -4.115416750447853E-02;
	venus << 3721127.5 << -5.583913293330165E-01 << -4.593338983264946E-01 <<  1.933381693638987E-02;
	venus << 3730258.5 <<  3.250923018253286E-02 <<  7.196495843412338E-01 <<  1.476305904970204E-02;
	venus << 3739389.5 <<  5.292324474934034E-01 << -4.969463778272684E-01 << -3.965621128016776E-02;
	venus << 3748520.5 << -7.156897629272441E-01 << -8.158468944428236E-02 <<  3.627088907136196E-02;
	venus << 3757651.5 <<  4.199155021899122E-01 <<  5.876975296350724E-01 << -8.784739576531224E-03;
	venus << 3766783.5 <<  1.984582097299740E-01 << -6.986216900811359E-01 << -2.675190566657023E-02;
	venus << 3775914.5 << -6.512260650973085E-01 <<  3.040307842349461E-01 <<  4.169855509389386E-02;
	venus << 3785045.5 <<  6.650233896941810E-01 <<  2.857663707107581E-01 << -2.868469232424502E-02;
	venus << 3794176.5 << -2.083540839329897E-01 << -6.948255073342241E-01 << -5.164340711630026E-03;
	venus << 3803307.5 << -3.782744740944860E-01 <<  6.112267491900609E-01 <<  3.438287150931403E-02;
	venus << 3812438.5 <<  7.152129934441651E-01 << -1.187308087207241E-01 << -4.070421675797326E-02;
	venus << 3821569.5 << -5.489747045036725E-01 << -4.707742770200057E-01 <<  1.799716809854163E-02;
	venus << 3830700.5 <<  1.733051342130281E-02 <<  7.201384812623701E-01 <<  1.608000981995501E-02;
	venus << 3839831.5 <<  5.392693435036233E-01 << -4.858806714560508E-01 << -4.001595954729566E-02;
	venus << 3848962.5 << -7.140139609778505E-01 << -9.645399928542614E-02 <<  3.544183955681905E-02;
	venus << 3858093.5 <<  4.074533890758961E-01 <<  5.963278758306680E-01 << -7.329684055013492E-03;
	venus << 3867224.5 <<  1.933803270224768E-01 << -6.999655416298255E-01 << -2.689259728595137E-02;
	venus << 3876355.5 << -6.490431019643685E-01 <<  3.088095501199462E-01 <<  4.159515602172097E-02;
	venus << 3885486.5 <<  6.670020781494582E-01 <<  2.810938366360880E-01 << -2.839438250905622E-02;
	venus << 3894617.5 << -2.134781529716526E-01 << -6.932060452276062E-01 << -5.420287658747383E-03;
	venus << 3903748.5 << -3.739255761494203E-01 <<  6.139737682332504E-01 <<  3.444935803867593E-02;

	// Mars:
	// Start time      : A.D. 1600-Jan-01 00:00:00.0000 TDB (2305447.500000000)
	// Stop  time      : A.D. 2500-Jan-04 00:00:00.0000 TDB (2634166.500000000)
	// Step-size       : 15 calendar years
	mars << 2305447.5 << -8.593163811347189E-01 <<  1.394720971789812E+00 <<  5.086469239544589E-02;
	mars << 2310926.5 << -6.608385310158910E-01 <<  1.484580298213753E+00 <<  4.767853167578042E-02;
	mars << 2316405.5 << -4.492437690982107E-01 <<  1.545809050964638E+00 <<  4.356832180463761E-02;
	mars << 2321884.5 << -2.287732768023697E-01 <<  1.575953282440993E+00 <<  3.860314826466391E-02;
	mars << 2327362.5 <<  9.416449875346710E-03 <<  1.572321437745818E+00 <<  3.250431469846549E-02;
	mars << 2332841.5 <<  2.348675837978696E-01 <<  1.533727952680114E+00 <<  2.600894805996008E-02;
	mars << 2338320.5 <<  4.546206835430203E-01 <<  1.461241105576332E+00 <<  1.897161638880356E-02;
	mars << 2343798.5 <<  6.758911299034948E-01 <<  1.347030047110919E+00 <<  1.102921716080467E-02;
	mars << 2349276.5 <<  8.781468005756334E-01 <<  1.195920407341249E+00 <<  2.823569141748553E-03;
	mars << 2354755.5 <<  1.046701213297760E+00 <<  1.021529810915667E+00 << -5.023350377853192E-03;
	mars << 2360234.5 <<  1.186793176120758E+00 <<  8.200142751943095E-01 << -1.270390707479196E-02;
	mars << 2365713.5 <<  1.293960241848084E+00 <<  5.959547595331234E-01 << -2.002737440553500E-02;
	mars << 2371191.5 <<  1.366450054008260E+00 <<  3.387443733833834E-01 << -2.716419338275639E-02;
	mars << 2376670.5 <<  1.393041422774622E+00 <<  8.633259244552194E-02 << -3.305198663254438E-02;
	mars << 2382148.5 <<  1.374101101957199E+00 << -1.844844337918968E-01 << -3.819172851560369E-02;
	mars << 2387627.5 <<  1.311861447610036E+00 << -4.334696090102491E-01 << -4.178973182422194E-02;
	mars << 2393105.5 <<  1.200500644214709E+00 << -6.834101498870278E-01 << -4.420055445520300E-02;
	mars << 2398584.5 <<  1.056458309765558E+00 << -8.959865829699956E-01 << -4.501846235916494E-02;
	mars << 2404063.5 <<  8.791295315549212E-01 << -1.080994511372132E+00 << -4.445231869756832E-02;
	mars << 2409542.5 <<  6.738614780097609E-01 << -1.232523627935425E+00 << -4.249915989413171E-02;
	mars << 2415020.5 <<  4.353672050963562E-01 << -1.352511654752771E+00 << -3.907972465526616E-02;
	mars << 2420498.5 <<  1.820439531138856E-01 << -1.428038993474950E+00 << -3.438117212463899E-02;
	mars << 2425977.5 << -6.186358333975374E-02 << -1.458040495805149E+00 << -2.897561251902949E-02;
	mars << 2431456.5 << -3.054065915114116E-01 << -1.449000201726634E+00 << -2.278894220061726E-02;
	mars << 2436934.5 << -5.532132503810797E-01 << -1.398669206344053E+00 << -1.563348528987718E-02;
	mars << 2442413.5 << -7.739534685311177E-01 << -1.315618913635878E+00 << -8.488135812440144E-03;
	mars << 2447892.5 << -9.763410874139239E-01 << -1.201258553556260E+00 << -1.141650368654698E-03;
	mars << 2453371.5 << -1.156778110504573E+00 << -1.060258468758684E+00 <<  6.202429444711275E-03;
	mars << 2458849.5 << -1.320107604952232E+00 << -8.857574644771996E-01 <<  1.382907562643052E-02;
	mars << 2464328.5 << -1.446075239841682E+00 << -7.019345693319942E-01 <<  2.072094587019319E-02;
	mars << 2469807.5 << -1.543231689349428E+00 << -5.035895177339794E-01 <<  2.720193315280650E-02;
	mars << 2475286.5 << -1.609793234117925E+00 << -2.962633754373760E-01 <<  3.312932519474993E-02;
	mars << 2480764.5 << -1.646918426631360E+00 << -7.000622720520019E-02 <<  3.872474791751875E-02;
	mars << 2486243.5 << -1.650278325515880E+00 <<  1.449307817505134E-01 <<  4.326720546724398E-02;
	mars << 2491721.5 << -1.620956535790368E+00 <<  3.698461063239862E-01 <<  4.721108712661447E-02;
	mars << 2497200.5 << -1.562858565497793E+00 <<  5.746975998134991E-01 <<  5.005607712699469E-02;
	mars << 2502678.5 << -1.470201271050181E+00 <<  7.808997267440548E-01 <<  5.208153263484457E-02;
	mars << 2508157.5 << -1.355471416761419E+00 <<  9.610186796432958E-01 <<  5.304049002258925E-02;
	mars << 2513636.5 << -1.216165355609577E+00 <<  1.123392975672690E+00 <<  5.303780429779852E-02;
	mars << 2519115.5 << -1.054325052130753E+00 <<  1.264963368692209E+00 <<  5.206681783536770E-02;
	mars << 2524593.5 << -8.612219738533241E-01 <<  1.389447834261136E+00 <<  4.999269528607678E-02;
	mars << 2530071.5 << -6.496933189902875E-01 <<  1.483970590077591E+00 <<  4.685211021925800E-02;
	mars << 2535550.5 << -4.372948546469792E-01 <<  1.543068753330718E+00 <<  4.297290854719427E-02;
	mars << 2541029.5 << -2.151043372122667E-01 <<  1.570815867920269E+00 <<  3.820070490364816E-02;
	mars << 2546507.5 <<  2.440627752607371E-02 <<  1.564510535088688E+00 <<  3.232220870361176E-02;
	mars << 2551986.5 <<  2.504497379365264E-01 <<  1.523226141835349E+00 <<  2.603505969210516E-02;
	mars << 2557465.5 <<  4.701505418366565E-01 <<  1.447186905180799E+00 <<  1.919541681216132E-02;
	mars << 2562943.5 <<  6.915277751833436E-01 <<  1.329164992946110E+00 <<  1.143504376044912E-02;
	mars << 2568421.5 <<  8.928533047630969E-01 <<  1.174193703982779E+00 <<  3.393412881519549E-03;
	mars << 2573900.5 <<  1.059113675107280E+00 <<  9.966690319476333E-01 << -4.274040983682852E-03;
	mars << 2579379.5 <<  1.196796073999594E+00 <<  7.910626174083361E-01 << -1.184612752263091E-02;
	mars << 2584858.5 <<  1.300581966484841E+00 <<  5.638781983933476E-01 << -1.905520916986478E-02;
	mars << 2590336.5 <<  1.368703503553623E+00 <<  3.046412263176584E-01 << -2.609545881017315E-02;
	mars << 2595815.5 <<  1.389943670232123E+00 <<  5.087973319907362E-02 << -3.189323265371180E-02;
	mars << 2601294.5 <<  1.368362025385807E+00 << -2.055320854106171E-01 << -3.674474399937577E-02;
	mars << 2606773.5 <<  1.303394013348758E+00 << -4.543132894102774E-01 << -4.040520256369984E-02;
	mars << 2612251.5 <<  1.189669998714514E+00 << -7.024400798766441E-01 << -4.291191462137037E-02;
	mars << 2617730.5 <<  1.042620167267976E+00 << -9.141238862980028E-01 << -4.386931957784559E-02;
	mars << 2623209.5 <<  8.637246388443391E-01 << -1.096902397356621E+00 << -4.347800221140578E-02;
	mars << 2628688.5 <<  6.577838518761338E-01 << -1.245954023680832E+00 << -4.175654783779491E-02;
	mars << 2634166.5 <<  4.187377283874138E-01 << -1.362827394334597E+00 << -3.858711334008324E-02;

	// Jupiter:
	// Start time      : A.D. 1600-Jan-10 00:00:00.0000 TDB (2305456.500000000)
	// Stop  time      : A.D. 2485-Jan-10 00:00:00.0000 TDB (2628697.500000000)
	// Step-size       : 15 calendar years
	jupiter << 2305456.5 << -4.111730701881532E+00 <<  3.417683252977552E+00 <<  7.952191676664057E-02;
	jupiter << 2310935.5 << -3.621318525925235E+00 << -4.006362483022776E+00 <<  9.719902531283665E-02;
	jupiter << 2316414.5 <<  3.971429301703552E+00 << -3.099974875322259E+00 << -7.745650917049707E-02;
	jupiter << 2321893.5 <<  2.051642564333571E+00 <<  4.604480647826734E+00 << -6.421197734657835E-02;
	jupiter << 2327371.5 << -5.081125193636537E+00 <<  1.852638350761014E+00 <<  1.072635767449819E-01;
	jupiter << 2332850.5 << -2.033874888636495E+00 << -4.936195568369103E+00 <<  6.512616358272698E-02;
	jupiter << 2338329.5 <<  4.801269040483668E+00 << -1.314548861549458E+00 << -1.029555931961769E-01;
	jupiter << 2343807.5 <<  1.863585672608799E-01 <<  5.119071484365518E+00 << -2.436385516781461E-02;
	jupiter << 2349285.5 << -5.444255871152979E+00 <<  7.593500508764311E-02 <<  1.222545306620374E-01;
	jupiter << 2354764.5 << -2.074815760353626E-01 << -5.254889916954029E+00 <<  2.541745608958439E-02;
	jupiter << 2360243.5 <<  4.905557655334567E+00 <<  6.568171711773217E-01 << -1.129623354514743E-01;
	jupiter << 2365722.5 << -1.710889184256227E+00 <<  4.924702208218476E+00 <<  1.888158745387208E-02;
	jupiter << 2371200.5 << -5.176654875379895E+00 << -1.712606258983425E+00 <<  1.232263769886522E-01;
	jupiter << 2376679.5 <<  1.646131008815441E+00 << -4.900652137412688E+00 << -1.746306512471713E-02;
	jupiter << 2382157.5 <<  4.265967543221164E+00 <<  2.523836406941355E+00 << -1.059836172779379E-01;
	jupiter << 2387636.5 << -3.375348787482535E+00 <<  4.087280762595189E+00 <<  5.939236727725588E-02;
	jupiter << 2393114.5 << -4.318457414728717E+00 << -3.293466188136708E+00 <<  1.102530023495243E-01;
	jupiter << 2398593.5 <<  3.266387932276998E+00 << -3.894816442872000E+00 << -5.765729008962973E-02;
	jupiter << 2404072.5 <<  2.980328522907179E+00 <<  4.013112735735738E+00 << -8.315795683278379E-02;
	jupiter << 2409551.5 << -4.624431607031957E+00 <<  2.739516343476889E+00 <<  9.262901293077581E-02;
	jupiter << 2415029.5 << -2.960296230505947E+00 << -4.494726626610772E+00 <<  8.469447534334884E-02;
	jupiter << 2420507.5 <<  4.426176412571971E+00 << -2.349625934274831E+00 << -8.971078909792447E-02;
	jupiter << 2425986.5 <<  1.264618599057650E+00 <<  4.907667912214100E+00 << -4.844753088629387E-02;
	jupiter << 2431465.5 << -5.319855754828637E+00 <<  1.074272556961103E+00 <<  1.147987253857985E-01;
	jupiter << 2436943.5 << -1.262506649826843E+00 << -5.158510204824173E+00 <<  4.949314029409294E-02;
	jupiter << 2442422.5 <<  4.943274510457927E+00 << -4.602286834185330E-01 << -1.088434032019346E-01;
	jupiter << 2447901.5 << -6.354894527338278E-01 <<  5.114675707121821E+00 << -6.934494716731627E-03;
	jupiter << 2453380.5 << -5.405232807389647E+00 << -7.195963587351745E-01 <<  1.239412774024903E-01;
	jupiter << 2458858.5 <<  5.929203381417349E-01 << -5.190535006689434E+00 <<  8.293161653316044E-03;
	jupiter << 2464337.5 <<  4.720272793239588E+00 <<  1.496376962567584E+00 << -1.118211172757597E-01;
	jupiter << 2469816.5 << -2.452205158555679E+00 <<  4.635815444924145E+00 <<  3.545134518680300E-02;
	jupiter << 2475295.5 << -4.876916996594808E+00 << -2.431240469335498E+00 <<  1.191613720479785E-01;
	jupiter << 2480773.5 <<  2.369234142760164E+00 << -4.553436459267464E+00 << -3.383136238348495E-02;
	jupiter << 2486252.5 <<  3.785005298911226E+00 <<  3.228427239075562E+00 << -9.811271869449179E-02;
	jupiter << 2491730.5 << -3.954260008606033E+00 <<  3.569980442849494E+00 <<  7.321749774353625E-02;
	jupiter << 2497209.5 << -3.798469894234790E+00 << -3.861759050829875E+00 <<  1.010816004107724E-01;
	jupiter << 2502687.5 <<  3.817995662654534E+00 << -3.303832748032003E+00 << -7.118393007932054E-02;
	jupiter << 2508166.5 <<  2.289651540467984E+00 <<  4.478511495117529E+00 << -7.005871907289378E-02;
	jupiter << 2513645.5 << -4.984076068706266E+00 <<  2.062449760819937E+00 <<  1.023249023812957E-01;
	jupiter << 2519124.5 << -2.274359193550020E+00 << -4.851188787538219E+00 <<  7.136282812078386E-02;
	jupiter << 2524602.5 <<  4.728187472258202E+00 << -1.576859979963561E+00 << -9.858671225864453E-02;
	jupiter << 2530080.5 <<  4.595036137825477E-01 <<  5.087589327465850E+00 << -3.199150398367770E-02;
	jupiter << 2535559.5 << -5.427551207456360E+00 <<  3.142910184724080E-01 <<  1.194626401037045E-01;
	jupiter << 2541038.5 << -4.820335759904149E-01 << -5.256700404156995E+00 <<  3.332977507648639E-02;
	jupiter << 2546516.5 <<  4.934994221407067E+00 <<  3.799884541809996E-01 << -1.114082353478660E-01;
	jupiter << 2551995.5 << -1.448391858502923E+00 <<  4.992948797644105E+00 <<  1.068734576376924E-02;
	jupiter << 2557474.5 << -5.246030989377603E+00 << -1.481580150498083E+00 <<  1.230149431937343E-01;
	jupiter << 2562952.5 <<  1.372587162306276E+00 << -5.002918342902835E+00 << -8.811726183901272E-03;
	jupiter << 2568430.5 <<  4.393191723098785E+00 <<  2.283416973538870E+00 << -1.075016857361124E-01;
	jupiter << 2573909.5 << -3.164189886976091E+00 <<  4.237272516183106E+00 <<  5.186936224528181E-02;
	jupiter << 2579388.5 << -4.462353877196473E+00 << -3.104329909637727E+00 <<  1.125839077178284E-01;
	jupiter << 2584867.5 <<  3.052436970434998E+00 << -4.080407993594633E+00 << -4.991283994036318E-02;
	jupiter << 2590345.5 <<  3.174464407902570E+00 <<  3.849053487649509E+00 << -8.726952484853780E-02;
	jupiter << 2595824.5 << -4.490329192156891E+00 <<  2.938830856372088E+00 <<  8.674669080980536E-02;
	jupiter << 2601303.5 << -3.154087392537798E+00 << -4.373136850140809E+00 <<  8.912795273949066E-02;
	jupiter << 2606782.5 <<  4.301008311284931E+00 << -2.581076699926234E+00 << -8.397239781494548E-02;
	jupiter << 2612260.5 <<  1.473845920425394E+00 <<  4.837963147672595E+00 << -5.399464764946989E-02;
	jupiter << 2617739.5 << -5.270971266172803E+00 <<  1.279087397760846E+00 <<  1.111603024938399E-01;
	jupiter << 2623218.5 << -1.466911930612735E+00 << -5.117237863642162E+00 <<  5.512027435292081E-02;
	jupiter << 2628697.5 <<  4.911578950512353E+00 << -6.995265971759733E-01 << -1.056522903395049E-01;

	// Saturn:
	// Start time      : A.D. 1600-Jan-01 00:00:00.0000 TDB (2305447.500000000)
	// Stop  time      : A.D. 2590-Jan-01 00:00:00.0000 TDB (2667038.500000000)
	// Step-size       : 15 calendar years
	saturn << 2305447.5 << -8.651414646080703E+00 << -4.493428750825409E+00 <<  4.216284751143584E-01;
	saturn << 2310926.5 <<  8.864533121532345E+00 <<  2.945506471081205E+00 << -4.024195322944089E-01;
	saturn << 2316405.5 << -8.135016422337120E+00 << -5.454321887538344E+00 <<  4.189193575560481E-01;
	saturn << 2321884.5 <<  8.420081755312316E+00 <<  3.970890237940459E+00 << -4.031481078308559E-01;
	saturn << 2327362.5 << -7.575869856244278E+00 << -6.292685196913841E+00 <<  4.118183132442944E-01;
	saturn << 2332841.5 <<  7.823577417612217E+00 <<  4.905774065996610E+00 << -3.969462366143903E-01;
	saturn << 2338320.5 << -6.864783118908137E+00 << -7.109467444097760E+00 <<  3.985813701189916E-01;
	saturn << 2343798.5 <<  7.142380507322883E+00 <<  5.812857789007003E+00 << -3.859435552151032E-01;
	saturn << 2349276.5 << -6.142732506777919E+00 << -7.804751021793130E+00 <<  3.822117865134503E-01;
	saturn << 2354755.5 <<  6.334351171308480E+00 <<  6.593123847098505E+00 << -3.683534891528235E-01;
	saturn << 2360234.5 << -5.270840719460922E+00 << -8.445827882971686E+00 <<  3.592426724502685E-01;
	saturn << 2365713.5 <<  5.431864019251545E+00 <<  7.331596551114441E+00 << -3.453200023425912E-01;
	saturn << 2371191.5 << -4.408328686660296E+00 << -8.973858043693706E+00 <<  3.340972499808945E-01;
	saturn << 2376670.5 <<  4.448092582626072E+00 <<  7.907923720402026E+00 << -3.168778988086379E-01;
	saturn << 2382148.5 << -3.420194178811991E+00 << -9.405417256496195E+00 <<  3.025120376895758E-01;
	saturn << 2387627.5 <<  3.380497040545513E+00 <<  8.410055470051692E+00 << -2.828630896652023E-01;
	saturn << 2393105.5 << -2.456128665406391E+00 << -9.738743996193447E+00 <<  2.697071692772534E-01;
	saturn << 2398584.5 <<  2.272593049928588E+00 <<  8.737286846826613E+00 << -2.447909260278776E-01;
	saturn << 2404063.5 << -1.389378515233124E+00 << -9.941798324620130E+00 <<  2.307158591718420E-01;
	saturn << 2409542.5 <<  1.087151640098880E+00 <<  8.966512835389418E+00 << -2.010988759552211E-01;
	saturn << 2415020.5 << -3.669668594754815E-01 << -1.005835161949959E+01 <<  1.915839285622428E-01;
	saturn << 2420498.5 << -6.541603229567929E-02 <<  9.012213029409427E+00 << -1.560384335482364E-01;
	saturn << 2425977.5 <<  7.181902248570151E-01 << -1.001969481029289E+01 <<  1.474003077041920E-01;
	saturn << 2431456.5 << -1.285542063859727E+00 <<  8.937579153462817E+00 << -1.054616584439862E-01;
	saturn << 2436934.5 <<  1.749178738142716E+00 << -9.908837304099741E+00 <<  1.038427985598214E-01;
	saturn << 2442413.5 << -2.423322062017020E+00 <<  8.696720824929725E+00 << -5.577995070673683E-02;
	saturn << 2447892.5 <<  2.807663768051146E+00 << -9.625880688786461E+00 <<  5.623321590502382E-02;
	saturn << 2453371.5 << -3.586777799138939E+00 <<  8.317500826234722E+00 << -2.042087875414097E-03;
	saturn << 2458849.5 <<  3.797244866117995E+00 << -9.288096503712179E+00 <<  1.033316914898575E-02;
	saturn << 2464328.5 << -4.624122504363694E+00 <<  7.805518897172681E+00 <<  4.851221131809263E-02;
	saturn << 2469807.5 <<  4.766226157279622E+00 << -8.773718285727414E+00 << -3.787765383411665E-02;
	saturn << 2475286.5 << -5.656038800942119E+00 <<  7.147678731997142E+00 <<  1.018739738797095E-01;
	saturn << 2480764.5 <<  5.661901368504909E+00 << -8.220853582865601E+00 << -8.375426701583114E-02;
	saturn << 2486243.5 << -6.522011469445991E+00 <<  6.408307632402502E+00 <<  1.495733480916664E-01;
	saturn << 2491721.5 <<  6.488449783176379E+00 << -7.504657774915292E+00 << -1.298535948071325E-01;
	saturn << 2497200.5 << -7.347153963206455E+00 <<  5.527508050487382E+00 <<  1.985116233972639E-01;
	saturn << 2502678.5 <<  7.241065081823880E+00 << -6.764369549645325E+00 << -1.731661803739120E-01;
	saturn << 2508157.5 << -7.999050057572356E+00 <<  4.615298922700425E+00 <<  2.406022467917713E-01;
	saturn << 2513636.5 <<  7.889342118807557E+00 << -5.875577683871818E+00 << -2.151313316768424E-01;
	saturn << 2519115.5 << -8.577431717298674E+00 <<  3.573207303618265E+00 <<  2.823483571926947E-01;
	saturn << 2524593.5 <<  8.460607280095115E+00 << -4.981664195031263E+00 << -2.536890281172585E-01;
	saturn << 2530071.5 << -8.982567205821336E+00 <<  2.560228102545022E+00 <<  3.161781534460064E-01;
	saturn << 2535550.5 <<  8.889826461686791E+00 << -3.977711199526808E+00 << -2.888173530215363E-01;
	saturn << 2541029.5 << -9.291054778229638E+00 <<  1.427974839087160E+00 <<  3.485199827299332E-01;
	saturn << 2546507.5 <<  9.247775274488049E+00 << -2.973522894469097E+00 << -3.205067829341330E-01;
	saturn << 2551986.5 << -9.447100082168777E+00 <<  3.697594126588916E-01 <<  3.729492167779909E-01;
	saturn << 2557465.5 <<  9.443514218178153E+00 << -1.892156376584142E+00 << -3.474890863789797E-01;
	saturn << 2562943.5 << -9.480671031316669E+00 << -7.738125203045413E-01 <<  3.942608121351757E-01;
	saturn << 2568421.5 <<  9.566527577503271E+00 << -8.364136973860692E-01 << -3.704051288063209E-01;
	saturn << 2573900.5 << -9.394547349011219E+00 << -1.819367332384804E+00 <<  4.085375352686307E-01;
	saturn << 2579379.5 <<  9.515096662756456E+00 <<  2.638835298290385E-01 << -3.876008208257148E-01;
	saturn << 2584858.5 << -9.169108506209692E+00 << -2.922603650305322E+00 <<  4.185377833257129E-01;
	saturn << 2590336.5 <<  9.394250027722988E+00 <<  1.335914051363698E+00 << -4.007315260383362E-01;
	saturn << 2595815.5 << -8.858558399516159E+00 << -3.902581871044317E+00 <<  4.224128878669121E-01;
	saturn << 2601294.5 <<  9.097464345380700E+00 <<  2.396581467530204E+00 << -4.071951032838818E-01;
	saturn << 2606773.5 << -8.397046697516000E+00 << -4.911478552897730E+00 <<  4.210929136319247E-01;
	saturn << 2612251.5 <<  8.730764705420102E+00 <<  3.422817983612839E+00 << -4.093720748320954E-01;
	saturn << 2617730.5 << -7.888427450018562E+00 << -5.786038977927599E+00 <<  4.149617984074606E-01;
	saturn << 2623209.5 <<  8.197682837050861E+00 <<  4.396876559417755E+00 << -4.045905385497410E-01;
	saturn << 2628688.5 << -7.223761463281973E+00 << -6.654261591543178E+00 <<  4.028103315258592E-01;
	saturn << 2634166.5 <<  7.599650603474343E+00 <<  5.319787113764992E+00 << -3.954711995870469E-01;
	saturn << 2639644.5 << -6.547531342969322E+00 << -7.388279385943359E+00 <<  3.873385885533113E-01;
	saturn << 2645123.5 <<  6.856308654781660E+00 <<  6.148366769559859E+00 << -3.794829295158329E-01;
	saturn << 2650602.5 << -5.717483445122559E+00 << -8.082864233028756E+00 <<  3.653513210364794E-01;
	saturn << 2656080.5 <<  6.045246612805867E+00 <<  6.919626417843433E+00 << -3.589807261373175E-01;
	saturn << 2661559.5 << -4.899621763048184E+00 << -8.655239211601083E+00 <<  3.412409246119955E-01;
	saturn << 2667038.5 <<  5.120577969621112E+00 <<  7.562770279546319E+00 << -3.323158474295848E-01;

	// Uranus:
	// Start time      : A.D. 1600-Jan-01 00:00:00.0000 TDB (2305447.500000000)
	// Stop  time      : A.D. 2600-Jan-01 00:00:00.0000 TDB (2670690.500000000)
	// Step-size       : 25 calendar years
	uranus << 2305447.5 <<  1.609655513752283E+01 <<  1.149738695712907E+01 << -1.663541040986954E-01;
	uranus << 2314579.5 << -1.494013779684271E+01 <<  1.073049881599604E+01 <<  2.363612334391751E-01;
	uranus << 2323710.5 << -3.687541658162147E+00 << -1.877920675120050E+01 << -2.338451195570254E-02;
	uranus << 2332841.5 <<  2.006643599184895E+01 << -8.663151849163465E-02 << -2.623685852994452E-01;
	uranus << 2341972.5 << -4.230857845811476E+00 <<  1.831342239112197E+01 <<  1.248737300764150E-01;
	uranus << 2351103.5 << -1.444900444261253E+01 << -1.168181290268731E+01 <<  1.443783159896584E-01;
	uranus << 2360234.5 <<  1.627270149830295E+01 << -1.162237353425167E+01 << -2.563162322316797E-01;
	uranus << 2369365.5 <<  8.509593914250294E+00 <<  1.743761690088057E+01 << -4.517107210563479E-02;
	uranus << 2378496.5 << -1.827116267741343E+01 <<  9.816834704638125E-01 <<  2.420118288666777E-01;
	uranus << 2387627.5 <<  6.136278685687443E+00 << -1.858892356085353E+01 << -1.497326601865856E-01;
	uranus << 2396758.5 <<  1.770199912429661E+01 <<  9.094956336852320E+00 << -1.964026192087592E-01;
	uranus << 2405889.5 << -1.293680088913710E+01 <<  1.314059289837574E+01 <<  2.175574464213456E-01;
	uranus << 2415020.5 << -6.479262777707929E+00 << -1.785344143015778E+01 <<  1.776365886977280E-02;
	uranus << 2424151.5 <<  1.988792712357311E+01 << -2.879959333100579E+00 << -2.689956547559065E-01;
	uranus << 2433282.5 << -1.241439189376733E+00 <<  1.890453731550977E+01 <<  8.647936650002305E-02;
	uranus << 2442413.5 << -1.608400740227782E+01 << -9.060619428768486E+00 <<  1.750343949273137E-01;
	uranus << 2451544.5 <<  1.443051758274212E+01 << -1.373565828163257E+01 << -2.381283040241605E-01;
	uranus << 2460676.5 <<  1.110363454620246E+01 <<  1.609448586116569E+01 << -8.419792041317865E-02;
	uranus << 2469807.5 << -1.782323647472261E+01 <<  4.071476789639214E+00 <<  2.458492165219814E-01;
	uranus << 2478938.5 <<  3.312279373107668E+00 << -1.915789452165341E+01 << -1.137695508190352E-01;
	uranus << 2488069.5 <<  1.886506879579828E+01 <<  6.548916397740944E+00 << -2.196567779514149E-01;
	uranus << 2497200.5 << -1.067001801638965E+01 <<  1.515705010191675E+01 <<  1.941997192191489E-01;
	uranus << 2506331.5 << -9.105347407477478E+00 << -1.649954628409343E+01 <<  5.669350389329137E-02;
	uranus << 2515462.5 <<  1.930739953144770E+01 << -5.580063254293695E+00 << -2.701877948510629E-01;
	uranus << 2524593.5 <<  1.705528062568980E+00 <<  1.900432841117166E+01 <<  4.811847851744051E-02;
	uranus << 2533724.5 << -1.725401410789971E+01 << -6.303087801106044E+00 <<  1.996744451711738E-01;
	uranus << 2542855.5 <<  1.233841228024825E+01 << -1.553696426468168E+01 << -2.165620904194872E-01;
	uranus << 2551986.5 <<  1.338068016208349E+01 <<  1.442101215119403E+01 << -1.197890129422582E-01;
	uranus << 2561117.5 << -1.692392318689846E+01 <<  6.950010519408915E+00 <<  2.440743697675641E-01;
	uranus << 2570248.5 <<  5.140246604191396E-01 << -1.929853537310378E+01 << -7.717891606465090E-02;
	uranus << 2579379.5 <<  1.967184158932658E+01 <<  3.913104976243669E+00 << -2.395365948050331E-01;
	uranus << 2588510.5 << -8.107651798478688E+00 <<  1.677627337887548E+01 <<  1.657232645807118E-01;
	uranus << 2597641.5 << -1.145651120051574E+01 << -1.477604825334926E+01 <<  9.417067598006251E-02;
	uranus << 2606773.5 <<  1.837389527195160E+01 << -8.103948480091271E+00 << -2.659529769662169E-01;
	uranus << 2615904.5 <<  4.595661240421637E+00 <<  1.869111599391315E+01 <<  8.323361157063979E-03;
	uranus << 2625035.5 << -1.798646910744177E+01 << -3.342751655867569E+00 <<  2.193448095006143E-01;
	uranus << 2634166.5 <<  1.001709689210739E+01 << -1.699064822924627E+01 << -1.900040695330853E-01;
	uranus << 2643297.5 <<  1.538576096185643E+01 <<  1.248305843814111E+01 << -1.528745218114268E-01;
	uranus << 2652428.5 << -1.555174966132888E+01 <<  9.723645761067342E+00 <<  2.346239004076392E-01;
	uranus << 2661559.5 << -2.350069690185516E+00 << -1.899829654255339E+01 << -3.779414040669535E-02;
	uranus << 2670690.5 <<  2.005159994815317E+01 <<  1.244399957010208E+00 << -2.525593045524173E-01;

	// Neptune:
	// Start time      : A.D. 1600-Jan-01 00:00:00.0000 TDB (2305447.500000000)
	// Stop  time      : A.D. 2600-Jan-01 00:00:00.0000 TDB (2670690.500000000)
	// Step-size       : 25 calendar years
	neptune << 2305447.5 << -2.658638611523556E+01 <<  1.418614327644317E+01 <<  3.199358181043331E-01;
	neptune << 2314579.5 << -2.724201498132395E+01 << -1.325883525312401E+01 <<  9.000041210435145E-01;
	neptune << 2323710.5 << -5.403142078900765E+00 << -2.978892586378095E+01 <<  7.375735181006099E-01;
	neptune << 2332841.5 <<  2.085526685902283E+01 << -2.167824328357317E+01 << -3.398699794515057E-02;
	neptune << 2341972.5 <<  2.950102341950099E+01 <<  4.606955447500906E+00 << -7.742013293441424E-01;
	neptune << 2351103.5 <<  1.288025521833979E+01 <<  2.688544444439241E+01 << -8.501704940821256E-01;
	neptune << 2360234.5 << -1.483663189390678E+01 <<  2.606322651529166E+01 << -1.949452682642910E-01;
	neptune << 2369365.5 << -3.005664361355939E+01 <<  3.112835292666493E+00 <<  6.281303224811613E-01;
	neptune << 2378496.5 << -2.031027370108721E+01 << -2.249413828951627E+01 <<  9.309117027238304E-01;
	neptune << 2387627.5 <<  6.166180615119696E+00 << -2.956758739241350E+01 <<  4.667663946049324E-01;
	neptune << 2396758.5 <<  2.744040481051559E+01 << -1.205873604497545E+01 << -3.838254494336560E-01;
	neptune << 2405889.5 <<  2.539127892308944E+01 <<  1.560460030896854E+01 << -9.062735921448720E-01;
	neptune << 2415020.5 <<  1.514907601019179E+00 <<  2.982555775886938E+01 << -6.491145380252272E-01;
	neptune << 2424151.5 << -2.372071583101292E+01 <<  1.852243077001534E+01 <<  1.650257146741907E-01;
	neptune << 2433282.5 << -2.909262026598094E+01 << -8.409023962116420E+00 <<  8.433793022474723E-01;
	neptune << 2442413.5 << -1.042584631735569E+01 << -2.844092534344397E+01 <<  8.258166196925245E-01;
	neptune << 2451544.5 <<  1.681075678673750E+01 << -2.499265126481678E+01 <<  1.272710177158801E-01;
	neptune << 2460676.5 <<  2.987992286565916E+01 << -6.341865793833307E-01 << -6.754986290623808E-01;
	neptune << 2469807.5 <<  1.739818295823252E+01 <<  2.419778405033324E+01 << -8.992402953885371E-01;
	neptune << 2478938.5 << -1.006623937436067E+01 <<  2.821929813844349E+01 << -3.491140999929938E-01;
	neptune << 2488069.5 << -2.905923513186021E+01 <<  8.235126251736197E+00 <<  5.001936967708620E-01;
	neptune << 2497200.5 << -2.388593721034401E+01 << -1.868200419148737E+01 <<  9.352564668778985E-01;
	neptune << 2506331.5 <<  9.548600271911174E-01 << -3.023213775893789E+01 <<  6.005011945178902E-01;
	neptune << 2515462.5 <<  2.493645484654239E+01 << -1.672108417969024E+01 << -2.306061076557726E-01;
	neptune << 2524593.5 <<  2.777575055651235E+01 <<  1.085627104602106E+01 << -8.640247106973676E-01;
	neptune << 2533724.5 <<  6.757901798609053E+00 <<  2.906592923388614E+01 << -7.544725471165495E-01;
	neptune << 2542855.5 << -2.010560712167247E+01 <<  2.235333070038314E+01 <<  3.233920930058762E-03;
	neptune << 2551986.5 << -3.010413417130249E+01 << -3.248602854287615E+00 <<  7.611673160544495E-01;
	neptune << 2561117.5 << -1.521670876661520E+01 << -2.622114143039981E+01 <<  8.910309860600673E-01;
	neptune << 2570248.5 <<  1.217559922209293E+01 << -2.759493876343767E+01 <<  2.876100122115683E-01;
	neptune << 2579379.5 <<  2.931591880502902E+01 << -5.959061886294943E+00 << -5.534228549572565E-01;
	neptune << 2588510.5 <<  2.143496372997629E+01 <<  2.068023838284075E+01 << -9.204587759036501E-01;
	neptune << 2597641.5 << -4.908739664459474E+00 <<  2.950335823656594E+01 << -4.946434211770721E-01;
	neptune << 2606773.5 << -2.714654053598140E+01 <<  1.315100166295488E+01 <<  3.552020296709817E-01;
	neptune << 2615904.5 << -2.676110676284328E+01 << -1.425907099629386E+01 <<  9.110667012483453E-01;
	neptune << 2625035.5 << -4.347212484633157E+00 << -2.996284458836965E+01 <<  7.175879518693152E-01;
	neptune << 2634166.5 <<  2.160272297266483E+01 << -2.090683987960262E+01 << -6.756629920140046E-02;
	neptune << 2643297.5 <<  2.928524773091281E+01 <<  5.692890105146974E+00 << -7.928265245329557E-01;
	neptune << 2652428.5 <<  1.183461314456774E+01 <<  2.735324675669263E+01 << -8.365215425062809E-01;
	neptune << 2661559.5 << -1.583326485146195E+01 <<  2.549828522494925E+01 << -1.600501215414427E-01;
	neptune << 2670690.5 << -3.019399440765067E+01 <<  2.025533659377639E+00 <<  6.548570322183085E-01;
}

void TestEphemeris::testMercuryHeliocentricEphemerisVsop87()
{
	const int planet_id = 0; // Mercury
	const double acceptableError = 1E-04;
	double jd, x, y, z;
	double xyz[6];

	while(mercury.count() >= 3)
	{
		jd = mercury.takeFirst().toDouble();
		x  = mercury.takeFirst().toDouble();
		y  = mercury.takeFirst().toDouble();
		z  = mercury.takeFirst().toDouble();

		GetVsop87Coor(jd, planet_id, xyz);

		double actualErrorX = qAbs(qAbs(x) - qAbs(xyz[0]));
		double actualErrorY = qAbs(qAbs(y) - qAbs(xyz[1]));
		double actualErrorZ = qAbs(qAbs(z) - qAbs(xyz[2]));

		QVERIFY2(actualErrorX <= acceptableError && actualErrorY <= acceptableError && actualErrorZ <= acceptableError,
			 QString("jd=%1 x=%2 (%5) y=%3 (%6) z=%4 (%7)")
			 .arg(QString::number(    jd, 'f', 15))
			 .arg(QString::number(xyz[0], 'f', 15))
			 .arg(QString::number(xyz[1], 'f', 15))
			 .arg(QString::number(xyz[2], 'f', 15))
			 .arg(QString::number(     x, 'f', 15))
			 .arg(QString::number(     y, 'f', 15))
			 .arg(QString::number(     z, 'f', 15))
			 .toUtf8());
	}
}

void TestEphemeris::testVenusHeliocentricEphemerisVsop87()
{
	const int planet_id = 1; // Venus
	const double acceptableError = 1E-05;
	double jd, x, y, z;
	double xyz[6];

	while(venus.count() >= 3)
	{
		jd = venus.takeFirst().toDouble();
		x  = venus.takeFirst().toDouble();
		y  = venus.takeFirst().toDouble();
		z  = venus.takeFirst().toDouble();

		GetVsop87Coor(jd, planet_id, xyz);

		double actualErrorX = qAbs(qAbs(x) - qAbs(xyz[0]));
		double actualErrorY = qAbs(qAbs(y) - qAbs(xyz[1]));
		double actualErrorZ = qAbs(qAbs(z) - qAbs(xyz[2]));

		QVERIFY2(actualErrorX <= acceptableError && actualErrorY <= acceptableError && actualErrorZ <= acceptableError,
			 QString("jd=%1 x=%2 (%5) y=%3 (%6) z=%4 (%7)")
			 .arg(QString::number(    jd, 'f', 15))
			 .arg(QString::number(xyz[0], 'f', 15))
			 .arg(QString::number(xyz[1], 'f', 15))
			 .arg(QString::number(xyz[2], 'f', 15))
			 .arg(QString::number(     x, 'f', 15))
			 .arg(QString::number(     y, 'f', 15))
			 .arg(QString::number(     z, 'f', 15))
			 .toUtf8());
	}
}

void TestEphemeris::testMarsHeliocentricEphemerisVsop87()
{
	const int planet_id = 3; // Mars
	const double acceptableError = 1E-04;
	double jd, x, y, z;
	double xyz[6];

	while(mars.count() >= 3)
	{
		jd = mars.takeFirst().toDouble();
		x  = mars.takeFirst().toDouble();
		y  = mars.takeFirst().toDouble();
		z  = mars.takeFirst().toDouble();

		GetVsop87Coor(jd, planet_id, xyz);

		double actualErrorX = qAbs(qAbs(x) - qAbs(xyz[0]));
		double actualErrorY = qAbs(qAbs(y) - qAbs(xyz[1]));
		double actualErrorZ = qAbs(qAbs(z) - qAbs(xyz[2]));

		QVERIFY2(actualErrorX <= acceptableError && actualErrorY <= acceptableError && actualErrorZ <= acceptableError,
			 QString("jd=%1 x=%2 (%5) y=%3 (%6) z=%4 (%7)")
			 .arg(QString::number(    jd, 'f', 15))
			 .arg(QString::number(xyz[0], 'f', 15))
			 .arg(QString::number(xyz[1], 'f', 15))
			 .arg(QString::number(xyz[2], 'f', 15))
			 .arg(QString::number(     x, 'f', 15))
			 .arg(QString::number(     y, 'f', 15))
			 .arg(QString::number(     z, 'f', 15))
			 .toUtf8());
	}
}

void TestEphemeris::testJupiterHeliocentricEphemerisVsop87()
{
	const int planet_id = 4; // Jupiter
	const double acceptableError = 1E-04;
	double jd, x, y, z;
	double xyz[6];

	while(jupiter.count() >= 3)
	{
		jd = jupiter.takeFirst().toDouble();
		x  = jupiter.takeFirst().toDouble();
		y  = jupiter.takeFirst().toDouble();
		z  = jupiter.takeFirst().toDouble();

		GetVsop87Coor(jd, planet_id, xyz);

		double actualErrorX = qAbs(qAbs(x) - qAbs(xyz[0]));
		double actualErrorY = qAbs(qAbs(y) - qAbs(xyz[1]));
		double actualErrorZ = qAbs(qAbs(z) - qAbs(xyz[2]));

		QVERIFY2(actualErrorX <= acceptableError && actualErrorY <= acceptableError && actualErrorZ <= acceptableError,
			 QString("jd=%1 x=%2 (%5) y=%3 (%6) z=%4 (%7)")
			 .arg(QString::number(    jd, 'f', 15))
			 .arg(QString::number(xyz[0], 'f', 15))
			 .arg(QString::number(xyz[1], 'f', 15))
			 .arg(QString::number(xyz[2], 'f', 15))
			 .arg(QString::number(     x, 'f', 15))
			 .arg(QString::number(     y, 'f', 15))
			 .arg(QString::number(     z, 'f', 15))
			 .toUtf8());
	}
}

void TestEphemeris::testSaturnHeliocentricEphemerisVsop87()
{
	const int planet_id = 5; // Saturn
	const double acceptableError = 1E-04;
	double jd, x, y, z;
	double xyz[6];

	while(saturn.count() >= 3)
	{
		jd = saturn.takeFirst().toDouble();
		x  = saturn.takeFirst().toDouble();
		y  = saturn.takeFirst().toDouble();
		z  = saturn.takeFirst().toDouble();

		GetVsop87Coor(jd, planet_id, xyz);

		double actualErrorX = qAbs(qAbs(x) - qAbs(xyz[0]));
		double actualErrorY = qAbs(qAbs(y) - qAbs(xyz[1]));
		double actualErrorZ = qAbs(qAbs(z) - qAbs(xyz[2]));

		QVERIFY2(actualErrorX <= acceptableError && actualErrorY <= acceptableError && actualErrorZ <= acceptableError,
			 QString("jd=%1 x=%2 (%5) y=%3 (%6) z=%4 (%7)")
			 .arg(QString::number(    jd, 'f', 15))
			 .arg(QString::number(xyz[0], 'f', 15))
			 .arg(QString::number(xyz[1], 'f', 15))
			 .arg(QString::number(xyz[2], 'f', 15))
			 .arg(QString::number(     x, 'f', 15))
			 .arg(QString::number(     y, 'f', 15))
			 .arg(QString::number(     z, 'f', 15))
			 .toUtf8());
	}
}

void TestEphemeris::testUranusHeliocentricEphemerisVsop87()
{
	const int planet_id = 6; // Uranus
	const double acceptableError = 1E-03;
	double jd, x, y, z;
	double xyz[6];

	while(uranus.count() >= 3)
	{
		jd = uranus.takeFirst().toDouble();
		x  = uranus.takeFirst().toDouble();
		y  = uranus.takeFirst().toDouble();
		z  = uranus.takeFirst().toDouble();

		GetVsop87Coor(jd, planet_id, xyz);

		double actualErrorX = qAbs(qAbs(x) - qAbs(xyz[0]));
		double actualErrorY = qAbs(qAbs(y) - qAbs(xyz[1]));
		double actualErrorZ = qAbs(qAbs(z) - qAbs(xyz[2]));

		QVERIFY2(actualErrorX <= acceptableError && actualErrorY <= acceptableError && actualErrorZ <= acceptableError,
			 QString("jd=%1 x=%2 (%5) y=%3 (%6) z=%4 (%7)")
			 .arg(QString::number(    jd, 'f', 15))
			 .arg(QString::number(xyz[0], 'f', 15))
			 .arg(QString::number(xyz[1], 'f', 15))
			 .arg(QString::number(xyz[2], 'f', 15))
			 .arg(QString::number(     x, 'f', 15))
			 .arg(QString::number(     y, 'f', 15))
			 .arg(QString::number(     z, 'f', 15))
			 .toUtf8());
	}
}

void TestEphemeris::testNeptuneHeliocentricEphemerisVsop87()
{
	const int planet_id = 7; // Neptune
	const double acceptableError = 1E-03;
	double jd, x, y, z;
	double xyz[6];

	while(neptune.count() >= 3)
	{
		jd = neptune.takeFirst().toDouble();
		x  = neptune.takeFirst().toDouble();
		y  = neptune.takeFirst().toDouble();
		z  = neptune.takeFirst().toDouble();

		GetVsop87Coor(jd, planet_id, xyz);

		double actualErrorX = qAbs(qAbs(x) - qAbs(xyz[0]));
		double actualErrorY = qAbs(qAbs(y) - qAbs(xyz[1]));
		double actualErrorZ = qAbs(qAbs(z) - qAbs(xyz[2]));

		QVERIFY2(actualErrorX <= acceptableError && actualErrorY <= acceptableError && actualErrorZ <= acceptableError,
			 QString("jd=%1 x=%2 (%5) y=%3 (%6) z=%4 (%7)")
			 .arg(QString::number(    jd, 'f', 15))
			 .arg(QString::number(xyz[0], 'f', 15))
			 .arg(QString::number(xyz[1], 'f', 15))
			 .arg(QString::number(xyz[2], 'f', 15))
			 .arg(QString::number(     x, 'f', 15))
			 .arg(QString::number(     y, 'f', 15))
			 .arg(QString::number(     z, 'f', 15))
			 .toUtf8());
	}
}

void TestEphemeris::testMercuryHeliocentricEphemerisDe430()
{
	if (de430FilePath.isEmpty())
		qWarning() << "Ephemeris JPL DE430 unit test has been marked as 'passed' (He cannot be passed, because DE430 file ephemeris is not exists)!";
	else
	{
		const int planet_id = 0; // Mercury
		const double acceptableError = 1E-04;
		double jd, x, y, z;
		double xyz[6];

		while(mercury.count() >= 3)
		{
			jd = mercury.takeFirst().toDouble();
			x  = mercury.takeFirst().toDouble();
			y  = mercury.takeFirst().toDouble();
			z  = mercury.takeFirst().toDouble();

			GetDe430Coor(jd, planet_id, xyz, CENTRAL_BODY_ID);

			double actualErrorX = qAbs(qAbs(x) - qAbs(xyz[0]));
			double actualErrorY = qAbs(qAbs(y) - qAbs(xyz[1]));
			double actualErrorZ = qAbs(qAbs(z) - qAbs(xyz[2]));

			QVERIFY2(actualErrorX <= acceptableError && actualErrorY <= acceptableError && actualErrorZ <= acceptableError,
				 QString("jd=%1 x=%2 (%5) y=%3 (%6) z=%4 (%7)")
				 .arg(QString::number(    jd, 'f', 15))
				 .arg(QString::number(xyz[0], 'f', 15))
				 .arg(QString::number(xyz[1], 'f', 15))
				 .arg(QString::number(xyz[2], 'f', 15))
				 .arg(QString::number(     x, 'f', 15))
				 .arg(QString::number(     y, 'f', 15))
				 .arg(QString::number(     z, 'f', 15))
				 .toUtf8());
		}
	}
}

void TestEphemeris::testVenusHeliocentricEphemerisDe430()
{
	if (de430FilePath.isEmpty())
		qWarning() << "Ephemeris JPL DE430 unit test has been marked as 'passed' (He cannot be passed, because DE430 file ephemeris is not exists)!";
	else
	{
		const int planet_id = 1; // Venus
		const double acceptableError = 1E-05;
		double jd, x, y, z;
		double xyz[6];

		while(venus.count() >= 3)
		{
			jd = venus.takeFirst().toDouble();
			x  = venus.takeFirst().toDouble();
			y  = venus.takeFirst().toDouble();
			z  = venus.takeFirst().toDouble();

			GetDe430Coor(jd, planet_id, xyz, CENTRAL_BODY_ID);

			double actualErrorX = qAbs(qAbs(x) - qAbs(xyz[0]));
			double actualErrorY = qAbs(qAbs(y) - qAbs(xyz[1]));
			double actualErrorZ = qAbs(qAbs(z) - qAbs(xyz[2]));

			QVERIFY2(actualErrorX <= acceptableError && actualErrorY <= acceptableError && actualErrorZ <= acceptableError,
				 QString("jd=%1 x=%2 (%5) y=%3 (%6) z=%4 (%7)")
				 .arg(QString::number(    jd, 'f', 15))
				 .arg(QString::number(xyz[0], 'f', 15))
				 .arg(QString::number(xyz[1], 'f', 15))
				 .arg(QString::number(xyz[2], 'f', 15))
				 .arg(QString::number(     x, 'f', 15))
				 .arg(QString::number(     y, 'f', 15))
				 .arg(QString::number(     z, 'f', 15))
				 .toUtf8());
		}
	}
}

void TestEphemeris::testMarsHeliocentricEphemerisDe430()
{
	if (de430FilePath.isEmpty())
		qWarning() << "Ephemeris JPL DE430 unit test has been marked as 'passed' (He cannot be passed, because DE430 file ephemeris is not exists)!";
	else
	{
		const int planet_id = 3; // Mars
		const double acceptableError = 1E-04;
		double jd, x, y, z;
		double xyz[6];

		while(mars.count() >= 3)
		{
			jd = mars.takeFirst().toDouble();
			x  = mars.takeFirst().toDouble();
			y  = mars.takeFirst().toDouble();
			z  = mars.takeFirst().toDouble();

			GetDe430Coor(jd, planet_id, xyz, CENTRAL_BODY_ID);

			double actualErrorX = qAbs(qAbs(x) - qAbs(xyz[0]));
			double actualErrorY = qAbs(qAbs(y) - qAbs(xyz[1]));
			double actualErrorZ = qAbs(qAbs(z) - qAbs(xyz[2]));

			QVERIFY2(actualErrorX <= acceptableError && actualErrorY <= acceptableError && actualErrorZ <= acceptableError,
				 QString("jd=%1 x=%2 (%5) y=%3 (%6) z=%4 (%7)")
				 .arg(QString::number(    jd, 'f', 15))
				 .arg(QString::number(xyz[0], 'f', 15))
				 .arg(QString::number(xyz[1], 'f', 15))
				 .arg(QString::number(xyz[2], 'f', 15))
				 .arg(QString::number(     x, 'f', 15))
				 .arg(QString::number(     y, 'f', 15))
				 .arg(QString::number(     z, 'f', 15))
				 .toUtf8());
		}
	}
}

void TestEphemeris::testJupiterHeliocentricEphemerisDe430()
{
	if (de430FilePath.isEmpty())
		qWarning() << "Ephemeris JPL DE430 unit test has been marked as 'passed' (He cannot be passed, because DE430 file ephemeris is not exists)!";
	else
	{
		const int planet_id = 4; // Jupiter
		const double acceptableError = 1E-04;
		double jd, x, y, z;
		double xyz[6];

		while(jupiter.count() >= 3)
		{
			jd = jupiter.takeFirst().toDouble();
			x  = jupiter.takeFirst().toDouble();
			y  = jupiter.takeFirst().toDouble();
			z  = jupiter.takeFirst().toDouble();

			GetDe430Coor(jd, planet_id, xyz, CENTRAL_BODY_ID);

			double actualErrorX = qAbs(qAbs(x) - qAbs(xyz[0]));
			double actualErrorY = qAbs(qAbs(y) - qAbs(xyz[1]));
			double actualErrorZ = qAbs(qAbs(z) - qAbs(xyz[2]));

			QVERIFY2(actualErrorX <= acceptableError && actualErrorY <= acceptableError && actualErrorZ <= acceptableError,
				 QString("jd=%1 x=%2 (%5) y=%3 (%6) z=%4 (%7)")
				 .arg(QString::number(    jd, 'f', 15))
				 .arg(QString::number(xyz[0], 'f', 15))
				 .arg(QString::number(xyz[1], 'f', 15))
				 .arg(QString::number(xyz[2], 'f', 15))
				 .arg(QString::number(     x, 'f', 15))
				 .arg(QString::number(     y, 'f', 15))
				 .arg(QString::number(     z, 'f', 15))
				 .toUtf8());
		}
	}
}

void TestEphemeris::testSaturnHeliocentricEphemerisDe430()
{
	if (de430FilePath.isEmpty())
		qWarning() << "Ephemeris JPL DE430 unit test has been marked as 'passed' (He cannot be passed, because DE430 file ephemeris is not exists)!";
	else
	{
		const int planet_id = 5; // Saturn
		const double acceptableError = 1E-04;
		double jd, x, y, z;
		double xyz[6];

		while(saturn.count() >= 3)
		{
			jd = saturn.takeFirst().toDouble();
			x  = saturn.takeFirst().toDouble();
			y  = saturn.takeFirst().toDouble();
			z  = saturn.takeFirst().toDouble();

			GetDe430Coor(jd, planet_id, xyz, CENTRAL_BODY_ID);

			double actualErrorX = qAbs(qAbs(x) - qAbs(xyz[0]));
			double actualErrorY = qAbs(qAbs(y) - qAbs(xyz[1]));
			double actualErrorZ = qAbs(qAbs(z) - qAbs(xyz[2]));

			QVERIFY2(actualErrorX <= acceptableError && actualErrorY <= acceptableError && actualErrorZ <= acceptableError,
				 QString("jd=%1 x=%2 (%5) y=%3 (%6) z=%4 (%7)")
				 .arg(QString::number(    jd, 'f', 15))
				 .arg(QString::number(xyz[0], 'f', 15))
				 .arg(QString::number(xyz[1], 'f', 15))
				 .arg(QString::number(xyz[2], 'f', 15))
				 .arg(QString::number(     x, 'f', 15))
				 .arg(QString::number(     y, 'f', 15))
				 .arg(QString::number(     z, 'f', 15))
				 .toUtf8());
		}
	}
}

void TestEphemeris::testUranusHeliocentricEphemerisDe430()
{
	if (de430FilePath.isEmpty())
		qWarning() << "Ephemeris JPL DE430 unit test has been marked as 'passed' (He cannot be passed, because DE430 file ephemeris is not exists)!";
	else
	{
		const int planet_id = 6; // Uranus
		const double acceptableError = 1E-03;
		double jd, x, y, z;
		double xyz[6];

		while(uranus.count() >= 3)
		{
			jd = uranus.takeFirst().toDouble();
			x  = uranus.takeFirst().toDouble();
			y  = uranus.takeFirst().toDouble();
			z  = uranus.takeFirst().toDouble();

			GetDe430Coor(jd, planet_id, xyz, CENTRAL_BODY_ID);

			double actualErrorX = qAbs(qAbs(x) - qAbs(xyz[0]));
			double actualErrorY = qAbs(qAbs(y) - qAbs(xyz[1]));
			double actualErrorZ = qAbs(qAbs(z) - qAbs(xyz[2]));

			QVERIFY2(actualErrorX <= acceptableError && actualErrorY <= acceptableError && actualErrorZ <= acceptableError,
				 QString("jd=%1 x=%2 (%5) y=%3 (%6) z=%4 (%7)")
				 .arg(QString::number(    jd, 'f', 15))
				 .arg(QString::number(xyz[0], 'f', 15))
				 .arg(QString::number(xyz[1], 'f', 15))
				 .arg(QString::number(xyz[2], 'f', 15))
				 .arg(QString::number(     x, 'f', 15))
				 .arg(QString::number(     y, 'f', 15))
				 .arg(QString::number(     z, 'f', 15))
				 .toUtf8());
		}
	}
}

void TestEphemeris::testNeptuneHeliocentricEphemerisDe430()
{
	if (de430FilePath.isEmpty())
		qWarning() << "Ephemeris JPL DE430 unit test has been marked as 'passed' (He cannot be passed, because DE430 file ephemeris is not exists)!";
	else
	{
		const int planet_id = 7; // Neptune
		const double acceptableError = 1E-03;
		double jd, x, y, z;
		double xyz[6];

		while(neptune.count() >= 3)
		{
			jd = neptune.takeFirst().toDouble();
			x  = neptune.takeFirst().toDouble();
			y  = neptune.takeFirst().toDouble();
			z  = neptune.takeFirst().toDouble();

			GetDe430Coor(jd, planet_id, xyz, CENTRAL_BODY_ID);

			double actualErrorX = qAbs(qAbs(x) - qAbs(xyz[0]));
			double actualErrorY = qAbs(qAbs(y) - qAbs(xyz[1]));
			double actualErrorZ = qAbs(qAbs(z) - qAbs(xyz[2]));

			QVERIFY2(actualErrorX <= acceptableError && actualErrorY <= acceptableError && actualErrorZ <= acceptableError,
				 QString("jd=%1 x=%2 (%5) y=%3 (%6) z=%4 (%7)")
				 .arg(QString::number(    jd, 'f', 15))
				 .arg(QString::number(xyz[0], 'f', 15))
				 .arg(QString::number(xyz[1], 'f', 15))
				 .arg(QString::number(xyz[2], 'f', 15))
				 .arg(QString::number(     x, 'f', 15))
				 .arg(QString::number(     y, 'f', 15))
				 .arg(QString::number(     z, 'f', 15))
				 .toUtf8());
		}
	}
}

void TestEphemeris::testMercuryHeliocentricEphemerisDe431()
{
	if (de431FilePath.isEmpty())
		qWarning() << "Ephemeris JPL DE431 unit test has been marked as 'passed' (He cannot be passed, because DE431 file ephemeris is not exists)!";
	else
	{
		const int planet_id = 0; // Mercury
		const double acceptableError = 1E-04;
		double jd, x, y, z;
		double xyz[6];

		while(mercury.count() >= 3)
		{
			jd = mercury.takeFirst().toDouble();
			x  = mercury.takeFirst().toDouble();
			y  = mercury.takeFirst().toDouble();
			z  = mercury.takeFirst().toDouble();

			GetDe431Coor(jd, planet_id, xyz, CENTRAL_BODY_ID);

			double actualErrorX = qAbs(qAbs(x) - qAbs(xyz[0]));
			double actualErrorY = qAbs(qAbs(y) - qAbs(xyz[1]));
			double actualErrorZ = qAbs(qAbs(z) - qAbs(xyz[2]));

			QVERIFY2(actualErrorX <= acceptableError && actualErrorY <= acceptableError && actualErrorZ <= acceptableError,
				 QString("jd=%1 x=%2 (%5) y=%3 (%6) z=%4 (%7)")
				 .arg(QString::number(    jd, 'f', 15))
				 .arg(QString::number(xyz[0], 'f', 15))
				 .arg(QString::number(xyz[1], 'f', 15))
				 .arg(QString::number(xyz[2], 'f', 15))
				 .arg(QString::number(     x, 'f', 15))
				 .arg(QString::number(     y, 'f', 15))
				 .arg(QString::number(     z, 'f', 15))
				 .toUtf8());
		}
	}
}

void TestEphemeris::testVenusHeliocentricEphemerisDe431()
{
	if (de431FilePath.isEmpty())
		qWarning() << "Ephemeris JPL DE431 unit test has been marked as 'passed' (He cannot be passed, because DE431 file ephemeris is not exists)!";
	else
	{
		const int planet_id = 1; // Venus
		const double acceptableError = 1E-05;
		double jd, x, y, z;
		double xyz[6];

		while(venus.count() >= 3)
		{
			jd = venus.takeFirst().toDouble();
			x  = venus.takeFirst().toDouble();
			y  = venus.takeFirst().toDouble();
			z  = venus.takeFirst().toDouble();

			GetDe431Coor(jd, planet_id, xyz, CENTRAL_BODY_ID);

			double actualErrorX = qAbs(qAbs(x) - qAbs(xyz[0]));
			double actualErrorY = qAbs(qAbs(y) - qAbs(xyz[1]));
			double actualErrorZ = qAbs(qAbs(z) - qAbs(xyz[2]));

			QVERIFY2(actualErrorX <= acceptableError && actualErrorY <= acceptableError && actualErrorZ <= acceptableError,
				 QString("jd=%1 x=%2 (%5) y=%3 (%6) z=%4 (%7)")
				 .arg(QString::number(    jd, 'f', 15))
				 .arg(QString::number(xyz[0], 'f', 15))
				 .arg(QString::number(xyz[1], 'f', 15))
				 .arg(QString::number(xyz[2], 'f', 15))
				 .arg(QString::number(     x, 'f', 15))
				 .arg(QString::number(     y, 'f', 15))
				 .arg(QString::number(     z, 'f', 15))
				 .toUtf8());
		}
	}
}

void TestEphemeris::testMarsHeliocentricEphemerisDe431()
{
	if (de431FilePath.isEmpty())
		qWarning() << "Ephemeris JPL DE431 unit test has been marked as 'passed' (He cannot be passed, because DE431 file ephemeris is not exists)!";
	else
	{
		const int planet_id = 3; // Mars
		const double acceptableError = 1E-04;
		double jd, x, y, z;
		double xyz[6];

		while(mars.count() >= 3)
		{
			jd = mars.takeFirst().toDouble();
			x  = mars.takeFirst().toDouble();
			y  = mars.takeFirst().toDouble();
			z  = mars.takeFirst().toDouble();

			GetDe431Coor(jd, planet_id, xyz, CENTRAL_BODY_ID);

			double actualErrorX = qAbs(qAbs(x) - qAbs(xyz[0]));
			double actualErrorY = qAbs(qAbs(y) - qAbs(xyz[1]));
			double actualErrorZ = qAbs(qAbs(z) - qAbs(xyz[2]));

			QVERIFY2(actualErrorX <= acceptableError && actualErrorY <= acceptableError && actualErrorZ <= acceptableError,
				 QString("jd=%1 x=%2 (%5) y=%3 (%6) z=%4 (%7)")
				 .arg(QString::number(    jd, 'f', 15))
				 .arg(QString::number(xyz[0], 'f', 15))
				 .arg(QString::number(xyz[1], 'f', 15))
				 .arg(QString::number(xyz[2], 'f', 15))
				 .arg(QString::number(     x, 'f', 15))
				 .arg(QString::number(     y, 'f', 15))
				 .arg(QString::number(     z, 'f', 15))
				 .toUtf8());
		}
	}
}

void TestEphemeris::testJupiterHeliocentricEphemerisDe431()
{
	if (de431FilePath.isEmpty())
		qWarning() << "Ephemeris JPL DE431 unit test has been marked as 'passed' (He cannot be passed, because DE431 file ephemeris is not exists)!";
	else
	{
		const int planet_id = 4; // Jupiter
		const double acceptableError = 1E-04;
		double jd, x, y, z;
		double xyz[6];

		while(jupiter.count() >= 3)
		{
			jd = jupiter.takeFirst().toDouble();
			x  = jupiter.takeFirst().toDouble();
			y  = jupiter.takeFirst().toDouble();
			z  = jupiter.takeFirst().toDouble();

			GetDe431Coor(jd, planet_id, xyz, CENTRAL_BODY_ID);

			double actualErrorX = qAbs(qAbs(x) - qAbs(xyz[0]));
			double actualErrorY = qAbs(qAbs(y) - qAbs(xyz[1]));
			double actualErrorZ = qAbs(qAbs(z) - qAbs(xyz[2]));

			QVERIFY2(actualErrorX <= acceptableError && actualErrorY <= acceptableError && actualErrorZ <= acceptableError,
				 QString("jd=%1 x=%2 (%5) y=%3 (%6) z=%4 (%7)")
				 .arg(QString::number(    jd, 'f', 15))
				 .arg(QString::number(xyz[0], 'f', 15))
				 .arg(QString::number(xyz[1], 'f', 15))
				 .arg(QString::number(xyz[2], 'f', 15))
				 .arg(QString::number(     x, 'f', 15))
				 .arg(QString::number(     y, 'f', 15))
				 .arg(QString::number(     z, 'f', 15))
				 .toUtf8());
		}
	}
}

void TestEphemeris::testSaturnHeliocentricEphemerisDe431()
{
	if (de431FilePath.isEmpty())
		qWarning() << "Ephemeris JPL DE431 unit test has been marked as 'passed' (He cannot be passed, because DE431 file ephemeris is not exists)!";
	else
	{
		const int planet_id = 5; // Saturn
		const double acceptableError = 1E-04;
		double jd, x, y, z;
		double xyz[6];

		while(saturn.count() >= 3)
		{
			jd = saturn.takeFirst().toDouble();
			x  = saturn.takeFirst().toDouble();
			y  = saturn.takeFirst().toDouble();
			z  = saturn.takeFirst().toDouble();

			GetDe431Coor(jd, planet_id, xyz, CENTRAL_BODY_ID);

			double actualErrorX = qAbs(qAbs(x) - qAbs(xyz[0]));
			double actualErrorY = qAbs(qAbs(y) - qAbs(xyz[1]));
			double actualErrorZ = qAbs(qAbs(z) - qAbs(xyz[2]));

			QVERIFY2(actualErrorX <= acceptableError && actualErrorY <= acceptableError && actualErrorZ <= acceptableError,
				 QString("jd=%1 x=%2 (%5) y=%3 (%6) z=%4 (%7)")
				 .arg(QString::number(    jd, 'f', 15))
				 .arg(QString::number(xyz[0], 'f', 15))
				 .arg(QString::number(xyz[1], 'f', 15))
				 .arg(QString::number(xyz[2], 'f', 15))
				 .arg(QString::number(     x, 'f', 15))
				 .arg(QString::number(     y, 'f', 15))
				 .arg(QString::number(     z, 'f', 15))
				 .toUtf8());
		}
	}
}

void TestEphemeris::testUranusHeliocentricEphemerisDe431()
{
	if (de431FilePath.isEmpty())
		qWarning() << "Ephemeris JPL DE431 unit test has been marked as 'passed' (He cannot be passed, because DE431 file ephemeris is not exists)!";
	else
	{
		const int planet_id = 6; // Uranus
		const double acceptableError = 1E-03;
		double jd, x, y, z;
		double xyz[6];

		while(uranus.count() >= 3)
		{
			jd = uranus.takeFirst().toDouble();
			x  = uranus.takeFirst().toDouble();
			y  = uranus.takeFirst().toDouble();
			z  = uranus.takeFirst().toDouble();

			GetDe431Coor(jd, planet_id, xyz, CENTRAL_BODY_ID);

			double actualErrorX = qAbs(qAbs(x) - qAbs(xyz[0]));
			double actualErrorY = qAbs(qAbs(y) - qAbs(xyz[1]));
			double actualErrorZ = qAbs(qAbs(z) - qAbs(xyz[2]));

			QVERIFY2(actualErrorX <= acceptableError && actualErrorY <= acceptableError && actualErrorZ <= acceptableError,
				 QString("jd=%1 x=%2 (%5) y=%3 (%6) z=%4 (%7)")
				 .arg(QString::number(    jd, 'f', 15))
				 .arg(QString::number(xyz[0], 'f', 15))
				 .arg(QString::number(xyz[1], 'f', 15))
				 .arg(QString::number(xyz[2], 'f', 15))
				 .arg(QString::number(     x, 'f', 15))
				 .arg(QString::number(     y, 'f', 15))
				 .arg(QString::number(     z, 'f', 15))
				 .toUtf8());
		}
	}
}

void TestEphemeris::testNeptuneHeliocentricEphemerisDe431()
{
	if (de431FilePath.isEmpty())
		qWarning() << "Ephemeris JPL DE431 unit test has been marked as 'passed' (He cannot be passed, because DE431 file ephemeris is not exists)!";
	else
	{
		const int planet_id = 7; // Neptune
		const double acceptableError = 1E-03;
		double jd, x, y, z;
		double xyz[6];

		while(neptune.count() >= 3)
		{
			jd = neptune.takeFirst().toDouble();
			x  = neptune.takeFirst().toDouble();
			y  = neptune.takeFirst().toDouble();
			z  = neptune.takeFirst().toDouble();

			GetDe431Coor(jd, planet_id, xyz, CENTRAL_BODY_ID);

			double actualErrorX = qAbs(qAbs(x) - qAbs(xyz[0]));
			double actualErrorY = qAbs(qAbs(y) - qAbs(xyz[1]));
			double actualErrorZ = qAbs(qAbs(z) - qAbs(xyz[2]));

			QVERIFY2(actualErrorX <= acceptableError && actualErrorY <= acceptableError && actualErrorZ <= acceptableError,
				 QString("jd=%1 x=%2 (%5) y=%3 (%6) z=%4 (%7)")
				 .arg(QString::number(    jd, 'f', 15))
				 .arg(QString::number(xyz[0], 'f', 15))
				 .arg(QString::number(xyz[1], 'f', 15))
				 .arg(QString::number(xyz[2], 'f', 15))
				 .arg(QString::number(     x, 'f', 15))
				 .arg(QString::number(     y, 'f', 15))
				 .arg(QString::number(     z, 'f', 15))
				 .toUtf8());
		}
	}
}

