/*
    libzint - the open source barcode library
    Copyright (C) 2019-2022 Robin Stuart <rstuart114@gmail.com>

    Redistribution and use in source and binary forms, with or without
    modification, are permitted provided that the following conditions
    are met:

    1. Redistributions of source code must retain the above copyright
       notice, this list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright
       notice, this list of conditions and the following disclaimer in the
       documentation and/or other materials provided with the distribution.
    3. Neither the name of the project nor the names of its contributors
       may be used to endorse or promote products derived from this software
       without specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
    ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
    IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
    ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
    FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
    DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
    OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
    HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
    LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
    OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
    SUCH DAMAGE.
 */
/* SPDX-License-Identifier: BSD-3-Clause */

#include "testcommon.h"

static void test_binary_div_modulo_divisor(const testCtx *const p_ctx) {
    int debug = p_ctx->debug;

    struct item {
        int symbology;
        char *data;
        float w;
        float h;

        int expected_rows;
        int expected_width;
        char *expected;
    };
    struct item data[] = {
        /*  0*/ { BARCODE_DBAR_OMN, "1234567890123", 100, 30, 1, 96, "010111010010000001001110000000010100001011111010110100011001100101111111110001011011000111000101" },
        /*  1*/ { BARCODE_DBAR_OMN, "0000004537076", 100, 30, 1, 96, "010101001000000001000100000000010111111100101010111101111101010101111111110111010100100000000101" },
        /*  2*/ { BARCODE_DBAR_OMN, "0000004537077", 100, 30, 1, 96, "010101001000000001000111000000010111111101001010101010110000000101111100000111011111111011010101" },
        /*  3*/ { BARCODE_DBAR_OMN, "0000004537078", 100, 30, 1, 96, "010101001000000001000111000000010111111101001010101011010000000101111000000011011111111011010101" },
        /*  4*/ { BARCODE_DBAR_OMN, "0000000001596", 100, 30, 1, 96, "010101001000000001001111100000010111111100101010111101111101010101111100000111011111111011010101" },
        /*  5*/ { BARCODE_DBAR_OMN, "0000000001597", 100, 30, 1, 96, "010101001000000001011111000000010111111100101010101010110000000101111100000111011111110111010101" },
        /*  6*/ { BARCODE_DBAR_OMN, "0000000001598", 100, 30, 1, 96, "010101001000000001011111000000010111111100101010101011010000000101111000000011011111110111010101" },
        /*  7*/ { BARCODE_DBAR_OMN, "0000000000000", 100, 30, 1, 96, "010101001000000001000111111110010111111100101010101010110000000101111111110111011111111011010101" },
        /*  8*/ { BARCODE_DBAR_OMN, "0000000257117", 100, 30, 1, 96, "010101001000000001001111100000010111111100101010101010110000000101111110000011000111111011010101" },
        /*  9*/ { BARCODE_DBAR_OMN, "0000000536592", 100, 30, 1, 96, "010101001000000001000111000000010111111100101010101010110000000101100000000111011101100011110101" },
        /* 10*/ { BARCODE_DBAR_OMN, "0000001644132", 100, 30, 1, 96, "010101001000000001001111100000010111111100101010100100001111010101111100000111000001101111010101" },
        /* 11*/ { BARCODE_DBAR_OMN, "0000002421052", 100, 30, 1, 96, "010101001000000001011111000000010111111100101010101010110000000101111100000111011100010001101101" },
        /* 12*/ { BARCODE_DBAR_OMN, "0000003217955", 100, 30, 1, 96, "010101001000000001000111000000010111111100101010101010110000000101111111000001000000111001010101" },
        /* 13*/ { BARCODE_DBAR_OMN, "0000004792584", 100, 30, 1, 96, "010101001000000001000111110000010111111101001010111011111101010101111111110111010110101111111101" },
        /* 14*/ { BARCODE_DBAR_OMN, "0000006062184", 100, 30, 1, 96, "010101001000000001000111111110010111111101001010110111111101010101111111110111001010011011111101" },
        /* 15*/ { BARCODE_DBAR_OMN, "0000007734882", 100, 30, 1, 96, "010101001000000001000111111110010111111101001010110111000010010101111111000111001010001100111101" },
        /* 16*/ { BARCODE_DBAR_OMN, "0000008845782", 100, 30, 1, 96, "010101001000000001011111000000010111111101001010111101111101010101111111000001000010101000011101" },
        /* 17*/ { BARCODE_DBAR_OMN, "0000009329661", 100, 30, 1, 96, "010101001000000001001111111000010111111101010010111011111101010101111000000011010110101111111101" },
        /* 18*/ { BARCODE_DBAR_OMN, "0000009607386", 100, 30, 1, 96, "010101001000000001011111000000010111111101010010100111001011110101111111110111011100011011110101" },
        /* 19*/ { BARCODE_DBAR_OMN, "0000010718286", 100, 30, 1, 96, "010101001000000001000111110000010111111101010010100100001111010101111100000111000001101111010101" },
        /* 20*/ { BARCODE_DBAR_OMN, "0000011607006", 100, 30, 1, 96, "010101001000000001000111111110010111111101010010101101000100000101111000000011010011100001101101" },
        /* 21*/ { BARCODE_DBAR_OMN, "0004360130997", 100, 30, 1, 96, "010101001000000001000100000000010110010010011110101010110000000101111000000011011111111011010101" },
        /* 22*/ { BARCODE_DBAR_OMN, "0004360386504", 100, 30, 1, 96, "010101001000000001000111000000010110010010011110111011111101010101111111100011010110101111111101" },
        /* 23*/ { BARCODE_DBAR_OMN, "0009142746747", 100, 30, 1, 96, "010101000100000001000111111110010100111110001010101010110000000101111111000111011101100011110101" },
        /* 24*/ { BARCODE_DBAR_OMN, "0012319818547", 100, 30, 1, 96, "010101000100000001000100000000010110001100101000101010110000000101111111000111001100011110010101" },
        /* 25*/ { BARCODE_DBAR_OMN, "0013775011335", 100, 30, 1, 96, "010101000100000001000100000000010101001100000110101010110000000101111100000111011111010011100101" },
        /* 26*/ { BARCODE_DBAR_OMN, "0018894538190", 100, 30, 1, 96, "010101000010000001000100000000010110100100111100101010110000000101111111110001010001100011110101" },
        /* 27*/ { BARCODE_DBAR_OMN, "0021059247735", 100, 30, 1, 96, "010101000010000001000100000000010101000011001100101010110000000101111111100011010110111000001101" },
        /* 28*/ { BARCODE_DBAR_OMN, "0024094346235", 100, 30, 1, 96, "010101000001000001000100000000010111010000111010101010110000000101100000000111000110011001101101" },
        /* 29*/ { BARCODE_DBAR_OMN, "1995000595035", 100, 30, 1, 96, "010100011011000001001110000000010111010111010000101010110000000101111000000011000110011001101101" },
        /* 30*/ { BARCODE_DBAR_OMN, "9999999999999", 100, 30, 1, 96, "010010111011100001000111111110010111101101001110100011111101010101111111000001000111110101011101" },
        /* 31*/ { BARCODE_DBAR_OMN_CC, "0000000000000", 100, 30, 5, 100, "0000010010111011100001001110000000010111101101001110110001010111110101111111110001000111110101011101" },
        /* 32*/ { BARCODE_DBAR_OMN_CC, "0000729476171", 100, 30, 5, 100, "0000010010111011100001011100000000010111100110100010101010110000000101111111100011011111111011010101" },
        /* 33*/ { BARCODE_DBAR_OMN_CC, "0004359674363", 100, 30, 5, 100, "0000010010111011100001011100000000010101110010000100101010110000000101111000000011011101100011110101" },
        /* 34*/ { BARCODE_DBAR_OMN_CC, "0009142871421", 100, 30, 5, 100, "0000010010111101000011000111111110010100010111110010101010110000000101111111000111001100011110010101" },
        /* 35*/ { BARCODE_DBAR_OMN_CC, "0012319591881", 100, 30, 5, 100, "0000010010111101000011000100000000010101000111010000101010110000000101111111000111011100010001101101" },
        /* 36*/ { BARCODE_DBAR_OMN_CC, "6975446373038", 100, 30, 5, 100, "0000010111001110010111001111111000010101000111010000101010110000000101111100000111011111111011010101" },
        /* 37*/ { BARCODE_DBAR_OMN_CC, "9999999999999", 100, 30, 5, 100, "0000010110101111011111001111111000010110111100100010100110111001110101111111110001001100110001110101" },
        /* 38*/ { BARCODE_DBAR_LTD, "1234567890123", 100, 30, 1, 79, "0100110011110010100010011101011010101100100101001010100101000001110001110100000" },
        /* 39*/ { BARCODE_DBAR_LTD, "0000002013570", 100, 30, 1, 79, "0101010101010000001000000111010010110101100101011111111011111101010101010100000" },
        /* 40*/ { BARCODE_DBAR_LTD, "0000002013571", 100, 30, 1, 79, "0101010101010000001100000011010101101010001101010101010100000010000001110100000" },
        /* 41*/ { BARCODE_DBAR_LTD, "0000002013572", 100, 30, 1, 79, "0101010101010000001100000011010101001011100101010101010100000011000000110100000" },
        /* 42*/ { BARCODE_DBAR_LTD, "0000000917879", 100, 30, 1, 79, "0101010101010000001000000111010011010101001101010011011101111101011001110100000" },
        /* 43*/ { BARCODE_DBAR_LTD, "0000000000000", 100, 30, 1, 79, "0101010101010000001000000111010111010100100101010101010100000010000001110100000" },
        /* 44*/ { BARCODE_DBAR_LTD, "0000000183063", 100, 30, 1, 79, "0101010101010000001000000111011010101001001101000000111000000101010101010100000" },
        /* 45*/ { BARCODE_DBAR_LTD, "0000000183064", 100, 30, 1, 79, "0101010101010000001000000111001010101011100101010101010100011110000011110100000" },
        /* 46*/ { BARCODE_DBAR_LTD, "0000000820063", 100, 30, 1, 79, "0101010101010000001000000111001010101001101101000001111000111101010101010100000" },
        /* 47*/ { BARCODE_DBAR_LTD, "0000000820064", 100, 30, 1, 79, "0101010101010000001000000111011010110100100101010101010101111110001111110100000" },
        /* 48*/ { BARCODE_DBAR_LTD, "0000001000775", 100, 30, 1, 79, "0101010101010000001000000111001011010101100101000111111011111101010101010100000" },
        /* 49*/ { BARCODE_DBAR_LTD, "0000001000776", 100, 30, 1, 79, "0101010101010000001000000111010111010101000101010101010100000110000011110100000" },
        /* 50*/ { BARCODE_DBAR_LTD, "0000001491020", 100, 30, 1, 79, "0101010101010000001000000111001010101011001101000001111000001101010101010100000" },
        /* 51*/ { BARCODE_DBAR_LTD, "0000001491021", 100, 30, 1, 79, "0101010101010000001000000111010100101011001101010101010100111110000111110100000" },
        /* 52*/ { BARCODE_DBAR_LTD, "0000001979844", 100, 30, 1, 79, "0101010101010000001000000111010101010001101101000011111001111101010101010100000" },
        /* 53*/ { BARCODE_DBAR_LTD, "0000001979845", 100, 30, 1, 79, "0101010101010000001000000111010111010010100101010101010100000010000000010100000" },
        /* 54*/ { BARCODE_DBAR_LTD, "0000001996938", 100, 30, 1, 79, "0101010101010000001000000111010110101000101101000000001000000101010101010100000" },
        /* 55*/ { BARCODE_DBAR_LTD, "0000001996939", 100, 30, 1, 79, "0101010101010000001000000111011011010101000101010101010101111110111111110100000" },
        /* 56*/ { BARCODE_DBAR_LTD, "0000012013571", 100, 30, 1, 79, "0101010101011100000010000001010101001010011101000111011110010100110110110100000" },
        /* 57*/ { BARCODE_DBAR_LTD, "0368610347973", 100, 30, 1, 79, "0100000011100000010101010101001010101110100101010101010100000010000001110100000" },
        /* 58*/ { BARCODE_DBAR_LTD, "0368612361544", 100, 30, 1, 79, "0101010101010001111000001111010100110101001101010101010100000010000001110100000" },
        /* 59*/ { BARCODE_DBAR_LTD, "1651255079973", 100, 30, 1, 79, "0100000111100011110101010101010101101100100101010110010010001001100000010100000" },
        /* 60*/ { BARCODE_DBAR_LTD, "1651257088544", 100, 30, 1, 79, "0101010101010111111000111111010110100110100101010101010100000010000001110100000" },
        /* 61*/ { BARCODE_DBAR_LTD, "1999999999999", 100, 30, 1, 79, "0100111100110110101101111101010101101011000101010000101110001101011110010100000" },
        /* 62*/ { BARCODE_DBAR_LTD_CC, "0000000000000", 100, 30, 6, 79, "0101010101010000011000001111010010101101001101010101010100000010000001110100000" },
        /* 63*/ { BARCODE_DBAR_LTD_CC, "0000002013571", 100, 30, 6, 79, "0101010101010000011100000111010110101000101101010101010100000010000001110100000" },
        /* 64*/ { BARCODE_DBAR_LTD_CC, "0987141101324", 100, 30, 6, 79, "0100000111100000110101010101011010101010001101010101010100000010000001110100000" },
        /* 65*/ { BARCODE_DBAR_LTD_CC, "0987143114895", 100, 30, 6, 79, "0101010101010011111000011111010111010101000101010101010100000010000001110100000" },
        /* 66*/ { BARCODE_DBAR_LTD_CC, "1971422931828", 100, 30, 6, 79, "0100001111100111110101010101010010101010011101010101010100000010000001110100000" },
        /* 67*/ { BARCODE_DBAR_LTD_CC, "1971424945399", 100, 30, 6, 79, "0101010101010000001000000001011010101001001101010101010100000010000001110100000" },
        /* 68*/ { BARCODE_DBAR_LTD_CC, "1999999999999", 100, 30, 6, 79, "0100001000000101010100000101011010110100100101010000101110001101011110010100000" },
        /* 69*/ { BARCODE_DBAR_LTD, "1651257071912", 100, 30, 1, 79, "0100000111100011110101010101010111010100100101010101010101111110111111110100000" },
        /* 70*/ { BARCODE_DBAR_LTD_CC, "0987144605916", 100, 30, 6, 79, "0101010101010011111000011111011010110100100101010101010100111110000111110100000" },
    };
    int data_size = ARRAY_SIZE(data);
    int i, length, ret;
    struct zint_symbol *symbol;

    char *text;

    char escaped[1024];
    char cmp_buf[1024];
    char cmp_msg[1024];

    int do_bwipp = (debug & ZINT_DEBUG_TEST_BWIPP) && testUtilHaveGhostscript(); /* Only do BWIPP test if asked, too slow otherwise */
    int do_zxingcpp = (debug & ZINT_DEBUG_TEST_ZXINGCPP) && testUtilHaveZXingCPPDecoder(); /* Only do ZXing-C++ test if asked, too slow otherwise */

    testStart("test_binary_div_modulo_divisor");

    for (i = 0; i < data_size; i++) {

        if (testContinue(p_ctx, i)) continue;

        symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        if (data[i].symbology == BARCODE_DBAR_OMN_CC || data[i].symbology == BARCODE_DBAR_LTD_CC) {
            text = "[20]01";
            strcpy(symbol->primary, data[i].data);
        } else {
            text = data[i].data;
        }
        length = testUtilSetSymbol(symbol, data[i].symbology, -1 /*input_mode*/, -1 /*eci*/, -1 /*option_1*/, -1, -1, -1 /*output_options*/, text, -1, debug);

        ret = ZBarcode_Encode(symbol, (const unsigned char *) text, length);
        assert_zero(ret, "i:%d ZBarcode_Encode ret %d != 0 (%s)\n", i, ret, symbol->errtxt);

        if (p_ctx->generate) {
            printf("        /*%3d*/ { %s, \"%s\", %.0f, %.0f, %d, %d, ",
                    i, testUtilBarcodeName(data[i].symbology), data[i].data, data[i].w, data[i].h, symbol->rows, symbol->width);
            testUtilModulesPrintRow(symbol, symbol->rows - 1, "", " },\n");
        } else {
            int width;

            assert_equal(symbol->rows, data[i].expected_rows, "i:%d symbol->rows %d != %d\n", i, symbol->rows, data[i].expected_rows);
            assert_equal(symbol->width, data[i].expected_width, "i:%d symbol->width %d != %d\n", i, symbol->width, data[i].expected_width);

            ret = testUtilModulesCmpRow(symbol, symbol->rows - 1, data[i].expected, &width);
            assert_zero(ret, "i:%d testUtilModulesCmpRow ret %d != 0 width %d row %d\n", i, ret, width, symbol->rows - 1);

            ret = ZBarcode_Buffer_Vector(symbol, 0);
            assert_zero(ret, "i:%d ZBarcode_Buffer_Vector ret %d != 0\n", i, ret);

            if (do_bwipp && testUtilCanBwipp(i, symbol, -1, -1, -1, debug)) {
                ret = testUtilBwipp(i, symbol, -1, -1, -1, text, length, symbol->primary, cmp_buf, sizeof(cmp_buf), NULL);
                assert_zero(ret, "i:%d %s testUtilBwipp ret %d != 0\n", i, testUtilBarcodeName(symbol->symbology), ret);

                ret = testUtilBwippCmpRow(symbol, symbol->rows - 1, cmp_msg, cmp_buf, data[i].expected);
                assert_zero(ret, "i:%d %s testUtilBwippCmp %d != 0 %s\n  actual: %s\nexpected: %s\n",
                               i, testUtilBarcodeName(symbol->symbology), ret, cmp_msg, cmp_buf, data[i].expected);
            }
            if (do_zxingcpp && testUtilCanZXingCPP(i, symbol, data[i].data, length, debug)) {
                int cmp_len, ret_len;
                char modules_dump[8192 + 1];
                assert_notequal(testUtilModulesDump(symbol, modules_dump, sizeof(modules_dump)), -1, "i:%d testUtilModulesDump == -1\n", i);
                ret = testUtilZXingCPP(i, symbol, data[i].data, length, modules_dump, cmp_buf, sizeof(cmp_buf), &cmp_len);
                assert_zero(ret, "i:%d %s testUtilZXingCPP ret %d != 0\n", i, testUtilBarcodeName(symbol->symbology), ret);

                ret = testUtilZXingCPPCmp(symbol, cmp_msg, cmp_buf, cmp_len, data[i].data, length, NULL /*primary*/, escaped, &ret_len);
                assert_zero(ret, "i:%d %s testUtilZXingCPPCmp %d != 0 %s\n  actual: %.*s\nexpected: %.*s\n",
                               i, testUtilBarcodeName(symbol->symbology), ret, cmp_msg, cmp_len, cmp_buf, ret_len, escaped);
            }
        }

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

/* Replicate examples from GS1 General Specifications 21.0.1 and ISO/IEC 24724:2011 */
static void test_examples(const testCtx *const p_ctx) {
    int debug = p_ctx->debug;

    struct item {
        int symbology;
        int input_mode;
        int option_2;
        int option_3;
        char *data;

        int ret;
        int expected_rows;
        int expected_width;
        int bwipp_cmp;
        char *comment;
        char *expected;
    };
    /* Verified manually against GS1 General Specifications 21.0.1 (GGS) and ISO/IEC 24724:2011, and verified via bwipp_dump.ps against BWIPP */
    struct item data[] = {
        /*  0*/ { BARCODE_DBAR_OMN, -1, -1, -1, "0950110153001", 0, 1, 96, 1, "GGS Figure 5.5.2.1.1-1. GS1 DataBar Omnidirectional",
                     "010000010100000101000111110000010111101101011100100011011101000101100000000111001110110111001101"
                },
        /*  1*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]90614141000015[3202]000150", 0, 1, 151, 1, "GGS Figure 5.5.2.3.1-1. GS1 DataBar Expanded",
                     "0101100011001100001011111111000010100100010000111101110011100010100010111100000011100111010111111011010100000100000110001111110000101000000100011010010"
                },
        /*  2*/ { BARCODE_DBAR_EXPSTK, -1, -1, -1, "[01]90614141000015[3202]000150", 0, 5, 102, 1, "GGS Figure 5.5.2.3.2-1. GS1 DataBar Expanded Stacked, same (tec-it separator differs)",
                     "010110001100110000101111111100001010010001000011110111001110001010001011110000001110011101011111101101"
                     "000001110011001111010000000010100101101110111100001000110001110101110100001010100001100010100000010000"
                     "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "000001011111011111001010000001010010111111011100100000000000000000000000000000000000000000000000000000"
                     "001010100000100000110001111110000101000000100011010010000000000000000000000000000000000000000000000000"
                },
        /*  3*/ { BARCODE_DBAR_OMN, -1, -1, -1, "2001234567890", 0, 1, 96, 1, "24724:2011 Figure 1 — GS1 DataBar Omnidirectional",
                     "010100011101000001001111111000010100110110111110110000010010100101100000000111000110110110001101"
                },
        /*  4*/ { BARCODE_DBAR_OMN, -1, -1, -1, "0441234567890", 0, 1, 96, 1, "24724:2011 Figure 2 — GS1 DataBar Omnidirectional",
                     "010010001000010001000111000000010101000001100110101100100100000101111110000011000010100011100101"
                },
        /*  5*/ { BARCODE_DBAR_OMN, -1, -1, -1, "0001234567890", 0, 1, 96, 1, "24724:2011 Figure 4 — GS1 DataBar Truncated",
                     "010101001000000001001111111000010111001011011110111001010110000101111111000111001100111101110101"
                },
        /*  6*/ { BARCODE_DBAR_STK, -1, -1, -1, "0001234567890", 0, 3, 50, 1, "24724:2011 Figure 5 — GS1 DataBar Stacked NOTE: Figure 5 separator differs from GGS Figure 5.5.2.1.3-1. which has ends set",
                     "01010100100000000100111111100001011100101101111010"
                     "00001010101011111010000000111010100011010010000000"
                     "10111001010110000101111111000111001100111101110101"
                },
        /*  7*/ { BARCODE_DBAR_OMNSTK, -1, -1, -1, "0003456789012", 0, 5, 50, 1, "24724:2011 Figure 6 — GS1 DataBar Stacked Omnidirectional",
                     "01010100100000000100111110000001010011100110011010"
                     "00001011011111111010000001010100101100011001100000"
                     "00000101010101010101010101010101010101010101010000"
                     "00001000100010111010010101010000111101001101110000"
                     "10110111011101000101100000000111000010110010001101"
                },
        /*  8*/ { BARCODE_DBAR_LTD, -1, -1, -1, "1501234567890", 0, 1, 79, 1, "24724:2011 Figure 7 — GS1 DataBar Limited",
                     "0100011001100011011010100111010010101101001101001001011000110111001100110100000"
                },
        /*  9*/ { BARCODE_DBAR_LTD, -1, -1, -1, "0031234567890", 0, 1, 79, 1, "24724:2011 Figure 8 — (a) GS1 DataBar Limited",
                     "0101010000010010001000010111001010110110100101011000001010010010110000010100000"
                },
        /* 10*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]98898765432106[3202]012345[15]991231", 0, 1, 200, 1, "24724:2011 Figure 10 — GS1 DataBar Expanded",
                     "01001000011000110110111111110000101110000110010100011010000001100010101111110000111010011100000010010100111110111001100011111100001011101100000100100100011110010110001011111111001110001101111010000101"
                },
        /* 11*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]90012345678908[3103]001750", 0, 1, 151, 1, "24724:2011 Figure 11 — GS1 DataBar Expanded",
                     "0101110010000010011011111111000010111000010011000101011110111001100010111100000011100101110001110111011110101111000110001111110000101011000010011111010"
                },
        /* 12*/ { BARCODE_DBAR_EXPSTK, -1, -1, -1, "[01]98898765432106[3202]012345[15]991231", 0, 5, 102, 1, "24724:2011 Figure 12 — GS1 DataBar Expanded Stacked symbol",
                     "010010000110001101101111111100001011100001100101000110100000011000101011111100001110100111000000100101"
                     "000001111001110010010000000010100100011110011010111001011111100111010100000010100001011000111111010000"
                     "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "000011101000010011100001000000001011100101100001110110110111110010001001010000001010011000100000110000"
                     "101000010111101100011100111111110100011010011110001001001000001101110100001111110001100111011111001010"
                },
        /* 13*/ { BARCODE_DBAR_EXPSTK, -1, -1, -1, "[01]95012345678903[3103]000123", 0, 5, 102, 1, "24724:2011 Figure 13 — GS1 DataBar Expanded Stacked",
                     "010100010001111000101111111100001010111000001100010111000110001001101011110000001110010111000111011101"
                     "000011101110000111010000000010100101000111110011101000111001110110010100001010100001101000111000100000"
                     "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "000000001010000111001010000001010010111011011111100000000000000000000000000000000000000000000000000000"
                     "001011110101111000110001111110000101000100100000011010000000000000000000000000000000000000000000000000"
                },
        /* 14*/ { BARCODE_DBAR_LTD, -1, -1, -1, "0009876543210", 0, 1, 79, 1, "24724:2011 Figure F.2 — GS1 DataBar Limited",
                     "0101010010010011000011000001010110100101100101000100010100010000010010010100000"
                },
        /* 15*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[10]12A", 0, 1, 102, 1, "24724:2011 Figure F.3 — GS1 DataBar Expanded",
                     "010100000110100000101111111100001010001000000010110101111100100111001011110000000010011101111111010101"
                },
        /* 16*/ { BARCODE_DBAR_STK, -1, -1, -1, "0000000000000", 0, 3, 50, 1, "#183 GS1 DataBar Stacked separator alternation; verified manually against tec-it.com",
                     "01010100100000000100011111111001011111110010101010"
                     "00000101011111111010100000001010100000001101010000"
                     "10101010110000000101111111110111011111111011010101"
                },
        /* 17*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[255]95011015340010123456789", 0, 1, 232, 1, "GGS 2.6.2.1 Example 1",
                     "0100011000110001011011111111000010100000010101100001100001100111001010111110000001100100001110100001001000011011111010001111110000101001011111100111011001000111100100101111111100111011111001100100110010011100010111100011110000001010"
                },
        /* 18*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[255]95011015340010123456789[3900]000", 0, 1, 298, 1, "GGS 2.6.2.1 Example 2",
                     "0101100011111010001011111111000010100001000001001101100001100111001010111110000001100100001110100001001000011011111010001111110000101001011111100111011001000111100100101111111100111011111001100100110010011100010111100011000000001010111111011101000100001000110001101011111111100110011110010010001101"
                },
        /* 19*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[255]9501101534001[17]160531[3902]050", 0, 1, 281, 1, "GGS 2.6.2.1 Example 3",
                     "01011001000110011110111111110000101000000101011000011000011001110010101111100000011001000011101000010010000110111110100011111100001010010111111001110111000010010100001011111111001110000100001100110100010000001101001000110000000010111010011110011101110010110001100010111111111001101"
                },
        /* 20*/ { BARCODE_DBAR_EXPSTK, -1, 3, -1, "[255]9501101534001012345[8111]0500", 0, 5, 151, 1, "GGS 2.6.2.1 Example 4, same (tec-it separator differs)",
                     "0101100111100011001011111111000010100000010101100001100001100111001010111110000001100100001110100001001000011011111010001111110000101001011111100111010"
                     "0000011000011100110100000000101001011111101010011110011110011000110101000001010100011011110001011110110111100100000101010000001010010110100000011000000"
                     "0000010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010100000"
                     "0000110111000011011010000000010000100000100111011001100001100100010010100101010100101110110001111001011101100011101110100000000010000000000000000000000"
                     "1011001000111100100101111111100111011111011000100110011110011011101100011000000001010001001110000110100010011100010001011111111100110100000000000000000"
                },
        /* 21*/ { BARCODE_DBAR_EXPSTK, -1, 3, -1, "[255]9501101534001[3941]0035", 0, 5, 151, 1, "GGS 2.6.2.1 Example 5, same (tec-it separator differs)",
                     "0100001101011000011011111111000010100000010101100001100001100111001010111110000001100100001110100001001000011011111010001111110000101001011111100111010"
                     "0000110010100111100100000000101001011111101010011110011110011000110101000001010100011011110001011110110111100100000101010000001010010110100000011000000"
                     "0000010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010100000"
                     "0000011011111011111010000000010000111100101011111001000100011100111010100001010100000000000000000000000000000000000000000000000000000000000000000000000"
                     "1010100100000100000101111111100111000011010100000110111011100011000100011110000001010000000000000000000000000000000000000000000000000000000000000000000"
                },
        /* 22*/ { BARCODE_DBAR_OMN, -1, -1, -1, "0950110153000", 0, 1, 96, 1, "https://www.gs1.org/standards/barcodes/databar, same, verified manually against tec-it",
                     "010000010100000101000111111110010111101101011100100011011011000101111110000011001110110111001101"
                },
        /* 23*/ { BARCODE_DBAR_STK, -1, -1, -1, "0950110153000", 0, 3, 50, 1, "https://www.gs1.org/standards/barcodes/databar, same, verified manually against tec-it",
                     "01000001010000010100011111111001011110110101110010"
                     "00001100101101101010100001010100100001001010100000"
                     "10100011011011000101111110000011001110110111001101"
                },
        /* 24*/ { BARCODE_DBAR_EXPSTK, -1, -1, -1, "[01]09501101530003[17]140704[10]AB-123", 0, 9, 102, 1, "https://www.gs1.org/standards/barcodes/databar, same (tec-it separator differs)",
                     "010101111100001001101111111100001011100001110110010100000011011010001011111000000110011010000001001101"
                     "000010000011110110010000000010100100011110001001101011111100100101110100000101010001100101111110110000"
                     "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "000011011001001111100001000000001011011111000110110110110000111110101001010000001010011101110100010000"
                     "101000100110110000011100111111110100100000111001001001001111000001010100001111110001100010001011100010"
                     "000011011001001111100001000000001011011111000110110110110000111110101001010000001010011101110100010000"
                     "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "000010111101011110010100101010100100111100110010111001001100011111010100000000010000000000000000000000"
                     "010001000010100001100011000000001011000011001101000110110011100000101011111111100110100000000000000000"
                },
        /* 25*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]09501101530003[17]140704[10]AB-123", 0, 1, 281, 1, "https://www.gs1.org/standards/barcodes/databar, same, verified manually against tec-it",
                     "01010111110000100110111111110000101110000111011001010000001101101000101111100000011001101000000100110001110100010001100011111100001010100000111100100100100111000001001011111111001110000011011001000100010000101000011000110000000010110000110011010001101100111000001010111111111001101"
                },
        /* 26*/ { BARCODE_DBAR_STK, -1, -1, -1, "07010001234567", 0, 3, 50, 1, "https://www.gs1.no/support/standardbibliotek/datafangst/gs1-databar, same, verified manually against tec-it",
                     "01000100001010000100011100000001011000100110001010"
                     "00000011010101011010101011111010100111010101010000"
                     "10111100101110100101100000000111011000001000110101"
                },
        /* 27*/ { BARCODE_DBAR_OMNSTK, -1, -1, -1, "12380000000008", 0, 5, 50, 1, "Example with finder values 3 & 3; for bottom row see 5.3.2.2, same as BWIPP (tec-it and IDAutomation differ (ie no shift))",
                     "01011101001000000100010000000001010000001101011010"
                     "00000010110111111010101010101010101111110010100000"
                     "00000101010101010101010101010101010101010101010000"
                     "00001101100011001010000000000100101100111011110000"
                     "10100010011100110101111111110111010011000100001101"
                },
        /* 28*/ { BARCODE_DBAR_OMNSTK, -1, -1, -1, "99991234912372", 0, 5, 50, 1, "Example with finder values 8 & 6, same as BWIPP, verified manually against tec-it and IDAutomation",
                     "01001011101110000101110000000001011111011100101010"
                     "00000100010001111010001010101010100000100011010000"
                     "00000101010101010101010101010101010101010101010000"
                     "00001000100011001010000000010100100001000100100000"
                     "10100111011100110101111111100011011110111011011101"
                },
        /* 29*/ { BARCODE_DBAR_OMNSTK, -1, -1, -1, "32219876543217", 0, 5, 50, 1, "Example with finder values 6 & 1, same as BWIPP, verified manually against tec-it and IDAutomation",
                     "01001011000010001100111000000001011100010101000010"
                     "00000100111101110010000101010100100011101010110000"
                     "00000101010101010101010101010101010101010101010000"
                     "00001110011100101010000010101000110100001000010000"
                     "10110001100011010101111100000111001011110111100101"
                },
        /* 30*/ { BARCODE_DBAR_OMNSTK, -1, -1, -1, "32219876543255", 0, 5, 50, 1, "Example with finder values 7 & 7, same as BWIPP, verified manually against tec-it and IDAutomation",
                     "01001011000010001101111100000001011100010101000010"
                     "00000100111101110010000010101010100011101010110000"
                     "00000101010101010101010101010101010101010101010000"
                     "00000111001110101010000000101010110100001000010000"
                     "10111000110001010101111111000001001011110111100101"
                },
        /* 31*/ { BARCODE_DBAR_OMNSTK, -1, -1, -1, "04072912296211", 0, 5, 50, 1, "Example with finder values 7 & 8, same as BWIPP, verified manually against tec-it and IDAutomation",
                     "01001001000000010101111100000001011111000100101010"
                     "00000110111111101010000010101010100000111011010000"
                     "00000101010101010101010101010101010101010101010000"
                     "00001110100010111010000000001010111010000111010000"
                     "10110001011101000101111111110001000101111000101101"
                },
        /* 32*/ { BARCODE_DBAR_OMNSTK, -1, -1, -1, "06666666666666", 0, 5, 50, 1, "Example with finder values 6 & 4, same as BWIPP, verified manually against tec-it and IDAutomation",
                     "01000100010010000100111000000001011110111100101010"
                     "00001011101101111010000101010100100001000011010000"
                     "00000101010101010101010101010101010101010101010000"
                     "00000100011111001010000101010100101001100001110000"
                     "10101011100000110101111000000011010110011110000101"
                },
        /* 33*/ { BARCODE_DBAR_EXPSTK, -1, -1, -1, "[90]12345678901234567", 0, 5, 102, 1, "Example with 7 chars, 1 full row, bottom 3 chars",
                     "010010100001111000101111111100001010000010001110110100111110001011101011111100001110001111010011000101"
                     "000001011110000111010000000010100101111101110001001011000001110100010100000010100001110000101100110000"
                     "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "000000100000000101111110110001101011110001110011010100101000000101000010010111000000000000000000000000"
                     "101110011111111010000001001110010100001110001100101010000111111000111101101000111101000000000000000000"
                },
        /* 34*/ { BARCODE_DBAR_EXPSTK, -1, -1, -1, "[90]123456789012345678901234567", 0, 9, 102, 1, "Example with 10 chars, 2 full rows, bottom 2 chars",
                     "010000111100100010101111111100001010001000100000110100111110001011101011111000000110001111010011000101"
                     "000011000011011101010000000010100101110111011111001011000001110100010100000101010001110000101100110000"
                     "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "000010111011100011100001000000001011101111100011010111100011100110101001010000001010000100101110000000"
                     "101001000100011100011100111111110100010000011100101000011100011001010100001111110001111011010001111010"
                     "000010111011100011100001000000001011101111100011010111100011100110101001010000001010000100101110000000"
                     "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "000010000110110001010100001010100100000111010011000000000000000000000000000000000000000000000000000000"
                     "010001111001001110100011110000001011111000101100100100000000000000000000000000000000000000000000000000"
                },
        /* 35*/ { BARCODE_DBAR_EXPSTK, -1, -1, -1, "[90]123456789012345678901234567890", 0, 9, 102, 1, "Example with 11 chars, 2 full rows, bottom 3 chars",
                     "010111011100010001101111111100001010000010001110110100111110001011101011111000000110001111010011000101"
                     "000000100011101110010000000010100101111101110001001011000001110100010100000101010001110000101100110000"
                     "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "000010111011100011100001000000001011101111100011010111100011100110101001010000001010000100101110000000"
                     "101001000100011100011100111111110100010000011100101000011100011001010100001111110001111011010001111010"
                     "000010111011100011100001000000001011101111100011010111100011100110101001010000001010000100101110000000"
                     "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "000010000110110001010100101010100100111000100011011011000110001101110100000000010000000000000000000000"
                     "010001111001001110100011000000001011000111011100100100111001110010001011111111100110100000000000000000"
                },
        /* 36*/ { BARCODE_DBAR_EXPSTK, -1, -1, -1, "[91]1234567890123456789012345678901234", 0, 9, 102, 1, "Example with 12 chars, 3 full rows",
                     "010100010011111001101111111100001011001000010000010100111110001011101011111000000110001111010011000101"
                     "000011101100000110010000000010100100110111101111101011000001110100010100000101010001110000101100110000"
                     "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "000010111011100011100001000000001011101111100011010111100011100110101001010000001010000100101110000000"
                     "101001000100011100011100111111110100010000011100101000011100011001010100001111110001111011010001111010"
                     "000010111011100011100001000000001011101111100011010111100011100110101001010000001010000100101110000000"
                     "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "000010000110110001010100101010100100111000100011011011000110001110110100000000010001101110100001000000"
                     "010001111001001110100011000000001011000111011100100100111001110001001011111111100110010001011110111101"
                },
        /* 37*/ { BARCODE_DBAR_EXPSTK, -1, -1, -1, "[91]123456789012345678901234567890123456789012", 0, 13, 102, 1, "Example with 15 chars, 3 full rows, bottom 7 chars",
                     "010010000111101011101111111100001011100000101100010100111110001011101011110000000010001111010011000101"
                     "000001111000010100010000000010100100011111010011101011000001110100010100001010101001110000101100110000"
                     "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "000010111011100011100001010100001011101111100011010111100011100110101001010000001010000100101110000000"
                     "101001000100011100011100000011110100010000011100101000011100011001010100001111110001111011010001111010"
                     "000010111011100011100001010100001011101111100011010111100011100110101001010000001010000100101110000000"
                     "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "000010000110110001010100001010100100111000100011011011000110001110110100000010100001011110100001000000"
                     "010001111001001110100011110000001011000111011100100100111001110001001011111100001110100001011110111101"
                     "000010000110110001010100001010100100111000100011011011000110001110110100000010100001011110100001000000"
                     "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "000000100000000101001001111101100011000010000110010100101010100101011111011100100000000000000000000000"
                     "101110011111111010110110000010011100111101111001101010000000011000100000100011011101000000000000000000"
                },
        /* 38*/ { BARCODE_DBAR_EXPSTK, -1, 3, -1, "[91]123456789012345678901234567890123456789012", 0, 9, 151, 1, "Example with 15 chars, 2 full rows, bottom 3 chars",
                     "0100100001111010111011111111000010111000001011000101001111100010111010111100000000100011110100110001011110001011011110001111110000101010011000111000010"
                     "0000011110000101000100000000101001000111110100111010110000011101000101000010101010011100001011001110100001110100100001010000001010010101100111000110000"
                     "0000010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010100000"
                     "0000011000111110111010000101010000111000111011101101110000110110001010100001010100100111000100011011011000110001110110100000010100001011110100001000000"
                     "1010100111000001000101111000000111000111000100010010001111001001110100011110000001011000111011100100100111001110001001011111100001110100001011110111101"
                     "0000011000111110111010000101010000111000111011101101110000110110001010100001010100100111000100011011011000110001110110100000010100001011110100001000000"
                     "0000010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010100000"
                     "0000001001110111110101001010101001010011000010000110001101111100100101000000001000000000000000000000000000000000000000000000000000000000000000000000000"
                     "0101110110001000001000110000000010101100111101111001110010000011011010111111110011101000000000000000000000000000000000000000000000000000000000000000000"
                },
        /* 39*/ { BARCODE_DBAR_EXPSTK, -1, -1, -1, "[91]ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFG", 0, 17, 102, 1, "Example with 19 chars, 4 full rows, bottom 3 chars",
                     "010101111100011101101111111100001011100000101100010101111110011110101011110000000010111000111110010101"
                     "000010000011100010010000000010100100011111010011101010000001100001010100001010101001000111000001100000"
                     "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "000011110000110101100001010100001011001011000011110100001001101111001001010000001010100011100111100000"
                     "101100001111001010011100000011110100110100111100001011110110010000110100001111110001011100011000010010"
                     "000011110000110101100001010100001011001011000011110100001001101111001001010000001010100011100111100000"
                     "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "000001000010000111010100001010100101000010000111001011010111011111110100000010100001100011000010010000"
                     "010110111101111000100011110000001010111101111000110100101000100000001011111100001110011100111101100101"
                     "000001000010000111010100001010100101000010000111001011010111011111110100000010100001100011000010010000"
                     "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "000010000110111011100010101000001011000011110110100101001111000011001001010101001010011111001010010000"
                     "101101111001000100011000000111110100111100001001011010110000111100110100000000110001100000110101100010"
                     "000010000110111011100010101000001011000011110110100101001111000011001001010101001010011111001010010000"
                     "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "000010011111000111010001010101010101000111001111011011110100110000110100000000010000000000000000000000"
                     "010101100000111000100110000000001010111000110000100100001011001111001011111111100110100000000000000000"
                },
        /* 40*/ { BARCODE_DBAR_EXPSTK, -1, -1, -1, "[91]12345678901234567890123456789012345678901234567890123456789012345678", 0, 21, 102, 1, "Example with 22 chars, 5 full rows, bottom 2 chars",
                     "010101011110111111101111111100001011001000000101000100111110001011101011110000000010001111010011000101"
                     "000010100001000000010000000010100100110111111010111011000001110100010100001010101001110000101100110000"
                     "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "000010111011100011100001010100001011101111100011010111100011100110101001010000001010000100101110000000"
                     "101001000100011100011100000011110100010000011100101000011100011001010100001111110001111011010001111010"
                     "000010111011100011100001010100001011101111100011010111100011100110101001010000001010000100101110000000"
                     "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "000010000110110001010100001010100100111000100011011011000110001110110100000000100001011110100001000000"
                     "010001111001001110100011110000001011000111011100100100111001110001001011111111001110100001011110111101"
                     "000010000110110001010100001010100100111000100011011011000110001110110100000000100001011110100001000000"
                     "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "000010110011000001100010101000001011101111010001000110000100001100101001010101001010111110111001000000"
                     "101101001100111110011000000111110100010000101110111001111011110011010100000000110001000001000110111010"
                     "000010110011000001100010101000001011101111010001000110000100001100101001010101001010111110111001000000"
                     "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "000000110001100010010000000101010100000111100110101011011100010001110100000000010001011111010001100000"
                     "010111001110011101100111111000001011111000011001010100100011101110001011111111100110100000101110011101"
                     "000000110001100010010000000101010100000111100110101011011100010001110100000000010001011111010001100000"
                     "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "000001000111010000101000101010101010100001011000110000000000000000000000000000000000000000000000000000"
                     "001000111000101111010011000000000101011110100111000010000000000000000000000000000000000000000000000000"
                },
        /* 41*/ { BARCODE_DBAR_EXPSTK, -1, 3, -1, "[91]12345678901234567890123456789012345678901234567890123456789012345678", 0, 13, 151, 1, "Example with 22 chars, 3 full rows, bottom 4 chars",
                     "0101010111101111111011111111000010110010000001010001001111100010111010111100000000100011110100110001011110001011011110001111110000101010011000111000010"
                     "0000101000010000000100000000101001001101111110101110110000011101000101000010101010011100001011001110100001110100100001010000001010010101100111000110000"
                     "0000010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010100000"
                     "0000011000111110111010000101010000111000111011101101110000110110001010100001010100100111000100011011011000110001110110100000000100001011110100001000000"
                     "1010100111000001000101111000000111000111000100010010001111001001110100011110000001011000111011100100100111001110001001011111111001110100001011110111101"
                     "0000011000111110111010000101010000111000111011101101110000110110001010100001010100100111000100011011011000110001110110100000000100001011110100001000000"
                     "0000010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010100000"
                     "0000001001110111110101001010101001010011000010000110001000101111011101000001010100011000001100110100100011000110001001000000010101010000011110011010000"
                     "0101110110001000001000110000000010101100111101111001110111010000100010111110000001100111110011001011011100111001110110011111100000101111100001100101010"
                     "0000001001110111110101001010101001010011000010000110001000101111011101000001010100011000001100110100100011000110001001000000010101010000011110011010000"
                     "0000010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010100000"
                     "0000101110001000111010000000001000101111101000110001110001110100001010001010101010101000010110001100000000000000000000000000000000000000000000000000000"
                     "1010010001110111000101111111110011010000010111001110001110001011110100110000000001010111101001110000100000000000000000000000000000000000000000000000000"
                },
        /* 42*/ { BARCODE_DBAR_EXPSTK, -1, 4, -1, "[91]12345678901234567890123456789012345678901234567890123456789012345678", 0, 9, 200, 1, "Example with 22 chars, 2 full rows, bottom 6 chars",
                     "01010101111011111110111111110000101100100000010100010011111000101110101111000000001000111101001100010111100010110111100011111100001010100110001110000101001110000010001011110000001110001110001000100101"
                     "00001010000100000001000000001010010011011111101011101100000111010001010000101010100111000010110011101000011101001000010100000010100101011001110001111010110001111101110100001010100001110001110111010000"
                     "00000101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "00001011001100000110001010100000101110111101000100011000010000110010100101010100101011111011100100010000100001011110100001000000001011011100011000110110110001000111001001010100001010100011011000010000"
                     "10110100110011111001100000011111010001000010111011100111101111001101010000000011000100000100011011101111011110100001011100111111110100100011100111001001001110111000110100000011110001011100100111100010"
                     "00001011001100000110001010100000101110111101000100011000010000110010100101010100101011111011100100010000100001011110100001000000001011011100011000110110110001000111001001010100001010100011011000010000"
                     "00000101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "00000011000110001001000000010101010000011110011010101101110001000111010000000001000101111101000110001110001110100001010001010101010101000010110001100000000000000000000000000000000000000000000000000000"
                     "01011100111001110110011111100000101111100001100101010010001110111000101111111110011010000010111001110001110001011110100110000000001010111101001110000100000000000000000000000000000000000000000000000000"
                },
        /* 43*/ { BARCODE_DBAR_EXPSTK, -1, 5, -1, "[91]12345678901234567890123456789012345678901234567890123456789012345678", 0, 9, 249, 1, "Example with 22 chars, 2 full rows, bottom 2 chars",
                     "010101011110111111101111111100001011001000000101000100111110001011101011110000000010001111010011000101111000101101111000111111000010101001100011100001010011100000100010111100000011100011100010001001000111100100111010001111000000101100011101110010010"
                     "000010100001000000010000000010100100110111111010111011000001110100010100001010101001110000101100111010000111010010000101000000101001010110011100011110101100011111011101000010101000011100011101110110111000011011000101010000101010010011100010001100000"
                     "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010100000"
                     "000010001100011101101000000001000010111101000010000100010011101111101010010101010010100110000100001100010001011110111010000010101000110000011001101001000110001100010010000000101010100000111100110101011011100010001110100000000010001011111010001100000"
                     "101001110011100010010111111110011101000010111101111011101100010000010001100000000101011001111011110011101110100001000101111100000011001111100110010110111001110011101100111111000001011111000011001010100100011101110001011111111100110100000101110011101"
                     "000010001100011101101000000001000010111101000010000100010011101111101010010101010010100110000100001100010001011110111010000010101000110000011001101001000110001100010010000000101010100000111100110101011011100010001110100000000010001011111010001100000"
                     "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010100000"
                     "000010001110100001010001010101010101000010110001100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                     "010001110001011110100110000000001010111101001110000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                },
        /* 44*/ { BARCODE_DBAR_EXPSTK, -1, 6, -1, "[91]12345678901234567890123456789012345678901234567890123456789012345678", 0, 5, 298, 1, "Example with 22 chars, 1 full row, bottom 10 chars",
                     "0101010111101111111011111111000010110010000001010001001111100010111010111100000000100011110100110001011110001011011110001111110000101010011000111000010100111000001000101111000000111000111000100010010001111001001110100011110000001011000111011100100100111001110001001011111111001110100001011110111101"
                     "0000101000010000000100000000101001001101111110101110110000011101000101000010101010011100001011001110100001110100100001010000001010010101100111000111101011000111110111010000101010000111000111011101101110000110110001010100001010100100111000100011011011000110001110110100000000100001011110100001000000"
                     "0000010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "0000000100111011111010100101010100101001100001000011000100010111101110100000101010001100000110011010010001100011000100100000001010101000001111001101010110111000100011101000000000100010111110100011000111000111010000101000101010101010100001011000110000000000000000000000000000000000000000000000000000"
                     "0010111011000100000100011000000001010110011110111100111011101000010001011111000000110011111001100101101110011100111011001111110000010111110000110010101001000111011100010111111111001101000001011100111000111000101111010011000000000101011110100111000010000000000000000000000000000000000000000000000000"
                },
        /* 45*/ { BARCODE_DBAR_EXPSTK, -1, 7, -1, "[91]12345678901234567890123456789012345678901234567890123456789012345678", 0, 5, 347, 1, "Example with 22 chars, 1 full row, bottom 8 chars",
                     "01010101111011111110111111110000101100100000010100010011111000101110101111000000001000111101001100010111100010110111100011111100001010100110001110000101001110000010001011110000001110001110001000100100011110010011101000111100000010110001110111001001001110011100010010111111110011101000010111101111011101100010000010001100000000101011001111011110010"
                     "00001010000100000001000000001010010011011111101011101100000111010001010000101010100111000010110011101000011101001000010100000010100101011001110001111010110001111101110100001010100001110001110111011011100001101100010101000010101001001110001000110110110001100011101101000000001000010111101000010000100010011101111101010010101010010100110000100000000"
                     "00000101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010100000"
                     "00000100010111101110100000101010001100000110011010010001100011000100100000001010101000001111001101010110111000100011101000000000100010111110100011000111000111010000101000101010101010100001011000110000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                     "10111011101000010001011111000000110011111001100101101110011100111011001111110000010111110000110010101001000111011100010111111111001101000001011100111000111000101111010011000000000101011110100111000010000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                },
        /* 46*/ { BARCODE_DBAR_EXPSTK, -1, 8, -1, "[91]12345678901234567890123456789012345678901234567890123456789012345678", 0, 5, 396, 1, "Example with 22 chars, 1 full row, bottom 6 chars",
                     "010101011110111111101111111100001011001000000101000100111110001011101011110000000010001111010011000101111000101101111000111111000010101001100011100001010011100000100010111100000011100011100010001001000111100100111010001111000000101100011101110010010011100111000100101111111100111010000101111011110111011000100000100011000000001010110011110111100111011101000010001011111000000110011111001100101101"
                     "000010100001000000010000000010100100110111111010111011000001110100010100001010101001110000101100111010000111010010000101000000101001010110011100011110101100011111011101000010101000011100011101110110111000011011000101010000101010010011100010001101101100011000111011010000000010000101111010000100001000100111011111010100101010100101001100001000011000100010111101110100000101010001100000110011010000"
                     "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "000000011000110001001000000010101010000011110011010101101110001000111010000000001000101111101000110001110001110100001010001010101010101000010110001100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                     "001011100111001110110011111100000101111100001100101010010001110111000101111111110011010000010111001110001110001011110100110000000001010111101001110000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                },
        /* 47*/ { BARCODE_DBAR_EXPSTK, -1, 9, -1, "[91]12345678901234567890123456789012345678901234567890123456789012345678", 0, 5, 445, 1, "Example with 22 chars, 1 full row, bottom 4 chars",
                     "0101010111101111111011111111000010110010000001010001001111100010111010111100000000100011110100110001011110001011011110001111110000101010011000111000010100111000001000101111000000111000111000100010010001111001001110100011110000001011000111011100100100111001110001001011111111001110100001011110111101110110001000001000110000000010101100111101111001110111010000100010111110000001100111110011001011011100111001110110011111100000101111100001100101010"
                     "0000101000010000000100000000101001001101111110101110110000011101000101000010101010011100001011001110100001110100100001010000001010010101100111000111101011000111110111010000101010000111000111011101101110000110110001010100001010100100111000100011011011000110001110110100000000100001011110100001000010001001110111110101001010101001010011000010000110001000101111011101000001010100011000001100110100100011000110001001000000010101010000011110011010000"
                     "0000010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010100000"
                     "0000101110001000111010000000001000101111101000110001110001110100001010001010101010101000010110001100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                     "1010010001110111000101111111110011010000010111001110001110001011110100110000000001010111101001110000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                },
        /* 48*/ { BARCODE_DBAR_EXPSTK, -1, 10, -1, "[91]12345678901234567890123456789012345678901234567890123456789012345678", 0, 5, 494, 1, "Example with 22 chars, 1 full row, bottom 2 chars",
                     "01010101111011111110111111110000101100100000010100010011111000101110101111000000001000111101001100010111100010110111100011111100001010100110001110000101001110000010001011110000001110001110001000100100011110010011101000111100000010110001110111001001001110011100010010111111110011101000010111101111011101100010000010001100000000101011001111011110011101110100001000101111100000011001111100110010110111001110011101100111111000001011111000011001010100100011101110001011111111100110100000101110011101"
                     "00001010000100000001000000001010010011011111101011101100000111010001010000101010100111000010110011101000011101001000010100000010100101011001110001111010110001111101110100001010100001110001110111011011100001101100010101000010101001001110001000110110110001100011101101000000001000010111101000010000100010011101111101010010101010010100110000100001100010001011110111010000010101000110000011001101001000110001100010010000000101010100000111100110101011011100010001110100000000010001011111010001100000"
                     "00000101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "00000100011101000010100010101010101010000101100011000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                     "00100011100010111101001100000000010101111010011100001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                },
        /* 49*/ { BARCODE_DBAR_EXPSTK, -1, 11, -1, "[91]12345678901234567890123456789012345678901234567890123456789012345678", 0, 1, 543, 1, "Example with 22 chars, 1 row",
                     "010101011110111111101111111100001011001000000101000100111110001011101011110000000010001111010011000101111000101101111000111111000010101001100011100001010011100000100010111100000011100011100010001001000111100100111010001111000000101100011101110010010011100111000100101111111100111010000101111011110111011000100000100011000000001010110011110111100111011101000010001011111000000110011111001100101101110011100111011001111110000010111110000110010101001000111011100010111111111001101000001011100111000111000101111010011000000000101011110100111000010"
                },
        /* 50*/ { BARCODE_DBAR_EXPSTK, -1, 1, -1, "[91]1234567890123456789012345678901234567890123456789012345678901", 0, 37, 53, 1, "",
                     "01011100010111111010111111110000101100100000010100010"
                     "00000011101000000101000000001010010011011111101010000"
                     "00000101010101010101010101010101010101010101010100000"
                     "00001000001110100010100001010101001110000101100110000"
                     "10100111110001011101011110000000010001111010011000101"
                     "00001000001110100010100001010101001110000101100110000"
                     "00000101010101010101010101010101010101010101010100000"
                     "00000001110100100001010000001010010101100111000110000"
                     "01011110001011011110001111110000101010011000111000010"
                     "00000001110100100001010000001010010101100111000110000"
                     "00000101010101010101010101010101010101010101010100000"
                     "00000110001111101110100001010100001110001110111010000"
                     "10101001110000010001011110000001110001110001000100101"
                     "00000110001111101110100001010100001110001110111010000"
                     "00000101010101010101010101010101010101010101010100000"
                     "00001000011011000101010000101010010011100010001100000"
                     "01000111100100111010001111000000101100011101110010010"
                     "00001000011011000101010000101010010011100010001100000"
                     "00000101010101010101010101010101010101010101010100000"
                     "00001000110001110110100000010100001011110100001000000"
                     "10100111001110001001011111100001110100001011110111101"
                     "00001000110001110110100000010100001011110100001000000"
                     "00000101010101010101010101010101010101010101010100000"
                     "00000010011101111101010010101010010100110000100000000"
                     "01011101100010000010001100000000101011001111011110010"
                     "00000010011101111101010010101010010100110000100000000"
                     "00000101010101010101010101010101010101010101010100000"
                     "00000100010111101110100000101010001100000110011010000"
                     "10111011101000010001011111000000110011111001100101101"
                     "00000100010111101110100000101010001100000110011010000"
                     "00000101010101010101010101010101010101010101010100000"
                     "00000011000110001001000101010101010000011110011010000"
                     "01011100111001110110011000000000101111100001100101010"
                     "00000011000110001001000101010101010000011110011010000"
                     "00000101010101010101010101010101010101010101010100000"
                     "00001011100010001110100000000010001110111010001100000"
                     "10100100011101110001011111111100110001000101110011101"
                },
        /* 51*/ { BARCODE_DBAR_EXPSTK, -1, 1, -1, "[91]12345678901234567890123456789012345678901234567890123456789012345678", 0, 41, 53, 1, "Example with 22 chars, 11 rows",
                     "01010101111011111110111111110000101100100000010100010"
                     "00001010000100000001000000001010010011011111101010000"
                     "00000101010101010101010101010101010101010101010100000"
                     "00001000001110100010100001010101001110000101100110000"
                     "10100111110001011101011110000000010001111010011000101"
                     "00001000001110100010100001010101001110000101100110000"
                     "00000101010101010101010101010101010101010101010100000"
                     "00000001110100100001010000001010010101100111000110000"
                     "01011110001011011110001111110000101010011000111000010"
                     "00000001110100100001010000001010010101100111000110000"
                     "00000101010101010101010101010101010101010101010100000"
                     "00000110001111101110100001010100001110001110111010000"
                     "10101001110000010001011110000001110001110001000100101"
                     "00000110001111101110100001010100001110001110111010000"
                     "00000101010101010101010101010101010101010101010100000"
                     "00001000011011000101010000101010010011100010001100000"
                     "01000111100100111010001111000000101100011101110010010"
                     "00001000011011000101010000101010010011100010001100000"
                     "00000101010101010101010101010101010101010101010100000"
                     "00001000110001110110100000000100001011110100001000000"
                     "10100111001110001001011111111001110100001011110111101"
                     "00001000110001110110100000000100001011110100001000000"
                     "00000101010101010101010101010101010101010101010100000"
                     "00000010011101111101010010101010010100110000100000000"
                     "01011101100010000010001100000000101011001111011110010"
                     "00000010011101111101010010101010010100110000100000000"
                     "00000101010101010101010101010101010101010101010100000"
                     "00000100010111101110100000101010001100000110011010000"
                     "10111011101000010001011111000000110011111001100101101"
                     "00000100010111101110100000101010001100000110011010000"
                     "00000101010101010101010101010101010101010101010100000"
                     "00000011000110001001000000010101010000011110011010000"
                     "01011100111001110110011111100000101111100001100101010"
                     "00000011000110001001000000010101010000011110011010000"
                     "00000101010101010101010101010101010101010101010100000"
                     "00001011100010001110100000000010001011111010001100000"
                     "10100100011101110001011111111100110100000101110011101"
                     "00001011100010001110100000000010001011111010001100000"
                     "00000101010101010101010101010101010101010101010100000"
                     "00001000111010000101000101010101010100001011000110000"
                     "01000111000101111010011000000000101011110100111000010"
                },
        /* 52*/ { BARCODE_DBAR_EXPSTK, -1, 6, -1, "[01]98898765432106[3202]012345[15]991231[3203]001234[17]010203", 0, 5, 298, 1, "#200 Daniel Gredler mostly empty last row, 16 chars, 2 rows, bottom row 4 chars",
                     "0100011101110001011011111111000010110000100101111101101000000110001010111100000000101001110000001001010011111011100110001111110000101110101000011000011010011100011000101111000000111001111000111101010111110010110000100011110000001010110010000010000111011111000100101011111100001110011110011000100101"
                     "0000100010001110100100000000101001001111011010000010010111111001110101000010101010010110001111110110101100000100011001010000001010010001010111100111100101100011100111010000101010000110000111000010101000001101001111010100001010100101001101111101111000100000111011010100000010100001100001100111010000"
                     "0000010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "0000110000010010011000010000000010111110011010001001111010011011100010010101010010101110100011111000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                     "1010001111101101100111001111111101000001100101110110000101100100011101000000001100010001011100000110100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                },
        /* 53*/ { BARCODE_DBAR_EXPSTK, -1, 3, -1, "[01]98898765432106[3202]012345[15]991231[3203]001234[17]010203", 0, 9, 151, 1, "#200 16 chars, 3 rows, bottom row 4 chars",
                     "0100011101110001011011111111000010110000100101111101101000000110001010111100000000101001110000001001010011111011100110001111110000101110101000011000010"
                     "0000100010001110100100000000101001001111011010000010010111111001110101000010101010010110001111110110101100000100011001010000001010010001010111100110000"
                     "0000010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010100000"
                     "0000101100011100111010000101010000110000111000010101000001101001111010100001010100101001101111101111000100000111011010100000010100001100001100111010000"
                     "1011010011100011000101111000000111001111000111101010111110010110000100011110000001010110010000010000111011111000100101011111100001110011110011000100101"
                     "0000101100011100111010000101010000110000111000010101000001101001111010100001010100101001101111101111000100000111011010100000010100001100001100111010000"
                     "0000010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010100000"
                     "0000011111000101110101001010101001000111011001011110010001011001111101000000001000011001001000001100000000000000000000000000000000000000000000000000000"
                     "0101100000111010001000110000000010111000100110100001101110100110000010111111110011100110110111110001010000000000000000000000000000000000000000000000000"
                },
        /* 54*/ { BARCODE_DBAR_EXPSTK, -1, 4, -1, "[01]98898765432106[3202]012345[15]991231[3203]001234[17]010203", 0, 5, 200, 1, "#200 16 chars, 2 full rows",
                     "01000111011100010110111111110000101100001001011111011010000001100010101111000000001010011100000010010100111110111001100011111100001011101010000110000110100111000110001011110000001110011110001111010101"
                     "00001000100011101001000000001010010011110110100000100101111110011101010000101010100101100011111101101011000001000110010100000010100100010101111001111001011000111001110100001010100001100001110000100000"
                     "00000101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "00001100000100100110000100000000101111100110100010011110100110111000100101010100101011101000111110010110111001100001100001010000001010110111000001000111101111101100101001010100001010111100101100000000"
                     "10100011111011011001110011111111010000011001011101100001011001000111010000000011000100010111000001101001000110011110011100001111110101001000111110111000010000010011010100000011110001000011010011111010"
                },
        /* 55*/ { BARCODE_DBAR_EXPSTK, -1, 5, -1, "[01]98898765432106[3202]012345[15]991231[3203]001234[17]010203", 0, 5, 249, 1, "#200 16 chars, 2 rows, bottom row 6 chars",
                     "010001110111000101101111111100001011000010010111110110100000011000101011110000000010100111000000100101001111101110011000111111000010111010100001100001101001110001100010111100000011100111100011110101011111001011000010001111000000101011001000001000010"
                     "000010001000111010010000000010100100111101101000001001011111100111010100001010101001011000111111011010110000010001100101000000101001000101011110011110010110001110011101000010101000011000011100001010100000110100111101010000101010010100110111110110000"
                     "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010100000"
                     "000001000001110110101000000101000011000011001110110100111110001011101010010101010010001110110010111100100010110011111010000000010000110010010000011000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                     "101110111110001001010111111000011100111100110001001011000001110100010001100000000101110001001101000011011101001100000101111111100111001101101111100010100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                },
        /* 56*/ { BARCODE_DBAR_EXPSTK, -1, 7, -1, "[01]98898765432106[3202]012345[15]991231[3203]001234[17]010203", 0, 5, 347, 1, "#200 16 chars, 2 rows, bottom row 2 chars",
                     "01000111011100010110111111110000101100001001011111011010000001100010101111000000001010011100000010010100111110111001100011111100001011101010000110000110100111000110001011110000001110011110001111010101111100101100001000111100000010101100100000100001110111110001001010111111000011100111100110001001011000001110100010001100000000101110001001101000010"
                     "00001000100011101001000000001010010011110110100000100101111110011101010000101010100101100011111101101011000001000110010100000010100100010101111001111001011000111001110100001010100001100001110000101010000011010011110101000010101001010011011111011110001000001110110101000000101000011000011001110110100111110001011101010010101010010001110110010110000"
                     "00000101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010100000"
                     "00001000101100111110100000000100001100100100000110000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                     "10110111010011000001011111111001110011011011111000101000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
                },
        /* 57*/ { BARCODE_DBAR_EXPSTK, -1, 8, -1, "[01]98898765432106[3202]012345[15]991231[3203]001234[17]010203", 0, 1, 396, 1, "#200 16 chars, 1 row",
                     "010001110111000101101111111100001011000010010111110110100000011000101011110000000010100111000000100101001111101110011000111111000010111010100001100001101001110001100010111100000011100111100011110101011111001011000010001111000000101011001000001000011101111100010010101111110000111001111001100010010110000011101000100011000000001011100010011010000110111010011000001011111111001110011011011111000101"
                },
        /* 58*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]00012345678905[10]ABC123", 0, 1, 232, 1, "24724:2011 7.2.5.4.1, encoding method 1 '1'",
                     "0100011000001011011011111111000010110011000010111101011110011011111010111110000001100010110000110111000111101101011110001111110000101110001100100001010011101111110110101111111100111001011011111101110011011100101111100011110000001010"
                },
        /* 59*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]90012345678908[3103]001750", 0, 1, 151, 1, "24724:2011 7.2.5.4.2, encoding method 3 '0100'",
                     "0101110010000010011011111111000010111000010011000101011110111001100010111100000011100101110001110111011110101111000110001111110000101011000010011111010"
                },
        /* 60*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]90012345678908[3103]032767", 0, 1, 151, 1, "Encoding method 3 '0100' with weight = 32767",
                     "0101001000111000011011111111000010111000010011000101011110111001100010111100000011100101110001110111011110111011000110001111110000101101111101110111010"
                },
        /* 61*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]90012345678908[3103]032768", 0, 1, 200, 1, "Possible encoding method 3 '0100' but weight > 32767 so encoding method 7 '0111000' with dummy date",
                     "01001100000101001110111111110000101000110111000010010111100110111110101111110000111000101100001101110001111011010111100011111100001011000110100110000110100000100110001011111111001110011001111010000101"
                },
        /* 62*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]90012345678908[3202]000156", 0, 1, 151, 1, "24724:2011 7.2.5.4.3, encoding method 4 '0101'",
                     "0101001000111100001011111111000010100111000100001101011110111001100010111100000011100101110001110111011110101111000110001111110000101100001000001010010"
                },
        /* 63*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]90012345678908[3202]009999", 0, 1, 151, 1, "Encoding method 4 '0101' with weight = 9999",
                     "0101110001000100011011111111000010100111000100001101011110111001100010111100000011100101110001110111011110110100011110001111110000101100111110010001010"
                },
        /* 64*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]90012345678908[3202]010000", 0, 1, 200, 1, "Possible encoding method 4 '0101' but weight > 9999 so encoding method 8 with dummy date",
                     "01001000101110000110111111110000101110100011000010010111100110111110101111110000111000101100001101110001111011010111100011111100001010000011101001100111101101001110001011111111001110011001111010000101"
                },
        /* 65*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]90012345678908[3203]022767", 0, 1, 151, 1, "Encoding method 4 '0101' with weight = 22767",
                     "0101110010011000001011111111000010100111000100001101011110111001100010111100000011100101110001110111011110111011000110001111110000101101111101110111010"
                },
        /* 66*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]90012345678908[3203]022768", 0, 1, 200, 1, "Possible encoding method 4 '0101' but weight > 22767 so encoding method 8 with dummy date",
                     "01000110111000100010111111110000101110100011000010010111100110111110101111110000111000101100001101110001111011010111100011111100001010011100010110000100001101000001101011111111001110011001111010000101"
                },
        /* 67*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]90012345678908[3922]795", 0, 1, 183, 1, "24724:2011 7.2.5.4.5, encoding method 5 '01100XX', no following AIs",
                     "010110000010001011101111111100001010011100000101100101111001101111101011111100001110001011000011011100011110110101111000111111000010100111101110100001100011011100100010111111110011101"
                },
        /* 68*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]90012345678908[3922]795[20]01", 0, 1, 200, 1, "Encoding method 5 '01100XX' with following AI",
                     "01000110110000110010111111110000101111000100001010010111100110111110101111110000111000101100001101110001111011010111100011111100001010011110111010000110001110001011001011111111001110100111110001110101"
                },
        /* 69*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]90012345678908[3932]0081234", 0, 1, 200, 1, "24724:2011 7.2.5.4.6, encoding method 6 '01101XX', no following AIs",
                     "01001110000101100010111111110000101110100000110010010111100110111110101111110000111000101100001101110001111011010111100011111100001011000011010111100100111110001011101011111111001110001101111001011101"
                },
        /* 70*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]90012345678908[3932]0081234[20]01", 0, 1, 232, 1, "Encoding method 6 '01101XX' with following AI",
                     "0100001101000100111011111111000010110000101000111001011110011011111010111110000001100010110000110111000111101101011110001111110000101100001101011110010011111000101110101111111100111000100110011110010011101111001000100011110000001010"
                },
        /* 71*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]90012345678908[3932]A401234", ZINT_WARN_NONCOMPLIANT, 1, 232, 0, "Possible encoding method 6 '01101XX' but invalid currency code so encoding method 1; BWIPP no check (craps out)",
                     "0100011000010011011011111111000010100100011111001101011110011011111010111110000001100010110000110111000111101101011110001111110000101100011001111001010001011011000000101111111100111001101011111110110001111001001110100011110000001010"
                },
        /* 72*/ { BARCODE_DBAR_EXP, GS1NOCHECK_MODE, -1, -1, "[01]90012345678908[3932]A401234", 0, 1, 232, 0, "Possible encoding method 6 '01101XX' but invalid currency code so encoding method 1; BWIPP no check (craps out)",
                     "0100011000010011011011111111000010100100011111001101011110011011111010111110000001100010110000110111000111101101011110001111110000101100011001111001010001011011000000101111111100111001101011111110110001111001001110100011110000001010"
                },
        /* 73*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]90012345678908[3102]099999[11]201209", 0, 1, 200, 1, "Encoding method 7 '0111000' with weight <= 99999 and valid date",
                     "01000101111001000010111111110000101000110111000010010111100110111110101111110000111000101100001101110001111011010111100011111100001010111100100001000100000011100101001011111111001110010000100100011101"
                },
        /* 74*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]90012345678908[3102]099999", 0, 1, 200, 1, "Encoding method 7 '0111000' with weight <= 99999 but no date",
                     "01000111011000010010111111110000101000110111000010010111100110111110101111110000111000101100001101110001111011010111100011111100001010111100100001000110100100011000001011111111001110011001111010000101"
                },
        /* 75*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]90012345678908[3102]100000[11]201209", 0, 1, 281, 1, "Possible encoding method 7 '0111000' but weight > 99999 so encoding method 1",
                     "01010011110001110010111111110000101001000111110011010111100110111110101111100000011000101100001101110001111011010111100011111100001010010110001110000110010000011110101011111111001110000011100110101100001001110100011000110000000010100101100000100001011100011001111010111111111001101"
                },
        /* 76*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]90012345678908[3102]099999[11]200000", ZINT_WARN_NONCOMPLIANT, 1, 281, 0, "Possible encoding method 7 '0111000' with weight <= 99999 but date invalid so encoding method 1; BWIPP requires `dontlint` (GS1NOCHECK_MODE)",
                     "01011001110001001110111111110000101001000111110011010111100110111110101111100000011000101100001101110001111011010111100011111100001010010110001110000110010011110100001011111111001110110011100010111100001001110100011000110000000010101100001000000101010111101111110010111111111001101"
                },
        /* 77*/ { BARCODE_DBAR_EXP, GS1NOCHECK_MODE, -1, -1, "[01]90012345678908[3102]099999[11]200000", 0, 1, 281, 1, "Possible encoding method 7 '0111000' with weight <= 99999 but date invalid so encoding method 1",
                     "01011001110001001110111111110000101001000111110011010111100110111110101111100000011000101100001101110001111011010111100011111100001010010110001110000110010011110100001011111111001110110011100010111100001001110100011000110000000010101100001000000101010111101111110010111111111001101"
                },
        /* 78*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]90012345678908[3201]099999[11]201209", 0, 1, 200, 1, "Encoding method 8 '0111001' with weight <= 99999 and valid date",
                     "01001000001101001110111111110000101110100011000010010111100110111110101111110000111000101100001101110001111011010111100011111100001011000100001101100111010111001110001011111111001110010000100100011101"
                },
        /* 79*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]90012345678908[3201]099999", 0, 1, 200, 1, "Encoding method 8 '0111001' with weight <= 99999 but no date",
                     "01000101110010000110111111110000101110100011000010010111100110111110101111110000111000101100001101110001111011010111100011111100001011000100001101100111010000111011101011111111001110011001111010000101"
                },
        /* 80*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]90012345678908[3201]100000[11]201209", 0, 1, 281, 1, "Possible encoding method 8 '0111001' but weight > 99999 so encoding method 1",
                     "01011101100011000110111111110000101001000111110011010111100110111110101111100000011000101100001101110001111011010111100011111100001011101010000110000111011110001100101011111111001110000011100110101100001001110100011000110000000010100101100000100001011100011001111010111111111001101"
                },
        /* 81*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]90012345678908[3201]099999[11]000000", ZINT_WARN_NONCOMPLIANT, 1, 281, 0, "Possible encoding method 8 '0111001' but date invalid so encoding method 1; BWIPP requires `dontlint` (GS1NOCHECK_MODE)",
                     "01011110100001001110111111110000101001000111110011010111100110111110101111100000011000101100001101110001111011010111100011111100001011101010000110000111000011110101101011111111001110110011100010111100001110100001011000110000000010101100001000000101010111101111110010111111111001101"
                },
        /* 82*/ { BARCODE_DBAR_EXP, GS1NOCHECK_MODE, -1, -1, "[01]90012345678908[3201]099999[11]000000", 0, 1, 281, 1, "Possible encoding method 8 '0111001' but date invalid so encoding method 1",
                     "01011110100001001110111111110000101001000111110011010111100110111110101111100000011000101100001101110001111011010111100011111100001011101010000110000111000011110101101011111111001110110011100010111100001110100001011000110000000010101100001000000101010111101111110010111111111001101"
                },
        /* 83*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]90012345678908[3100]099999[13]201209", 0, 1, 200, 1, "Encoding method 9 '0111010' with weight <= 99999 and valid date",
                     "01001000001101001110111111110000101111001010000010010111100110111110101111110000111000101100001101110001111011010111100011111100001011000111000001010111010000110110001011111111001110010000100100011101"
                },
        /* 84*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]90012345678908[3204]099999[13]201209", 0, 1, 200, 1, "Encoding method 10 '0111011' with weight <= 99999 and valid date",
                     "01001000111000010110111111110000101100101000001110010111100110111110101111110000111000101100001101110001111011010111100011111100001010011101000011110101110000101111101011111111001110010000100100011101"
                },
        /* 85*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]90012345678908[3103]012233[15]991231", 0, 1, 200, 1, "24724:2011 7.2.5.4.4, encoding method 11 '0111100' with weight <= 99999 and valid date",
                     "01001100000100111010111111110000101011100100000110010111100110111110101111110000111000101100001101110001111011010111100011111100001011000011010110000111001100110001001011111111001110001101111010000101"
                },
        /* 86*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]90012345678908[3205]099999[15]201209", 0, 1, 200, 1, "Encoding method 12 '0111101' with weight <= 99999 and valid date",
                     "01001110000010011010111111110000101000001101110100010111100110111110101111110000111000101100001101110001111011010111100011111100001011110011010001100101001100011000001011111111001110010000100100011101"
                },
        /* 87*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]90012345678908[3105]099999[17]201209", 0, 1, 200, 1, "Encoding method 13 '0111110' with weight <= 99999 and valid date",
                     "01000111010000110010111111110000101110000100110100010111100110111110101111110000111000101100001101110001111011010111100011111100001011110011010001100101001100011000001011111111001110010000100100011101"
                },
        /* 88*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]90012345678908[3200]099999[17]201209", 0, 1, 200, 1, "Encoding method 14 '0111111' with weight <= 99999 and valid date",
                     "01001110000010100110111111110000101111000100010100010111100110111110101111110000111000101100001101110001111011010111100011111100001011000111000001010111010000110110001011111111001110010000100100011101"
                },
        /* 89*/ { BARCODE_DBAR_EXPSTK, -1, 4, -1, "[8110]106141416543213500110000310123196000", 0, 5, 200, 1, "North American Coupon Application Guideline (NACAG) Figure 1 (& Appendix C: Example 6)",
                     "01001100101110001110111111110000101000001101000001010111100110111000101111100000011011010001000000110000111100001010100011111100001010000010101111000111010000011001101011111111001110000001101101000101"
                     "00000011010001110001000000001010010111110010111110101000011001000111010000010101000100101110111111001111000011110101010100000010100101111101010000111000101111100110010100000000100001111110010010110000"
                     "00000101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "00000100010110000110100000001010101010111100111101110010110001110011101000000000100010000011010010000100011110111011001000101010101010011101111111010000000000000000000000000000000000000000000000000000"
                     "00100011101001111001001111110000010101000011000010001101001110001100010111111111001101111100101101111011100001000100110011000000000101100010000000101010000000000000000000000000000000000000000000000000"
                },
        /* 90*/ { BARCODE_DBAR_EXPSTK, -1, -1, 2, "[8110]106141416543213500110000310123196000", 0, 5, 200, 1, "NACAG Figure 1 (& Appendix C: Example 6) specifying max rows",
                     "01001100101110001110111111110000101000001101000001010111100110111000101111100000011011010001000000110000111100001010100011111100001010000010101111000111010000011001101011111111001110000001101101000101"
                     "00000011010001110001000000001010010111110010111110101000011001000111010000010101000100101110111111001111000011110101010100000010100101111101010000111000101111100110010100000000100001111110010010110000"
                     "00000101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "00000100010110000110100000001010101010111100111101110010110001110011101000000000100010000011010010000100011110111011001000101010101010011101111111010000000000000000000000000000000000000000000000000000"
                     "00100011101001111001001111110000010101000011000010001101001110001100010111111111001101111100101101111011100001000100110011000000000101100010000000101010000000000000000000000000000000000000000000000000"
                },
        /* 91*/ { BARCODE_DBAR_EXPSTK, -1, 3, -1, "[8110]10614141011111275110111", 0, 5, 151, 1, "NACAG Figure 3",
                     "0101011110011100001011111111000010100000110100000101011110011011100010111110000001101101000100000011000011100001101010001111110000101010000011000111010"
                     "0000100001100011110100000000101001011111001011111010100001100100011101000001010100010010111011111100111100011110010101010000001010010101111100111000000"
                     "0000010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010100000"
                     "0000001000100001011010000000010000111001101101100001100111100100100010100001010100100001101011111000000000000000000000000000000000000000000000000000000"
                     "1011110111011110100101111111100111000110010010011110011000011011011100011110000001011110010100000100100000000000000000000000000000000000000000000000000"
                },
        /* 92*/ { BARCODE_DBAR_EXPSTK, -1, -1, 2, "[8110]10614141011111275110111", 0, 5, 151, 1, "NACAG Figure 3 specifying max rows",
                     "0101011110011100001011111111000010100000110100000101011110011011100010111110000001101101000100000011000011100001101010001111110000101010000011000111010"
                     "0000100001100011110100000000101001011111001011111010100001100100011101000001010100010010111011111100111100011110010101010000001010010101111100111000000"
                     "0000010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010100000"
                     "0000001000100001011010000000010000111001101101100001100111100100100010100001010100100001101011111000000000000000000000000000000000000000000000000000000"
                     "1011110111011110100101111111100111000110010010011110011000011011011100011110000001011110010100000100100000000000000000000000000000000000000000000000000"
                },
        /* 93*/ { BARCODE_DBAR_EXPSTK, -1, 6, -1, "[8110]106141410222223100110222111101231023456721104561045678991201", 0, 5, 298, 1, "NACAG Figure 4",
                     "0100101111101001111011111111000010110010000011001101011110011011100010111100000000101101000100000011000011100001101010001111110000101011000011100001010111000011011110101111000000111010011111000101110001110110011001100011110000001011100110000101000110101111110111001011111111001110011010011100001101"
                     "0000010000010110000100000000101001001101111100110010100001100100011101000010101010010010111011111100111100011110010101010000001010010100111100011110101000111100100001010000101010000101100000111010001110001001100110010100001010100100011001111010111001010000001000110100000000100001100101100011110000"
                     "0000010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "0000011010011011100010100101010100100111000100000101010000111000110010100000101010001111000101110100011001110001101000100000001010101011011110000111010111110001101011101000000000100011110100100010000110101100111000001000101010101000000000000000000000000000000000000000000000000000000000000000000000"
                     "0010000101100100011100011000000001011000111011111010101111000111001101011111000000110000111010001011100110001110010111001111110000010100100001111000101000001110010100010111111111001100001011011101111001010011000111110011000000000101000000000000000000000000000000000000000000000000000000000000000000"
                },
        /* 94*/ { BARCODE_DBAR_EXPSTK, -1, -1, 2, "[8110]106141410222223100110222111101231023456721104561045678991201", 0, 5, 298, 1, "NACAG Figure 4, specifying max rows",
                     "0100101111101001111011111111000010110010000011001101011110011011100010111100000000101101000100000011000011100001101010001111110000101011000011100001010111000011011110101111000000111010011111000101110001110110011001100011110000001011100110000101000110101111110111001011111111001110011010011100001101"
                     "0000010000010110000100000000101001001101111100110010100001100100011101000010101010010010111011111100111100011110010101010000001010010100111100011110101000111100100001010000101010000101100000111010001110001001100110010100001010100100011001111010111001010000001000110100000000100001100101100011110000"
                     "0000010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "0000011010011011100010100101010100100111000100000101010000111000110010100000101010001111000101110100011001110001101000100000001010101011011110000111010111110001101011101000000000100011110100100010000110101100111000001000101010101000000000000000000000000000000000000000000000000000000000000000000000"
                     "0010000101100100011100011000000001011000111011111010101111000111001101011111000000110000111010001011100110001110010111001111110000010100100001111000101000001110010100010111111111001100001011011101111001010011000111110011000000000101000000000000000000000000000000000000000000000000000000000000000000"
                },
        /* 95*/ { BARCODE_DBAR_EXPSTK, -1, 5, -1, "[8110]1061414165432131501101201211014092110256100126663101231", 0, 5, 249, 1, "NACAG Appendix C: Example 1",
                     "010111101100011011101111111100001011001000001100110101111001101110001011110000000010110100010000001100001111000010101000111111000010100000101011110001110100000110011010111100000011101001111000101111000111011001001110001111000000101011001000001110010"
                     "000000010011100100010000000010100100110111110011001010000110010001110100001010101001001011101111110011110000111101010101000000101001011111010100001110001011111001100101000010101000010110000111010000111000100110110001010000101010010100110111110000000"
                     "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010100000"
                     "000001110000100101101000000101000011110100100011100111100001001101001010010101010010011111110101101100010010110011111010000010101000111101010000010001110001001100110010001010101010100011000111001011000010011010000010100000000010000000000000000000000"
                     "101110001111011010010111111000011100001011011100011000011110110010110001100000000101100000001010010011101101001100000101111100000011000010101111101110001110110011001100110000000001011100111000110100111101100101111101011111111100110100000000000000000"
                },
        /* 96*/ { BARCODE_DBAR_EXPSTK, -1, -1, 2, "[8110]1061414165432131501101201211014092110256100126663101231", 0, 5, 249, 1, "NACAG Appendix C: Example 1, specifying max row",
                     "010111101100011011101111111100001011001000001100110101111001101110001011110000000010110100010000001100001111000010101000111111000010100000101011110001110100000110011010111100000011101001111000101111000111011001001110001111000000101011001000001110010"
                     "000000010011100100010000000010100100110111110011001010000110010001110100001010101001001011101111110011110000111101010101000000101001011111010100001110001011111001100101000010101000010110000111010000111000100110110001010000101010010100110111110000000"
                     "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010100000"
                     "000001110000100101101000000101000011110100100011100111100001001101001010010101010010011111110101101100010010110011111010000010101000111101010000010001110001001100110010001010101010100011000111001011000010011010000010100000000010000000000000000000000"
                     "101110001111011010010111111000011100001011011100011000011110110010110001100000000101100000001010010011101101001100000101111100000011000010101111101110001110110011001100110000000001011100111000110100111101100101111101011111111100110100000000000000000"
                },
        /* 97*/ { BARCODE_DBAR_EXPSTK, -1, 4, -1, "[8110]106141410012342501106501013085093101231", 0, 5, 200, 1, "NACAG Appendix C: Example 2",
                     "01001100101110001110111111110000101000001101000001010111100110111000101111100000011011010001000000110000111000011010100011111100001010000110011000110111001110000010101011111111001110011110010001100101"
                     "00000011010001110001000000001010010111110010111110101000011001000111010000010101000100101110111111001111000111100101010100000010100101111001100111001000110001111101010100000000100001100001101110010000"
                     "00000101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "00000100100001001110100000001010101011111101001011110000110001001011101000000000100010111000110010000111011001101000001000101010101010011011001000010000000000000000000000000000000000000000000000000000"
                     "00100011011110110001001111110000010100000010110100001111001110110100010111111111001101000111001101111000100110010111110011000000000101100100110111100010000000000000000000000000000000000000000000000000"
                },
        /* 98*/ { BARCODE_DBAR_EXPSTK, -1, 5, -1, "[8110]106141410012471011076011110850921108609310123191000", 0, 5, 249, 1, "NACAG Appendix C: Example 3",
                     "010101111101111000101111111100001010000011000101000101111001101110001011110000000010110100010000001100001110000110101000111111000010100001100110001101001000010000111010111100000011100011100011011001011100001001101110001111000000101001100010000111010"
                     "000010000010000111010000000010100101111100111010111010000110010001110100001010101001001011101111110011110001111001010101000000101001011110011001110010110111101111000101000010101000011100011100100110100011110110010001010000101010010110011101111000000"
                     "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010100000"
                     "000010000001000010101000000101000011110111011110110111101101011100001010010101010010101000011111011101111101110011001010000010101000101001111001110001000001100110001010000000101010100111111101101000000000000000000000000000000000000000000000000000000"
                     "101001111110111101010111111000011100001000100001001000010010100011110001100000000101010111100000100010000010001100110101111100000011010110000110001110111110011001110100111111000001011000000010010100100000000000000000000000000000000000000000000000000"
                },
        /* 99*/ { BARCODE_DBAR_EXPSTK, -1, 6, -1, "[8110]106141411234562891101201212085010048000214025610048000310123191000", 0, 5, 298, 1, "NACAG Appendix C: Example 4",
                     "0100111111001110101011111111000010100000110001010001011110011011100010111100000000101101000100000011000011100001101010001111110000101010011000111000010101110001000000101111000000111001000001001110010011000011011101100011110000001010000010010110000100011001011000001011111111001110001110101100111101"
                     "0000000000110001010100000000101001011111001110101110100001100100011101000010101010010010111011111100111100011110010101010000001010010101100111000111101010001110111111010000101010000110111110110001101100111100100010010100001010100101111101101001111011100110100111110100000000100001110001010011000000"
                     "0000010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                     "0000001110011100111010100101010100101001111011111101000100011101100010100000101010001110110010001000010111101100001110100000001010101000011101111010110010111110001001101000000000100011100001011000010111001010111000001000101010101010001111001001100000000000000000000000000000000000000000000000000000"
                     "0010110001100011000100011000000001010110000100000010111011100010011101011111000000110001001101110111101000010011110001001111110000010111100010000101001101000001110110010111111111001100011110100111101000110101000111110011000000000101110000110110011010000000000000000000000000000000000000000000000000"
                },
        /*100*/ { BARCODE_DBAR_EXPSTK, -1, 5, -1, "[8110]1061414154321031501101201211014092110256100126663101231", 0, 5, 249, 1, "NACAG Appendix C: Example 5",
                     "010100111111010011101111111100001011001000001100110101111001101110001011110000000010110100010000001100001111000010101000111111000010101100100000001001100011101011000010111100000011101001111000101111000111011001001110001111000000101011001000001110010"
                     "000011000000101100010000000010100100110111110011001010000110010001110100001010101001001011101111110011110000111101010101000000101001010011011111110110011100010100111101000010101000010110000111010000111000100110110001010000101010010100110111110000000"
                     "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010100000"
                     "000001110000100101101000000101000011110100100011100111100001001101001010010101010010011111110101101100010010110011111010000010101000111101010000010001110001001100110010001010101010100011000111001011000010011010000010100000000010000000000000000000000"
                     "101110001111011010010111111000011100001011011100011000011110110010110001100000000101100000001010010011101101001100000101111100000011000010101111101110001110110011001100110000000001011100111000110100111101100101111101011111111100110100000000000000000"
                },
    };
    int data_size = ARRAY_SIZE(data);
    int i, length, ret;
    struct zint_symbol *symbol;

    char escaped[4096];
    char cmp_buf[16384];
    char cmp_msg[1024];

    int do_bwipp = (debug & ZINT_DEBUG_TEST_BWIPP) && testUtilHaveGhostscript(); /* Only do BWIPP test if asked, too slow otherwise */
    int do_zxingcpp = (debug & ZINT_DEBUG_TEST_ZXINGCPP) && testUtilHaveZXingCPPDecoder(); /* Only do ZXing-C++ test if asked, too slow otherwise */

    testStart("test_examples");

    for (i = 0; i < data_size; i++) {

        if (testContinue(p_ctx, i)) continue;

        symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        length = testUtilSetSymbol(symbol, data[i].symbology, data[i].input_mode, -1 /*eci*/, -1 /*option_1*/, data[i].option_2, data[i].option_3, -1 /*output_options*/, data[i].data, -1, debug);

        ret = ZBarcode_Encode(symbol, (const unsigned char *) data[i].data, length);
        assert_equal(ret, data[i].ret, "i:%d ZBarcode_Encode ret %d != %d (%s)\n", i, ret, data[i].ret, symbol->errtxt);

        if (p_ctx->generate) {
            printf("        /*%3d*/ { %s, %s, %d, %d, \"%s\", %d, %d, %d, %d, \"%s\",\n",
                    i, testUtilBarcodeName(symbol->symbology), testUtilInputModeName(data[i].input_mode),
                    data[i].option_2, data[i].option_3,
                    data[i].data, data[i].ret, symbol->rows, symbol->width, data[i].bwipp_cmp, data[i].comment);
            testUtilModulesPrint(symbol, "                     ", "\n");
            printf("                },\n");
        } else {
            int width, row;

            assert_equal(symbol->rows, data[i].expected_rows, "i:%d %s symbol->rows %d != %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), symbol->rows, data[i].expected_rows, data[i].data);
            assert_equal(symbol->width, data[i].expected_width, "i:%d %s symbol->width %d != %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), symbol->width, data[i].expected_width, data[i].data);

            ret = testUtilModulesCmp(symbol, data[i].expected, &width, &row);
            assert_zero(ret, "i:%d %s testUtilModulesCmp ret %d != 0 width %d row %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), ret, width, row, data[i].data);

            if (do_bwipp && testUtilCanBwipp(i, symbol, -1, data[i].option_2, data[i].option_3, debug)) {
                if (!data[i].bwipp_cmp) {
                    if (debug & ZINT_DEBUG_TEST_PRINT) printf("i:%d %s not BWIPP compatible (%s)\n", i, testUtilBarcodeName(symbol->symbology), data[i].comment);
                } else {
                    ret = testUtilBwipp(i, symbol, -1, data[i].option_2, data[i].option_3, data[i].data, length, NULL, cmp_buf, sizeof(cmp_buf), NULL);
                    assert_zero(ret, "i:%d %s testUtilBwipp ret %d != 0\n", i, testUtilBarcodeName(symbol->symbology), ret);

                    ret = testUtilBwippCmp(symbol, cmp_msg, cmp_buf, data[i].expected);
                    assert_zero(ret, "i:%d %s testUtilBwippCmp %d != 0 %s\n  actual: %s\nexpected: %s\n",
                                   i, testUtilBarcodeName(symbol->symbology), ret, cmp_msg, cmp_buf, data[i].expected);
                }
            }
            if (do_zxingcpp && testUtilCanZXingCPP(i, symbol, data[i].data, length, debug)) {
                int cmp_len, ret_len;
                char modules_dump[8192 + 1];
                assert_notequal(testUtilModulesDump(symbol, modules_dump, sizeof(modules_dump)), -1, "i:%d testUtilModulesDump == -1\n", i);
                ret = testUtilZXingCPP(i, symbol, data[i].data, length, modules_dump, cmp_buf, sizeof(cmp_buf), &cmp_len);
                assert_zero(ret, "i:%d %s testUtilZXingCPP ret %d != 0\n", i, testUtilBarcodeName(symbol->symbology), ret);

                ret = testUtilZXingCPPCmp(symbol, cmp_msg, cmp_buf, cmp_len, data[i].data, length, NULL /*primary*/, escaped, &ret_len);
                assert_zero(ret, "i:%d %s testUtilZXingCPPCmp %d != 0 %s\n  actual: %.*s\nexpected: %.*s\n",
                               i, testUtilBarcodeName(symbol->symbology), ret, cmp_msg, cmp_len, cmp_buf, ret_len, escaped);
            }
        }

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

/* Test general-purpose data compaction */
static void test_general_field(const testCtx *const p_ctx) {
    int debug = p_ctx->debug;

    struct item {
        int symbology;
        char *data;

        int expected_rows;
        int expected_width;
        char *comment;
        char *expected;
    };
    /* Verified via bwipp_dump.ps against BWIPP and manually against tec-it.com (some separators differ from tec-it.com where noted) */
    struct item data[] = {
        /* 0*/ { BARCODE_DBAR_EXP, "[91]1", 1, 102, "Single numeric",
                    "010100000001000101101111111100001011001000010000010110111110101100001011110000000010101111100001011101"
               },
        /* 1*/ { BARCODE_DBAR_EXP, "[91]12", 1, 102, "2 numerics",
                    "010010000010000101101111111100001011001000010000010101111100101110001011110000000010101111100001011101"
               },
        /* 2*/ { BARCODE_DBAR_EXP, "[91]123", 1, 102, "Odd-numbered numeric",
                    "010100000110000100101111111100001011001000010000010100011111010111001011110000000010000011000110100101"
               },
        /* 3*/ { BARCODE_DBAR_EXP, "[91]1234", 1, 102, "Even-numbered numeric",
                    "010110010000001000101111111100001011001000010000010100111110001011101011110000000010001101111001011101"
               },
        /* 4*/ { BARCODE_DBAR_EXP, "[91]A1234567C", 1, 183, "Alphanumeric followed by 7 digits and alphanumeric",
                    "010100000111001001101111111100001011000001000101110101111110111101001011111100001110100110111110111100001111010110011000111111000010110001000001101101110111101111010010111111110011101"
               },
        /* 5*/ { BARCODE_DBAR_EXP, "[91]A123456C", 1, 151, "Alphanumeric followed by 6 digits and alphanumeric",
                    "0101100111001000001011111111000010110010000100000101011111101111010010111100000011101001101111101111000011110101100110001111110000101000100011000111010"
               },
        /* 6*/ { BARCODE_DBAR_EXP, "[91]A12345B", 1, 151, "Alphanumeric followed by 5 digits and alphanumeric",
                    "0101111001000001001011111111000010110010000100000101011111101111010010111100000011100000010111001001010000111101000010001111110000101100000001001010010"
               },
        /* 7*/ { BARCODE_DBAR_EXP, "[91]A1234567", 1, 151, "Alphanumeric followed by 7 digits, terminating",
                    "0101100100011100001011111111000010110010000100000101011111101111010010111100000011101001101111101111000011110110100110001111110000101101011110111100010"
               },
        /* 8*/ { BARCODE_DBAR_EXP, "[91]A123456", 1, 134, "Alphanumeric followed by 6 digits, terminating",
                    "01000101001100000010111111110000101100000100010111010111111011110100101111000000111010011011111011110000111101011001100011111100001010"
               },
        /* 9*/ { BARCODE_DBAR_EXP, "[91]A12345", 1, 134, "Alphanumeric followed by 5 digits, terminating",
                    "01000110010100000010111111110000101100000100010111010111111011110100101111000000111010011011111011110000101100111110100011111100001010"
               },
        /*10*/ { BARCODE_DBAR_EXP, "[91]A1234", 1, 134, "Alphanumeric followed by 4 digits, terminating",
                    "01011101000010000110111111110000101100000100010111010111111011110100101111000000111010011011111011110001101111100100100011111100001010"
               },
        /*11*/ { BARCODE_DBAR_EXP, "[91]A123", 1, 134, "Alphanumeric followed by 3 digits, terminating",
                    "01000010110010000010111111110000101100000100010111010111111011110100101111000000111000000101110010010001000010000101100011111100001010"
               },
        /*12*/ { BARCODE_DBAR_EXP, "[91]a1234ABCDEFGb", 1, 249, "ISO-646 followed by 11 non-ISO-646 non-terminating, starting 4 digits",
                    "010000100011100110101111111100001011001000010000010101101111110011101011111000000110000110110100011100001110000101011000111111000010100111000010110001000000100110110010111111110011101000001110010001010011011111100110001111000000101110111010011000010"
               },
        /*13*/ { BARCODE_DBAR_EXP, "[91]a1234ABCDEFb", 1, 249, "ISO-646 followed by 10 non-ISO-646 non-terminating, starting 4 digits",
                    "010111010001110001101111111100001011001000010000010101101111110011101011111000000110000110110100011100001110000101011000111111000010100111000010110001000000100110110010111111110011101000001110010001001100011011100110001111000000101101000111001110010"
               },
        /*14*/ { BARCODE_DBAR_EXP, "[91]a1234ABCDEF", 1, 232, "ISO-646 followed by 10 non-ISO-646 terminating, starting 4 digits",
                    "0100001011010000111011111111000010110000010001011101011011111100111010111110000001100001101101000111000011100001010110001111110000101001110000101100010000001001101100101111111100111010000011100100010011100111000101100011110000001010"
               },
        /*15*/ { BARCODE_DBAR_EXP, "[91]a1234ABCDEb", 1, 232, "ISO-646 followed by 9 non-ISO-646 non-terminating, starting 4 digits",
                    "0100001011011000011011111111000010110000010001011101011011111100111010111110000001100100011100100111011110010111110010001111110000101111100011001010010110001000011110101111111100111011100101111000110111001000011110100011110000001010"
               },
        /*16*/ { BARCODE_DBAR_EXP, "[91]a1234ABCDE", 1, 200, "ISO-646 followed by 9 non-ISO-646 terminating, starting 4 digits",
                    "01001000011000111010111111110000101100100001000001010110111111001110101111110000111000011011010001110000111000010101100011111100001010011100001011000100000010011011001011111111001110010011100000100101"
               },
        /*17*/ { BARCODE_DBAR_EXP, "[91]aABCDEF12345b", 1, 249, "ISO-646 followed by 11 non-ISO-646 non-terminating, starting 6 letters",
                    "010000100010100111101111111100001011001000010000010101101111110011101011111000000110100001110001011100010000010100011000111111000010111101000100100001100111010000110010111111110011101000001110011011000100000110101110001111000000101011110010001110010"
               },
        /*18*/ { BARCODE_DBAR_EXP, "[91]aABCDEF1234b", 1, 249, "ISO-646 followed by 10 non-ISO-646 non-terminating, starting 6 letters",
                    "010110111100110000101111111100001011001000010000010101101111110011101011111000000110100001110001011100010000010100011000111111000010111101000100100001100111010000110010111111110011101000001110011011000100001001110110001111000000101111110110110001010"
               },
        /*19*/ { BARCODE_DBAR_EXP, "[91]aABCDE12345b", 1, 249, "ISO-646 followed by 10 non-ISO-646 non-terminating, starting 5 letters",
                    "010000100010011011101111111100001011001000010000010101101111110011101011111000000110100001110001011100010000010100011000111111000010111101000100100001100100001110011010111111110011100110110010000111000101111000101110001111000000101111011001101000010"
               },
        /*20*/ { BARCODE_DBAR_EXP, "[91]aABCDE1234", 1, 200, "ISO-646 followed by 10 non-ISO-646 terminating, starting 5 letters",
                    "01000101100011100010111111110000101100100001000001010110111111001110101111110000111010000111000101110001000001010001100011111100001011110100010010000110000100101111001011111111001110010010001110011101"
               },
        /*21*/ { BARCODE_DBAR_EXP, "[91]aABCDE1234b", 1, 232, "ISO-646 followed by 9 non-ISO-646 non-terminating, starting 5 letters",
                    "0100010000110011011011111111000010110000010001011101011011111100111010111110000001100001000110110111010000010110000010001111110000101000011101001100011001000011100110101111111100111011001100100001110111001000011110100011110000001010"
               },
        /*22*/ { BARCODE_DBAR_EXP, "[91]aABCDE1234", 1, 200, "ISO-646 followed by 9 non-ISO-646 terminating, starting 5 letters",
                    "01000101100011100010111111110000101100100001000001010110111111001110101111110000111010000111000101110001000001010001100011111100001011110100010010000110000100101111001011111111001110010010001110011101"
               },
        /*23*/ { BARCODE_DBAR_EXPSTK, "[91]1", 1, 102, "Single numeric",
                    "010100000001000101101111111100001011001000010000010110111110101100001011110000000010101111100001011101"
               },
        /*24*/ { BARCODE_DBAR_EXPSTK, "[91]12", 1, 102, "2 numerics",
                    "010010000010000101101111111100001011001000010000010101111100101110001011110000000010101111100001011101"
               },
        /*25*/ { BARCODE_DBAR_EXPSTK, "[91]123", 1, 102, "Odd-numbered numeric",
                    "010100000110000100101111111100001011001000010000010100011111010111001011110000000010000011000110100101"
               },
        /*26*/ { BARCODE_DBAR_EXPSTK, "[91]1234", 1, 102, "Even-numbered numeric",
                    "010110010000001000101111111100001011001000010000010100111110001011101011110000000010001101111001011101"
               },
        /*27*/ { BARCODE_DBAR_EXPSTK, "[91]A1234567C", 5, 102, "Alphanumeric followed by 7 digits and alphanumeric **NOTE** bottom separator differs from tec-it, same as BWIPP",
                    "010100000111001001101111111100001011000001000101110101111110111101001011111100001110100110111110111101"
                    "000011111000110110010000000010100100111110111010001010000001000010110100000010100001011001000001000000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000000100000000101101000010000100010010011111011100100101000000101001100101000011000000000000000000000"
                    "101110011111111010010111101111011101101100000100011010000111111000110011010111100001000000000000000000"
               },
        /*28*/ { BARCODE_DBAR_EXPSTK, "[91]A123456C", 5, 102, "Alphanumeric followed by 6 digits and alphanumeric **NOTE** bottom separator differs from tec-it, same as BWIPP",
                    "010110011100100000101111111100001011001000010000010101111110111101001011110000001110100110111110111101"
                    "000001100011011111010000000010100100110111101111101010000001000010110100001010100001011001000001000000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000001100001010011001010000001010010111011100111000000000000000000000000000000000000000000000000000000"
                    "001000011110101100110001111110000101000100011000111010000000000000000000000000000000000000000000000000"
               },
        /*29*/ { BARCODE_DBAR_EXPSTK, "[91]A12345B", 5, 102, "Alphanumeric followed by 5 digits and alphanumeric **NOTE** ditto",
                    "010111100100000100101111111100001011001000010000010101111110111101001011110000001110000001011100100101"
                    "000000011011111011010000000010100100110111101111101010000001000010110100001010100001111110100011010000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000001111000010111101010000001010010011111110110100000000000000000000000000000000000000000000000000000"
                    "001010000111101000010001111110000101100000001001010010000000000000000000000000000000000000000000000000"
               },
        /*30*/ { BARCODE_DBAR_EXPSTK, "[91]A1234567", 5, 102, "Alphanumeric followed by 7 digits, terminating **NOTE** ditto",
                    "010110010001110000101111111100001011001000010000010101111110111101001011110000001110100110111110111101"
                    "000001101110001111010000000010100100110111101111101010000001000010110100001010100001011001000001000000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000001100001001011001010000001010010010100001000010000000000000000000000000000000000000000000000000000"
                    "001000011110110100110001111110000101101011110111100010000000000000000000000000000000000000000000000000"
               },
        /*31*/ { BARCODE_DBAR_EXPSTK, "[91]A123456", 5, 102, "Alphanumeric followed by 6 digits, terminating **NOTE** ditto",
                    "010100001100111000101111111100001011001000010000010101111110111101001011110000001110100110111110111101"
                    "000011110011000111010000000010100100110111101111101010000001000010110100001010100001011001000001000000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000001100001010011001010000001010010011101011111110000000000000000000000000000000000000000000000000000"
                    "001000011110101100110001111110000101100010100000001010000000000000000000000000000000000000000000000000"
               },
        /*32*/ { BARCODE_DBAR_EXPSTK, "[91]A12345", 5, 102, "Alphanumeric followed by 5 digits, terminating **NOTE** ditto",
                    "010100100011000011101111111100001011001000010000010101111110111101001011110000001110100110111110111101"
                    "000011011100111100010000000010100100110111101111101010000001000010110100001010100001011001000001000000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000001101001100000101010000001010010011101011111110000000000000000000000000000000000000000000000000000"
                    "001000010110011111010001111110000101100010100000001010000000000000000000000000000000000000000000000000"
               },
        /*33*/ { BARCODE_DBAR_EXPSTK, "[91]A1234", 5, 102, "Alphanumeric followed by 4 digits, terminating **NOTE** ditto",
                    "010111000010010001101111111100001011001000010000010101111110111101001011110000001110100110111110111101"
                    "000000111101101110010000000010100100110111101111101010000001000010110100001010100001011001000001000000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000001001000001101101010000001010010000111011101110000000000000000000000000000000000000000000000000000"
                    "001000110111110010010001111110000101111000100010001010000000000000000000000000000000000000000000000000"
               },
        /*34*/ { BARCODE_DBAR_EXPSTK, "[91]A123", 5, 102, "Alphanumeric followed by 3 digits, terminating **NOTE** ditto",
                    "010110000100111000101111111100001011001000010000010101111110111101001011110000001110000001011100100101"
                    "000001111011000111010000000010100100110111101111101010000001000010110100001010100001111110100011010000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000001011110111101001010000001010010101111100001010000000000000000000000000000000000000000000000000000"
                    "001000100001000010110001111110000101010000011110100010000000000000000000000000000000000000000000000000"
               },
        /*35*/ { BARCODE_DBAR_EXPSTK, "[91]a1234ABCDEFGb", 9, 102, "ISO-646 followed by 11 non-ISO-646 non-terminating, starting 4 digits **NOTE** separators differ from tec-it, same as BWIPP",
                    "010000100011100110101111111100001011001000010000010101101111110011101011111000000110000110110100011101"
                    "000011011100011001010000000010100100110111101111101010010000001100010100000101010001111001001011100000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000011011000111110100001000000001011001001101111110111001011110001101001010000001010010101111000110000"
                    "101000100111000001011100111111110100110110010000001000110100001110010100001111110001101010000111000010"
                    "000011011000111110100001000000001011001001101111110111001011110001101001010000001010010101111000110000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000011001000000110010100001010100100010001011001100000000000000000000000000000000000000000000000000000"
                    "010100110111111001100011110000001011101110100110000100000000000000000000000000000000000000000000000000"
               },
        /*36*/ { BARCODE_DBAR_EXPSTK, "[91]a1234ABCDEFb", 9, 102, "ISO-646 followed by 10 non-ISO-646 non-terminating, starting 4 digits **NOTE** top separator differs from tec-it, same as BWIPP",
                    "010111010001110001101111111100001011001000010000010101101111110011101011111000000110000110110100011101"
                    "000000101110001110010000000010100100110111101111101010010000001100010100000101010001111001001011100000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000011011000111110100001000000001011001001101111110111001011110001101001010000001010010101111000110000"
                    "101000100111000001011100111111110100110110010000001000110100001110010100001111110001101010000111000010"
                    "000011011000111110100001000000001011001001101111110111001011110001101001010000001010010101111000110000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000000111001000110010100001010100100101110001100000000000000000000000000000000000000000000000000000000"
                    "010011000110111001100011110000001011010001110011100100000000000000000000000000000000000000000000000000"
               },
        /*37*/ { BARCODE_DBAR_EXPSTK, "[91]a1234ABCDEF", 9, 102, "ISO-646 followed by 10 non-ISO-646 terminating, starting 4 digits **NOTE** separators differ from tec-it, same as BWIPP",
                    "010110111001000011101111111100001011001000010000010101101111110011101011111000000110000110110100011101"
                    "000001000110111100010000000010100100110111101111101010010000001100010100000101010001111001001011100000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000011011000111110100001000000001011001001101111110111001011110001101001010000001010010101111000110000"
                    "101000100111000001011100111111110100110110010000001000110100001110010100001111110001101010000111000010"
                    "000011011000111110100001000000001011001001101111110111001011110001101001010000001010010101111000110000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000000011000111010010100001010100101011111000010100000000000000000000000000000000000000000000000000000"
                    "010011100111000101100011110000001010100000111101000100000000000000000000000000000000000000000000000000"
               },
        /*38*/ { BARCODE_DBAR_EXPSTK, "[91]a1234ABCDEb", 9, 102, "ISO-646 followed by 9 non-ISO-646 non-terminating, starting 4 digits **NOTE** ditto",
                    "010110111001100001101111111100001011001000010000010101101111110011101011111000000110010001110010011101"
                    "000001000110011110010000000010100100110111101111101010010000001100010100000101010001101110001101100000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000011100001011000100001000000001010000111101110010110101100111000001001010000001010110000010110000000"
                    "101100011110100111011100111111110101111000010001101001010011000111110100001111110001001111101001111010"
                    "000011100001011000100001000000001010000111101110010110101100111000001001010000001010110000010110000000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000000110111100001010100001010100101011111000010100000000000000000000000000000000000000000000000000000"
                    "010111001000011110100011110000001010100000111101000100000000000000000000000000000000000000000000000000"
               },
        /*39*/ { BARCODE_DBAR_EXPSTK, "[91]a1234ABCDE", 5, 102, "ISO-646 followed by 9 non-ISO-646 terminating, starting 4 digits **NOTE** bottom separator differs from tec-it, same as BWIPP",
                    "010010000110001110101111111100001011001000010000010101101111110011101011111100001110000110110100011101"
                    "000001111001110001010000000010100100110111101111101010010000001100010100000010100001111001001011100000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000010111110001101100001000000001011001001101111110111001011110001101001010000001010010101111000110000"
                    "101001000001110010011100111111110100110110010000001000110100001110010100001111110001101010000111000010"
               },
        /*40*/ { BARCODE_DBAR_EXPSTK, "[91]aABCDEF12345b", 9, 102, "ISO-646 followed by 11 non-ISO-646 non-terminating, starting 6 letters **NOTE** top separator differs from tec-it, same as BWIPP",
                    "010000100010100111101111111100001011001000010000010101101111110011101011111000000110100001110001011101"
                    "000011011101011000010000000010100100110111101111101010010000001100010100000101010001011110001110100000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000010011000111110100001000000001011001111010001100111101101110100001001010000001010011101011111010000"
                    "101101100111000001011100111111110100110000101110011000010010001011110100001111110001100010100000100010"
                    "000010011000111110100001000000001011001111010001100111101101110100001001010000001010011101011111010000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000010111110010100010100001010100101000011011100000000000000000000000000000000000000000000000000000000"
                    "010001000001101011100011110000001010111100100011100100000000000000000000000000000000000000000000000000"
               },
        /*41*/ { BARCODE_DBAR_EXPSTK, "[91]aABCDEF1234b", 9, 102, "ISO-646 followed by 10 non-ISO-646 non-terminating, starting 6 letters **NOTE** separators differ from tec-it, same as BWIPP",
                    "010110111100110000101111111100001011001000010000010101101111110011101011111000000110100001110001011101"
                    "000001000011001111010000000010100100110111101111101010010000001100010100000101010001011110001110100000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000010011000111110100001000000001011001111010001100111101101110100001001010000001010011101011111010000"
                    "101101100111000001011100111111110100110000101110011000010010001011110100001111110001100010100000100010"
                    "000010011000111110100001000000001011001111010001100111101101110100001001010000001010011101011111010000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000010111101100010010100001010100100000010010011100000000000000000000000000000000000000000000000000000"
                    "010001000010011101100011110000001011111101101100010100000000000000000000000000000000000000000000000000"
               },
        /*42*/ { BARCODE_DBAR_EXPSTK, "[91]aABCDE12345b", 9, 102, "ISO-646 followed by 10 non-ISO-646 non-terminating, starting 5 letters **NOTE** ditto",
                    "010000100010011011101111111100001011001000010000010101101111110011101011111000000110100001110001011101"
                    "000011011101100100010000000010100100110111101111101010010000001100010100000101010001011110001110100000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000001111011001001100001000000001010011000111101100111101101110100001001010000001010011101011111010000"
                    "101110000100110110011100111111110101100111000010011000010010001011110100001111110001100010100000100010"
                    "000001111011001001100001000000001010011000111101100111101101110100001001010000001010011101011111010000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000010100001110100010100001010100100001001100101100000000000000000000000000000000000000000000000000000"
                    "010001011110001011100011110000001011110110011010000100000000000000000000000000000000000000000000000000"
               },
        /*43*/ { BARCODE_DBAR_EXPSTK, "[91]aABCDE1234", 5, 102, "ISO-646 followed by 10 non-ISO-646 terminating, starting 5 letters **NOTE** bottom separator differs from tec-it, same as BWIPP",
                    "010001011000111000101111111100001011001000010000010101101111110011101011111100001110100001110001011101"
                    "000010100111000111010000000010100100110111101111101010010000001100010100000010100001011110001110100000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000001100011101101100001000000001011000010110111100111101101110100001001010000001010011101011111010000"
                    "101110011100010010011100111111110100111101001000011000010010001011110100001111110001100010100000100010"
               },
        /*44*/ { BARCODE_DBAR_EXPSTK, "[91]aABCDE1234b", 9, 102, "ISO-646 followed by 9 non-ISO-646 non-terminating, starting 5 letters **NOTE** separators differ from tec-it, same as BWIPP",
                    "010000100001110110101111111100001011001000010000010101101111110011101011111000000110000100011011011101"
                    "000011011110001001010000000010100100110111101111101010010000001100010100000101010001111011100100100000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000001111011001100100001000000001010011000111101100111001101000111101001010000001010111110010111110000"
                    "101110000100110011011100111111110101100111000010011000110010111000010100001111110001000001101000001010"
                    "000001111011001100100001000000001010011000111101100111001101000111101001010000001010111110010111110000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000000110111100001010100001010100101011111000010100000000000000000000000000000000000000000000000000000"
                    "010111001000011110100011110000001010100000111101000100000000000000000000000000000000000000000000000000"
               },
        /*45*/ { BARCODE_DBAR_EXPSTK, "[91]aABCDE1234", 5, 102, "ISO-646 followed by 9 non-ISO-646 terminating, starting 5 letters **NOTE** bottom separator differs from tec-it, same as BWIPP",
                    "010001011000111000101111111100001011001000010000010101101111110011101011111100001110100001110001011101"
                    "000010100111000111010000000010100100110111101111101010010000001100010100000010100001011110001110100000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000001100011101101100001000000001011000010110111100111101101110100001001010000001010011101011111010000"
                    "101110011100010010011100111111110100111101001000011000010010001011110100001111110001100010100000100010"
               },
    };
    int data_size = ARRAY_SIZE(data);
    int i, length, ret;
    struct zint_symbol *symbol;

    testStart("test_general_field");

    for (i = 0; i < data_size; i++) {

        if (testContinue(p_ctx, i)) continue;

        symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        length = testUtilSetSymbol(symbol, data[i].symbology, -1 /*input_mode*/, -1 /*eci*/, -1 /*option_1*/, -1, -1, -1 /*output_options*/, data[i].data, -1, debug);

        ret = ZBarcode_Encode(symbol, (const unsigned char *) data[i].data, length);
        assert_zero(ret, "i:%d ZBarcode_Encode ret %d != 0 (%s)\n", i, ret, symbol->errtxt);

        if (p_ctx->generate) {
            printf("        /*%2d*/ { %s, \"%s\", %d, %d, %d, \"%s\",\n",
                    i, testUtilBarcodeName(symbol->symbology), data[i].data, ret, symbol->rows, symbol->width, data[i].comment);
            testUtilModulesPrint(symbol, "                    ", "\n");
            printf("               },\n");
        } else {
            int width, row;

            assert_equal(symbol->rows, data[i].expected_rows, "i:%d %s symbol->rows %d != %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), symbol->rows, data[i].expected_rows, data[i].data);
            assert_equal(symbol->width, data[i].expected_width, "i:%d %s symbol->width %d != %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), symbol->width, data[i].expected_width, data[i].data);

            ret = testUtilModulesCmp(symbol, data[i].expected, &width, &row);
            assert_zero(ret, "i:%d %s testUtilModulesCmp ret %d != 0 width %d row %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), ret, width, row, data[i].data);
        }

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

static void test_binary_buffer_size(const testCtx *const p_ctx) {
    int debug = p_ctx->debug;

    struct item {
        int input_mode;
        char *data;
        int ret;

        int expected_rows;
        int expected_width;
        char *comment;
    };
    struct item data[] = {
        /*  0*/ { -1, "[91]1", 0, 1, 102, "Minimum digit" },
        /*  1*/ { -1, "[91]+", 0, 1, 102, "Minimum ISO-646" },
        /*  2*/ { -1, "[00]123456789012345675[00]123456789012345675[00]123456789012345675[91]12345678", 0, 1, 543, "70 == any AIs max" },
        /*  3*/ { -1, "[00]123456789012345675[00]123456789012345675[00]123456789012345675[91]123456789", ZINT_ERROR_TOO_LONG, 0, 0, "71 > any AIs max" },
        /*  4*/ { GS1NOCHECK_MODE, "[00]123456789012345675[00]123456789012345675[00]123456789012345675[91]123456789", ZINT_ERROR_TOO_LONG, 0, 0, "No check doesn't affect limit" },
        /*  5*/ { -1, "[01]12345678901231[00]123456789012345675[00]123456789012345675[91]1234567890123456", 0, 1, 543, "74 == 01 + other AIs max" },
        /*  6*/ { -1, "[01]12345678901231[00]123456789012345675[00]123456789012345675[91]12345678901234567", ZINT_ERROR_TOO_LONG, 0, 0, "75 > 01 + other AIs max" },
        /*  7*/ { GS1NOCHECK_MODE, "[01]12345678901231[00]123456789012345675[00]123456789012345675[91]12345678901234567", ZINT_ERROR_TOO_LONG, 0, 0, "No check doesn't affect limit" },
        /*  8*/ { -1, "[01]92345678901237[3920]123456789012345[00]123456789012345675[91]1234567890123456789", 0, 1, 543, "77 (incl. FNC1 after 3920) == 01 + 392x + other AIs max" },
        /*  9*/ { -1, "[01]92345678901237[3920]123456789012345[00]123456789012345675[91]12345678901234567890", ZINT_ERROR_TOO_LONG, 0, 0, "78 > 01 + 392x + other AIs max" },
        /* 10*/ { GS1NOCHECK_MODE, "[01]92345678901237[3920]123456789012345[00]123456789012345675[91]12345678901234567890", ZINT_ERROR_TOO_LONG, 0, 0, "No check doesn't affect limit" },
    };
    int data_size = ARRAY_SIZE(data);
    int i, length, ret;
    struct zint_symbol *symbol;

    testStart("test_binary_buffer_size");

    for (i = 0; i < data_size; i++) {

        if (testContinue(p_ctx, i)) continue;

        symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        length = testUtilSetSymbol(symbol, BARCODE_DBAR_EXP, data[i].input_mode, -1 /*eci*/, -1 /*option_1*/, -1, -1, -1 /*output_options*/, data[i].data, -1, debug);

        ret = ZBarcode_Encode(symbol, (const unsigned char *) data[i].data, length);
        assert_equal(ret, data[i].ret, "i:%d ZBarcode_Encode ret %d != %d (%s)\n", i, ret, data[i].ret, symbol->errtxt);

        if (p_ctx->generate) {
            printf("        /*%3d*/ { %s, \"%s\", %s, %d, %d, \"%s\" },\n",
                    i, testUtilInputModeName(data[i].input_mode), data[i].data, testUtilErrorName(ret), symbol->rows, symbol->width, data[i].comment);
        } else {
            assert_equal(symbol->rows, data[i].expected_rows, "i:%d symbol->rows %d != %d (%s)\n", i, symbol->rows, data[i].expected_rows, data[i].data);
            assert_equal(symbol->width, data[i].expected_width, "i:%d symbol->width %d != %d (%s)\n", i, symbol->width, data[i].expected_width, data[i].data);
        }

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

static void test_hrt(const testCtx *const p_ctx) {
    int debug = p_ctx->debug;

    struct item {
        int symbology;
        int input_mode;
        char *data;

        int ret;
        char *expected;
    };
    /* s/\/\*[ 0-9]*\*\//\=printf("\/\*%3d*\/", line(".") - line("'<")): */
    struct item data[] = {
        /*  0*/ { BARCODE_DBAR_OMN, -1, "1234567890123", 0, "(01)12345678901231" },
        /*  1*/ { BARCODE_DBAR_OMN, -1, "12345678901231", 0, "(01)12345678901231" },
        /*  4*/ { BARCODE_DBAR_OMN, -1, "1000000000009", 0, "(01)10000000000090" },
        /*  5*/ { BARCODE_DBAR_LTD, -1, "1341056790138", 0, "(01)13410567901384" },
        /*  6*/ { BARCODE_DBAR_LTD, -1, "13410567901384", 0, "(01)13410567901384" },
        /*  9*/ { BARCODE_DBAR_EXP, -1, "[01]12345678901231", 0, "(01)12345678901231" }, /* See test_hrt() in "test_gs1.c" for full HRT tests */
        /* 10*/ { BARCODE_DBAR_STK, -1, "12345678901231", 0, "" }, /* No HRT for stacked */
        /* 11*/ { BARCODE_DBAR_OMNSTK, -1, "10000000000090", 0, "" },
        /* 12*/ { BARCODE_DBAR_EXPSTK, -1, "[01]12345678901231", 0, "" },
    };
    int data_size = ARRAY_SIZE(data);
    int i, length, ret;
    struct zint_symbol *symbol;

    testStart("test_hrt");

    for (i = 0; i < data_size; i++) {

        if (testContinue(p_ctx, i)) continue;

        symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        length = testUtilSetSymbol(symbol, data[i].symbology, data[i].input_mode, -1 /*eci*/, -1 /*option_1*/, -1, -1, -1 /*output_options*/, data[i].data, -1, debug);

        ret = ZBarcode_Encode(symbol, (const unsigned char *) data[i].data, length);
        assert_equal(ret, data[i].ret, "i:%d ZBarcode_Encode ret %d != %d (%s)\n", i, data[i].ret, ret, symbol->errtxt);

        assert_zero(strcmp((const char *) symbol->text, data[i].expected), "i:%d strcmp(%s, %s) != 0\n", i, symbol->text, data[i].expected);

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

static void test_input(const testCtx *const p_ctx) {
    int debug = p_ctx->debug;

    struct item {
        int symbology;
        int input_mode;
        int option_2;
        int option_3;
        char *data;
        int ret;
        int expected_rows;
        int expected_width;
        char *expected_errtxt;
    };
    /* s/\/\*[ 0-9]*\*\//\=printf("\/\*%3d*\/", line(".") - line("'<")): */
    struct item data[] = {
        /*  0*/ { BARCODE_DBAR_OMN, -1, -1, -1, "1234567890123", 0, 1, 96, "" },
        /*  1*/ { BARCODE_DBAR_OMN, -1, -1, -1, "123456789012A", ZINT_ERROR_INVALID_DATA, -1, -1, "Error 381: Invalid character in data (digits only)" },
        /*  2*/ { BARCODE_DBAR_OMN, GS1NOCHECK_MODE, -1, -1, "123456789012A", ZINT_ERROR_INVALID_DATA, -1, -1, "Error 381: Invalid character in data (digits only)" },
        /*  3*/ { BARCODE_DBAR_OMN, -1, -1, -1, "12345678901234", ZINT_ERROR_INVALID_CHECK, -1, -1, "Error 388: Invalid check digit '4', expecting '1'" },
        /*  4*/ { BARCODE_DBAR_OMN, GS1NOCHECK_MODE, -1, -1, "12345678901234", ZINT_ERROR_INVALID_CHECK, -1, -1, "Error 388: Invalid check digit '4', expecting '1'" }, /* Still checked */
        /*  5*/ { BARCODE_DBAR_OMN, -1, -1, -1, "123456789012315", ZINT_ERROR_TOO_LONG, -1, -1, "Error 380: Input too long (14 character maximum)" },
        /*  6*/ { BARCODE_DBAR_OMN, GS1NOCHECK_MODE, -1, -1, "123456789012315", ZINT_ERROR_TOO_LONG, -1, -1, "Error 380: Input too long (14 character maximum)" },
        /*  7*/ { BARCODE_DBAR_LTD, -1, -1, -1, "1234567890123", 0, 1, 79, "" },
        /*  8*/ { BARCODE_DBAR_LTD, -1, -1, -1, "123456789012A", ZINT_ERROR_INVALID_DATA, -1, -1, "Error 383: Invalid character in data (digits only)" },
        /*  9*/ { BARCODE_DBAR_LTD, GS1NOCHECK_MODE, -1, -1, "123456789012A", ZINT_ERROR_INVALID_DATA, -1, -1, "Error 383: Invalid character in data (digits only)" },
        /* 10*/ { BARCODE_DBAR_LTD, -1, -1, -1, "12345678901235", ZINT_ERROR_INVALID_CHECK, -1, -1, "Error 389: Invalid check digit '5', expecting '1'" },
        /* 11*/ { BARCODE_DBAR_LTD, GS1NOCHECK_MODE, -1, -1, "12345678901235", ZINT_ERROR_INVALID_CHECK, -1, -1, "Error 389: Invalid check digit '5', expecting '1'" }, /* Still checked */
        /* 12*/ { BARCODE_DBAR_LTD, -1, -1, -1, "123456789012315", ZINT_ERROR_TOO_LONG, -1, -1, "Error 382: Input too long (14 character maximum)" },
        /* 13*/ { BARCODE_DBAR_LTD, GS1NOCHECK_MODE, -1, -1, "123456789012315", ZINT_ERROR_TOO_LONG, -1, -1, "Error 382: Input too long (14 character maximum)" },
        /* 14*/ { BARCODE_DBAR_LTD, -1, -1, -1, "2234567890123", ZINT_ERROR_INVALID_DATA, -1, -1, "Error 384: Input out of range (0 to 1999999999999)" },
        /* 15*/ { BARCODE_DBAR_LTD, GS1NOCHECK_MODE, -1, -1, "2234567890123", ZINT_ERROR_INVALID_DATA, -1, -1, "Error 384: Input out of range (0 to 1999999999999)" },
        /* 16*/ { BARCODE_DBAR_LTD, -1, -1, -1, "22345678901238", ZINT_ERROR_INVALID_DATA, -1, -1, "Error 384: Input out of range (0 to 1999999999999)" },
        /* 17*/ { BARCODE_DBAR_LTD, GS1NOCHECK_MODE, -1, -1, "22345678901238", ZINT_ERROR_INVALID_DATA, -1, -1, "Error 384: Input out of range (0 to 1999999999999)" },
        /* 18*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]12345678901234", ZINT_WARN_NONCOMPLIANT, 1, 134, "Warning 261: AI (01) position 14: Bad checksum '4', expected '1'" },
        /* 19*/ { BARCODE_DBAR_EXP, GS1NOCHECK_MODE, -1, -1, "[01]12345678901234", 0, 1, 134, "" },
        /* 20*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]12345678901231", 0, 1, 134, "" },
        /* 21*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]1234567890123A", ZINT_ERROR_INVALID_DATA, -1, -1, "Error 385: Invalid character in Compressed Field data (digits only)" },
        /* 22*/ { BARCODE_DBAR_EXP, GS1NOCHECK_MODE, -1, -1, "[01]1234567890123A", ZINT_ERROR_INVALID_DATA, -1, -1, "Error 385: Invalid character in Compressed Field data (digits only)" },
        /* 23*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]123456789012315", ZINT_ERROR_INVALID_DATA, -1, -1, "Error 259: Invalid data length for AI (01)" },
        /* 24*/ { BARCODE_DBAR_EXP, GS1NOCHECK_MODE, -1, -1, "[01]123456789012315", 0, 1, 151, "" },
        /* 25*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]12345678901234", ZINT_WARN_NONCOMPLIANT, 1, 134, "Warning 261: AI (01) position 14: Bad checksum '4', expected '1'" },
        /* 26*/ { BARCODE_DBAR_EXP, GS1NOCHECK_MODE, -1, -1, "[01]12345678901234", 0, 1, 134, "" },
        /* 27*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]12345678901231[91]!\"%&'()*+,-./:;<=>?_ ", ZINT_WARN_NONCOMPLIANT, 1, 526, "Warning 261: AI (91) position 21: Invalid CSET 82 character ' '" }, /* ISOIEC punc */
        /* 28*/ { BARCODE_DBAR_EXP, GS1NOCHECK_MODE, -1, -1, "[01]12345678901231[91]!\"%&'()*+,-./:;<=>?_ ", 0, 1, 526, "" },
        /* 29*/ { BARCODE_DBAR_EXP, -1, -1, -1, "[01]12345678901231[91]!\"%&'()*+,-./:;<=>?_", 0, 1, 494, "" }, /* ISOIEC punc less space */
        /* 30*/ { BARCODE_DBAR_EXP, GS1NOCHECK_MODE, -1, -1, "[01]12345678901231[91]!\"%&'()*+,-./:;<=>?_", 0, 1, 494, "" },
        /* 31*/ { BARCODE_DBAR_STK, -1, -1, -1, "1234567890123", 0, 3, 50, "" },
        /* 32*/ { BARCODE_DBAR_STK, GS1NOCHECK_MODE, -1, -1, "123456789012A", ZINT_ERROR_INVALID_DATA, -1, -1, "Error 381: Invalid character in data (digits only)" },
        /* 33*/ { BARCODE_DBAR_STK, -1, -1, -1, "123456789012A", ZINT_ERROR_INVALID_DATA, -1, -1, "Error 381: Invalid character in data (digits only)" },
        /* 34*/ { BARCODE_DBAR_STK, GS1NOCHECK_MODE, -1, -1, "123456789012A", ZINT_ERROR_INVALID_DATA, -1, -1, "Error 381: Invalid character in data (digits only)" },
        /* 35*/ { BARCODE_DBAR_STK, -1, -1, -1, "12345678901235", ZINT_ERROR_INVALID_CHECK, -1, -1, "Error 388: Invalid check digit '5', expecting '1'" },
        /* 36*/ { BARCODE_DBAR_STK, GS1NOCHECK_MODE, -1, -1, "12345678901235", ZINT_ERROR_INVALID_CHECK, -1, -1, "Error 388: Invalid check digit '5', expecting '1'" }, /* Still checked */
        /* 37*/ { BARCODE_DBAR_STK, -1, -1, -1, "123456789012315", ZINT_ERROR_TOO_LONG, -1, -1, "Error 380: Input too long (14 character maximum)" },
        /* 38*/ { BARCODE_DBAR_STK, GS1NOCHECK_MODE, -1, -1, "123456789012315", ZINT_ERROR_TOO_LONG, -1, -1, "Error 380: Input too long (14 character maximum)" },
        /* 39*/ { BARCODE_DBAR_OMNSTK, -1, -1, -1, "1234567890123", 0, 5, 50, "" },
        /* 40*/ { BARCODE_DBAR_OMNSTK, -1, -1, -1, "123456789012A", ZINT_ERROR_INVALID_DATA, -1, -1, "Error 381: Invalid character in data (digits only)" },
        /* 41*/ { BARCODE_DBAR_OMNSTK, GS1NOCHECK_MODE, -1, -1, "123456789012A", ZINT_ERROR_INVALID_DATA, -1, -1, "Error 381: Invalid character in data (digits only)" },
        /* 42*/ { BARCODE_DBAR_OMNSTK, -1, -1, -1, "12345678901236", ZINT_ERROR_INVALID_CHECK, -1, -1, "Error 388: Invalid check digit '6', expecting '1'" },
        /* 43*/ { BARCODE_DBAR_OMNSTK, GS1NOCHECK_MODE, -1, -1, "12345678901236", ZINT_ERROR_INVALID_CHECK, -1, -1, "Error 388: Invalid check digit '6', expecting '1'" }, /* Still checked */
        /* 44*/ { BARCODE_DBAR_OMNSTK, -1, -1, -1, "123456789012315", ZINT_ERROR_TOO_LONG, -1, -1, "Error 380: Input too long (14 character maximum)" },
        /* 45*/ { BARCODE_DBAR_OMNSTK, GS1NOCHECK_MODE, -1, -1, "123456789012315", ZINT_ERROR_TOO_LONG, -1, -1, "Error 380: Input too long (14 character maximum)" },
        /* 46*/ { BARCODE_DBAR_EXPSTK, -1, -1, -1, "[01]12345678901234", ZINT_WARN_NONCOMPLIANT, 5, 102, "Warning 261: AI (01) position 14: Bad checksum '4', expected '1'" },
        /* 47*/ { BARCODE_DBAR_EXPSTK, GS1NOCHECK_MODE, -1, -1, "[01]12345678901234", 0, 5, 102, "" },
        /* 48*/ { BARCODE_DBAR_EXPSTK, -1, -1, -1, "[01]12345678901231", 0, 5, 102, "" },
        /* 49*/ { BARCODE_DBAR_EXPSTK, -1, -1, -1, "[01]1234567890123A", ZINT_ERROR_INVALID_DATA, -1, -1, "Error 385: Invalid character in Compressed Field data (digits only)" },
        /* 50*/ { BARCODE_DBAR_EXPSTK, GS1NOCHECK_MODE, -1, -1, "[01]1234567890123A", ZINT_ERROR_INVALID_DATA, -1, -1, "Error 385: Invalid character in Compressed Field data (digits only)" },
        /* 51*/ { BARCODE_DBAR_EXPSTK, -1, -1, -1, "[01]123456789012315", ZINT_ERROR_INVALID_DATA, -1, -1, "Error 259: Invalid data length for AI (01)" },
        /* 52*/ { BARCODE_DBAR_EXPSTK, GS1NOCHECK_MODE, -1, -1, "[01]123456789012315", 0, 5, 102, "" },
        /* 53*/ { BARCODE_DBAR_EXPSTK, -1, 12, -1, "[01]12345678901231", 0, 5, 102, "" }, /* Cols > 11 ignored */
        /* 54*/ { BARCODE_DBAR_EXPSTK, -1, -1, 12, "[01]12345678901231", 0, 5, 102, "" }, /* Rows > 11 ignored */
        /* 55*/ { BARCODE_DBAR_EXPSTK, -1, 1, -1, "[01]12345678901231", 0, 9, 53, "" },
        /* 56*/ { BARCODE_DBAR_EXPSTK, -1, 2, -1, "[01]12345678901231", 0, 5, 102, "" },
        /* 57*/ { BARCODE_DBAR_EXPSTK, -1, 3, -1, "[01]12345678901231", 0, 1, 134, "" },
        /* 58*/ { BARCODE_DBAR_EXPSTK, -1, 4, -1, "[01]12345678901231", 0, 1, 134, "" },
        /* 59*/ { BARCODE_DBAR_EXPSTK, -1, -1, 2, "[01]12345678901231", 0, 5, 102, "" },
        /* 60*/ { BARCODE_DBAR_EXPSTK, -1, -1, 3, "[01]12345678901231", 0, 5, 102, "" },
        /* 61*/ { BARCODE_DBAR_EXPSTK, -1, -1, -1, "[8110]106141416543213500110000310123196000", 0, 13, 102, "" },
        /* 62*/ { BARCODE_DBAR_EXPSTK, -1, 1, -1, "[8110]106141416543213500110000310123196000", 0, 25, 53, "" },
        /* 63*/ { BARCODE_DBAR_EXPSTK, -1, 2, -1, "[8110]106141416543213500110000310123196000", 0, 13, 102, "" },
        /* 64*/ { BARCODE_DBAR_EXPSTK, -1, 3, -1, "[8110]106141416543213500110000310123196000", 0, 9, 151, "" },
        /* 65*/ { BARCODE_DBAR_EXPSTK, -1, 4, -1, "[8110]106141416543213500110000310123196000", 0, 5, 200, "" },
        /* 66*/ { BARCODE_DBAR_EXPSTK, -1, 5, -1, "[8110]106141416543213500110000310123196000", 0, 5, 249, "" },
        /* 67*/ { BARCODE_DBAR_EXPSTK, -1, -1, 2, "[8110]106141416543213500110000310123196000", 0, 5, 200, "" },
        /* 68*/ { BARCODE_DBAR_EXPSTK, -1, -1, 3, "[8110]106141416543213500110000310123196000", 0, 9, 151, "" },
        /* 69*/ { BARCODE_DBAR_EXPSTK, -1, -1, 4, "[8110]106141416543213500110000310123196000", 0, 13, 102, "" },
        /* 70*/ { BARCODE_DBAR_EXPSTK, -1, -1, 5, "[8110]106141416543213500110000310123196000", 0, 13, 102, "" },
        /* 71*/ { BARCODE_DBAR_EXPSTK, -1, -1, -1, "[91]123456789012345678901", 0, 9, 102, "" },
        /* 72*/ { BARCODE_DBAR_EXPSTK, -1, 1, -1, "[91]123456789012345678901", 0, 17, 53, "" },
        /* 73*/ { BARCODE_DBAR_EXPSTK, -1, 2, -1, "[91]123456789012345678901", 0, 9, 102, "" },
        /* 74*/ { BARCODE_DBAR_EXPSTK, -1, 3, -1, "[91]123456789012345678901", 0, 5, 151, "" },
        /* 75*/ { BARCODE_DBAR_EXPSTK, -1, 4, -1, "[91]123456789012345678901", 0, 5, 200, "" },
        /* 76*/ { BARCODE_DBAR_EXPSTK, -1, -1, 2, "[91]123456789012345678901", 0, 5, 151, "" },
        /* 77*/ { BARCODE_DBAR_EXPSTK, -1, -1, 3, "[91]123456789012345678901", 0, 9, 102, "" },
        /* 78*/ { BARCODE_DBAR_EXPSTK, -1, -1, 4, "[91]123456789012345678901", 0, 9, 102, "" },
        /* 79*/ { BARCODE_DBAR_EXPSTK, -1, -1, -1, "[91]123456789012345678901", 0, 9, 102, "" },
        /* 80*/ { BARCODE_DBAR_EXPSTK, -1, -1, -1, "[91]1234567890123456789012345678901234567890123456789", 0, 17, 102, "" },
        /* 81*/ { BARCODE_DBAR_EXPSTK, -1, 1, -1, "[91]1234567890123456789012345678901234567890123456789", 0, 33, 53, "" },
        /* 82*/ { BARCODE_DBAR_EXPSTK, -1, 2, -1, "[91]1234567890123456789012345678901234567890123456789", 0, 17, 102, "" },
        /* 83*/ { BARCODE_DBAR_EXPSTK, -1, 3, -1, "[91]1234567890123456789012345678901234567890123456789", 0, 9, 151, "" },
        /* 84*/ { BARCODE_DBAR_EXPSTK, -1, 4, -1, "[91]1234567890123456789012345678901234567890123456789", 0, 9, 200, "" },
        /* 85*/ { BARCODE_DBAR_EXPSTK, -1, 4, -1, "[91]1234567890123456789012345678901234567890123456789", 0, 9, 200, "" },
        /* 86*/ { BARCODE_DBAR_EXPSTK, -1, 5, -1, "[91]1234567890123456789012345678901234567890123456789", 0, 5, 249, "" },
        /* 87*/ { BARCODE_DBAR_EXPSTK, -1, 6, -1, "[91]1234567890123456789012345678901234567890123456789", 0, 5, 298, "" },
        /* 88*/ { BARCODE_DBAR_EXPSTK, -1, 7, -1, "[91]1234567890123456789012345678901234567890123456789", 0, 5, 347, "" },
        /* 89*/ { BARCODE_DBAR_EXPSTK, -1, 8, -1, "[91]1234567890123456789012345678901234567890123456789", 0, 5, 396, "" },
        /* 90*/ { BARCODE_DBAR_EXPSTK, -1, 9, -1, "[91]1234567890123456789012345678901234567890123456789", 0, 1, 428, "" },
        /* 91*/ { BARCODE_DBAR_EXPSTK, -1, -1, 2, "[91]1234567890123456789012345678901234567890123456789", 0, 5, 249, "" },
        /* 92*/ { BARCODE_DBAR_EXPSTK, -1, -1, 3, "[91]1234567890123456789012345678901234567890123456789", 0, 9, 151, "" },
        /* 93*/ { BARCODE_DBAR_EXPSTK, -1, -1, 4, "[91]1234567890123456789012345678901234567890123456789", 0, 9, 151, "" },
        /* 94*/ { BARCODE_DBAR_EXPSTK, -1, -1, 5, "[91]1234567890123456789012345678901234567890123456789", 0, 17, 102, "" },
        /* 95*/ { BARCODE_DBAR_EXPSTK, -1, -1, 6, "[91]1234567890123456789012345678901234567890123456789", 0, 17, 102, "" },
        /* 96*/ { BARCODE_DBAR_EXPSTK, -1, -1, -1, "[91]12345678901234567890123456789012345678901234567890123456789012", 0, 21, 102, "" },
        /* 97*/ { BARCODE_DBAR_EXPSTK, -1, 1, -1, "[91]12345678901234567890123456789012345678901234567890123456789012", 0, 41, 53, "" },
        /* 98*/ { BARCODE_DBAR_EXPSTK, -1, 2, -1, "[91]12345678901234567890123456789012345678901234567890123456789012", 0, 21, 102, "" },
        /* 99*/ { BARCODE_DBAR_EXPSTK, -1, 3, -1, "[91]12345678901234567890123456789012345678901234567890123456789012", 0, 13, 151, "" },
        /*100*/ { BARCODE_DBAR_EXPSTK, -1, 4, -1, "[91]12345678901234567890123456789012345678901234567890123456789012", 0, 9, 200, "" },
        /*101*/ { BARCODE_DBAR_EXPSTK, -1, 5, -1, "[91]12345678901234567890123456789012345678901234567890123456789012", 0, 9, 249, "" },
        /*102*/ { BARCODE_DBAR_EXPSTK, -1, 6, -1, "[91]12345678901234567890123456789012345678901234567890123456789012", 0, 5, 298, "" },
        /*103*/ { BARCODE_DBAR_EXPSTK, -1, 7, -1, "[91]12345678901234567890123456789012345678901234567890123456789012", 0, 5, 347, "" },
        /*104*/ { BARCODE_DBAR_EXPSTK, -1, -1, 1, "[91]12345678901234567890123456789012345678901234567890123456789012", 0, 21, 102, "" },
        /*105*/ { BARCODE_DBAR_EXPSTK, -1, -1, 2, "[91]12345678901234567890123456789012345678901234567890123456789012", 0, 5, 298, "" },
        /*106*/ { BARCODE_DBAR_EXPSTK, -1, -1, 3, "[91]12345678901234567890123456789012345678901234567890123456789012", 0, 9, 200, "" },
        /*107*/ { BARCODE_DBAR_EXPSTK, -1, -1, 4, "[91]12345678901234567890123456789012345678901234567890123456789012", 0, 13, 151, "" },
        /*108*/ { BARCODE_DBAR_EXPSTK, -1, -1, 5, "[91]12345678901234567890123456789012345678901234567890123456789012", 0, 13, 151, "" },
        /*109*/ { BARCODE_DBAR_EXPSTK, -1, -1, 6, "[91]12345678901234567890123456789012345678901234567890123456789012", 0, 21, 102, "" },
        /*110*/ { BARCODE_DBAR_EXPSTK, -1, -1, 7, "[91]12345678901234567890123456789012345678901234567890123456789012", 0, 21, 102, "" },
    };
    int data_size = ARRAY_SIZE(data);
    int i, length, ret;
    struct zint_symbol *symbol;

    testStart("test_input");

    for (i = 0; i < data_size; i++) {

        if (testContinue(p_ctx, i)) continue;

        symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        length = testUtilSetSymbol(symbol, data[i].symbology, data[i].input_mode, -1 /*eci*/, -1 /*option_1*/, data[i].option_2, data[i].option_3, -1 /*output_options*/, data[i].data, -1, debug);

        ret = ZBarcode_Encode(symbol, (const unsigned char *) data[i].data, length);
        assert_equal(ret, data[i].ret, "i:%d ZBarcode_Encode ret %d != %d (%s)\n", i, ret, data[i].ret, symbol->errtxt);
        assert_zero(strcmp(symbol->errtxt, data[i].expected_errtxt), "i:%d strcmp(%s, %s) != 0\n", i, symbol->errtxt, data[i].expected_errtxt);

        if (ret < ZINT_ERROR) {
            assert_equal(symbol->rows, data[i].expected_rows, "i:%d symbol->rows %d != %d (symbol->width %d)\n", i, symbol->rows, data[i].expected_rows, symbol->width);
            assert_equal(symbol->width, data[i].expected_width, "i:%d symbol->width %d != %d\n", i, symbol->width, data[i].expected_width);
        }

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

int main(int argc, char *argv[]) {

    testFunction funcs[] = { /* name, func */
        { "test_binary_div_modulo_divisor", test_binary_div_modulo_divisor },
        { "test_examples", test_examples },
        { "test_general_field", test_general_field },
        { "test_binary_buffer_size", test_binary_buffer_size },
        { "test_hrt", test_hrt },
        { "test_input", test_input },
    };

    testRun(argc, argv, funcs, ARRAY_SIZE(funcs));

    testReport();

    return 0;
}

/* vim: set ts=4 sw=4 et : */
