/*
    $Id: opcodes.c 2709 2021-09-18 18:40:01Z soci $

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

*/
#include "opcodes.h"

static const uint8_t opcode_w65816[] = {
    0, 1, 2, 3, 4, 5, 4, 6, 3, 7, 8, 9, 10, 11, 12, 13,
    14, 15, 16, 17, 18, 19, 20, 21, 22, 21, 23, 24, 22, 25, 26, 27,
    28, 29, 30, 31, 32, 31, 30, 33, 34, 35, 36, 37, 38, 39, 40, 41,
    42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57,
    58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 18, 72,
    73, 74, 75, 76, 77, 78, 79, 80, 81, 2, 51, 82, 39, 83, 84, 19,
    85, 86, 87, 88, 89, 86, 87, 90, 90, 91, 92, 93, 92, 94, 95, 96,
    97, 98, 99, 100, 101, 85, 102
};

static const uint32_t mnemonic_w65816[] = {
    0x616463, 0x616e64, 0x61736c, 0x626363, 0x626373, 0x626571, 0x626765, 0x626974,
    0x626c74, 0x626d69, 0x626e65, 0x62706c, 0x627261, 0x62726b, 0x62726c, 0x627663,
    0x627673, 0x636c63, 0x636c64, 0x636c69, 0x636c70, 0x636c72, 0x636c76, 0x636d70,
    0x636f70, 0x637061, 0x637078, 0x637079, 0x637370, 0x646561, 0x646563, 0x646578,
    0x646579, 0x656f72, 0x676363, 0x676373, 0x676571, 0x676765, 0x676c74, 0x676d69,
    0x676e65, 0x67706c, 0x677261, 0x677663, 0x677673, 0x686c74, 0x696e61, 0x696e63,
    0x696e78, 0x696e79, 0x6a6d6c, 0x6a6d70, 0x6a736c, 0x6a7372, 0x6c6461, 0x6c6478,
    0x6c6479, 0x6c7372, 0x6d766e, 0x6d7670, 0x6e6f70, 0x6f7261, 0x706561, 0x706569,
    0x706572, 0x706861, 0x706862, 0x706864, 0x70686b, 0x706870, 0x706878, 0x706879,
    0x706c61, 0x706c62, 0x706c64, 0x706c70, 0x706c78, 0x706c79, 0x726570, 0x726f6c,
    0x726f72, 0x727469, 0x72746c, 0x727473, 0x736263, 0x736563, 0x736564, 0x736569,
    0x736570, 0x73686c, 0x736872, 0x737461, 0x737470, 0x737478, 0x737479, 0x73747a,
    0x737761, 0x746164, 0x746173, 0x746178, 0x746179, 0x746364, 0x746373, 0x746461,
    0x746463, 0x747262, 0x747361, 0x747362, 0x747363, 0x747378, 0x747861, 0x747873,
    0x747879, 0x747961, 0x747978, 0x776169, 0x77646d, 0x786261, 0x786365
};

static const uint16_t disasm_w65816[] = {
    0x020d, 0x0a3d, 0x0218, 0x0b3d, 0x056b, 0x053d, 0x0502, 0x123d,
    0x0145, 0x023d, 0x0002, 0x0143, 0x046b, 0x043d, 0x0402, 0x033d,
    0x160b, 0x103d, 0x143d, 0x0c3d, 0x0569, 0x083d, 0x0802, 0x0f3d,
    0x0111, 0x0d3d, 0x002f, 0x0166, 0x0469, 0x073d, 0x0702, 0x063d,
    0x0435, 0x0a01, 0x0334, 0x0b01, 0x0507, 0x0501, 0x054f, 0x1201,
    0x014b, 0x0201, 0x004f, 0x014a, 0x0407, 0x0401, 0x044f, 0x0301,
    0x1609, 0x1001, 0x1401, 0x0c01, 0x0807, 0x0801, 0x084f, 0x0f01,
    0x0155, 0x0d01, 0x001e, 0x016c, 0x0707, 0x0701, 0x074f, 0x0601,
    0x0151, 0x0a21, 0x0274, 0x0b21, 0x173b, 0x0521, 0x0539, 0x1221,
    0x0141, 0x0221, 0x0039, 0x0144, 0x0433, 0x0421, 0x0439, 0x0321,
    0x160f, 0x1021, 0x1421, 0x0c21, 0x173a, 0x0821, 0x0839, 0x0f21,
    0x0113, 0x0d21, 0x0147, 0x0165, 0x0332, 0x0721, 0x0739, 0x0621,
    0x0153, 0x0a00, 0x1540, 0x0b00, 0x055f, 0x0500, 0x0550, 0x1200,
    0x0148, 0x0200, 0x0050, 0x0152, 0x1333, 0x0400, 0x0450, 0x0300,
    0x1610, 0x1000, 0x1400, 0x0c00, 0x085f, 0x0800, 0x0850, 0x0f00,
    0x0157, 0x0d00, 0x014d, 0x0168, 0x0933, 0x0700, 0x0750, 0x0600,
    0x160c, 0x0a5b, 0x150e, 0x0b5b, 0x055e, 0x055b, 0x055d, 0x125b,
    0x0120, 0x0207, 0x016e, 0x0142, 0x045e, 0x045b, 0x045d, 0x035b,
    0x1603, 0x105b, 0x145b, 0x0c5b, 0x085e, 0x085b, 0x0e5d, 0x0f5b,
    0x0171, 0x0d5b, 0x016f, 0x0170, 0x045f, 0x075b, 0x075f, 0x065b,
    0x0238, 0x0a36, 0x0237, 0x0b36, 0x0538, 0x0536, 0x0537, 0x1236,
    0x0164, 0x0236, 0x0163, 0x0149, 0x0438, 0x0436, 0x0437, 0x0336,
    0x1604, 0x1036, 0x1436, 0x0c36, 0x0838, 0x0836, 0x0e37, 0x0f36,
    0x0116, 0x0d36, 0x016d, 0x0172, 0x0738, 0x0736, 0x0d37, 0x0636,
    0x021b, 0x0a17, 0x024e, 0x0b17, 0x051b, 0x0517, 0x051e, 0x1217,
    0x0131, 0x0217, 0x011f, 0x0173, 0x041b, 0x0417, 0x041e, 0x0317,
    0x160a, 0x1017, 0x1417, 0x0c17, 0x053f, 0x0817, 0x081e, 0x0f17,
    0x0112, 0x0d17, 0x0146, 0x015c, 0x1133, 0x0717, 0x071e, 0x0617,
    0x021a, 0x0a54, 0x0258, 0x0b54, 0x051a, 0x0554, 0x052f, 0x1254,
    0x0130, 0x0254, 0x013c, 0x0175, 0x041a, 0x0454, 0x042f, 0x0354,
    0x1605, 0x1054, 0x1454, 0x0c54, 0x023e, 0x0854, 0x082f, 0x0f54,
    0x0156, 0x0d54, 0x014c, 0x0176, 0x0935, 0x0754, 0x072f, 0x0654
};

static const uint8_t alias_w65816[] = {
    0, 1, 2, 3, 4, 5, 4, 7, 3, 9, 10, 11, 12, 13, 14, 15,
    16, 17, 18, 19, 78, 95, 22, 23, 24, 23, 26, 27, 24, 30, 30, 31,
    32, 33, 34, 35, 36, 35, 34, 39, 40, 41, 42, 43, 44, 92, 47, 47,
    48, 49, 51, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63,
    64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79,
    80, 81, 82, 83, 84, 85, 86, 87, 88, 2, 57, 91, 92, 93, 94, 95,
    117, 101, 102, 99, 100, 101, 102, 104, 104, 105, 108, 107, 108, 109, 110, 111,
    112, 113, 114, 115, 116, 117, 118
};

const struct cpu_s w65816 = {
    "65816",
    mnemonic_w65816,
    opcode_w65816,
    disasm_w65816,
    alias_w65816,
    0x184840b,
    119,
    0xffffff,
    51,
    14
};

static const uint8_t opcode_c6502[] = {
    103, 104, 2, 3, 4, 5, 4, 105, 3, 7, 8, 9, 11, 13, 14, 15,
    16, 17, 20, 106, 106, 23, 24, 107, 27, 28, 108, 30, 31, 32, 31, 30,
    33, 34, 35, 37, 38, 109, 42, 43, 110, 111, 112, 113, 114, 51, 54, 115,
    59, 63, 66, 69, 72, 73, 74, 76, 116, 78, 79, 80, 2, 51, 117, 83,
    84, 88, 89, 94, 95, 96, 98
};

static const uint32_t mnemonic_c6502[] = {
    0x616463, 0x616e64, 0x61736c, 0x626363, 0x626373, 0x626571, 0x626765, 0x626974,
    0x626c74, 0x626d69, 0x626e65, 0x62706c, 0x62726b, 0x627663, 0x627673, 0x636c63,
    0x636c64, 0x636c69, 0x636c76, 0x636d70, 0x637061, 0x637078, 0x637079, 0x646563,
    0x646578, 0x646579, 0x656f72, 0x676363, 0x676373, 0x676571, 0x676765, 0x676c74,
    0x676d69, 0x676e65, 0x67706c, 0x677663, 0x677673, 0x696e63, 0x696e78, 0x696e79,
    0x6a6d70, 0x6a7372, 0x6c6461, 0x6c6478, 0x6c6479, 0x6c7372, 0x6e6f70, 0x6f7261,
    0x706861, 0x706870, 0x706c61, 0x706c70, 0x726f6c, 0x726f72, 0x727469, 0x727473,
    0x736263, 0x736563, 0x736564, 0x736569, 0x73686c, 0x736872, 0x737461, 0x737478,
    0x737479, 0x746178, 0x746179, 0x747378, 0x747861, 0x747873, 0x747961
};

static const uint16_t disasm_c6502[] = {
    0x020c, 0x0a2f, 0x012e, 0x012e, 0x012e, 0x052f, 0x0502, 0x012e,
    0x0131, 0x022f, 0x0002, 0x012e, 0x012e, 0x042f, 0x0402, 0x012e,
    0x160b, 0x102f, 0x012e, 0x012e, 0x012e, 0x082f, 0x0802, 0x012e,
    0x010f, 0x0d2f, 0x012e, 0x012e, 0x012e, 0x072f, 0x0702, 0x012e,
    0x0429, 0x0a01, 0x012e, 0x012e, 0x0507, 0x0501, 0x0534, 0x012e,
    0x0133, 0x0201, 0x0034, 0x012e, 0x0407, 0x0401, 0x0434, 0x012e,
    0x1609, 0x1001, 0x012e, 0x012e, 0x012e, 0x0801, 0x0834, 0x012e,
    0x0139, 0x0d01, 0x012e, 0x012e, 0x012e, 0x0701, 0x0734, 0x012e,
    0x0136, 0x0a1a, 0x012e, 0x012e, 0x012e, 0x051a, 0x052d, 0x012e,
    0x0130, 0x021a, 0x002d, 0x012e, 0x0428, 0x041a, 0x042d, 0x012e,
    0x160d, 0x101a, 0x012e, 0x012e, 0x012e, 0x081a, 0x082d, 0x012e,
    0x0111, 0x0d1a, 0x012e, 0x012e, 0x012e, 0x071a, 0x072d, 0x012e,
    0x0137, 0x0a00, 0x012e, 0x012e, 0x012e, 0x0500, 0x0535, 0x012e,
    0x0132, 0x0200, 0x0035, 0x012e, 0x1328, 0x0400, 0x0435, 0x012e,
    0x160e, 0x1000, 0x012e, 0x012e, 0x012e, 0x0800, 0x0835, 0x012e,
    0x013b, 0x0d00, 0x012e, 0x012e, 0x012e, 0x0700, 0x0735, 0x012e,
    0x012e, 0x0a3e, 0x012e, 0x012e, 0x0540, 0x053e, 0x053f, 0x012e,
    0x0119, 0x012e, 0x0144, 0x012e, 0x0440, 0x043e, 0x043f, 0x012e,
    0x1603, 0x103e, 0x012e, 0x012e, 0x0840, 0x083e, 0x0e3f, 0x012e,
    0x0146, 0x0d3e, 0x0145, 0x012e, 0x012e, 0x073e, 0x012e, 0x012e,
    0x022c, 0x0a2a, 0x022b, 0x012e, 0x052c, 0x052a, 0x052b, 0x012e,
    0x0142, 0x022a, 0x0141, 0x012e, 0x042c, 0x042a, 0x042b, 0x012e,
    0x1604, 0x102a, 0x012e, 0x012e, 0x082c, 0x082a, 0x0e2b, 0x012e,
    0x0112, 0x0d2a, 0x0143, 0x012e, 0x072c, 0x072a, 0x0d2b, 0x012e,
    0x0216, 0x0a13, 0x012e, 0x012e, 0x0516, 0x0513, 0x0517, 0x012e,
    0x0127, 0x0213, 0x0118, 0x012e, 0x0416, 0x0413, 0x0417, 0x012e,
    0x160a, 0x1013, 0x012e, 0x012e, 0x012e, 0x0813, 0x0817, 0x012e,
    0x0110, 0x0d13, 0x012e, 0x012e, 0x012e, 0x0713, 0x0717, 0x012e,
    0x0215, 0x0a38, 0x012e, 0x012e, 0x0515, 0x0538, 0x0525, 0x012e,
    0x0126, 0x0238, 0x012e, 0x012e, 0x0415, 0x0438, 0x0425, 0x012e,
    0x1605, 0x1038, 0x012e, 0x012e, 0x012e, 0x0838, 0x0825, 0x012e,
    0x013a, 0x0d38, 0x012e, 0x012e, 0x012e, 0x0738, 0x0725, 0x012e
};

static const uint8_t alias_c6502[] = {
    0, 1, 2, 3, 4, 5, 4, 7, 3, 9, 10, 11, 12, 13, 14, 15,
    16, 17, 18, 19, 19, 21, 22, 23, 24, 25, 26, 27, 28, 29, 28, 27,
    32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47,
    48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 2, 45, 62, 63,
    64, 65, 66, 67, 68, 69, 70
};

const struct cpu_s c6502 = {
    "6502",
    mnemonic_c6502,
    opcode_c6502,
    disasm_c6502,
    alias_c6502,
    0x1848001,
    71,
    0xffff,
    40,
    -1
};

static const uint8_t opcode_c65c02[] = {
    118, 119, 2, 3, 4, 5, 4, 6, 3, 7, 8, 9, 10, 11, 13, 14,
    15, 16, 17, 19, 20, 120, 120, 23, 24, 25, 26, 27, 28, 121, 30, 31,
    32, 31, 30, 33, 34, 35, 36, 37, 38, 40, 41, 42, 43, 122, 111, 123,
    113, 114, 51, 54, 124, 59, 63, 64, 65, 66, 69, 70, 71, 72, 73, 74,
    76, 125, 78, 79, 80, 2, 51, 126, 83, 84, 19, 88, 89, 91, 93, 94,
    95, 96, 98
};

static const uint32_t mnemonic_c65c02[] = {
    0x616463, 0x616e64, 0x61736c, 0x626363, 0x626373, 0x626571, 0x626765, 0x626974,
    0x626c74, 0x626d69, 0x626e65, 0x62706c, 0x627261, 0x62726b, 0x627663, 0x627673,
    0x636c63, 0x636c64, 0x636c69, 0x636c72, 0x636c76, 0x636d70, 0x637061, 0x637078,
    0x637079, 0x646561, 0x646563, 0x646578, 0x646579, 0x656f72, 0x676363, 0x676373,
    0x676571, 0x676765, 0x676c74, 0x676d69, 0x676e65, 0x67706c, 0x677261, 0x677663,
    0x677673, 0x696e61, 0x696e63, 0x696e78, 0x696e79, 0x6a6d70, 0x6a7372, 0x6c6461,
    0x6c6478, 0x6c6479, 0x6c7372, 0x6e6f70, 0x6f7261, 0x706861, 0x706870, 0x706878,
    0x706879, 0x706c61, 0x706c70, 0x706c78, 0x706c79, 0x726f6c, 0x726f72, 0x727469,
    0x727473, 0x736263, 0x736563, 0x736564, 0x736569, 0x73686c, 0x736872, 0x737461,
    0x737478, 0x737479, 0x73747a, 0x746178, 0x746179, 0x747262, 0x747362, 0x747378,
    0x747861, 0x747873, 0x747961
};

static const uint16_t disasm_c65c02[] = {
    0x020d, 0x0a34, 0x0133, 0x0133, 0x054e, 0x0534, 0x0502, 0x0133,
    0x0136, 0x0234, 0x0002, 0x0133, 0x044e, 0x0434, 0x0402, 0x0133,
    0x160b, 0x1034, 0x1434, 0x0133, 0x054d, 0x0834, 0x0802, 0x0133,
    0x0110, 0x0d34, 0x002a, 0x0133, 0x044d, 0x0734, 0x0702, 0x0133,
    0x042e, 0x0a01, 0x0133, 0x0133, 0x0507, 0x0501, 0x053d, 0x0133,
    0x013a, 0x0201, 0x003d, 0x0133, 0x0407, 0x0401, 0x043d, 0x0133,
    0x1609, 0x1001, 0x1401, 0x0133, 0x0807, 0x0801, 0x083d, 0x0133,
    0x0142, 0x0d01, 0x001a, 0x0133, 0x0707, 0x0701, 0x073d, 0x0133,
    0x013f, 0x0a1d, 0x0133, 0x0133, 0x0133, 0x051d, 0x0532, 0x0133,
    0x0135, 0x021d, 0x0032, 0x0133, 0x042d, 0x041d, 0x0432, 0x0133,
    0x160e, 0x101d, 0x141d, 0x0133, 0x0133, 0x081d, 0x0832, 0x0133,
    0x0112, 0x0d1d, 0x0138, 0x0133, 0x0133, 0x071d, 0x0732, 0x0133,
    0x0140, 0x0a00, 0x0133, 0x0133, 0x054a, 0x0500, 0x053e, 0x0133,
    0x0139, 0x0200, 0x003e, 0x0133, 0x132d, 0x0400, 0x043e, 0x0133,
    0x160f, 0x1000, 0x1400, 0x0133, 0x084a, 0x0800, 0x083e, 0x0133,
    0x0144, 0x0d00, 0x013c, 0x0133, 0x092d, 0x0700, 0x073e, 0x0133,
    0x160c, 0x0a47, 0x0133, 0x0133, 0x0549, 0x0547, 0x0548, 0x0133,
    0x011c, 0x0207, 0x0150, 0x0133, 0x0449, 0x0447, 0x0448, 0x0133,
    0x1603, 0x1047, 0x1447, 0x0133, 0x0849, 0x0847, 0x0e48, 0x0133,
    0x0152, 0x0d47, 0x0151, 0x0133, 0x044a, 0x0747, 0x074a, 0x0133,
    0x0231, 0x0a2f, 0x0230, 0x0133, 0x0531, 0x052f, 0x0530, 0x0133,
    0x014c, 0x022f, 0x014b, 0x0133, 0x0431, 0x042f, 0x0430, 0x0133,
    0x1604, 0x102f, 0x142f, 0x0133, 0x0831, 0x082f, 0x0e30, 0x0133,
    0x0114, 0x0d2f, 0x014f, 0x0133, 0x0731, 0x072f, 0x0d30, 0x0133,
    0x0218, 0x0a15, 0x0133, 0x0133, 0x0518, 0x0515, 0x051a, 0x0133,
    0x012c, 0x0215, 0x011b, 0x0133, 0x0418, 0x0415, 0x041a, 0x0133,
    0x160a, 0x1015, 0x1415, 0x0133, 0x0133, 0x0815, 0x081a, 0x0133,
    0x0111, 0x0d15, 0x0137, 0x0133, 0x0133, 0x0715, 0x071a, 0x0133,
    0x0217, 0x0a41, 0x0133, 0x0133, 0x0517, 0x0541, 0x052a, 0x0133,
    0x012b, 0x0241, 0x0133, 0x0133, 0x0417, 0x0441, 0x042a, 0x0133,
    0x1605, 0x1041, 0x1441, 0x0133, 0x0133, 0x0841, 0x082a, 0x0133,
    0x0143, 0x0d41, 0x013b, 0x0133, 0x0133, 0x0741, 0x072a, 0x0133
};

static const uint8_t alias_c65c02[] = {
    0, 1, 2, 3, 4, 5, 4, 7, 3, 9, 10, 11, 12, 13, 14, 15,
    16, 17, 18, 74, 20, 21, 21, 23, 24, 26, 26, 27, 28, 29, 30, 31,
    32, 31, 30, 35, 36, 37, 38, 39, 40, 42, 42, 43, 44, 45, 46, 47,
    48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63,
    64, 65, 66, 67, 68, 2, 50, 71, 72, 73, 74, 75, 76, 77, 78, 79,
    80, 81, 82
};

const struct cpu_s c65c02 = {
    "65c02",
    mnemonic_c65c02,
    opcode_c65c02,
    disasm_c65c02,
    alias_c65c02,
    0x1848001,
    83,
    0xffff,
    45,
    -1
};

static const uint8_t opcode_c6502i[] = {
    103, 127, 128, 129, 104, 130, 131, 2, 128, 132, 3, 4, 5, 4, 105, 3,
    7, 8, 9, 11, 13, 14, 15, 16, 17, 20, 106, 106, 23, 24, 133, 133,
    107, 27, 28, 108, 30, 31, 32, 31, 30, 33, 34, 35, 37, 38, 109, 134,
    42, 43, 134, 134, 135, 110, 111, 136, 136, 137, 112, 136, 113, 114, 51, 138,
    139, 115, 59, 63, 66, 69, 140, 72, 73, 141, 74, 76, 142, 116, 132, 78,
    79, 80, 127, 2, 51, 143, 144, 145, 146, 147, 117, 83, 84, 143, 88, 89,
    94, 95, 96, 98, 130
};

static const uint32_t mnemonic_c6502i[] = {
    0x616463, 0x616878, 0x616c72, 0x616e63, 0x616e64, 0x616e65, 0x617272, 0x61736c,
    0x617372, 0x617873, 0x626363, 0x626373, 0x626571, 0x626765, 0x626974, 0x626c74,
    0x626d69, 0x626e65, 0x62706c, 0x62726b, 0x627663, 0x627673, 0x636c63, 0x636c64,
    0x636c69, 0x636c76, 0x636d70, 0x637061, 0x637078, 0x637079, 0x64636d, 0x646370,
    0x646563, 0x646578, 0x646579, 0x656f72, 0x676363, 0x676373, 0x676571, 0x676765,
    0x676c74, 0x676d69, 0x676e65, 0x67706c, 0x677663, 0x677673, 0x696e63, 0x696e73,
    0x696e78, 0x696e79, 0x697362, 0x697363, 0x6a616d, 0x6a6d70, 0x6a7372, 0x6c6165,
    0x6c6173, 0x6c6178, 0x6c6461, 0x6c6473, 0x6c6478, 0x6c6479, 0x6c7372, 0x6c7861,
    0x6e6f70, 0x6f7261, 0x706861, 0x706870, 0x706c61, 0x706c70, 0x726c61, 0x726f6c,
    0x726f72, 0x727261, 0x727469, 0x727473, 0x736178, 0x736263, 0x736278, 0x736563,
    0x736564, 0x736569, 0x736861, 0x73686c, 0x736872, 0x736873, 0x736878, 0x736879,
    0x736c6f, 0x737265, 0x737461, 0x737478, 0x737479, 0x746173, 0x746178, 0x746179,
    0x747378, 0x747861, 0x747873, 0x747961, 0x786161
};

static const uint16_t disasm_c6502i[] = {
    0x0213, 0x0a41, 0x0134, 0x0a58, 0x0540, 0x0541, 0x0507, 0x0558,
    0x0143, 0x0241, 0x0007, 0x0203, 0x0440, 0x0441, 0x0407, 0x0458,
    0x1612, 0x1041, 0x0140, 0x1058, 0x0840, 0x0841, 0x0807, 0x0858,
    0x0116, 0x0d41, 0x0140, 0x0d58, 0x0740, 0x0741, 0x0707, 0x0758,
    0x0436, 0x0a04, 0x0140, 0x0a46, 0x050e, 0x0504, 0x0547, 0x0546,
    0x0145, 0x0204, 0x0047, 0x0140, 0x040e, 0x0404, 0x0447, 0x0446,
    0x1610, 0x1004, 0x0140, 0x1046, 0x0140, 0x0804, 0x0847, 0x0846,
    0x014f, 0x0d04, 0x0140, 0x0d46, 0x0140, 0x0704, 0x0747, 0x0746,
    0x014a, 0x0a23, 0x0140, 0x0a59, 0x0140, 0x0523, 0x053e, 0x0559,
    0x0142, 0x0223, 0x003e, 0x0208, 0x0435, 0x0423, 0x043e, 0x0459,
    0x1614, 0x1023, 0x0140, 0x1059, 0x0140, 0x0823, 0x083e, 0x0859,
    0x0118, 0x0d23, 0x0140, 0x0d59, 0x0140, 0x0723, 0x073e, 0x0759,
    0x014b, 0x0a00, 0x0140, 0x0a49, 0x0140, 0x0500, 0x0548, 0x0549,
    0x0144, 0x0200, 0x0048, 0x0206, 0x1335, 0x0400, 0x0448, 0x0449,
    0x1615, 0x1000, 0x0140, 0x1049, 0x0140, 0x0800, 0x0848, 0x0849,
    0x0151, 0x0d00, 0x0140, 0x0d49, 0x0140, 0x0700, 0x0748, 0x0749,
    0x0240, 0x0a5a, 0x0140, 0x0a4c, 0x055c, 0x055a, 0x055b, 0x054c,
    0x0122, 0x0140, 0x0161, 0x0205, 0x045c, 0x045a, 0x045b, 0x044c,
    0x160a, 0x105a, 0x0140, 0x1052, 0x085c, 0x085a, 0x0e5b, 0x0e4c,
    0x0163, 0x0d5a, 0x0162, 0x0d55, 0x0757, 0x075a, 0x0d56, 0x0d52,
    0x023d, 0x0a3a, 0x023c, 0x0a39, 0x053d, 0x053a, 0x053c, 0x0539,
    0x015f, 0x023a, 0x015e, 0x0239, 0x043d, 0x043a, 0x043c, 0x0439,
    0x160b, 0x103a, 0x0140, 0x1039, 0x083d, 0x083a, 0x0e3c, 0x0e39,
    0x0119, 0x0d3a, 0x0160, 0x0d3b, 0x073d, 0x073a, 0x0d3c, 0x0d39,
    0x021d, 0x0a1a, 0x0140, 0x0a1f, 0x051d, 0x051a, 0x0520, 0x051f,
    0x0131, 0x021a, 0x0121, 0x024e, 0x041d, 0x041a, 0x0420, 0x041f,
    0x1611, 0x101a, 0x0140, 0x101f, 0x0140, 0x081a, 0x0820, 0x081f,
    0x0117, 0x0d1a, 0x0140, 0x0d1f, 0x0140, 0x071a, 0x0720, 0x071f,
    0x021c, 0x0a4d, 0x0140, 0x0a32, 0x051c, 0x054d, 0x052e, 0x0532,
    0x0130, 0x024d, 0x0140, 0x0140, 0x041c, 0x044d, 0x042e, 0x0432,
    0x160c, 0x104d, 0x0140, 0x1032, 0x0140, 0x084d, 0x082e, 0x0832,
    0x0150, 0x0d4d, 0x0140, 0x0d32, 0x0140, 0x074d, 0x072e, 0x0732
};

static const uint8_t alias_c6502i[] = {
    0, 82, 8, 3, 4, 5, 6, 7, 8, 78, 10, 11, 12, 11, 14, 10,
    16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 26, 28, 29, 31, 31,
    32, 33, 34, 35, 36, 37, 38, 37, 36, 41, 42, 43, 44, 45, 46, 50,
    48, 49, 50, 50, 52, 53, 54, 59, 59, 57, 58, 59, 60, 61, 62, 57,
    64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79,
    80, 81, 82, 7, 62, 85, 86, 87, 88, 89, 90, 91, 92, 85, 94, 95,
    96, 97, 98, 99, 5
};

const struct cpu_s c6502i = {
    "6502i",
    mnemonic_c6502i,
    opcode_c6502i,
    disasm_c6502i,
    alias_c6502i,
    0x1848001,
    101,
    0xffff,
    53,
    -1
};

static const uint8_t opcode_c65dtv02[] = {
    103, 128, 104, 130, 131, 2, 128, 3, 4, 5, 4, 105, 3, 7, 8, 9,
    148, 11, 13, 14, 15, 16, 17, 20, 106, 106, 23, 24, 133, 133, 107, 27,
    28, 108, 30, 31, 32, 31, 30, 33, 34, 35, 149, 37, 38, 109, 134, 42,
    43, 134, 134, 110, 111, 137, 112, 113, 114, 51, 138, 54, 115, 59, 63, 66,
    69, 140, 72, 73, 141, 74, 76, 150, 142, 116, 78, 79, 80, 2, 51, 101,
    146, 147, 117, 83, 84, 88, 89, 94, 95, 96, 98, 130
};

static const uint32_t mnemonic_c65dtv02[] = {
    0x616463, 0x616c72, 0x616e64, 0x616e65, 0x617272, 0x61736c, 0x617372, 0x626363,
    0x626373, 0x626571, 0x626765, 0x626974, 0x626c74, 0x626d69, 0x626e65, 0x62706c,
    0x627261, 0x62726b, 0x627663, 0x627673, 0x636c63, 0x636c64, 0x636c69, 0x636c76,
    0x636d70, 0x637061, 0x637078, 0x637079, 0x64636d, 0x646370, 0x646563, 0x646578,
    0x646579, 0x656f72, 0x676363, 0x676373, 0x676571, 0x676765, 0x676c74, 0x676d69,
    0x676e65, 0x67706c, 0x677261, 0x677663, 0x677673, 0x696e63, 0x696e73, 0x696e78,
    0x696e79, 0x697362, 0x697363, 0x6a6d70, 0x6a7372, 0x6c6178, 0x6c6461, 0x6c6478,
    0x6c6479, 0x6c7372, 0x6c7861, 0x6e6f70, 0x6f7261, 0x706861, 0x706870, 0x706c61,
    0x706c70, 0x726c61, 0x726f6c, 0x726f72, 0x727261, 0x727469, 0x727473, 0x736163,
    0x736178, 0x736263, 0x736563, 0x736564, 0x736569, 0x73686c, 0x736872, 0x736972,
    0x736c6f, 0x737265, 0x737461, 0x737478, 0x737479, 0x746178, 0x746179, 0x747378,
    0x747861, 0x747873, 0x747961, 0x786161
};

static const uint16_t disasm_c65dtv02[] = {
    0x0211, 0x0a3c, 0x013b, 0x0a50, 0x013b, 0x053c, 0x0505, 0x0550,
    0x013e, 0x023c, 0x0005, 0x013b, 0x013b, 0x043c, 0x0405, 0x0450,
    0x160f, 0x103c, 0x1610, 0x1050, 0x013b, 0x083c, 0x0805, 0x0850,
    0x0114, 0x0d3c, 0x013b, 0x0d50, 0x013b, 0x073c, 0x0705, 0x0750,
    0x0434, 0x0a02, 0x013b, 0x0a41, 0x050b, 0x0502, 0x0542, 0x0541,
    0x0140, 0x0202, 0x0042, 0x013b, 0x040b, 0x0402, 0x0442, 0x0441,
    0x160d, 0x1002, 0x0247, 0x1041, 0x013b, 0x0802, 0x0842, 0x0841,
    0x014a, 0x0d02, 0x013b, 0x0d41, 0x013b, 0x0702, 0x0742, 0x0741,
    0x0145, 0x0a21, 0x024f, 0x0a51, 0x013b, 0x0521, 0x0539, 0x0551,
    0x013d, 0x0221, 0x0039, 0x0206, 0x0433, 0x0421, 0x0439, 0x0451,
    0x1612, 0x1021, 0x013b, 0x1051, 0x013b, 0x0821, 0x0839, 0x0851,
    0x0116, 0x0d21, 0x013b, 0x0d51, 0x013b, 0x0721, 0x0739, 0x0751,
    0x0146, 0x0a00, 0x013b, 0x0a44, 0x013b, 0x0500, 0x0543, 0x0544,
    0x013f, 0x0200, 0x0043, 0x0204, 0x1333, 0x0400, 0x0443, 0x0444,
    0x1613, 0x1000, 0x013b, 0x1044, 0x013b, 0x0800, 0x0843, 0x0844,
    0x014c, 0x0d00, 0x013b, 0x0d44, 0x013b, 0x0700, 0x0743, 0x0744,
    0x013b, 0x0a52, 0x013b, 0x0a48, 0x0554, 0x0552, 0x0553, 0x0548,
    0x0120, 0x013b, 0x0158, 0x0203, 0x0454, 0x0452, 0x0453, 0x0448,
    0x1607, 0x1052, 0x013b, 0x013b, 0x0854, 0x0852, 0x0e53, 0x0e48,
    0x015a, 0x0d52, 0x0159, 0x013b, 0x013b, 0x0752, 0x013b, 0x013b,
    0x0238, 0x0a36, 0x0237, 0x0a35, 0x0538, 0x0536, 0x0537, 0x0535,
    0x0156, 0x0236, 0x0155, 0x0235, 0x0438, 0x0436, 0x0437, 0x0435,
    0x1608, 0x1036, 0x013b, 0x1035, 0x0838, 0x0836, 0x0e37, 0x0e35,
    0x0117, 0x0d36, 0x0157, 0x013b, 0x0738, 0x0736, 0x0d37, 0x0d35,
    0x021b, 0x0a18, 0x013b, 0x0a1d, 0x051b, 0x0518, 0x051e, 0x051d,
    0x0130, 0x0218, 0x011f, 0x013b, 0x041b, 0x0418, 0x041e, 0x041d,
    0x160e, 0x1018, 0x013b, 0x101d, 0x013b, 0x0818, 0x081e, 0x081d,
    0x0115, 0x0d18, 0x013b, 0x0d1d, 0x013b, 0x0718, 0x071e, 0x071d,
    0x021a, 0x0a49, 0x013b, 0x0a31, 0x051a, 0x0549, 0x052d, 0x0531,
    0x012f, 0x0249, 0x013b, 0x013b, 0x041a, 0x0449, 0x042d, 0x0431,
    0x1609, 0x1049, 0x013b, 0x1031, 0x013b, 0x0849, 0x082d, 0x0831,
    0x014b, 0x0d49, 0x013b, 0x0d31, 0x013b, 0x0749, 0x072d, 0x0731
};

static const uint8_t alias_c65dtv02[] = {
    0, 6, 2, 3, 4, 5, 6, 7, 8, 9, 8, 11, 7, 13, 14, 15,
    16, 17, 18, 19, 20, 21, 22, 23, 24, 24, 26, 27, 29, 29, 30, 31,
    32, 33, 34, 35, 36, 35, 34, 39, 40, 41, 42, 43, 44, 45, 49, 47,
    48, 49, 49, 51, 52, 53, 54, 55, 56, 57, 53, 59, 60, 61, 62, 63,
    64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 5, 57, 79,
    80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 3
};

const struct cpu_s c65dtv02 = {
    "65dtv02",
    mnemonic_c65dtv02,
    opcode_c65dtv02,
    disasm_c65dtv02,
    alias_c65dtv02,
    0x1848001,
    92,
    0xffff,
    51,
    -1
};

static const uint8_t opcode_c65el02[] = {
    151, 152, 2, 3, 4, 5, 4, 6, 3, 7, 8, 9, 10, 11, 13, 14,
    15, 16, 17, 18, 19, 20, 153, 153, 23, 24, 25, 26, 27, 28, 154, 155,
    156, 30, 31, 32, 31, 30, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42,
    43, 122, 47, 157, 158, 50, 51, 159, 160, 54, 161, 135, 162, 56, 57, 58,
    59, 163, 63, 64, 65, 66, 164, 69, 70, 71, 165, 166, 18, 167, 62, 61,
    87, 86, 75, 68, 92, 90, 72, 73, 74, 76, 168, 169, 78, 79, 80, 81,
    2, 51, 170, 39, 83, 84, 19, 85, 171, 88, 89, 172, 173, 91, 67, 93,
    94, 95, 174, 60, 96, 97, 98, 99, 100, 85, 102, 175
};

static const uint32_t mnemonic_c65el02[] = {
    0x616463, 0x616e64, 0x61736c, 0x626363, 0x626373, 0x626571, 0x626765, 0x626974,
    0x626c74, 0x626d69, 0x626e65, 0x62706c, 0x627261, 0x62726b, 0x627663, 0x627673,
    0x636c63, 0x636c64, 0x636c69, 0x636c70, 0x636c72, 0x636c76, 0x636d70, 0x637061,
    0x637078, 0x637079, 0x646561, 0x646563, 0x646578, 0x646579, 0x646976, 0x656e74,
    0x656f72, 0x676363, 0x676373, 0x676571, 0x676765, 0x676c74, 0x676d69, 0x676e65,
    0x67706c, 0x677261, 0x677663, 0x677673, 0x686c74, 0x696e61, 0x696e63, 0x696e78,
    0x696e79, 0x6a6d70, 0x6a7372, 0x6c6461, 0x6c6478, 0x6c6479, 0x6c7372, 0x6d6d75,
    0x6d756c, 0x6e6f70, 0x6e7861, 0x6e7874, 0x6f7261, 0x706561, 0x706569, 0x706572,
    0x706861, 0x706864, 0x706870, 0x706878, 0x706879, 0x706c61, 0x706c64, 0x706c70,
    0x706c78, 0x706c79, 0x726561, 0x726569, 0x726570, 0x726572, 0x726861, 0x726869,
    0x726878, 0x726879, 0x726c61, 0x726c69, 0x726c78, 0x726c79, 0x726f6c, 0x726f72,
    0x727469, 0x727473, 0x736263, 0x736561, 0x736563, 0x736564, 0x736569, 0x736570,
    0x73686c, 0x736872, 0x737461, 0x737470, 0x737478, 0x737479, 0x73747a, 0x737761,
    0x746164, 0x746178, 0x746179, 0x746461, 0x746978, 0x747262, 0x747278, 0x747362,
    0x747378, 0x747861, 0x747869, 0x747872, 0x747873, 0x747879, 0x747961, 0x747978,
    0x776169, 0x786261, 0x786365, 0x7a6561
};

static const uint16_t disasm_c65el02[] = {
    0x020d, 0x0a3c, 0x013b, 0x0b3c, 0x056f, 0x053c, 0x0502, 0x183c,
    0x0142, 0x023c, 0x0002, 0x014f, 0x046f, 0x043c, 0x0402, 0x0538,
    0x160b, 0x103c, 0x143c, 0x0c3c, 0x056d, 0x083c, 0x0802, 0x193c,
    0x0110, 0x0d3c, 0x002e, 0x0150, 0x046d, 0x073c, 0x0702, 0x0838,
    0x0432, 0x0a01, 0x011f, 0x0b01, 0x0507, 0x0501, 0x0556, 0x1801,
    0x0147, 0x0201, 0x0056, 0x0153, 0x0407, 0x0401, 0x0456, 0x0438,
    0x1609, 0x1001, 0x1401, 0x0c01, 0x0807, 0x0801, 0x0856, 0x1901,
    0x015c, 0x0d01, 0x001b, 0x0154, 0x0707, 0x0701, 0x0756, 0x0738,
    0x0158, 0x0a20, 0x013a, 0x0b20, 0x024a, 0x0520, 0x0536, 0x1820,
    0x0140, 0x0220, 0x0036, 0x014e, 0x0431, 0x0420, 0x0436, 0x051e,
    0x160e, 0x1020, 0x1420, 0x0c20, 0x054b, 0x0820, 0x0836, 0x1920,
    0x0112, 0x0d20, 0x0144, 0x0151, 0x0172, 0x0720, 0x0736, 0x081e,
    0x0159, 0x0a00, 0x153f, 0x0b00, 0x0566, 0x0500, 0x0557, 0x1800,
    0x0145, 0x0200, 0x0057, 0x0152, 0x1331, 0x0400, 0x0457, 0x041e,
    0x160f, 0x1000, 0x1400, 0x0c00, 0x0866, 0x0800, 0x0857, 0x1900,
    0x015e, 0x0d00, 0x0149, 0x0155, 0x0931, 0x0700, 0x0757, 0x071e,
    0x160c, 0x0a62, 0x164d, 0x0b62, 0x0565, 0x0562, 0x0564, 0x1862,
    0x011d, 0x0207, 0x0171, 0x0173, 0x0465, 0x0462, 0x0464, 0x017b,
    0x1603, 0x1062, 0x1462, 0x0c62, 0x0865, 0x0862, 0x0e64, 0x1962,
    0x0176, 0x0d62, 0x0174, 0x0175, 0x0466, 0x0762, 0x0766, 0x015b,
    0x0235, 0x0a33, 0x0234, 0x0b33, 0x0535, 0x0533, 0x0534, 0x1833,
    0x016a, 0x0233, 0x0169, 0x016e, 0x0435, 0x0433, 0x0434, 0x016b,
    0x1604, 0x1033, 0x1433, 0x0c33, 0x0835, 0x0833, 0x0e34, 0x1933,
    0x0115, 0x0d33, 0x0170, 0x0177, 0x0735, 0x0733, 0x0d34, 0x0168,
    0x0219, 0x0a16, 0x024c, 0x0b16, 0x0519, 0x0516, 0x051b, 0x1816,
    0x0130, 0x0216, 0x011c, 0x0178, 0x0419, 0x0416, 0x041b, 0x0146,
    0x160a, 0x1016, 0x1416, 0x0c16, 0x053e, 0x0816, 0x081b, 0x1916,
    0x0111, 0x0d16, 0x0143, 0x0163, 0x016c, 0x0716, 0x071b, 0x0141,
    0x0218, 0x0a5a, 0x025f, 0x0b5a, 0x0518, 0x055a, 0x052e, 0x185a,
    0x012f, 0x025a, 0x0139, 0x0179, 0x0418, 0x045a, 0x042e, 0x0237,
    0x1605, 0x105a, 0x145a, 0x0c5a, 0x023d, 0x085a, 0x082e, 0x195a,
    0x015d, 0x0d5a, 0x0148, 0x017a, 0x0932, 0x075a, 0x072e, 0x0139
};

static const uint8_t alias_c65el02[] = {
    0, 1, 2, 3, 4, 5, 4, 7, 3, 9, 10, 11, 12, 13, 14, 15,
    16, 17, 18, 76, 102, 21, 22, 22, 24, 25, 27, 27, 28, 29, 30, 31,
    32, 33, 34, 35, 34, 33, 38, 39, 40, 41, 42, 43, 99, 46, 46, 47,
    48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63,
    64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79,
    80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95,
    2, 54, 98, 99, 100, 101, 102, 121, 104, 105, 106, 107, 108, 109, 110, 111,
    112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 123
};

const struct cpu_s c65el02 = {
    "65el02",
    mnemonic_c65el02,
    opcode_c65el02,
    disasm_c65el02,
    alias_c65el02,
    0x1868109,
    124,
    0xffff,
    49,
    -1
};

static const uint8_t opcode_r65c02[] = {
    118, 119, 2, 176, 177, 3, 4, 5, 4, 6, 3, 7, 8, 9, 10, 11,
    13, 14, 15, 16, 17, 19, 20, 120, 120, 23, 24, 25, 26, 27, 28, 121,
    30, 31, 32, 31, 30, 33, 34, 35, 36, 37, 38, 40, 41, 42, 43, 122,
    111, 123, 113, 114, 51, 178, 124, 59, 63, 64, 65, 66, 69, 70, 71, 179,
    72, 73, 74, 76, 125, 78, 79, 80, 2, 51, 180, 126, 83, 84, 19, 88,
    89, 91, 93, 94, 95, 96, 98
};

static const uint32_t mnemonic_r65c02[] = {
    0x616463, 0x616e64, 0x61736c, 0x626272, 0x626273, 0x626363, 0x626373, 0x626571,
    0x626765, 0x626974, 0x626c74, 0x626d69, 0x626e65, 0x62706c, 0x627261, 0x62726b,
    0x627663, 0x627673, 0x636c63, 0x636c64, 0x636c69, 0x636c72, 0x636c76, 0x636d70,
    0x637061, 0x637078, 0x637079, 0x646561, 0x646563, 0x646578, 0x646579, 0x656f72,
    0x676363, 0x676373, 0x676571, 0x676765, 0x676c74, 0x676d69, 0x676e65, 0x67706c,
    0x677261, 0x677663, 0x677673, 0x696e61, 0x696e63, 0x696e78, 0x696e79, 0x6a6d70,
    0x6a7372, 0x6c6461, 0x6c6478, 0x6c6479, 0x6c7372, 0x6e6f70, 0x6f7261, 0x706861,
    0x706870, 0x706878, 0x706879, 0x706c61, 0x706c70, 0x706c78, 0x706c79, 0x726d62,
    0x726f6c, 0x726f72, 0x727469, 0x727473, 0x736263, 0x736563, 0x736564, 0x736569,
    0x73686c, 0x736872, 0x736d62, 0x737461, 0x737478, 0x737479, 0x73747a, 0x746178,
    0x746179, 0x747262, 0x747362, 0x747378, 0x747861, 0x747873, 0x747961
};

static const uint16_t disasm_r65c02[] = {
    0x020f, 0x0a36, 0x0135, 0x0135, 0x0552, 0x0536, 0x0502, 0x1a3f,
    0x0138, 0x0236, 0x0002, 0x0135, 0x0452, 0x0436, 0x0402, 0x1b03,
    0x160d, 0x1036, 0x1436, 0x0135, 0x0551, 0x0836, 0x0802, 0x1a3f,
    0x0112, 0x0d36, 0x002c, 0x0135, 0x0451, 0x0736, 0x0702, 0x1b03,
    0x0430, 0x0a01, 0x0135, 0x0135, 0x0509, 0x0501, 0x0540, 0x1a3f,
    0x013c, 0x0201, 0x0040, 0x0135, 0x0409, 0x0401, 0x0440, 0x1b03,
    0x160b, 0x1001, 0x1401, 0x0135, 0x0809, 0x0801, 0x0840, 0x1a3f,
    0x0145, 0x0d01, 0x001c, 0x0135, 0x0709, 0x0701, 0x0740, 0x1b03,
    0x0142, 0x0a1f, 0x0135, 0x0135, 0x0535, 0x051f, 0x0534, 0x1a3f,
    0x0137, 0x021f, 0x0034, 0x0135, 0x042f, 0x041f, 0x0434, 0x1b03,
    0x1610, 0x101f, 0x141f, 0x0135, 0x0835, 0x081f, 0x0834, 0x1a3f,
    0x0114, 0x0d1f, 0x013a, 0x0135, 0x0135, 0x071f, 0x0734, 0x1b03,
    0x0143, 0x0a00, 0x0135, 0x0135, 0x054e, 0x0500, 0x0541, 0x1a3f,
    0x013b, 0x0200, 0x0041, 0x0135, 0x132f, 0x0400, 0x0441, 0x1b03,
    0x1611, 0x1000, 0x1400, 0x0135, 0x084e, 0x0800, 0x0841, 0x1a3f,
    0x0147, 0x0d00, 0x013e, 0x0135, 0x092f, 0x0700, 0x0741, 0x1b03,
    0x160e, 0x0a4b, 0x0235, 0x0135, 0x054d, 0x054b, 0x054c, 0x1a4a,
    0x011e, 0x0209, 0x0154, 0x0135, 0x044d, 0x044b, 0x044c, 0x1b04,
    0x1605, 0x104b, 0x144b, 0x0135, 0x084d, 0x084b, 0x0e4c, 0x1a4a,
    0x0156, 0x0d4b, 0x0155, 0x0135, 0x044e, 0x074b, 0x074e, 0x1b04,
    0x0233, 0x0a31, 0x0232, 0x0135, 0x0533, 0x0531, 0x0532, 0x1a4a,
    0x0150, 0x0231, 0x014f, 0x0135, 0x0433, 0x0431, 0x0432, 0x1b04,
    0x1606, 0x1031, 0x1431, 0x0135, 0x0833, 0x0831, 0x0e32, 0x1a4a,
    0x0116, 0x0d31, 0x0153, 0x0135, 0x0733, 0x0731, 0x0d32, 0x1b04,
    0x021a, 0x0a17, 0x0135, 0x0135, 0x051a, 0x0517, 0x051c, 0x1a4a,
    0x012e, 0x0217, 0x011d, 0x0135, 0x041a, 0x0417, 0x041c, 0x1b04,
    0x160c, 0x1017, 0x1417, 0x0135, 0x0135, 0x0817, 0x081c, 0x1a4a,
    0x0113, 0x0d17, 0x0139, 0x0135, 0x0435, 0x0717, 0x071c, 0x1b04,
    0x0219, 0x0a44, 0x0135, 0x0135, 0x0519, 0x0544, 0x052c, 0x1a4a,
    0x012d, 0x0244, 0x0135, 0x0135, 0x0419, 0x0444, 0x042c, 0x1b04,
    0x1607, 0x1044, 0x1444, 0x0135, 0x0135, 0x0844, 0x082c, 0x1a4a,
    0x0146, 0x0d44, 0x013d, 0x0135, 0x0135, 0x0744, 0x072c, 0x1b04
};

static const uint8_t alias_r65c02[] = {
    0, 1, 2, 3, 4, 5, 6, 7, 6, 9, 5, 11, 12, 13, 14, 15,
    16, 17, 18, 19, 20, 78, 22, 23, 23, 25, 26, 28, 28, 29, 30, 31,
    32, 33, 34, 33, 32, 37, 38, 39, 40, 41, 42, 44, 44, 45, 46, 47,
    48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63,
    64, 65, 66, 67, 68, 69, 70, 71, 2, 52, 74, 75, 76, 77, 78, 79,
    80, 81, 82, 83, 84, 85, 86
};

const struct cpu_s r65c02 = {
    "r65c02",
    mnemonic_r65c02,
    opcode_r65c02,
    disasm_r65c02,
    alias_r65c02,
    0x1848001,
    87,
    0xffff,
    47,
    -1
};

static const uint8_t opcode_w65c02[] = {
    118, 119, 2, 176, 177, 3, 4, 5, 4, 6, 3, 7, 8, 9, 10, 11,
    13, 14, 15, 16, 17, 19, 20, 120, 120, 23, 24, 25, 26, 27, 28, 121,
    30, 31, 32, 31, 30, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43,
    122, 111, 123, 113, 114, 51, 178, 124, 59, 63, 64, 65, 66, 69, 70, 71,
    179, 72, 73, 74, 76, 125, 78, 79, 80, 2, 51, 180, 126, 39, 83, 84,
    19, 88, 89, 91, 93, 94, 95, 96, 98, 100
};

static const uint32_t mnemonic_w65c02[] = {
    0x616463, 0x616e64, 0x61736c, 0x626272, 0x626273, 0x626363, 0x626373, 0x626571,
    0x626765, 0x626974, 0x626c74, 0x626d69, 0x626e65, 0x62706c, 0x627261, 0x62726b,
    0x627663, 0x627673, 0x636c63, 0x636c64, 0x636c69, 0x636c72, 0x636c76, 0x636d70,
    0x637061, 0x637078, 0x637079, 0x646561, 0x646563, 0x646578, 0x646579, 0x656f72,
    0x676363, 0x676373, 0x676571, 0x676765, 0x676c74, 0x676d69, 0x676e65, 0x67706c,
    0x677261, 0x677663, 0x677673, 0x686c74, 0x696e61, 0x696e63, 0x696e78, 0x696e79,
    0x6a6d70, 0x6a7372, 0x6c6461, 0x6c6478, 0x6c6479, 0x6c7372, 0x6e6f70, 0x6f7261,
    0x706861, 0x706870, 0x706878, 0x706879, 0x706c61, 0x706c70, 0x706c78, 0x706c79,
    0x726d62, 0x726f6c, 0x726f72, 0x727469, 0x727473, 0x736263, 0x736563, 0x736564,
    0x736569, 0x73686c, 0x736872, 0x736d62, 0x737461, 0x737470, 0x737478, 0x737479,
    0x73747a, 0x746178, 0x746179, 0x747262, 0x747362, 0x747378, 0x747861, 0x747873,
    0x747961, 0x776169
};

static const uint16_t disasm_w65c02[] = {
    0x020f, 0x0a37, 0x0136, 0x0136, 0x0554, 0x0537, 0x0502, 0x1a40,
    0x0139, 0x0237, 0x0002, 0x0136, 0x0454, 0x0437, 0x0402, 0x1b03,
    0x160d, 0x1037, 0x1437, 0x0136, 0x0553, 0x0837, 0x0802, 0x1a40,
    0x0112, 0x0d37, 0x002d, 0x0136, 0x0453, 0x0737, 0x0702, 0x1b03,
    0x0431, 0x0a01, 0x0136, 0x0136, 0x0509, 0x0501, 0x0541, 0x1a40,
    0x013d, 0x0201, 0x0041, 0x0136, 0x0409, 0x0401, 0x0441, 0x1b03,
    0x160b, 0x1001, 0x1401, 0x0136, 0x0809, 0x0801, 0x0841, 0x1a40,
    0x0146, 0x0d01, 0x001c, 0x0136, 0x0709, 0x0701, 0x0741, 0x1b03,
    0x0143, 0x0a1f, 0x0136, 0x0136, 0x0536, 0x051f, 0x0535, 0x1a40,
    0x0138, 0x021f, 0x0035, 0x0136, 0x0430, 0x041f, 0x0435, 0x1b03,
    0x1610, 0x101f, 0x141f, 0x0136, 0x0836, 0x081f, 0x0835, 0x1a40,
    0x0114, 0x0d1f, 0x013b, 0x0136, 0x0136, 0x071f, 0x0735, 0x1b03,
    0x0144, 0x0a00, 0x0136, 0x0136, 0x0550, 0x0500, 0x0542, 0x1a40,
    0x013c, 0x0200, 0x0042, 0x0136, 0x1330, 0x0400, 0x0442, 0x1b03,
    0x1611, 0x1000, 0x1400, 0x0136, 0x0850, 0x0800, 0x0842, 0x1a40,
    0x0148, 0x0d00, 0x013f, 0x0136, 0x0930, 0x0700, 0x0742, 0x1b03,
    0x160e, 0x0a4c, 0x0236, 0x0136, 0x054f, 0x054c, 0x054e, 0x1a4b,
    0x011e, 0x0209, 0x0156, 0x0136, 0x044f, 0x044c, 0x044e, 0x1b04,
    0x1605, 0x104c, 0x144c, 0x0136, 0x084f, 0x084c, 0x0e4e, 0x1a4b,
    0x0158, 0x0d4c, 0x0157, 0x0136, 0x0450, 0x074c, 0x0750, 0x1b04,
    0x0234, 0x0a32, 0x0233, 0x0136, 0x0534, 0x0532, 0x0533, 0x1a4b,
    0x0152, 0x0232, 0x0151, 0x0136, 0x0434, 0x0432, 0x0433, 0x1b04,
    0x1606, 0x1032, 0x1432, 0x0136, 0x0834, 0x0832, 0x0e33, 0x1a4b,
    0x0116, 0x0d32, 0x0155, 0x0136, 0x0734, 0x0732, 0x0d33, 0x1b04,
    0x021a, 0x0a17, 0x0136, 0x0136, 0x051a, 0x0517, 0x051c, 0x1a4b,
    0x012f, 0x0217, 0x011d, 0x0159, 0x041a, 0x0417, 0x041c, 0x1b04,
    0x160c, 0x1017, 0x1417, 0x0136, 0x0136, 0x0817, 0x081c, 0x1a4b,
    0x0113, 0x0d17, 0x013a, 0x014d, 0x0436, 0x0717, 0x071c, 0x1b04,
    0x0219, 0x0a45, 0x0136, 0x0136, 0x0519, 0x0545, 0x052d, 0x1a4b,
    0x012e, 0x0245, 0x0136, 0x0136, 0x0419, 0x0445, 0x042d, 0x1b04,
    0x1607, 0x1045, 0x1445, 0x0136, 0x0136, 0x0845, 0x082d, 0x1a4b,
    0x0147, 0x0d45, 0x013e, 0x0136, 0x0136, 0x0745, 0x072d, 0x1b04
};

static const uint8_t alias_w65c02[] = {
    0, 1, 2, 3, 4, 5, 6, 7, 6, 9, 5, 11, 12, 13, 14, 15,
    16, 17, 18, 19, 20, 80, 22, 23, 23, 25, 26, 28, 28, 29, 30, 31,
    32, 33, 34, 33, 32, 37, 38, 39, 40, 41, 42, 77, 45, 45, 46, 47,
    48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63,
    64, 65, 66, 67, 68, 69, 70, 71, 72, 2, 53, 75, 76, 77, 78, 79,
    80, 81, 82, 83, 84, 85, 86, 87, 88, 89
};

const struct cpu_s w65c02 = {
    "w65c02",
    mnemonic_w65c02,
    opcode_w65c02,
    disasm_w65c02,
    alias_w65c02,
    0x1848001,
    90,
    0xffff,
    48,
    -1
};

static const uint8_t opcode_c65ce02[] = {
    118, 119, 2, 181, 182, 176, 177, 183, 184, 185, 184, 6, 183, 186, 187, 188,
    189, 11, 190, 191, 192, 15, 16, 135, 17, 20, 120, 120, 23, 24, 193, 25,
    194, 195, 27, 28, 92, 121, 30, 31, 32, 31, 30, 33, 34, 35, 36, 37,
    38, 40, 196, 197, 42, 43, 87, 122, 198, 199, 113, 200, 201, 51, 202, 54,
    124, 59, 63, 203, 64, 65, 39, 66, 69, 70, 71, 102, 179, 72, 73, 204,
    74, 205, 206, 125, 78, 79, 207, 80, 2, 51, 180, 208, 209, 210, 19, 86,
    88, 89, 62, 90, 91, 93, 94, 61, 95, 96, 98, 68, 75
};

static const uint32_t mnemonic_c65ce02[] = {
    0x616463, 0x616e64, 0x61736c, 0x617372, 0x617377, 0x626272, 0x626273, 0x626363,
    0x626373, 0x626571, 0x626765, 0x626974, 0x626c74, 0x626d69, 0x626e65, 0x62706c,
    0x627261, 0x62726b, 0x627372, 0x627663, 0x627673, 0x636c63, 0x636c64, 0x636c65,
    0x636c69, 0x636c76, 0x636d70, 0x637061, 0x637078, 0x637079, 0x63707a, 0x646561,
    0x646563, 0x646577, 0x646578, 0x646579, 0x64657a, 0x656f72, 0x676363, 0x676373,
    0x676571, 0x676765, 0x676c74, 0x676d69, 0x676e65, 0x67706c, 0x677261, 0x677663,
    0x677673, 0x696e61, 0x696e63, 0x696e77, 0x696e78, 0x696e79, 0x696e7a, 0x6a6d70,
    0x6a7372, 0x6c6461, 0x6c6478, 0x6c6479, 0x6c647a, 0x6c7372, 0x6e6567, 0x6e6f70,
    0x6f7261, 0x706861, 0x706870, 0x706877, 0x706878, 0x706879, 0x70687a, 0x706c61,
    0x706c70, 0x706c78, 0x706c79, 0x706c7a, 0x726d62, 0x726f6c, 0x726f72, 0x726f77,
    0x727469, 0x72746e, 0x727473, 0x736263, 0x736563, 0x736564, 0x736565, 0x736569,
    0x73686c, 0x736872, 0x736d62, 0x737461, 0x737478, 0x737479, 0x73747a, 0x746162,
    0x746178, 0x746179, 0x74617a, 0x746261, 0x747262, 0x747362, 0x747378, 0x747379,
    0x747861, 0x747873, 0x747961, 0x747973, 0x747a61
};

static const uint16_t disasm_c65ce02[] = {
    0x0211, 0x0a40, 0x0117, 0x0156, 0x0565, 0x0540, 0x0502, 0x1a4c,
    0x0142, 0x0240, 0x0002, 0x0167, 0x0465, 0x0440, 0x0402, 0x1b05,
    0x160f, 0x1040, 0x1c40, 0x150f, 0x0564, 0x0840, 0x0802, 0x1a4c,
    0x0115, 0x0d40, 0x0032, 0x0136, 0x0464, 0x0740, 0x0702, 0x1b05,
    0x0438, 0x0a01, 0x1338, 0x0938, 0x050b, 0x0501, 0x054d, 0x1a4c,
    0x0148, 0x0201, 0x004d, 0x016b, 0x040b, 0x0401, 0x044d, 0x1b05,
    0x160d, 0x1001, 0x1c01, 0x150d, 0x080b, 0x0801, 0x084d, 0x1a4c,
    0x0154, 0x0d01, 0x0020, 0x0124, 0x070b, 0x0701, 0x074d, 0x1b05,
    0x0150, 0x0a25, 0x003e, 0x0003, 0x0503, 0x0525, 0x053d, 0x1a4c,
    0x0141, 0x0225, 0x003d, 0x0162, 0x0437, 0x0425, 0x043d, 0x1b05,
    0x1613, 0x1025, 0x1c25, 0x1513, 0x0803, 0x0825, 0x083d, 0x1a4c,
    0x0118, 0x0d25, 0x0145, 0x015f, 0x013f, 0x0725, 0x073d, 0x1b05,
    0x0152, 0x0a00, 0x0252, 0x1512, 0x055e, 0x0500, 0x054e, 0x1a4c,
    0x0147, 0x0200, 0x004e, 0x016c, 0x1337, 0x0400, 0x044e, 0x1b05,
    0x1614, 0x1000, 0x1c00, 0x1514, 0x085e, 0x0800, 0x084e, 0x1a4c,
    0x0157, 0x0d00, 0x014a, 0x0163, 0x0937, 0x0700, 0x074e, 0x1b05,
    0x1610, 0x0a5b, 0x0c5b, 0x1510, 0x055d, 0x055b, 0x055c, 0x1a5a,
    0x0123, 0x020b, 0x0168, 0x075d, 0x045d, 0x045b, 0x045c, 0x1b06,
    0x1607, 0x105b, 0x1c5b, 0x1507, 0x085d, 0x085b, 0x0e5c, 0x1a5a,
    0x016a, 0x0d5b, 0x0169, 0x0d5c, 0x045e, 0x075b, 0x075e, 0x1b06,
    0x023b, 0x0a39, 0x023a, 0x023c, 0x053b, 0x0539, 0x053a, 0x1a5a,
    0x0161, 0x0239, 0x0160, 0x043c, 0x043b, 0x0439, 0x043a, 0x1b06,
    0x1608, 0x1039, 0x1c39, 0x1508, 0x083b, 0x0839, 0x0e3a, 0x1a5a,
    0x0119, 0x0d39, 0x0166, 0x073c, 0x073b, 0x0739, 0x0d3a, 0x1b06,
    0x021d, 0x0a1a, 0x021e, 0x0521, 0x051d, 0x051a, 0x0520, 0x1a5a,
    0x0135, 0x021a, 0x0122, 0x0404, 0x041d, 0x041a, 0x0420, 0x1b06,
    0x160e, 0x101a, 0x1c1a, 0x150e, 0x051e, 0x081a, 0x0820, 0x1a5a,
    0x0116, 0x0d1a, 0x0144, 0x0146, 0x041e, 0x071a, 0x0720, 0x1b06,
    0x021c, 0x0a53, 0x0c39, 0x0533, 0x051c, 0x0553, 0x0532, 0x1a5a,
    0x0134, 0x0253, 0x013f, 0x044f, 0x041c, 0x0453, 0x0432, 0x1b06,
    0x1609, 0x1053, 0x1c53, 0x1509, 0x0243, 0x0853, 0x0832, 0x1a5a,
    0x0155, 0x0d53, 0x0149, 0x014b, 0x0443, 0x0753, 0x0732, 0x1b06
};

static const uint8_t alias_c65ce02[] = {
    0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 8, 11, 7, 13, 14, 15,
    16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 26, 28, 29, 30, 32,
    32, 33, 34, 35, 36, 37, 38, 39, 40, 39, 38, 43, 44, 45, 46, 47,
    48, 50, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63,
    64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79,
    80, 82, 82, 83, 84, 85, 86, 87, 2, 61, 90, 91, 92, 93, 94, 95,
    96, 97, 98, 99, 100, 101, 102, 103, 104, 105, 106, 107, 108
};

const struct cpu_s c65ce02 = {
    "65ce02",
    mnemonic_c65ce02,
    opcode_c65ce02,
    disasm_c65ce02,
    alias_c65ce02,
    0x3848003,
    109,
    0xffff,
    55,
    -1
};

static const uint8_t opcode_c4510[] = {
    118, 119, 2, 181, 182, 176, 177, 183, 184, 185, 184, 6, 183, 186, 187, 188,
    189, 11, 190, 191, 192, 15, 16, 135, 17, 20, 120, 120, 23, 24, 193, 25,
    194, 195, 27, 28, 92, 54, 121, 30, 31, 32, 31, 30, 33, 34, 35, 36,
    37, 38, 40, 196, 197, 42, 43, 87, 122, 198, 199, 113, 200, 201, 51, 174,
    202, 54, 124, 59, 63, 203, 64, 65, 39, 66, 69, 70, 71, 102, 179, 72,
    73, 204, 74, 205, 206, 125, 78, 79, 207, 80, 2, 51, 180, 208, 209, 210,
    19, 86, 88, 89, 62, 90, 91, 93, 94, 61, 95, 96, 98, 68, 75
};

static const uint32_t mnemonic_c4510[] = {
    0x616463, 0x616e64, 0x61736c, 0x617372, 0x617377, 0x626272, 0x626273, 0x626363,
    0x626373, 0x626571, 0x626765, 0x626974, 0x626c74, 0x626d69, 0x626e65, 0x62706c,
    0x627261, 0x62726b, 0x627372, 0x627663, 0x627673, 0x636c63, 0x636c64, 0x636c65,
    0x636c69, 0x636c76, 0x636d70, 0x637061, 0x637078, 0x637079, 0x63707a, 0x646561,
    0x646563, 0x646577, 0x646578, 0x646579, 0x64657a, 0x656f6d, 0x656f72, 0x676363,
    0x676373, 0x676571, 0x676765, 0x676c74, 0x676d69, 0x676e65, 0x67706c, 0x677261,
    0x677663, 0x677673, 0x696e61, 0x696e63, 0x696e77, 0x696e78, 0x696e79, 0x696e7a,
    0x6a6d70, 0x6a7372, 0x6c6461, 0x6c6478, 0x6c6479, 0x6c647a, 0x6c7372, 0x6d6170,
    0x6e6567, 0x6e6f70, 0x6f7261, 0x706861, 0x706870, 0x706877, 0x706878, 0x706879,
    0x70687a, 0x706c61, 0x706c70, 0x706c78, 0x706c79, 0x706c7a, 0x726d62, 0x726f6c,
    0x726f72, 0x726f77, 0x727469, 0x72746e, 0x727473, 0x736263, 0x736563, 0x736564,
    0x736565, 0x736569, 0x73686c, 0x736872, 0x736d62, 0x737461, 0x737478, 0x737479,
    0x73747a, 0x746162, 0x746178, 0x746179, 0x74617a, 0x746261, 0x747262, 0x747362,
    0x747378, 0x747379, 0x747861, 0x747873, 0x747961, 0x747973, 0x747a61
};

static const uint16_t disasm_c4510[] = {
    0x0211, 0x0a42, 0x0117, 0x0158, 0x0567, 0x0542, 0x0502, 0x1a4e,
    0x0144, 0x0242, 0x0002, 0x0169, 0x0467, 0x0442, 0x0402, 0x1b05,
    0x160f, 0x1042, 0x1c42, 0x150f, 0x0566, 0x0842, 0x0802, 0x1a4e,
    0x0115, 0x0d42, 0x0033, 0x0137, 0x0466, 0x0742, 0x0702, 0x1b05,
    0x0439, 0x0a01, 0x1339, 0x0939, 0x050b, 0x0501, 0x054f, 0x1a4e,
    0x014a, 0x0201, 0x004f, 0x016d, 0x040b, 0x0401, 0x044f, 0x1b05,
    0x160d, 0x1001, 0x1c01, 0x150d, 0x080b, 0x0801, 0x084f, 0x1a4e,
    0x0156, 0x0d01, 0x0020, 0x0124, 0x070b, 0x0701, 0x074f, 0x1b05,
    0x0152, 0x0a26, 0x0040, 0x0003, 0x0503, 0x0526, 0x053e, 0x1a4e,
    0x0143, 0x0226, 0x003e, 0x0164, 0x0438, 0x0426, 0x043e, 0x1b05,
    0x1613, 0x1026, 0x1c26, 0x1513, 0x0803, 0x0826, 0x083e, 0x1a4e,
    0x0118, 0x0d26, 0x0147, 0x0161, 0x013f, 0x0726, 0x073e, 0x1b05,
    0x0154, 0x0a00, 0x0254, 0x1512, 0x0560, 0x0500, 0x0550, 0x1a4e,
    0x0149, 0x0200, 0x0050, 0x016e, 0x1338, 0x0400, 0x0450, 0x1b05,
    0x1614, 0x1000, 0x1c00, 0x1514, 0x0860, 0x0800, 0x0850, 0x1a4e,
    0x0159, 0x0d00, 0x014c, 0x0165, 0x0938, 0x0700, 0x0750, 0x1b05,
    0x1610, 0x0a5d, 0x0c5d, 0x1510, 0x055f, 0x055d, 0x055e, 0x1a5c,
    0x0123, 0x020b, 0x016a, 0x075f, 0x045f, 0x045d, 0x045e, 0x1b06,
    0x1607, 0x105d, 0x1c5d, 0x1507, 0x085f, 0x085d, 0x0e5e, 0x1a5c,
    0x016c, 0x0d5d, 0x016b, 0x0d5e, 0x0460, 0x075d, 0x0760, 0x1b06,
    0x023c, 0x0a3a, 0x023b, 0x023d, 0x053c, 0x053a, 0x053b, 0x1a5c,
    0x0163, 0x023a, 0x0162, 0x043d, 0x043c, 0x043a, 0x043b, 0x1b06,
    0x1608, 0x103a, 0x1c3a, 0x1508, 0x083c, 0x083a, 0x0e3b, 0x1a5c,
    0x0119, 0x0d3a, 0x0168, 0x073d, 0x073c, 0x073a, 0x0d3b, 0x1b06,
    0x021d, 0x0a1a, 0x021e, 0x0521, 0x051d, 0x051a, 0x0520, 0x1a5c,
    0x0136, 0x021a, 0x0122, 0x0404, 0x041d, 0x041a, 0x0420, 0x1b06,
    0x160e, 0x101a, 0x1c1a, 0x150e, 0x051e, 0x081a, 0x0820, 0x1a5c,
    0x0116, 0x0d1a, 0x0146, 0x0148, 0x041e, 0x071a, 0x0720, 0x1b06,
    0x021c, 0x0a55, 0x0c3a, 0x0534, 0x051c, 0x0555, 0x0533, 0x1a5c,
    0x0135, 0x0255, 0x0141, 0x0451, 0x041c, 0x0455, 0x0433, 0x1b06,
    0x1609, 0x1055, 0x1c55, 0x1509, 0x0245, 0x0855, 0x0833, 0x1a5c,
    0x0157, 0x0d55, 0x014b, 0x014d, 0x0445, 0x0755, 0x0733, 0x1b06
};

static const uint8_t alias_c4510[] = {
    0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 8, 11, 7, 13, 14, 15,
    16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 26, 28, 29, 30, 32,
    32, 33, 34, 35, 36, 65, 38, 39, 40, 41, 40, 39, 44, 45, 46, 47,
    48, 49, 51, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63,
    64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79,
    80, 81, 82, 84, 84, 85, 86, 87, 88, 89, 2, 62, 92, 93, 94, 95,
    96, 97, 98, 99, 100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110
};

const struct cpu_s c4510 = {
    "4510",
    mnemonic_c4510,
    opcode_c4510,
    disasm_c4510,
    alias_c4510,
    0x3848003,
    111,
    0xffff,
    56,
    -1
};

const char *reg_names = "axysdrizbkp";

const char *const addr_modes[ADR_LEN+1] = {
    "a",      /* ADR_REG */
    " ",      /* ADR_IMPLIED */
    "#",      /* ADR_IMMEDIATE */
    " ",      /* ADR_LONG */
    " ",      /* ADR_ADDR */
    " ",      /* ADR_ZP */
    " ,x",    /* ADR_LONG_X */
    " ,x",    /* ADR_ADDR_X */
    " ,x",    /* ADR_ZP_X */
    "(,x)",   /* ADR_ADDR_X_I */
    "(,x)",   /* ADR_ZP_X_I */
    " ,s",    /* ADR_ZP_S */
    "(,s),y", /* ADR_ZP_S_I_Y */
    " ,y",    /* ADR_ADDR_Y */
    " ,y",    /* ADR_ZP_Y */
    "[],y",   /* ADR_ZP_LI_Y */
    "(),y",   /* ADR_ZP_I_Y */
    "[]",     /* ADR_ADDR_LI */
    "[]",     /* ADR_ZP_LI */
    "()",     /* ADR_ADDR_I */
    "()",     /* ADR_ZP_I */
    " ",      /* ADR_REL_L */
    " ",      /* ADR_REL */
    "#",      /* ADR_MOVE */
    " ,r",    /* ADR_ZP_R */
    "(,r),y", /* ADR_ZP_R_I_Y */
    " ",      /* ADR_BIT_ZP */
    " ",      /* ADR_BIT_ZP_REL */
    "(),z",   /* ADR_ZP_I_Z */
};

const uint8_t regopcode_table[][REG_LEN] = {
    { ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* ___ */
    { 0x0A, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* ASL, SHL */
    { 0x3A, 0xCA, 0x88, ____, ____, ____, ____, ____, ____, ____, ____ }, /* DEC */
    { 0x1A, 0xE8, 0xC8, ____, ____, ____, ____, ____, ____, ____, ____ }, /* INC */
    { ____, 0x8A, 0x98, 0x3B, 0x7B, ____, ____, ____, ____, ____, ____ }, /* LDA */
    { 0xAA, ____, 0xBB, 0xBA, ____, ____, ____, ____, ____, ____, ____ }, /* LDX */
    { 0xA8, 0x9B, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* LDY */
    { 0x4A, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* LSR, SHR */
    { 0x2A, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* ROL */
    { 0x6A, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* ROR */
    { ____, ____, ____, 0x1B, ____, ____, ____, ____, ____, ____, ____ }, /* STA */
    { ____, ____, ____, 0x9A, ____, ____, ____, ____, ____, ____, ____ }, /* STX */
    { ____, 0xCA, 0x88, ____, ____, ____, ____, ____, ____, ____, ____ }, /* DEC */
    { ____, 0xE8, 0xC8, ____, ____, ____, ____, ____, ____, ____, ____ }, /* INC */
    { ____, 0x8A, 0x98, ____, ____, ____, ____, ____, ____, ____, ____ }, /* LDA */
    { 0xAA, ____, ____, 0xBA, ____, ____, ____, ____, ____, ____, ____ }, /* LDX */
    { 0xA8, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* LDY */
    { ____, 0x8A, 0x98, ____, 0xAF, ____, ____, ____, ____, ____, ____ }, /* LDA */
    { 0xAA, ____, 0xBB, 0xBA, ____, 0xAB, 0xDC, ____, ____, ____, ____ }, /* LDX */
    { 0x43, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* ASR */
    { 0x3A, 0xCA, 0x88, ____, ____, ____, ____, 0x3B, ____, ____, ____ }, /* DEC */
    { 0x1A, 0xE8, 0xC8, ____, ____, ____, ____, 0x1B, ____, ____, ____ }, /* INC */
    { ____, 0x8A, 0x98, ____, ____, ____, ____, 0x6B, 0x7B, ____, ____ }, /* LDA */
    { 0xA8, ____, ____, 0x0B, ____, ____, ____, ____, ____, ____, ____ }, /* LDY */
    { 0x4B, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* LDZ */
    { 0x42, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* NEG */
    { ____, ____, ____, ____, ____, ____, ____, ____, 0x5B, ____, ____ }, /* STA */
    { ____, ____, ____, 0x2B, ____, ____, ____, ____, ____, ____, ____ }  /* STY */
};

const uint8_t opcode_table[][ADR_LEN] = {
    {  0, ____, 0x69, 0x6F, 0x6D, 0x65, 0x7F, 0x7D, 0x75, ____, 0x61, 0x63, 0x73, 0x79, ____, 0x77, 0x71, ____, 0x67, ____, 0x72, ____, ____, ____, ____, ____, ____, ____ }, /* ADC */
    {  0, ____, 0x29, 0x2F, 0x2D, 0x25, 0x3F, 0x3D, 0x35, ____, 0x21, 0x23, 0x33, 0x39, ____, 0x37, 0x31, ____, 0x27, ____, 0x32, ____, ____, ____, ____, ____, ____, ____ }, /* AND */
    {  1, 0x0A, ____, ____, 0x0E, 0x06, ____, 0x1E, 0x16, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* ASL, SHL */
    {  0, ____, 0x90, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x90, ____, ____, ____, ____, ____ }, /* BCC, BLT */
    {  0, ____, 0xB0, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0xB0, ____, ____, ____, ____, ____ }, /* BCS, BGE */
    {  0, ____, 0xF0, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0xF0, ____, ____, ____, ____, ____ }, /* BEQ */
    {  0, ____, 0x89, ____, 0x2C, 0x24, ____, 0x3C, 0x34, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* BIT */
    {  0, ____, 0x30, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x30, ____, ____, ____, ____, ____ }, /* BMI */
    {  0, ____, 0xD0, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0xD0, ____, ____, ____, ____, ____ }, /* BNE */
    {  0, ____, 0x10, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x10, ____, ____, ____, ____, ____ }, /* BPL */
    {  0, ____, 0x80, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x80, ____, ____, ____, ____, ____ }, /* BRA */
    {  0, 0x00, 0x00, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* BRK */
    {  0, ____, 0x82, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x82, ____, ____, ____, ____, ____, ____ }, /* BRL */
    {  0, ____, 0x50, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x50, ____, ____, ____, ____, ____ }, /* BVC */
    {  0, ____, 0x70, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x70, ____, ____, ____, ____, ____ }, /* BVS */
    {  0, 0x18, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* CLC */
    {  0, 0xD8, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* CLD */
    {  0, 0x58, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* CLI */
    {  0, ____, 0xC2, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* CLP, REP */
    {  0, ____, ____, ____, 0x9C, 0x64, ____, 0x9E, 0x74, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* CLR, STZ */
    {  0, 0xB8, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* CLV */
    {  0, ____, 0xC9, 0xCF, 0xCD, 0xC5, 0xDF, 0xDD, 0xD5, ____, 0xC1, 0xC3, 0xD3, 0xD9, ____, 0xD7, 0xD1, ____, 0xC7, ____, 0xD2, ____, ____, ____, ____, ____, ____, ____ }, /* CMP, CPA */
    {  0, 0x02, 0x02, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* COP, CSP */
    {  0, ____, 0xE0, ____, 0xEC, 0xE4, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* CPX */
    {  0, ____, 0xC0, ____, 0xCC, 0xC4, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* CPY */
    {  0, 0x3A, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* DEA */
    {  2, ____, ____, ____, 0xCE, 0xC6, ____, 0xDE, 0xD6, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* DEC */
    {  0, 0xCA, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* DEX */
    {  0, 0x88, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* DEY */
    {  0, ____, 0x49, 0x4F, 0x4D, 0x45, 0x5F, 0x5D, 0x55, ____, 0x41, 0x43, 0x53, 0x59, ____, 0x57, 0x51, ____, 0x47, ____, 0x52, ____, ____, ____, ____, ____, ____, ____ }, /* EOR */
    {  0, ____, ____, ____, 0x4C, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x90, ____, ____, ____, ____, ____ }, /* GCC, GLT */
    {  0, ____, ____, ____, 0x4C, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0xB0, ____, ____, ____, ____, ____ }, /* GCS, GGE */
    {  0, ____, ____, ____, 0x4C, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0xF0, ____, ____, ____, ____, ____ }, /* GEQ */
    {  0, ____, ____, ____, 0x4C, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x30, ____, ____, ____, ____, ____ }, /* GMI */
    {  0, ____, ____, ____, 0x4C, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0xD0, ____, ____, ____, ____, ____ }, /* GNE */
    {  0, ____, ____, ____, 0x4C, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x10, ____, ____, ____, ____, ____ }, /* GPL */
    {  0, ____, ____, ____, 0x4C, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x80, ____, ____, ____, ____, ____ }, /* GRA */
    {  0, ____, ____, ____, 0x4C, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x50, ____, ____, ____, ____, ____ }, /* GVC */
    {  0, ____, ____, ____, 0x4C, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x70, ____, ____, ____, ____, ____ }, /* GVS */
    {  0, 0xDB, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* HLT, PHZ, STP */
    {  0, 0x1A, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* INA */
    {  3, ____, ____, ____, 0xEE, 0xE6, ____, 0xFE, 0xF6, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* INC */
    {  0, 0xE8, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* INX */
    {  0, 0xC8, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* INY */
    {  0, ____, ____, 0x5C, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0xDC, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* JML */
    {  0, ____, ____, ____, 0x4C, ____, ____, ____, ____, 0x7C, ____, ____, ____, ____, ____, ____, ____, 0xDC, ____, 0x6C, ____, ____, ____, ____, ____, ____, ____, ____ }, /* JMP */
    {  0, ____, ____, 0x22, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* JSL */
    {  0, ____, ____, ____, 0x20, ____, ____, ____, ____, 0xFC, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* JSR */
    {  4, ____, 0xA9, 0xAF, 0xAD, 0xA5, 0xBF, 0xBD, 0xB5, ____, 0xA1, 0xA3, 0xB3, 0xB9, ____, 0xB7, 0xB1, ____, 0xA7, ____, 0xB2, ____, ____, ____, ____, ____, ____, ____ }, /* LDA */
    {  5, ____, 0xA2, ____, 0xAE, 0xA6, ____, ____, ____, ____, ____, ____, ____, 0xBE, 0xB6, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* LDX */
    {  6, ____, 0xA0, ____, 0xAC, 0xA4, ____, 0xBC, 0xB4, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* LDY */
    {  7, 0x4A, ____, ____, 0x4E, 0x46, ____, 0x5E, 0x56, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* LSR, SHR */
    {  0, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x54, ____, ____, ____, ____ }, /* MVN */
    {  0, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x44, ____, ____, ____, ____ }, /* MVP */
    {  0, 0xEA, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* EOM, NOP */
    {  0, ____, 0x09, 0x0F, 0x0D, 0x05, 0x1F, 0x1D, 0x15, ____, 0x01, 0x03, 0x13, 0x19, ____, 0x17, 0x11, ____, 0x07, ____, 0x12, ____, ____, ____, ____, ____, ____, ____ }, /* ORA */
    {  0, ____, 0xF4, ____, 0xF4, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* PEA */
    {  0, ____, ____, ____, ____, 0xD4, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0xD4, ____, ____, ____, ____, ____, ____, ____ }, /* PEI */
    {  0, ____, 0x62, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x62, ____, ____, ____, ____, ____, ____ }, /* PER */
    {  0, 0x48, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* PHA */
    {  0, 0x8B, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* PHB, TXR */
    {  0, 0x0B, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* PHD, RHI, TSY */
    {  0, 0x4B, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* PHK, RHA, TAZ */
    {  0, 0x08, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* PHP */
    {  0, 0xDA, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* PHX */
    {  0, 0x5A, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* PHY */
    {  0, 0x68, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* PLA */
    {  0, 0xAB, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* PLB, TRX */
    {  0, 0x2B, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* PLD, RLI, TYS */
    {  0, 0x28, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* PLP */
    {  0, 0xFA, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* PLX */
    {  0, 0x7A, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* PLY */
    {  8, 0x2A, ____, ____, 0x2E, 0x26, ____, 0x3E, 0x36, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* ROL */
    {  9, 0x6A, ____, ____, 0x6E, 0x66, ____, 0x7E, 0x76, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* ROR */
    {  0, 0x40, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* RTI */
    {  0, 0x6B, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* RLA, RTL, TZA */
    {  0, 0x60, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* RTS */
    {  0, ____, 0xE9, 0xEF, 0xED, 0xE5, 0xFF, 0xFD, 0xF5, ____, 0xE1, 0xE3, 0xF3, 0xF9, ____, 0xF7, 0xF1, ____, 0xE7, ____, 0xF2, ____, ____, ____, ____, ____, ____, ____ }, /* SBC */
    {  0, 0x38, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* SEC */
    {  0, 0xF8, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* SED */
    {  0, 0x78, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* SEI */
    {  0, ____, 0xE2, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* SEP */
    { 10, ____, ____, 0x8F, 0x8D, 0x85, 0x9F, 0x9D, 0x95, ____, 0x81, 0x83, 0x93, 0x99, ____, 0x97, 0x91, ____, 0x87, ____, 0x92, ____, ____, ____, ____, ____, ____, ____ }, /* STA */
    { 11, ____, ____, ____, 0x8E, 0x86, ____, ____, ____, ____, ____, ____, ____, ____, 0x96, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* STX */
    {  0, ____, ____, ____, 0x8C, 0x84, ____, ____, 0x94, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* STY */
    {  0, 0xEB, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* SWA, XBA */
    {  0, 0x5B, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* RHY, TAB, TAD, TCD */
    {  0, 0x1B, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* INZ, RHX, TAS, TCS */
    {  0, 0xAA, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* TAX */
    {  0, 0xA8, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* TAY */
    {  0, 0x7B, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* RLY, TBA, TDA, TDC */
    {  0, ____, ____, ____, 0x1C, 0x14, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* TRB */
    {  0, 0x3B, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* DEZ, RLX, TSA, TSC */
    {  0, ____, ____, ____, 0x0C, 0x04, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* TSB */
    {  0, 0xBA, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* TSX */
    {  0, 0x8A, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* TXA */
    {  0, 0x9A, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* TXS */
    {  0, 0x9B, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* TXY */
    {  0, 0x98, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* TYA */
    {  0, 0xBB, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* TYX */
    {  0, 0xCB, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* WAI */
    {  0, ____, 0x42, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* SIR, WDM */
    {  0, 0xFB, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* PLZ, XCE */
    {  0, ____, 0x69, ____, 0x6D, 0x65, ____, 0x7D, 0x75, ____, 0x61, ____, ____, 0x79, ____, ____, 0x71, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* ADC */
    {  0, ____, 0x29, ____, 0x2D, 0x25, ____, 0x3D, 0x35, ____, 0x21, ____, ____, 0x39, ____, ____, 0x31, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* AND */
    {  0, ____, ____, ____, 0x2C, 0x24, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* BIT */
    {  0, ____, 0xC9, ____, 0xCD, 0xC5, ____, 0xDD, 0xD5, ____, 0xC1, ____, ____, 0xD9, ____, ____, 0xD1, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* CMP, CPA */
    { 12, ____, ____, ____, 0xCE, 0xC6, ____, 0xDE, 0xD6, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* DEC */
    {  0, ____, 0x49, ____, 0x4D, 0x45, ____, 0x5D, 0x55, ____, 0x41, ____, ____, 0x59, ____, ____, 0x51, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* EOR */
    { 13, ____, ____, ____, 0xEE, 0xE6, ____, 0xFE, 0xF6, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* INC */
    {  0, ____, ____, ____, 0x4C, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x6C, ____, ____, ____, ____, ____, ____, ____, ____ }, /* JMP */
    {  0, ____, ____, ____, 0x20, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* JSR */
    { 14, ____, 0xA9, ____, 0xAD, 0xA5, ____, 0xBD, 0xB5, ____, 0xA1, ____, ____, 0xB9, ____, ____, 0xB1, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* LDA */
    { 15, ____, 0xA2, ____, 0xAE, 0xA6, ____, ____, ____, ____, ____, ____, ____, 0xBE, 0xB6, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* LDX */
    { 16, ____, 0xA0, ____, 0xAC, 0xA4, ____, 0xBC, 0xB4, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* LDY */
    {  0, ____, 0x09, ____, 0x0D, 0x05, ____, 0x1D, 0x15, ____, 0x01, ____, ____, 0x19, ____, ____, 0x11, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* ORA */
    {  0, ____, 0xE9, ____, 0xED, 0xE5, ____, 0xFD, 0xF5, ____, 0xE1, ____, ____, 0xF9, ____, ____, 0xF1, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* SBC */
    {  0, ____, ____, ____, 0x8D, 0x85, ____, 0x9D, 0x95, ____, 0x81, ____, ____, 0x99, ____, ____, 0x91, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* STA */
    {  0, ____, 0x69, ____, 0x6D, 0x65, ____, 0x7D, 0x75, ____, 0x61, ____, ____, 0x79, ____, ____, 0x71, ____, ____, ____, 0x72, ____, ____, ____, ____, ____, ____, ____ }, /* ADC */
    {  0, ____, 0x29, ____, 0x2D, 0x25, ____, 0x3D, 0x35, ____, 0x21, ____, ____, 0x39, ____, ____, 0x31, ____, ____, ____, 0x32, ____, ____, ____, ____, ____, ____, ____ }, /* AND */
    {  0, ____, 0xC9, ____, 0xCD, 0xC5, ____, 0xDD, 0xD5, ____, 0xC1, ____, ____, 0xD9, ____, ____, 0xD1, ____, ____, ____, 0xD2, ____, ____, ____, ____, ____, ____, ____ }, /* CMP, CPA */
    {  0, ____, 0x49, ____, 0x4D, 0x45, ____, 0x5D, 0x55, ____, 0x41, ____, ____, 0x59, ____, ____, 0x51, ____, ____, ____, 0x52, ____, ____, ____, ____, ____, ____, ____ }, /* EOR */
    {  0, ____, ____, ____, 0x4C, ____, ____, ____, ____, 0x7C, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x6C, ____, ____, ____, ____, ____, ____, ____, ____ }, /* JMP */
    { 14, ____, 0xA9, ____, 0xAD, 0xA5, ____, 0xBD, 0xB5, ____, 0xA1, ____, ____, 0xB9, ____, ____, 0xB1, ____, ____, ____, 0xB2, ____, ____, ____, ____, ____, ____, ____ }, /* LDA */
    {  0, ____, 0x09, ____, 0x0D, 0x05, ____, 0x1D, 0x15, ____, 0x01, ____, ____, 0x19, ____, ____, 0x11, ____, ____, ____, 0x12, ____, ____, ____, ____, ____, ____, ____ }, /* ORA */
    {  0, ____, 0xE9, ____, 0xED, 0xE5, ____, 0xFD, 0xF5, ____, 0xE1, ____, ____, 0xF9, ____, ____, 0xF1, ____, ____, ____, 0xF2, ____, ____, ____, ____, ____, ____, ____ }, /* SBC */
    {  0, ____, ____, ____, 0x8D, 0x85, ____, 0x9D, 0x95, ____, 0x81, ____, ____, 0x99, ____, ____, 0x91, ____, ____, ____, 0x92, ____, ____, ____, ____, ____, ____, ____ }, /* STA */
    {  0, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x9F, ____, ____, 0x93, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* AHX, SHA */
    {  0, ____, 0x4B, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* ALR, ASR */
    {  0, ____, 0x0B, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* ANC */
    {  0, ____, 0x8B, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* ANE, XAA */
    {  0, ____, 0x6B, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* ARR */
    {  0, ____, 0xCB, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* AXS, SBX */
    {  0, ____, ____, ____, 0xCF, 0xC7, ____, 0xDF, 0xD7, ____, 0xC3, ____, ____, 0xDB, ____, ____, 0xD3, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* DCM, DCP */
    {  0, ____, ____, ____, 0xEF, 0xE7, ____, 0xFF, 0xF7, ____, 0xE3, ____, ____, 0xFB, ____, ____, 0xF3, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* INS, ISB, ISC */
    {  0, 0x02, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* CLE, JAM, NXT */
    {  0, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0xBB, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* LAE, LAS, LDS */
    {  0, ____, 0xAB, ____, 0xAF, 0xA7, ____, ____, ____, ____, 0xA3, ____, ____, 0xBF, 0xB7, ____, 0xB3, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* LAX */
    {  0, ____, 0xAB, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* LXA */
    {  0, 0xEA, 0x80, ____, 0x0C, 0x04, ____, 0x1C, 0x14, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* NOP */
    {  0, ____, ____, ____, 0x2F, 0x27, ____, 0x3F, 0x37, ____, 0x23, ____, ____, 0x3B, ____, ____, 0x33, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* RLA */
    {  0, ____, ____, ____, 0x6F, 0x67, ____, 0x7F, 0x77, ____, 0x63, ____, ____, 0x7B, ____, ____, 0x73, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* RRA */
    {  0, ____, ____, ____, 0x8F, 0x87, ____, ____, ____, ____, 0x83, ____, ____, ____, 0x97, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* SAX */
    {  0, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x9B, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* SHS, TAS */
    {  0, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x9E, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* SHX */
    {  0, ____, ____, ____, ____, ____, ____, 0x9C, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* SHY */
    {  0, ____, ____, ____, 0x0F, 0x07, ____, 0x1F, 0x17, ____, 0x03, ____, ____, 0x1B, ____, ____, 0x13, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* SLO */
    {  0, ____, ____, ____, 0x4F, 0x47, ____, 0x5F, 0x57, ____, 0x43, ____, ____, 0x5B, ____, ____, 0x53, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* SRE */
    {  0, ____, 0x12, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x12, ____, ____, ____, ____, ____ }, /* BRA */
    {  0, ____, ____, ____, 0x4C, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x12, ____, ____, ____, ____, ____ }, /* GRA */
    {  0, ____, 0x32, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* SAC */
    {  0, ____, 0x69, ____, 0x6D, 0x65, ____, 0x7D, 0x75, ____, 0x61, 0x63, 0x73, 0x79, ____, ____, 0x71, ____, ____, ____, 0x72, ____, ____, ____, 0x67, 0x77, ____, ____ }, /* ADC */
    {  0, ____, 0x29, ____, 0x2D, 0x25, ____, 0x3D, 0x35, ____, 0x21, 0x23, 0x33, 0x39, ____, ____, 0x31, ____, ____, ____, 0x32, ____, ____, ____, 0x27, 0x37, ____, ____ }, /* AND */
    {  0, ____, 0xC9, ____, 0xCD, 0xC5, ____, 0xDD, 0xD5, ____, 0xC1, 0xC3, 0xD3, 0xD9, ____, ____, 0xD1, ____, ____, ____, 0xD2, ____, ____, ____, 0xC7, 0xD7, ____, ____ }, /* CMP, CPA */
    {  0, ____, ____, ____, 0x6F, 0x4F, ____, 0x7F, 0x5F, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* DIV */
    {  0, 0x22, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* ENT */
    {  0, ____, 0x49, ____, 0x4D, 0x45, ____, 0x5D, 0x55, ____, 0x41, 0x43, 0x53, 0x59, ____, ____, 0x51, ____, ____, ____, 0x52, ____, ____, ____, 0x47, 0x57, ____, ____ }, /* EOR */
    { 17, ____, 0xA9, ____, 0xAD, 0xA5, ____, 0xBD, 0xB5, ____, 0xA1, 0xA3, 0xB3, 0xB9, ____, ____, 0xB1, ____, ____, ____, 0xB2, ____, ____, ____, 0xA7, 0xB7, ____, ____ }, /* LDA */
    { 18, ____, 0xA2, ____, 0xAE, 0xA6, ____, ____, ____, ____, ____, ____, ____, 0xBE, 0xB6, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* LDX */
    {  0, ____, 0xEF, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* MMU */
    {  0, ____, ____, ____, 0x2F, 0x0F, ____, 0x3F, 0x1F, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* MUL */
    {  0, 0x42, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* NXA */
    {  0, ____, 0x09, ____, 0x0D, 0x05, ____, 0x1D, 0x15, ____, 0x01, 0x03, 0x13, 0x19, ____, ____, 0x11, ____, ____, ____, 0x12, ____, ____, ____, 0x07, 0x17, ____, ____ }, /* ORA */
    {  0, 0xDF, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* PHD */
    {  0, 0xCF, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* PLD */
    {  0, ____, 0x44, ____, 0x44, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* REA */
    {  0, ____, ____, ____, ____, 0x54, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x54, ____, ____, ____, ____, ____, ____, ____ }, /* REI */
    {  0, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x82, ____, ____, ____, ____, ____ }, /* RER */
    {  0, ____, 0xE9, ____, 0xED, 0xE5, ____, 0xFD, 0xF5, ____, 0xE1, 0xE3, 0xF3, 0xF9, ____, ____, 0xF1, ____, ____, ____, 0xF2, ____, ____, ____, 0xE7, 0xF7, ____, ____ }, /* SBC */
    {  0, 0x9F, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* SEA */
    {  0, ____, ____, ____, 0x8D, 0x85, ____, 0x9D, 0x95, ____, 0x81, 0x83, 0x93, 0x99, ____, ____, 0x91, ____, ____, ____, 0x92, ____, ____, ____, 0x87, 0x97, ____, ____ }, /* STA */
    {  0, 0xBF, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* TAD */
    {  0, 0xAF, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* TDA */
    {  0, 0xDC, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* TIX */
    {  0, 0x5C, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* MAP, TXI */
    {  0, 0x8F, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* ZEA */
    {  0, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x0F }, /* BBR */
    {  0, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x8F }, /* BBS */
    {  0, 0xEA, 0x82, ____, 0xDC, 0x44, ____, ____, 0x54, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* NOP */
    {  0, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x07, ____ }, /* RMB */
    {  0, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x87, ____ }, /* SMB */
    { 19, 0x43, ____, ____, ____, 0x44, ____, ____, 0x54, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* ASR */
    {  0, ____, ____, ____, 0xCB, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* ASW */
    {  0, ____, 0x90, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x93, 0x90, ____, ____, ____, ____, ____ }, /* BCC, BLT */
    {  0, ____, 0xB0, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0xB3, 0xB0, ____, ____, ____, ____, ____ }, /* BCS, BGE */
    {  0, ____, 0xF0, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0xF3, 0xF0, ____, ____, ____, ____, ____ }, /* BEQ */
    {  0, ____, 0x30, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x33, 0x30, ____, ____, ____, ____, ____ }, /* BMI */
    {  0, ____, 0xD0, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0xD3, 0xD0, ____, ____, ____, ____, ____ }, /* BNE */
    {  0, ____, 0x10, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x13, 0x10, ____, ____, ____, ____, ____ }, /* BPL */
    {  0, ____, 0x80, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x83, 0x80, ____, ____, ____, ____, ____ }, /* BRA */
    {  0, ____, 0x63, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x63, ____, ____, ____, ____, ____, ____ }, /* BSR */
    {  0, ____, 0x50, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x53, 0x50, ____, ____, ____, ____, ____ }, /* BVC */
    {  0, ____, 0x70, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x73, 0x70, ____, ____, ____, ____, ____ }, /* BVS */
    {  0, ____, 0xC2, ____, 0xDC, 0xD4, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* CPZ */
    { 20, ____, ____, ____, 0xCE, 0xC6, ____, 0xDE, 0xD6, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* DEC */
    {  0, ____, ____, ____, ____, 0xC3, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* DEW */
    { 21, ____, ____, ____, 0xEE, 0xE6, ____, 0xFE, 0xF6, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* INC */
    {  0, ____, ____, ____, ____, 0xE3, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* INW */
    {  0, ____, ____, ____, 0x20, ____, ____, ____, ____, 0x23, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x22, ____, ____, ____, ____, ____, ____, ____, ____ }, /* JSR */
    { 22, ____, 0xA9, ____, 0xAD, 0xA5, ____, 0xBD, 0xB5, ____, 0xA1, ____, 0xE2, 0xB9, ____, ____, 0xB1, ____, ____, ____, 0xB2, ____, ____, ____, ____, ____, ____, ____ }, /* LDA */
    { 23, ____, 0xA0, ____, 0xAC, 0xA4, ____, 0xBC, 0xB4, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* LDY */
    { 24, ____, 0xA3, ____, 0xAB, ____, ____, 0xBB, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* LDZ */
    { 25, 0x42, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* NEG */
    {  0, ____, 0xF4, ____, 0xFC, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* PHW */
    {  0, ____, ____, ____, 0xEB, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* ROW */
    {  0, ____, 0x62, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* RTN */
    {  0, 0x60, 0x62, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* RTS */
    {  0, 0x03, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* SEE */
    { 26, ____, ____, ____, 0x8D, 0x85, ____, 0x9D, 0x95, ____, 0x81, ____, 0x82, 0x99, ____, ____, 0x91, ____, ____, ____, 0x92, ____, ____, ____, ____, ____, ____, ____ }, /* STA */
    { 11, ____, ____, ____, 0x8E, 0x86, ____, ____, ____, ____, ____, ____, ____, 0x9B, 0x96, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* STX */
    { 27, ____, ____, ____, 0x8C, 0x84, ____, 0x8B, 0x94, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }  /* STY */
};
