//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Import/RealDataMaskWidget.cpp
//! @brief     Implements class RealDataMaskWidget
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Import/RealDataMaskWidget.h"
#include "GUI/Model/Data/IntensityDataItem.h"
#include "GUI/Model/Data/MaskItems.h"
#include "GUI/View/Mask/MaskEditorActions.h"
#include "GUI/View/Mask/MaskEditorCanvas.h"
#include "GUI/View/Mask/MaskEditorPropertyPanel.h"
#include "GUI/View/Mask/MaskEditorToolbar.h"
#include "GUI/View/Mask/MaskGraphicsScene.h"
#include <QSplitter>

RealDataMaskWidget::RealDataMaskWidget(QWidget* parent)
    : DataAccessWidget(parent)
    , m_editorActions(new MaskEditorActions(this))
    , m_toolbar(new MaskEditorToolbar(m_editorActions))
    , m_editorPropertyPanel(new MaskEditorPropertyPanel)
    , m_editorCanvas(new MaskEditorCanvas)
{
    setObjectName("MaskEditor");
    setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);

    auto* splitter = new QSplitter;
    splitter->addWidget(m_editorCanvas);
    splitter->addWidget(m_editorPropertyPanel);
    splitter->setCollapsible(1, true);

    QHBoxLayout* mainLayout = new QHBoxLayout(this);
    mainLayout->addWidget(splitter);
    mainLayout->setSpacing(0);
    mainLayout->setContentsMargins(0, 0, 0, 0);

    m_toolbar->setOrientation(Qt::Vertical);
    m_toolbar->setStyleSheet("QToolBar{border-left:1px solid rgb(180,180,180);}");
    mainLayout->addWidget(m_toolbar);

    setup_connections();

    m_editorPropertyPanel->setPanelHidden(true);
}

void RealDataMaskWidget::setJobOrRealItem(QObject* job_or_real_item)
{
    DataAccessWidget::setJobOrRealItem(job_or_real_item);
    m_editorPropertyPanel->setJobOrRealItem(job_or_real_item);
    setContext();
}

//! shows/hides right panel with properties
void RealDataMaskWidget::onPropertyPanelRequest()
{
    m_editorPropertyPanel->setPanelHidden(!m_editorPropertyPanel->isHidden());
}

//! Returns list of actions intended for styled toolbar (on the top).

QList<QAction*> RealDataMaskWidget::actionList()
{
    return m_editorActions->topToolbarActions();
}

void RealDataMaskWidget::setContext()
{
    ASSERT(currentIntensityDataItem());

    MaskContainerItem* containerItem = currentIntensityDataItem()->getOrCreateMaskContainerItem();
    ASSERT(containerItem);

    MaskContainerModel* containerModel = containerItem->model();
    ASSERT(containerModel);

    m_editorPropertyPanel->setMaskContext(containerModel);

    m_editorCanvas->setSelectionModel(m_editorPropertyPanel->selectionModel());
    m_editorCanvas->setMaskContext(currentIntensityDataItem());

    m_editorActions->setModel(containerModel);
    m_editorActions->setSelectionModel(m_editorPropertyPanel->selectionModel());

    update();
}

void RealDataMaskWidget::resetContext()
{
    m_editorPropertyPanel->resetContext();
    m_editorCanvas->resetContext();
}

void RealDataMaskWidget::showEvent(QShowEvent*)
{
    if (currentIntensityDataItem())
        setContext();
}

void RealDataMaskWidget::hideEvent(QHideEvent*)
{
    resetContext();
}

void RealDataMaskWidget::setup_connections()
{
    // reset view request is propagated from editorActions to graphics view
    connect(m_editorActions, &MaskEditorActions::resetViewRequest, m_editorCanvas,
            &MaskEditorCanvas::onResetViewRequest);

    // tool panel request is propagated from editorActions to this MaskEditor
    connect(m_editorActions, &MaskEditorActions::propertyPanelRequest, this,
            &RealDataMaskWidget::onPropertyPanelRequest);

    // save plot request is propagated from editorActions to graphics scene
    connect(m_editorActions, &MaskEditorActions::savePlotRequest, m_editorCanvas,
            &MaskEditorCanvas::onSavePlotRequest);

    // selection/drawing activity is propagated from Toolbar to graphics scene
    connect(m_toolbar, &MaskEditorToolbar::activityModeChanged, m_editorCanvas->getScene(),
            &MaskGraphicsScene::onActivityModeChanged);

    // mask value is propagated from Toolbar to graphics scene
    connect(m_toolbar, &MaskEditorToolbar::maskValueChanged, m_editorCanvas->getScene(),
            &MaskGraphicsScene::onMaskValueChanged);

    // show results request is propagated from Toolbar to Canvas
    connect(m_toolbar, &MaskEditorToolbar::presentationTypeRequest, m_editorCanvas,
            &MaskEditorCanvas::onPresentationTypeRequest);

    // space bar push (request for zoom mode) is propagated from graphics view to Toolbar
    connect(m_editorCanvas, &MaskEditorCanvas::changeActivityRequest, m_toolbar,
            &MaskEditorToolbar::onChangeActivityRequest);

    // Delete request is propagated from canvas to actions
    connect(m_editorCanvas, &MaskEditorCanvas::deleteSelectedRequest, m_editorActions,
            &MaskEditorActions::onDeleteMaskAction);

    // context menu request is propagated from graphics scene to MaskEditorActions
    connect(m_editorCanvas->getScene(), &MaskGraphicsScene::itemContextMenuRequest, m_editorActions,
            &MaskEditorActions::onItemContextMenuRequest);

    // context menu request is propagated from PropertyPanel to MaskEditorActions
    connect(m_editorPropertyPanel, &MaskEditorPropertyPanel::itemContextMenuRequest,
            m_editorActions, &MaskEditorActions::onItemContextMenuRequest);
}
