'''OpenGL extension ARB.imaging

This module customises the behaviour of the 
OpenGL.raw.GL.ARB.imaging to provide a more 
Python-friendly API

The official definition of this extension is available here:
http://www.opengl.org/registry/specs/ARB/imaging.txt
'''
from OpenGL import platform, constant, arrays
from OpenGL import extensions, wrapper
from OpenGL.GL import glget
import ctypes
from OpenGL.raw.GL import _types
from OpenGL.raw.GL.ARB.imaging import *
from OpenGL.raw.GL.ARB.imaging import _EXTENSION_NAME

def glInitImagingARB():
    '''Return boolean indicating whether this extension is available'''
    from OpenGL import extensions
    return extensions.hasGLExtension( _EXTENSION_NAME )

### END AUTOGENERATED SECTION
from OpenGL.GL import images
from OpenGL.lazywrapper import lazy as _lazy
glGetHistogramParameterfv = wrapper.wrapper(glGetHistogramParameterfv).setOutput(
    "params",(1,),
)
glGetHistogramParameteriv = wrapper.wrapper(glGetHistogramParameteriv).setOutput(
    "params",(1,),
)
glColorTable = images.setDimensionsAsInts(
    images.setImageInput(
        glColorTable,
        pixelName = 'table',
        typeName = 'type',
    )
)
glColorSubTable = images.setDimensionsAsInts(
    images.setImageInput(
        glColorSubTable,
        pixelName = 'data',
    )
)
glSeparableFilter2D = images.setDimensionsAsInts(
    images.setImageInput(
        images.setImageInput(
            glSeparableFilter2D,
            pixelName = 'row',
            typeName = 'type',
        ),
        pixelName = 'column',
        typeName = 'type',
    )
)
glConvolutionFilter1D = images.setDimensionsAsInts(
    images.setImageInput(
        glConvolutionFilter1D,
        pixelName = 'image',
        typeName = 'type',
    )
)
glConvolutionFilter2D = images.setDimensionsAsInts(
    images.setImageInput(
        glConvolutionFilter2D,
        pixelName = 'image',
        typeName = 'type',
    )
)

@_lazy( glGetConvolutionFilter )
def glGetConvolutionFilter( baseFunction, target, format, type ):
    """Retrieve 1 or 2D convolution parameter "kernels" as pixel data"""
    dims = (
        glGetConvolutionParameteriv( target, GL_CONVOLUTION_WIDTH )[0],
    )
    if target != GL_CONVOLUTION_1D:
        dims += (
            glGetConvolutionParameteriv( target, GL_CONVOLUTION_HEIGHT )[0],
        )
    # is it always 4?  Seems to be, but the spec/man-page isn't really clear about it...
    dims += (4,)
    array = images.images.SetupPixelRead( format, dims, type )
    arrayType = arrays.GL_CONSTANT_TO_ARRAY_TYPE[
        images.images.TYPE_TO_ARRAYTYPE.get(type,type)
    ]
    baseFunction(
        target, format, type,
        ctypes.c_void_p( arrayType.dataPointer(array))
    )
    return array
@_lazy( glGetSeparableFilter )
def glGetSeparableFilter( baseFunction, target, format, type ):
    """Retrieve 2 1D convolution parameter "kernels" as pixel data"""
    rowDims = (
        glGetConvolutionParameteriv( GL_CONVOLUTION_WIDTH )[0],
        4,
    )
    columnDims = (
        glGetConvolutionParameteriv( GL_CONVOLUTION_HEIGHT )[0],
        4,
    )
    arrayType = arrays.GL_CONSTANT_TO_ARRAY_TYPE[
        images.images.TYPE_TO_ARRAYTYPE.get(type,type)
    ]
    row = images.images.SetupPixelRead( format, rowDims, type )
    column = images.images.SetupPixelRead( format, columnDims, type )
    baseFunction(
        target, format, type,
        ctypes.c_void_p( arrayType.dataPointer(row)),
        ctypes.c_void_p( arrayType.dataPointer(column)),
        None # span
    )
    return row, column
@_lazy( glGetColorTable )
def glGetColorTable( baseFunction, target, format, type ):
    """Retrieve the current 1D color table as a bitmap"""
    dims = (
        glGetColorTableParameteriv(target, GL_COLOR_TABLE_WIDTH),
        4, # Grr, spec *seems* to say that it's different sizes, but it doesn't really say...
    )
    array = images.images.SetupPixelRead( format, dims, type )
    arrayType = arrays.GL_CONSTANT_TO_ARRAY_TYPE[
        images.images.TYPE_TO_ARRAYTYPE.get(type,type)
    ]
    baseFunction(
        target, format, type,
        ctypes.c_void_p( arrayType.dataPointer(array))
    )
    return array
@_lazy( glGetHistogram )
def glGetHistogram( baseFunction, target, reset, format, type, values=None):
    """Retrieve current 1D histogram as a 1D bitmap"""
    if values is None:
        width = glGetHistogramParameteriv(
            target,
            GL_HISTOGRAM_WIDTH,
        )
        values = images.images.SetupPixelRead( format, (width,4), type )
    arrayType = arrays.GL_CONSTANT_TO_ARRAY_TYPE[
        images.images.TYPE_TO_ARRAYTYPE.get(type,type)
    ]
    baseFunction(
        target, reset, format, type,
        ctypes.c_void_p( arrayType.dataPointer(values))
    )
    return values

@_lazy( glGetMinmax )
def glGetMinmax( baseFunction, target, reset, format, type, values=None):
    """Retrieve minimum and maximum values as a 2-element image"""
    if values is None:
        width = 2
        values = images.images.SetupPixelRead( format, (width,4), type )
    arrayType = arrays.GL_CONSTANT_TO_ARRAY_TYPE[
        images.images.TYPE_TO_ARRAYTYPE.get(type,type)
    ]
    baseFunction(
        target, reset, format, type,
        ctypes.c_void_p( arrayType.dataPointer(values))
    )
    return values

# 4-items, specified in spec...
glColorTableParameterfv = arrays.setInputArraySizeType(
    glColorTableParameterfv,
    4,
    arrays.GLfloatArray,
    'params',
)
GL_GET_CTP_SIZES = {
    GL_COLOR_TABLE_FORMAT :1,
    GL_COLOR_TABLE_WIDTH  :1,
    GL_COLOR_TABLE_RED_SIZE :1,
    GL_COLOR_TABLE_GREEN_SIZE :1,
    GL_COLOR_TABLE_BLUE_SIZE :1,
    GL_COLOR_TABLE_ALPHA_SIZE :1,
    GL_COLOR_TABLE_LUMINANCE_SIZE :1,
    GL_COLOR_TABLE_INTENSITY_SIZE :1,
}
glGetColorTableParameterfv = wrapper.wrapper(glGetColorTableParameterfv).setOutput(
    "params",GL_GET_CTP_SIZES, "pname",
)
glGetColorTableParameteriv = wrapper.wrapper(glGetColorTableParameteriv).setOutput(
    "params",GL_GET_CTP_SIZES, "pname",
)
GL_GET_CP_SIZES = {
    GL_CONVOLUTION_BORDER_MODE: 1,
    GL_CONVOLUTION_BORDER_COLOR: 4,
    GL_CONVOLUTION_FILTER_SCALE: 4,
    GL_CONVOLUTION_FILTER_BIAS: 4,
    GL_CONVOLUTION_FORMAT: 1,
    GL_CONVOLUTION_WIDTH: 1,
    GL_CONVOLUTION_HEIGHT: 1,
    GL_MAX_CONVOLUTION_WIDTH: 1,
    GL_MAX_CONVOLUTION_HEIGHT: 1,
}
glGetConvolutionParameteriv = wrapper.wrapper(glGetConvolutionParameteriv).setOutput(
    "params",GL_GET_CP_SIZES, "pname",
)
glGetConvolutionParameterfv = wrapper.wrapper(glGetConvolutionParameterfv).setOutput(
    "params",GL_GET_CP_SIZES, "pname",
)
