/*************************************************************************
* Copyright (C) 2023 Intel Corporation
*
* Licensed under the Apache License,  Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* 	http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law  or agreed  to  in  writing,  software
* distributed under  the License  is  distributed  on  an  "AS IS"  BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the  specific  language  governing  permissions  and
* limitations under the License.
*************************************************************************/
#ifdef MBX_FIPS_MODE

#include <crypto_mb/fips_cert.h>
#include <internal/fips_cert/common.h>
#include <internal/rsa/ifma_rsa_method.h>

#include <crypto_mb/rsa.h>

/* KAT TEST (generated via internal tests) */
/* plaintext */
static const int8u plaintext[MBX_RSA3K_DATA_BYTE_LEN] = {
  0x43,0x9a,0x10,0x7a,0xf5,0x23,0x48,0x2a,0xb9,0xf7,0x67,0xe7,0x57,0x65,0xbe,0x90,
  0xb9,0x50,0x8a,0xc1,0xb0,0x97,0xf5,0x64,0x93,0x8a,0xb0,0xe6,0xdf,0xd9,0x44,0x17,
  0xd0,0x29,0x84,0xad,0xf8,0xb0,0xad,0xe2,0xd3,0xda,0xb7,0xf5,0x12,0x47,0x6c,0x25,
  0xb2,0x9f,0x7f,0xac,0x04,0x33,0x90,0x9f,0x2c,0x96,0xe9,0x1c,0xae,0x02,0x96,0x12,
  0x6c,0x6a,0x88,0x10,0x06,0x9b,0x54,0xc7,0x9d,0x1b,0xd0,0xfd,0x5c,0x46,0xb4,0xe1,
  0x13,0x09,0x12,0xfc,0xfb,0xc2,0x2a,0xe8,0x74,0x95,0xda,0xfa,0xd3,0x69,0x2f,0xc7,
  0x8a,0xbe,0xea,0x0b,0xd7,0x8e,0xd1,0x39,0x58,0xe6,0x0d,0xa9,0xad,0x22,0x64,0x27,
  0x13,0x37,0x12,0x16,0xd6,0xb7,0x8f,0x47,0x86,0x18,0xbb,0x6e,0x79,0xc2,0x92,0xfc,
  0x68,0xbc,0x8a,0xc5,0x14,0xe2,0xa0,0x2f,0xd8,0xa1,0x4c,0xf9,0x39,0xb5,0xe5,0x3e,
  0xcb,0x55,0x11,0x22,0xd7,0x19,0xc4,0x02,0x33,0xea,0x34,0x6e,0x23,0xde,0x1d,0x6e,
  0x3c,0x6f,0x13,0x20,0xb3,0x19,0x97,0xd7,0x14,0xc3,0x3b,0x18,0x13,0xd9,0x1f,0x13,
  0x22,0x34,0xc0,0x17,0xa3,0x54,0x2f,0x37,0x35,0xcc,0x53,0xa9,0xc9,0x6f,0x5a,0xaf,
  0x52,0x4c,0xe0,0x5e,0x77,0xa6,0x2f,0xba,0xf9,0x15,0x3e,0x37,0x65,0x28,0x2d,0x69,
  0x2f,0xcc,0x6e,0xb3,0x68,0x8b,0xe5,0xfb,0xd4,0xb1,0xb0,0x41,0xc8,0xb7,0x8a,0xd5,
  0xc5,0x83,0x49,0x8c,0xfc,0x96,0x21,0xab,0x94,0xd8,0x4c,0x82,0x67,0x2a,0x6f,0x95,
  0x42,0x83,0xe0,0xb1,0xc2,0xa0,0x10,0xed,0x31,0x26,0xbd,0xe2,0x73,0xb3,0x92,0x91,
  0x06,0x70,0xf8,0x10,0xb8,0x01,0xc9,0x34,0x3d,0xd7,0x8e,0x61,0x4e,0x37,0x9e,0x1c,
  0xd4,0x62,0x76,0xb2,0x15,0x15,0xd8,0x44,0x46,0xe5,0xc3,0x5d,0xf0,0xb1,0xe5,0xa0,
  0xbd,0x70,0x17,0x6f,0x84,0x81,0xa7,0xd4,0x64,0xda,0x0a,0xe0,0xf0,0xd7,0x85,0xe9,
  0x86,0x6f,0xe6,0x09,0xed,0x41,0xe3,0x3f,0xfc,0xa6,0xa1,0x30,0x5b,0x36,0x77,0x4d,
  0x96,0xf1,0x58,0x99,0x00,0x27,0x4d,0xdf,0xf2,0x9e,0x94,0x0c,0xe3,0xf3,0xe0,0xeb,
  0x2b,0xa2,0x02,0x6b,0x88,0xa8,0x10,0xb9,0x36,0xcf,0x41,0xb5,0x8c,0xca,0x33,0x69,
  0xeb,0x22,0x6d,0x7d,0xf0,0x66,0x29,0x08,0x79,0x21,0x30,0x47,0x93,0x9d,0x7b,0xbb,
  0xa3,0x05,0x8c,0xc6,0x80,0x04,0x78,0x2c,0x27,0xc5,0xfc,0xfd,0xd6,0x12,0x82,0x2c};
/* ciphertext */
static const int8u ciphertext[MBX_RSA3K_DATA_BYTE_LEN] = {
  0x14,0x65,0xbe,0x8a,0x00,0xef,0x3e,0xc3,0x36,0xfd,0xa0,0x30,0x4d,0x34,0x22,0xd2,
  0x12,0xde,0x7c,0xfa,0x47,0x0c,0x66,0x6b,0x1c,0x2a,0xd2,0x48,0xee,0xf6,0x7d,0x0e,
  0x75,0x68,0xf4,0x50,0xa6,0x14,0x5b,0xec,0xc2,0xd4,0xc0,0xfd,0x01,0xdc,0x9d,0xa6,
  0x79,0x5d,0x3c,0xc2,0xf7,0x8e,0xb9,0xdc,0x5b,0xd6,0xc9,0xe3,0xb3,0xe5,0xad,0xb6,
  0xfc,0x77,0x22,0x48,0xf9,0x5d,0x7d,0xfe,0x71,0x5a,0xb5,0x1e,0x9f,0x48,0x13,0xd0,
  0xa6,0x3c,0xf4,0xed,0x7e,0x45,0x21,0x46,0x08,0x99,0x04,0x94,0xc5,0x35,0x9a,0xe7,
  0x2f,0x13,0xc4,0x24,0xf9,0xb4,0x79,0x03,0xd5,0x68,0x13,0x89,0x6b,0x31,0x78,0x5a,
  0x84,0x11,0x50,0x18,0x2d,0x3b,0x02,0x48,0x44,0xbc,0xaf,0x48,0xe0,0xe9,0x56,0xef,
  0xfa,0xaf,0xef,0xb7,0x05,0xac,0xdb,0xf7,0x42,0x0f,0x9b,0xc0,0xaa,0xc7,0x50,0x79,
  0x25,0x86,0x60,0x53,0x12,0x37,0x58,0xcd,0xbc,0x29,0x19,0x0f,0xb3,0x66,0xdb,0x36,
  0x31,0x55,0x69,0xf5,0xb8,0x8f,0x4f,0x7d,0xdf,0x3a,0xda,0xe0,0xc2,0xe4,0xcb,0x28,
  0x70,0x72,0xf5,0x32,0x3a,0xb8,0xbb,0x78,0x7e,0x3a,0xfa,0x1f,0xe7,0x14,0xb4,0x4e,
  0x77,0xf1,0xbe,0x80,0x19,0x10,0x75,0xaa,0x1f,0xd0,0x80,0x78,0xcf,0xce,0x9c,0x8e,
  0x70,0xd9,0x24,0xa2,0xbb,0x6b,0xd6,0x5c,0xdb,0xe4,0xc5,0x51,0x49,0x1a,0xba,0xaf,
  0x94,0x37,0xc2,0xb7,0xfe,0x85,0xcc,0xff,0xcb,0x81,0xb4,0x47,0x5a,0x46,0xd8,0xff,
  0x36,0xc9,0x14,0x19,0x25,0x9e,0x6b,0x41,0x0c,0x73,0xe9,0xc6,0xd8,0xcf,0x61,0xdd,
  0x82,0xd4,0xc5,0xae,0x53,0x89,0x83,0xc5,0x39,0x47,0x56,0x0d,0x77,0xf0,0xa8,0x76,
  0x04,0x07,0xe8,0x34,0xbf,0x54,0x02,0x84,0xf8,0x00,0x36,0xe0,0x46,0x01,0x96,0xca,
  0x0a,0xdb,0x99,0x61,0x3f,0x1b,0x86,0x36,0xea,0x6b,0xdb,0xe4,0x1b,0xcf,0x3c,0xe6,
  0x16,0x6d,0x48,0x20,0xd7,0xfa,0x4f,0xf3,0xdc,0xd8,0x5c,0x68,0x28,0xe6,0x8a,0x6d,
  0x96,0x9f,0xa5,0x71,0x86,0x90,0xc4,0x97,0x4c,0xe5,0x66,0x25,0xed,0xf7,0x07,0xd6,
  0xbd,0xbb,0xb7,0xef,0x43,0xc2,0xba,0x6e,0xcc,0x00,0xca,0xb2,0x57,0x3f,0xef,0xdb,
  0x08,0x24,0x65,0x7d,0x09,0xd1,0x2f,0x1c,0x1a,0xed,0x35,0x63,0xf6,0xea,0xec,0xea,
  0xc3,0x2a,0xe8,0xb0,0xf0,0xa5,0x8f,0xb6,0x21,0xfb,0xf0,0x91,0x72,0x35,0x7e,0xb2};
/* p, q primes */
static const int8u p[MBX_RSA3K_DATA_BYTE_LEN / 2] = {
  0x71,0xa4,0x32,0xb5,0x2f,0xde,0x65,0x86,0x93,0x47,0xf7,0x27,0x0c,0x6a,0x58,0x89,
  0x94,0xf7,0xa2,0xdf,0xcf,0x7e,0xae,0xa2,0x11,0x60,0x84,0xea,0x40,0x53,0xc3,0x66,
  0xc9,0x5e,0x0b,0xf6,0x0e,0x73,0x7c,0xa2,0x0b,0xbe,0x6b,0x8e,0x9a,0x7e,0x2c,0x66,
  0xad,0x6c,0x7e,0x85,0x4a,0x83,0x10,0x38,0xcc,0x8e,0xf1,0xe3,0xc2,0xa5,0x34,0x90,
  0x7d,0xe8,0x85,0x67,0x21,0x57,0xc0,0x2b,0xd9,0xd1,0xd3,0x59,0x83,0x20,0xac,0x8c,
  0x8a,0x59,0xe0,0x7e,0x85,0xad,0x5f,0xd8,0xbb,0xeb,0x6b,0xf8,0xbd,0x86,0xd7,0xfa,
  0x2f,0x34,0xb0,0x31,0xbb,0x2f,0xcb,0x00,0x89,0x35,0x2e,0x2a,0xea,0x0f,0xf9,0xb3,
  0xe7,0x9c,0x98,0x61,0x86,0xb2,0xc2,0x37,0x15,0x59,0x72,0x06,0x2d,0xbc,0xe4,0x2f,
  0xb9,0x50,0xd4,0x29,0x16,0xbb,0x65,0x36,0x1d,0xf4,0xbb,0x8f,0x9b,0xe2,0x87,0xe6,
  0xdf,0x7c,0xdd,0xad,0x4f,0x2a,0xe6,0xbe,0xe0,0x94,0x64,0xd8,0xee,0x68,0xb0,0x5a,
  0x54,0xf7,0x61,0xad,0x52,0xbe,0x66,0x8b,0x4c,0x19,0x6e,0x97,0xc1,0xde,0xaa,0xa2,
  0x49,0xe6,0xc9,0xf3,0x04,0x03,0x91,0x7e,0xd7,0xaf,0xc4,0x31,0x16,0xf5,0xc5,0xea};
static const int8u q[MBX_RSA3K_DATA_BYTE_LEN / 2] = {
  0xd5,0xa9,0x2d,0xdb,0x06,0xda,0x5d,0xb7,0xb4,0x29,0x2f,0x5e,0xa9,0xfd,0x60,0xf9,
  0x73,0x70,0x26,0x6b,0xbe,0x91,0x4a,0xe9,0x7f,0xfc,0x0e,0xf0,0x55,0x2a,0x95,0x27,
  0x39,0x67,0xec,0x49,0x7e,0x38,0x29,0x13,0x27,0x69,0xef,0xee,0x82,0xcf,0x3d,0xa7,
  0xee,0x42,0x68,0xe1,0xf6,0x0f,0x4a,0x7a,0xae,0x3c,0xd2,0xbe,0xbd,0x8b,0x9b,0xa0,
  0xee,0x33,0x02,0x15,0x77,0x49,0x38,0x6e,0xd1,0x4c,0xb0,0x24,0xb0,0xa9,0x8b,0x72,
  0x29,0x7d,0x44,0x4d,0x20,0xf9,0x78,0x9f,0xe5,0x8a,0x68,0x91,0x27,0x78,0x90,0xd6,
  0xf8,0x80,0x4c,0xf6,0x87,0x34,0xa4,0xcb,0xe8,0xce,0xda,0x8f,0x40,0xcd,0x17,0x2d,
  0x9b,0x16,0x9b,0x41,0xce,0xba,0x18,0x94,0xe8,0x89,0x39,0x33,0x10,0x4e,0x01,0xd6,
  0x58,0xfb,0xb3,0xa0,0x6f,0xa1,0xad,0x28,0x39,0x86,0xc0,0xda,0x13,0x73,0xbd,0x2e,
  0x28,0x16,0xf1,0xc6,0x30,0x30,0x11,0x58,0x37,0x90,0x0c,0x6e,0x27,0x72,0xe3,0xd8,
  0xb9,0x69,0x69,0x33,0xd8,0x39,0x6d,0x37,0xe4,0xe8,0x69,0xd8,0xbf,0x7c,0x9a,0xd0,
  0x74,0xc5,0x44,0xc7,0xec,0xbd,0x06,0x61,0xc3,0xe9,0x47,0x9c,0x90,0x71,0x3b,0xcf};
/* p's, q's CRT private exponent */
static const int8u dp[MBX_RSA3K_DATA_BYTE_LEN / 2] = {
  0xe1,0xaa,0x76,0x5d,0x3d,0x1c,0x9d,0x05,0x6c,0xc2,0x60,0x58,0x15,0x7e,0xef,0x40,
  0xbe,0x81,0x20,0xaf,0x2b,0xb1,0x2b,0x88,0xfd,0xbf,0xe9,0xe1,0xa3,0x28,0x3a,0x89,
  0x96,0xb9,0xba,0x3c,0x91,0x38,0x9d,0x1d,0x22,0x91,0xd2,0x22,0x2f,0x66,0x44,0x38,
  0x7a,0xad,0xa8,0x3c,0x64,0xea,0x23,0x62,0x6f,0xef,0x99,0xb8,0xca,0x54,0x1b,0x3c,
  0x5d,0x90,0xe4,0x55,0x10,0x6b,0x78,0x68,0x3a,0xcb,0x5d,0x46,0xed,0x6a,0x55,0xf8,
  0x80,0x57,0xb8,0xdf,0x76,0x93,0x07,0x40,0x46,0x64,0xf7,0xfe,0x3d,0xc0,0x07,0xaa,
  0x43,0xe5,0x46,0x97,0x7b,0xb1,0x3b,0xa6,0x2f,0xf5,0x64,0xb7,0x5f,0x5e,0xf3,0x64,
  0x75,0xb2,0x6e,0x13,0x26,0x51,0xbb,0x82,0x1b,0xbb,0xbb,0xda,0xd7,0x00,0xd1,0x16,
  0x1b,0x21,0x42,0xdc,0x24,0x76,0xbf,0xf6,0x56,0xe3,0x87,0x99,0x24,0xd1,0x5d,0x61,
  0x2c,0xcd,0x80,0x32,0x12,0x77,0x3e,0xb9,0x82,0x26,0xae,0x9a,0xa8,0xd8,0x61,0xf5,
  0xee,0x78,0x67,0xc2,0x87,0x59,0x2d,0xb8,0x2a,0x10,0x0d,0xfe,0xc3,0x9a,0x57,0x74,
  0x3f,0x91,0xb9,0x87,0x65,0x13,0x6a,0xc2,0xf9,0x6c,0x11,0x4b,0x13,0x40,0x34,0x8c};
static const int8u dq[MBX_RSA3K_DATA_BYTE_LEN / 2] = {
  0xb5,0xbf,0x91,0x58,0x9c,0x87,0xa0,0x50,0x5d,0x71,0xb5,0x2a,0x0a,0x0c,0x07,0xcd,
  0xe7,0xa2,0x6f,0xa2,0x6d,0x2f,0x8e,0xd4,0x70,0x01,0xac,0xa4,0xbc,0x7b,0xcf,0x5e,
  0x82,0xc2,0xa5,0x7f,0xe4,0xdf,0xa4,0x11,0x47,0x86,0x3e,0xcc,0x24,0x9b,0x66,0x91,
  0x1d,0x57,0x94,0x16,0x1d,0x0b,0x1f,0x38,0x84,0xcf,0x6f,0x0d,0x02,0xa2,0x7e,0xbc,
  0x9d,0x2f,0xe5,0x28,0x58,0x80,0xfc,0x1a,0xbc,0xf7,0x0b,0x34,0x09,0x54,0x4e,0x9b,
  0x88,0xdf,0x17,0xf4,0xfd,0x2e,0xb9,0x6a,0x15,0x2a,0x40,0xae,0x8a,0x47,0x6d,0x56,
  0xa0,0xec,0xe2,0x6f,0x7d,0x6d,0x54,0xb2,0x42,0x1c,0xa0,0xa5,0xd7,0x7a,0x26,0xcf,
  0x3a,0xc8,0xd3,0x6f,0xbf,0xca,0xcf,0xcf,0x84,0x94,0xeb,0xc8,0x61,0x08,0xd3,0xed,
  0x25,0x78,0xa3,0xcf,0xa5,0x5e,0x4a,0x04,0xec,0xa4,0x32,0x85,0x67,0x63,0x52,0x79,
  0xab,0x6a,0x6b,0x5d,0x5f,0x11,0x0e,0x88,0x93,0xa0,0x1b,0x5a,0xdf,0xe8,0xcf,0x83,
  0x7b,0xac,0x6b,0xb3,0x90,0x88,0xb6,0x56,0xb7,0x46,0xce,0x1d,0xb0,0x50,0x00,0xe1,
  0x4d,0x0b,0x3b,0xed,0x2f,0xf7,0x27,0x80,0x72,0xf4,0x76,0xe0,0xf0,0x7d,0xdb,0xa8};
/* CRT coefficient */
static const int8u inv_q[MBX_RSA3K_DATA_BYTE_LEN / 2] = {
  0xc4,0xd1,0x5e,0xb9,0x75,0x8d,0x12,0x3c,0xf1,0x3b,0x74,0xf5,0xc6,0xff,0x1f,0xca,
  0xa3,0xc1,0xed,0x4f,0xe1,0x88,0xef,0xd2,0x7f,0x81,0x26,0x7e,0x34,0x17,0x3d,0x2b,
  0x44,0x4d,0xee,0x91,0x32,0x2e,0x1f,0x14,0xa5,0x6f,0x89,0x4f,0x2a,0x5d,0xbb,0xc0,
  0xf6,0xf2,0xf7,0xfb,0x68,0xdb,0xc7,0xb6,0xe9,0x14,0x96,0xd7,0x3b,0xcb,0x05,0xbe,
  0x9d,0xa2,0xb3,0xd7,0x03,0x8a,0x84,0x72,0x03,0x50,0x3d,0x24,0xac,0x8d,0x08,0x21,
  0x2e,0xfd,0x0a,0xec,0x8d,0x8e,0xbd,0xb4,0x54,0x7f,0xaf,0x7b,0xab,0xf1,0x23,0xec,
  0x76,0x35,0xe9,0xd0,0xd5,0x72,0x9e,0xab,0xe0,0x6c,0xbc,0xfd,0x77,0x93,0x76,0x60,
  0x64,0x89,0xbc,0x60,0x0a,0x89,0xf1,0x41,0xa4,0x36,0xad,0x97,0x54,0x96,0x85,0xc0,
  0x84,0x1e,0x69,0x71,0x3f,0xe6,0x55,0x1e,0xde,0x3d,0x9b,0xf1,0xa1,0xe8,0x8c,0x24,
  0x77,0x75,0x12,0x5f,0x70,0xb0,0x5d,0xdc,0x43,0xff,0xa8,0x63,0xf1,0x9b,0x61,0x2c,
  0x5c,0xc1,0xef,0x95,0x37,0xdd,0xf5,0x51,0x54,0x3e,0xbc,0xa5,0x2f,0xd8,0x76,0x1c,
  0xc3,0x05,0x16,0x1c,0x96,0x8c,0xa7,0x52,0xc0,0xca,0x97,0x96,0x84,0xf2,0x27,0x5b};

DLL_PUBLIC
fips_test_status fips_selftest_mbx_rsa3k_private_crt_mb8(void) {
  fips_test_status test_result = MBX_ALGO_SELFTEST_OK;

  /* output plaintext */
  int8u out_plaintext[MBX_LANES][MBX_RSA3K_DATA_BYTE_LEN];
  /* key operation */
  const mbx_RSA_Method* method = mbx_RSA3K_private_crt_Method();

  /* function input parameters */
  // ciphertext
  const int8u *pa_ciphertext[MBX_LANES] = {
    ciphertext, ciphertext, ciphertext, ciphertext,
    ciphertext, ciphertext, ciphertext, ciphertext};
  // plaintext
  int8u *pa_plaintext[MBX_LANES] = {
    out_plaintext[0], out_plaintext[1], out_plaintext[2], out_plaintext[3],
    out_plaintext[4], out_plaintext[5], out_plaintext[6], out_plaintext[7]};
  // p, q primes
  const int64u *pa_p[MBX_LANES]= {
    (int64u *)p, (int64u *)p, (int64u *)p, (int64u *)p,
    (int64u *)p, (int64u *)p, (int64u *)p, (int64u *)p};
  const int64u *pa_q[MBX_LANES]= {
    (int64u *)q, (int64u *)q, (int64u *)q, (int64u *)q,
    (int64u *)q, (int64u *)q, (int64u *)q, (int64u *)q};
  // p's, q's CRT private exponent
  const int64u *pa_dp[MBX_LANES]= {
    (int64u *)dp, (int64u *)dp, (int64u *)dp, (int64u *)dp,
    (int64u *)dp, (int64u *)dp, (int64u *)dp, (int64u *)dp};
  const int64u *pa_dq[MBX_LANES]= {
    (int64u *)dq, (int64u *)dq, (int64u *)dq, (int64u *)dq,
    (int64u *)dq, (int64u *)dq, (int64u *)dq, (int64u *)dq};
  // CRT coefficient
  const int64u *pa_inv_q[MBX_LANES]= {
    (int64u *)inv_q, (int64u *)inv_q, (int64u *)inv_q, (int64u *)inv_q,
    (int64u *)inv_q, (int64u *)inv_q, (int64u *)inv_q, (int64u *)inv_q};

  /* test function */
  mbx_status expected_status_mb8 = MBX_SET_STS_ALL(MBX_STATUS_OK);

  mbx_status sts;
  sts = mbx_rsa_private_crt_mb8(pa_ciphertext, pa_plaintext, pa_p, pa_q, pa_dp, pa_dq, pa_inv_q,
    MBX_RSA3K_DATA_BIT_LEN, method, NULL);
  if (expected_status_mb8 != sts) {
    test_result = MBX_ALGO_SELFTEST_BAD_ARGS_ERR;
  }
  // compare output plaintext to known answer
  int output_status;
  for (int i = 0; (i < MBX_LANES) && (MBX_ALGO_SELFTEST_OK == test_result); ++i) {
    output_status = mbx_is_mem_eq(pa_plaintext[i], MBX_RSA3K_DATA_BYTE_LEN, plaintext, MBX_RSA3K_DATA_BYTE_LEN);
    if (!output_status) { // wrong output
      test_result = MBX_ALGO_SELFTEST_KAT_ERR;
    }
  }

  return test_result;
}

#ifndef BN_OPENSSL_DISABLE

// memory free macro
#define MEM_FREE(BN_PTR1, BN_PTR2, BN_PTR3, BN_PTR4, BN_PTR5) { \
  BN_free(BN_PTR1);                                             \
  BN_free(BN_PTR2);                                             \
  BN_free(BN_PTR3);                                             \
  BN_free(BN_PTR4);                                             \
  BN_free(BN_PTR5); }

DLL_PUBLIC
fips_test_status fips_selftest_mbx_rsa3k_private_crt_ssl_mb8(void) {

  fips_test_status test_result = MBX_ALGO_SELFTEST_OK;

  /* output plaintext */
  int8u out_plaintext[MBX_LANES][MBX_RSA3K_DATA_BYTE_LEN];
  /* ssl parameters */
  // p, q primes
  BIGNUM* BN_p  = BN_new();
  BIGNUM* BN_q  = BN_new();
  // p's, q's CRT private exponent
  BIGNUM* BN_dp = BN_new();
  BIGNUM* BN_dq = BN_new();
  // CRT coefficient
  BIGNUM* BN_inv_q = BN_new();
  /* check if allocated memory is valid */
  if(NULL == BN_p || NULL == BN_q || NULL == BN_dp || NULL == BN_dq || NULL == BN_inv_q) {
    test_result = MBX_ALGO_SELFTEST_BAD_ARGS_ERR;
    MEM_FREE(BN_p, BN_q, BN_dp, BN_dq, BN_inv_q)
    return test_result;
  }

  /* function status and expected status */
  mbx_status sts;
  mbx_status expected_status_mb8 = MBX_SET_STS_ALL(MBX_STATUS_OK);
  /* output validity status */
  int output_status;

  /* set ssl parameters */
  BN_lebin2bn(p, MBX_RSA3K_DATA_BYTE_LEN / 2, BN_p);
  BN_lebin2bn(q, MBX_RSA3K_DATA_BYTE_LEN / 2, BN_q);
  BN_lebin2bn(dp, MBX_RSA3K_DATA_BYTE_LEN / 2, BN_dp);
  BN_lebin2bn(dq, MBX_RSA3K_DATA_BYTE_LEN / 2, BN_dq);
  BN_lebin2bn(inv_q, MBX_RSA3K_DATA_BYTE_LEN / 2, BN_inv_q);

  /* function input parameters */
  // ciphertext
  const int8u *pa_ciphertext[MBX_LANES] = {
    ciphertext, ciphertext, ciphertext, ciphertext,
    ciphertext, ciphertext, ciphertext, ciphertext};
  // plaintext
  int8u *pa_plaintext[MBX_LANES] = {
    out_plaintext[0], out_plaintext[1], out_plaintext[2], out_plaintext[3],
    out_plaintext[4], out_plaintext[5], out_plaintext[6], out_plaintext[7]};
  // p, q primes
  const BIGNUM *pa_p[MBX_LANES] = {
    (const BIGNUM *)BN_p, (const BIGNUM *)BN_p, (const BIGNUM *)BN_p, (const BIGNUM *)BN_p,
    (const BIGNUM *)BN_p, (const BIGNUM *)BN_p, (const BIGNUM *)BN_p, (const BIGNUM *)BN_p};
  const BIGNUM *pa_q[MBX_LANES] = {
    (const BIGNUM *)BN_q, (const BIGNUM *)BN_q, (const BIGNUM *)BN_q, (const BIGNUM *)BN_q,
    (const BIGNUM *)BN_q, (const BIGNUM *)BN_q, (const BIGNUM *)BN_q, (const BIGNUM *)BN_q};
  // p's, q's CRT private exponent
  const BIGNUM *pa_dp[MBX_LANES] = {
    (const BIGNUM *)BN_dp, (const BIGNUM *)BN_dp, (const BIGNUM *)BN_dp, (const BIGNUM *)BN_dp,
    (const BIGNUM *)BN_dp, (const BIGNUM *)BN_dp, (const BIGNUM *)BN_dp, (const BIGNUM *)BN_dp};
  const BIGNUM *pa_dq[MBX_LANES] = {
    (const BIGNUM *)BN_dq, (const BIGNUM *)BN_dq, (const BIGNUM *)BN_dq, (const BIGNUM *)BN_dq,
    (const BIGNUM *)BN_dq, (const BIGNUM *)BN_dq, (const BIGNUM *)BN_dq, (const BIGNUM *)BN_dq};
  // CRT coefficient
  const BIGNUM *pa_inv_q[MBX_LANES] = {
    (const BIGNUM *)BN_inv_q, (const BIGNUM *)BN_inv_q, (const BIGNUM *)BN_inv_q, (const BIGNUM *)BN_inv_q,
    (const BIGNUM *)BN_inv_q, (const BIGNUM *)BN_inv_q, (const BIGNUM *)BN_inv_q, (const BIGNUM *)BN_inv_q};

  /* test function */
  sts = mbx_rsa_private_crt_ssl_mb8(pa_ciphertext, pa_plaintext,
    pa_p, pa_q, pa_dp, pa_dq, pa_inv_q, MBX_RSA3K_DATA_BIT_LEN);
  if (expected_status_mb8 != sts) {
    test_result = MBX_ALGO_SELFTEST_BAD_ARGS_ERR;
  }
  // compare output signature to known answer
  for (int i = 0; (i < MBX_LANES) && (MBX_ALGO_SELFTEST_OK == test_result); ++i) {
    output_status = mbx_is_mem_eq(pa_plaintext[i], MBX_RSA3K_DATA_BYTE_LEN, plaintext, MBX_RSA3K_DATA_BYTE_LEN);
    if (!output_status) { // wrong output
      test_result = MBX_ALGO_SELFTEST_KAT_ERR;
    }
  }

  // memory free
  MEM_FREE(BN_p, BN_q, BN_dp, BN_dq, BN_inv_q)

  return test_result;
}

#endif // BN_OPENSSL_DISABLE
#endif // MBX_FIPS_MODE
