/*
 * @(#)AnalysisModuleSetUTest.java
 *
 * Copyright (C) 2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.datastore;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import net.sourceforge.groboutils.autodoc.v1.AutoDoc;
import net.sourceforge.groboutils.codecoverage.v2.CCCreatorUtil;
import net.sourceforge.groboutils.codecoverage.v2.IAnalysisModule;


/**
 * Tests the AnalysisModuleSet class.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2004/04/15 05:48:28 $
 * @since     January 22, 2003
 */
public class AnalysisModuleSetUTest extends TestCase
{
    //-------------------------------------------------------------------------
    // Standard JUnit Class-specific declarations
    
    private static final Class THIS_CLASS = AnalysisModuleSetUTest.class;
    private static final AutoDoc DOC = new AutoDoc( THIS_CLASS );
    
    public AnalysisModuleSetUTest( String name )
    {
        super( name );
    }


    //-------------------------------------------------------------------------
    // Tests
    
    
    public void testConstructor1()
    {
        try
        {
            new AnalysisModuleSet( (IAnalysisModule[])null );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException ex)
        {
            // check exception
        }
    }
    
    
    public void testConstructor2()
    {
        try
        {
            new AnalysisModuleSet( (AnalysisModuleSet)null );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException ex)
        {
            // check exception
        }
    }
    
    
    public void testConstructor3()
    {
        try
        {
            new AnalysisModuleSet( new IAnalysisModule[1] );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException ex)
        {
            // check exception
        }
    }
    
    
    public void testConstructor4()
    {
        try
        {
            new AnalysisModuleSet( new IAnalysisModule[] {
                createIAnalysisModule( "0" ), null } );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException ex)
        {
            // check exception
        }
    }
    
    
    public void testConstructor5()
    {
        new AnalysisModuleSet( new AnalysisModuleSet() );
    }
    
    
    public void testAddAnalysisModule1()
    {
        AnalysisModuleSet ams = new AnalysisModuleSet();
        IAnalysisModule a1 = createIAnalysisModule( "a" );
        IAnalysisModule a2 = createIAnalysisModule( "a" );
        assertEquals( a1.getMeasureName(), a2.getMeasureName() );
        ams.addAnalysisModule( a1 );
        try
        {
            ams.addAnalysisModule( a2 );
            fail( "Did not throw IllegalStateException." );
        }
        catch (IllegalStateException ex)
        {
            // test exception
        }
    }
    
    
    public void testAddAnalysisModule2()
    {
        AnalysisModuleSet ams = new AnalysisModuleSet();
        try
        {
            ams.addAnalysisModule( null );
            fail( "Did not throw IllegalStateException." );
        }
        catch (IllegalArgumentException iae)
        {
            // check exception
        }
    }
    
    
    public void testJoinAnalysisModuleSet1()
    {
        AnalysisModuleSet ams = new AnalysisModuleSet();
        try
        {
            ams.joinAnalysisModuleSet( null );
            fail( "Did not throw IllegalStateException." );
        }
        catch (IllegalArgumentException iae)
        {
            // check exception
        }
    }
    
    
    public void testJoinAnalysisModuleSet2()
    {
        IAnalysisModule a1 = createIAnalysisModule( "a" );
        AnalysisModuleSet ams = new AnalysisModuleSet(
            new IAnalysisModule[] { a1 } );
        ams.joinAnalysisModuleSet( ams );
    }
    
    
    public void testJoinAnalysisModuleSet3()
    {
        IAnalysisModule a1 = createIAnalysisModule( "a" );
        IAnalysisModule a2 = createIAnalysisModule( "b" );
        AnalysisModuleSet ams = new AnalysisModuleSet(
            new IAnalysisModule[] { a1 } );
        ams.joinAnalysisModuleSet( new AnalysisModuleSet(
            new IAnalysisModule[] { a2 } ) );
        assertEquals(
            "Did not store correct number of modules.",
            2,
            ams.getAnalysisModuleCount() );
    }
    
    
    /* this test takes a VERY long time, and as such is not suitable for
    unit tests.  If you want to increase unit test coverage, though,
    include this test to get 1 more branch and 1 more line.  Not only that,
    but this is a rarely encountered situation, and encountering this situation
    may be a sign of a bug elsewhere.
    public void testAddAnalysisModule2()
    {
        // test the limits of the short index vs. int internal structure
        // counting system.
        AnalysisModuleSet ams = new AnalysisModuleSet();
        IAnalysisModule amL[] = CCCreatorUtil.createAnalysisModules(
            (int)Short.MAX_VALUE );
        ams.addAnalysisModules( amL );
        IAnalysisModule a1 = createIAnalysisModule( "a" );
        try
        {
            ams.addAnalysisModule( a1 );
            fail( "Did not throw IllegalStateException." );
        }
        catch (IllegalStateException ex)
        {
            // test exception
        }
    }
    */
    
    
    public void testGetAnalysisModules1()
    {
        AnalysisModuleSet ams = new AnalysisModuleSet();
        IAnalysisModule[] amL = ams.getAnalysisModules();
        assertNotNull(
            "Returned null module list.",
            amL );
        assertEquals(
            "Returned invalid module array length.",
            0,
            amL.length );
    }
    
    
    public void testGetAnalysisModules2()
    {
        IAnalysisModule orig = createIAnalysisModule( "a" );
        AnalysisModuleSet ams = new AnalysisModuleSet(
            new IAnalysisModule[] { orig } );
        IAnalysisModule[] amL = ams.getAnalysisModules();
        assertNotNull(
            "Returned null module list.",
            amL );
        assertEquals(
            "Returned invalid module array length.",
            1,
            amL.length );
        assertSame(
            "Returned invalid module.",
            orig,
            amL[0] );
    }
    
    
    public void testGetMeasureIndex1()
    {
        AnalysisModuleSet ams = new AnalysisModuleSet();
        short i = ams.getMeasureIndex( "b" );
        assertEquals(
            "Returned invalid index.",
            -1,
            i );
    }
    
    
    public void testGetMeasureIndex2()
    {
        IAnalysisModule orig = createIAnalysisModule( "b" );
        AnalysisModuleSet ams = new AnalysisModuleSet(
            new IAnalysisModule[] { orig } );
        short i = ams.getMeasureIndex( "b" );
        assertEquals(
            "Returned invalid index.",
            0,
            i );
    }
    
    
    public void testGetMeasureIndex3()
    {
        AnalysisModuleSet ams = new AnalysisModuleSet();
        try
        {
            ams.getMeasureIndex( null );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException ex)
        {
            // test exception
        }
    }
    
    
    public void testGetAnalysisModuleIndex1()
    {
        AnalysisModuleSet ams = new AnalysisModuleSet();
        try
        {
            ams.getAnalysisModuleIndex( null );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException ex)
        {
            // test exception
        }
    }
    
    
    public void testGetAnalysisModuleIndex2()
    {
        IAnalysisModule orig = createIAnalysisModule( "b" );
        AnalysisModuleSet ams = new AnalysisModuleSet();
        short i = ams.getAnalysisModuleIndex( orig );
        assertEquals(
            "Did not return correct index.",
            -1,
            i );
    }
    
    
    public void testGetAnalysisModuleIndex3()
    {
        IAnalysisModule orig = createIAnalysisModule( "b" );
        IAnalysisModule a = createIAnalysisModule( "a" );
        AnalysisModuleSet ams = new AnalysisModuleSet(
            new IAnalysisModule[] { a } );
        short i = ams.getAnalysisModuleIndex( orig );
        assertEquals(
            "Did not return correct index.",
            -1,
            i );
    }
    
    
    public void testGetAnalysisModuleIndex4()
    {
        IAnalysisModule orig1 = createIAnalysisModule( "b" );
        AnalysisModuleSet ams = new AnalysisModuleSet(
            new IAnalysisModule[] { orig1 } );
        short i = ams.getAnalysisModuleIndex( orig1 );
        assertEquals(
            "Did not return correct index.",
            0,
            i );
    }
    
    
    public void testGetAnalysisModuleCount1()
    {
        AnalysisModuleSet ams = new AnalysisModuleSet();
        assertEquals(
            "Did not return correct count.",
            0,
            ams.getAnalysisModuleCount() );
    }
    
    
    public void testGetAnalysisModuleCount2()
    {
        AnalysisModuleSet ams = new AnalysisModuleSet(
            CCCreatorUtil.createAnalysisModules( 15 ) );
        assertEquals(
            "Did not return correct count.",
            15,
            ams.getAnalysisModuleCount() );
    }
    
    
    
    
    
    
    //-------------------------------------------------------------------------
    // Helpers
    
    
    protected IAnalysisModule createIAnalysisModule( String name )
    {
        return CCCreatorUtil.createIAnalysisModule( name, "u", "m" );
    }
    
    
    
    //-------------------------------------------------------------------------
    // Standard JUnit declarations
    
    
    public static Test suite()
    {
        TestSuite suite = new TestSuite( THIS_CLASS );
        
        return suite;
    }
    
    public static void main( String[] args )
    {
        String[] name = { THIS_CLASS.getName() };
        
        // junit.textui.TestRunner.main( name );
        // junit.swingui.TestRunner.main( name );
        
        junit.textui.TestRunner.main( name );
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void setUp() throws Exception
    {
        super.setUp();

       
        // set ourself up
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void tearDown() throws Exception
    {
        // tear ourself down
        
        
        super.tearDown();
    }
}

