/*
 * @(#)ConvertSingleLogUTest.java
 *
 * Copyright (C) 2004 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.util;

import net.sourceforge.groboutils.codecoverage.v2.IChannelLogger;
import java.io.Reader;
import java.io.BufferedReader;
import java.io.IOException;
import java.util.*;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import net.sourceforge.groboutils.autodoc.v1.AutoDoc;


/**
 * Tests the ConvertSingleLog class.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2004/07/07 09:39:14 $
 * @since     April 16, 2004
 */
public class ConvertSingleLogUTest extends TestCase
{
    //-------------------------------------------------------------------------
    // Standard JUnit Class-specific declarations
    
    private static final Class THIS_CLASS = ConvertSingleLogUTest.class;
    private static final AutoDoc DOC = new AutoDoc( THIS_CLASS );
    
    public ConvertSingleLogUTest( String name )
    {
        super( name );
    }



    //-------------------------------------------------------------------------
    // Tests
    
    
    public void testConstructor1()
    {
        try
        {
            new ConvertSingleLog( null );
            fail( "Didn't throw IllegalArgumentException" );
        }
        catch (IllegalArgumentException e)
        {
            // check exception
        }
    }
    
    
    public void testConstructor2()
    {
        try
        {
            new ConvertSingleLog( new IChannelLogger[0] );
            fail( "Didn't throw IllegalArgumentException" );
        }
        catch (IllegalArgumentException e)
        {
            // check exception
        }
    }
    
    
    public void testConstructor3()
    {
        try
        {
            new ConvertSingleLog( new IChannelLogger[1] );
            fail( "Didn't throw IllegalArgumentException" );
        }
        catch (IllegalArgumentException e)
        {
            // check exception
        }
    }
    
    
    public void testParseShort1()
    {
        MyChannelLogger mcl = new MyChannelLogger();
        ConvertSingleLog csl = new ConvertSingleLog( new IChannelLogger[] { mcl } );
        
        try
        {
            csl.parseShort( "1,", "0,1,2" );
            fail( "Did not throw IOE" );
        }
        catch (IOException e)
        {
            // check output
        }
    }
    
    
    public void testParseShort2()
    {
        MyChannelLogger mcl = new MyChannelLogger();
        ConvertSingleLog csl = new ConvertSingleLog( new IChannelLogger[] { mcl } );
        
        try
        {
            csl.parseShort( "asdfasdf", "0,1,2" );
            fail( "Did not throw IOE" );
        }
        catch (IOException e)
        {
            // check output
        }
    }
    
    
    public void testParseShort3() throws IOException
    {
        MyChannelLogger mcl = new MyChannelLogger();
        ConvertSingleLog csl = new ConvertSingleLog( new IChannelLogger[] { mcl } );
        
        assertEquals(
            "Didn't convert right",
            12,
            csl.parseShort( "12", "0,1,2" ) );
    }
    
    
    public void testNextElement1()
    {
        MyChannelLogger mcl = new MyChannelLogger();
        ConvertSingleLog csl = new ConvertSingleLog( new IChannelLogger[] { mcl } );
        
        try
        {
            int pos[] = { 0, 0 };
            csl.nextElement( pos, "asdf" );
            fail("Did not throw IOE.");
        }
        catch (IOException e)
        {
            // check exception
        }
    }
    
    
    public void testProcessLine1()
    {
        MyChannelLogger mcl = new MyChannelLogger();
        ConvertSingleLog csl = new ConvertSingleLog( new IChannelLogger[] { mcl } );
        
        try
        {
            csl.processLine( null, true );
            fail("Did not throw IOE");
        }
        catch (IOException e)
        {
            assertTrue(
                "Bad exception text",
                e.getMessage().indexOf("End of stream: line is null") >= 0 );
        }
    }
    
    
    public void testProcessLine2()
    {
        MyChannelLogger mcl = new MyChannelLogger();
        ConvertSingleLog csl = new ConvertSingleLog( new IChannelLogger[] { mcl } );
        
        try
        {
            csl.processLine( "1,asdf,1,1", false );
            fail("Did not throw IOE");
        }
        catch (IOException e)
        {
            assertTrue(
                "Bad exception text",
                e.getMessage().indexOf("Invalid channel:") >= 0 );
        }
    }
    
    
    public void testProcessLine3() throws IOException
    {
        MyChannelLogger mclL[] = {
                new MyChannelLogger(),
                new MyChannelLogger(),
                new MyChannelLogger()
        };
        ConvertSingleLog csl = new ConvertSingleLog( mclL );
        
        csl.processLine( "1,asdf1,1 2", false );
        csl.processLine( "0,asdf2,0006 000C", false );
        csl.processLine( "2,asdf3,9 3", false );
        
        assertEquals( "did not store all class IDs for c0",
            new String[] { "asdf2" }, 
            mclL[0].getClassIDs() );
        assertEquals( "did not store all marks for c0",
            new String[] { "6,12" }, 
            mclL[0].getMarks("asdf2") );
        
        assertEquals( "did not store all class IDs for c1",
            new String[] { "asdf1" }, 
            mclL[1].getClassIDs() );
        assertEquals( "did not store all marks for c1",
            new String[] { "1,2" }, 
            mclL[1].getMarks("asdf1") );
        
        assertEquals( "did not store all class IDs for c2",
            new String[] { "asdf3" }, 
            mclL[2].getClassIDs() );
        assertEquals( "did not store all marks for c2",
            new String[] { "9,3" }, 
            mclL[2].getMarks("asdf3") );
    }
    
    
    public void testProcessLine4() throws IOException
    {
        MyChannelLogger mcl = new MyChannelLogger();
        ConvertSingleLog csl = new ConvertSingleLog( new IChannelLogger[] { mcl } );
        
        csl.processLine( "0,A,0001 0002", false );
        csl.processLine( "0,B,0005 000C", false );
        csl.processLine( "0,C,03E7 0006", false );
        csl.processLine( "0,D,0012 0004", false );
        csl.processLine( "0,D,0017 004E", false );
        csl.processLine( "0,D,0017 0000", false );
        
        assertEquals( "did not store all class IDs",
            new String[] { "A", "B", "C", "D" }, 
            mcl.getClassIDs() );
        assertEquals( "did not store all marks for A",
            new String[] { "1,2" }, 
            mcl.getMarks("A") );
        assertEquals( "did not store all marks for B",
            new String[] { "5,12" }, 
            mcl.getMarks("B") );
        assertEquals( "did not store all marks for C",
            new String[] { "999,6" }, 
            mcl.getMarks("C") );
        assertEquals( "did not store all marks",
            new String[] { "23,78", "23,0", "18,4" }, 
            mcl.getMarks("D") );
    }
    
    
    
    //-------------------------------------------------------------------------
    // Helpers
    
    private static class MyChannelLogger implements IChannelLogger
    {
        private Map classToMark = new HashMap();
        
        public void cover( String classID, short methodIndex, short markIndex )
        {
            List s = (List)this.classToMark.get( classID );
            if (s == null)
            {
                s = new ArrayList();
                this.classToMark.put( classID, s );
            }
            s.add( ""+methodIndex+','+markIndex );
        }
        
        
        public String[] getClassIDs()
        {
            Set s = this.classToMark.keySet();
            return (String[])s.toArray( new String[ s.size() ] );
        }
        
        public String[] getMarks( String classID )
        {
            List s = (List)this.classToMark.get( classID );
            if (s == null)
            {
                return new String[0];
            }
            return (String[])s.toArray( new String[ s.size() ] );
        }
    }
    
    
    private static void assertEquals( String msg, String[] a, String[] b )
    {
        if (a == null)
        {
            assertNull(
                msg, b );
            return;
        }
        if (b == null)
        {
            fail(msg+": actual array is null, but expected isn't."); 
            return;
        }
        
        assertEquals(
            msg+": lengths don't match;",
            a.length,
            b.length );
        Arrays.sort( a );
        Arrays.sort( b );
        for (int i = 0; i < a.length; ++i)
        {
            assertEquals(
                msg+": contents aren't equal",
                a[i], b[i] );
        }
    }
    
    
    //-------------------------------------------------------------------------
    // Standard JUnit declarations
    
    
    public static Test suite()
    {
        TestSuite suite = new TestSuite( THIS_CLASS );
        
        return suite;
    }
    
    public static void main( String[] args )
    {
        String[] name = { THIS_CLASS.getName() };
        
        // junit.textui.TestRunner.main( name );
        // junit.swingui.TestRunner.main( name );
        
        junit.textui.TestRunner.main( name );
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void setUp() throws Exception
    {
        super.setUp();
        
        // set ourself up
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void tearDown() throws Exception
    {
        // tear ourself down
        
        
        super.tearDown();
    }
}

