/*
 * basiclog.cpp - basic log formats implementation
 * $Id: basiclog.cpp 176 2005-09-03 06:49:55Z rdenisc $
 */

/***********************************************************************
 *  Copyright (C) 2002-2005 Remi Denis-Courmont.                       *
 *  This program is free software; you can redistribute and/or modify  *
 *  it under the terms of the GNU General Public License as published  *
 *  by the Free Software Foundation; version 2 of the license.         *
 *                                                                     *
 *  This program is distributed in the hope that it will be useful,    *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of     *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.               *
 *  See the GNU General Public License for more details.               *
 *                                                                     *
 *  You should have received a copy of the GNU General Public License  *
 *  along with this program; if not, you can get it from:              *
 *  http://www.gnu.org/copyleft/gpl.html                               *
 ***********************************************************************/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include "gettext.h"

#include "output.h"
#include "basiclog.h"


/*** BasicDataLog class implementation ***/
GenericDataLog::~GenericDataLog (void)
{
	if (out != NULL)
		fputs (_("End of transmission.\n"), out);
}



void
GenericDataLog::Connect (const char *server, const char *service,
				const char *client, const char *port)
{
	fprintf (out, _("Connection from: %s port %s\n"), client, port);
	fprintf (out, _("Connection to: %s port %s\n"), server, service);
	fputs (_("Transmission follows...\n"), out);
}


int
GenericDataLog::WriteServerData (const void *, int length, int)
{
	if (outcount >= 0)
		outcount += length;
	return length;
}


int
BasicDataLog::WriteServerData (const void *data, int length, int oob)
{
	GenericDataLog::WriteServerData (data, length, oob);

	if ((oob && (fputs ("(OOB)", out) < 0))
	 || (fputs (">>> ", out) < 0))
		return -1;
	return logwrite (data, length, out);
}


int
GenericDataLog::WriteClientData (const void *, int length, int)
{
	if (incount >= 0)
		incount += length;
	return length;
}


int
BasicDataLog::WriteClientData (const void *data, int length, int oob)
{
	GenericDataLog::WriteClientData (data, length, oob);

	if ((oob && (fputs ("(OOB)", out) < 0))
	 || (fputs ("<<< ", out) < 0))
		return -1;
	return logwrite (data, length, out);
}


void
GenericDataLog::ShutdownServer (void)
{
	fputs (_("End of output"), out);
	if (outcount >= 0)
		fprintf (out, ngettext (": %ld byte sent.\n", ": %ld bytes sent.\n",
		                        outcount), outcount);
	else
		fputs (".\n", out);
}


void
GenericDataLog::ShutdownClient (void)
{
	fputs (_("End of input"), out);
	if (incount >= 0)
		fprintf (out, ngettext (": %ld byte received.\n",
		                        ": %ld bytes received.\n", incount), incount);
	else
		fputs (".\n", out);
}


/* Makers */
DataLog *
CountDataLogMaker (void)
{
	return new BasicDataLog (fwrite_count);
}


DataLog *
CDataLogMaker (void)
{
	return new BasicDataLog (fwrite_C);
}


DataLog *
HexDataLogMaker (void)
{
	return new BasicDataLog (fwrite_hex);
}


DataLog *
RawDataLogMaker (void)
{
	return new BasicDataLog (fwrite_raw);
}


DataLog *
StrippedDataLogMaker (void)
{
	return new BasicDataLog (fwrite_strip);
}

DataLog *
NullDataLogMaker (void)
{
	return new GenericDataLog;
}
