# -*- coding: utf-8 -*-

"""
On Screen Display (OSD) for Tegaki
"""

import gtk
import gobject

from tegaki.character import Writing
from tegakigtk.fakekey import send_unicode
from tegaki.recognizer import Recognizer


class Window():
    def __init__(self, color='red', line_width=5):
        """
            :param color: The foreground color we want to use. can be named
('black', 'red', etc.) or hex ('#ff0000', '#abcdef')
            :param line_width: The width of a line
        """
        # create the window (without decorations and display in the taskbar
#through override_redirect)
        self.window = gtk.Window(gtk.WINDOW_TOPLEVEL)
        # get screen
        self.screen = self.window.get_screen()
        # get size
        self.size = (self.screen.get_width(), self.screen.get_height())
        # get the colormap and the fg color
        self.colormap = self.screen.get_rgb_colormap()
        self.color = self.colormap.alloc_color(color)
        self.c_black = self.colormap.alloc_color('black')
        self.c_white = self.colormap.alloc_color('white')
        self.window.set_default_size(self.size[0], self.size[1])
        self.window.set_decorated(False)
        self.window.set_colormap(self.colormap)
        self.window.show()
        self.window.window.set_override_redirect(True)
        # create the pixmap
        self.pixmap = gtk.gdk.Pixmap(self.window.window, self.size[0],
self.size[1])
        self.pixmap.set_colormap(self.colormap)
        self.pixmap.gc = self.pixmap.new_gc(foreground=self.color,
line_width=line_width)
        self.pixmap.gc.set_colormap(self.colormap)
        # create the mask
        self.mask = gtk.gdk.Pixmap(None, self.size[0], self.size[1], 1)
        self.mask.gc = self.mask.new_gc(foreground=self.c_white,
line_width=line_width)
        # reset everything
        self.clear()
        self.window.stick()
        # set `old_pos`
        self.old_pos = None

    def destroy(self):
        """ destroy the window """
        self.window.destroy()

    def move_to(self, x, y):
        self.old_pos = (x, y)
        self.writing.move_to(x,y)

    def line_to(self, x, y):
        """ paint a new station in the gesture's path """
        if self.old_pos:
            self.draw_line(self.old_pos[0], self.old_pos[1], x, y)
            self.writing.line_to(x,y)
        self.old_pos = (x, y)

    def clear(self):
        """ clear + remove all """
        self.writing = Writing()
        self.writing.set_size(*self.size)
        self.window.hide()
        self.reset_mask()
        self.reset_pixmap()
        self.update_mask(False)
        self.update_pixmap()
        self.window.hide()
        self.old_pos = None

    def draw_line(self, x1, y1, x2, y2):
        """ wrapper for a line """
        # draw on pixmap
        self.pixmap.draw_line(self.pixmap.gc, x1, y1, x2, y2)
        # draw on mask
        self.mask.draw_line(self.mask.gc, x1, y1, x2, y2)
        # update all
        self.update_mask()
        self.update_pixmap()

    def destroy(self):
        """ destroy the window """
        self.window.destroy()

    def reset_mask(self):
        """ reset the shape mask to fully transparent """
        # fill black
        self.mask.gc.set_foreground(self.c_black) #todo?
        self.mask.draw_rectangle(self.mask.gc, True, 0, 0, self.size[0],
self.size[1])
        self.mask.gc.set_foreground(self.c_white)

    def reset_pixmap(self):
        """ reset the pixmap to fully black """
        # fill black
        self.pixmap.gc.set_foreground(self.c_black) #todo?
        self.pixmap.draw_rectangle(self.pixmap.gc, True, 0, 0, self.size[0],
self.size[1])
        self.pixmap.gc.set_foreground(self.color)

    def update_mask(self, do_show=True):
        """ update the mask """
        self.window.window.input_shape_combine_mask(self.mask, 0, 0) # vllt nach
#.show()?
        self.window.window.shape_combine_mask(self.mask, 0, 0) # vllt nach
#.show()?
        if do_show:
            self.window.show()

    def update_pixmap(self):
        """ update the pixmap """
        self.window.window.begin_paint_rect((0, 0, self.size[0], self.size[1]))
        self.window.window.set_back_pixmap(self.pixmap, False)
        self.window.window.end_paint()

    def get_writing(self):
        return self.writing.copy()


class Listener:
    def __init__(self, recognizer):
        self.root_window = gtk.gdk.get_default_root_window()
        #self.create_osd_window()
        self.running = False
        self._window = Window()
        self._drawing = False
        self._new_stroke = True
        self._recognizer = recognizer

    def stop(self):
        self.running = False

    def start(self):
        self.running = True
        gobject.timeout_add(100, self._listen)

    def _listen(self):

        x, y, state = self.root_window.get_pointer()
       
        ctrl = gtk.gdk.CONTROL_MASK
        ctrl_left = gtk.gdk.CONTROL_MASK|gtk.gdk.BUTTON1_MASK

        if state & ctrl_left == ctrl_left:

            if self._new_stroke:
                self._window.move_to(x,y)
            else:
                self._window.line_to(x,y)

            self._new_stroke = False
            self._drawing = True

        elif state & ctrl == ctrl:
            self._new_stroke = True

        else:
            if self._drawing:
                print "drawing done"
                writing = self._window.get_writing()
                print writing
                writing.normalize()
                writing.upsample_threshold(10)

                # should not be necessary but...
                writing.resize(1000.0/writing.get_width(),
                               1000.0/writing.get_height())
                writing.set_size(1000, 1000)                
              
                res = recognizer.recognize(writing)
                best = res[0][0]

                self._window.clear()

                send_unicode(unicode(best, "utf8"))

                clipboard = gtk.Clipboard()
                clipboard.set_text(best)
                
                #win = gtk.Window()
                #from tegakigtk.canvas import Canvas
                #canvas = Canvas()
                #canvas.set_writing(writing)
                #win.add(canvas)
                #win.show_all()

                self._drawing = None
    

        return self.running

if __name__ == "__main__":
    import sys

    #win = Window()
    ##win.show_all()
    #gtk.main()

    recognizer = sys.argv[1] # name of recognizer
    model = sys.argv[2] # name of model file

    recognizers = Recognizer.get_available_recognizers()
    print "Available recognizers", recognizers.keys()

    if not recognizer in recognizers:
        raise Exception, "Not an available recognizer"

    recognizer_klass = recognizers[recognizer]
    recognizer = recognizer_klass()

    models = recognizer_klass.get_available_models()
    print "Available models", models.keys()

    if not model in models:
        raise Exception, "Not an available model"

    recognizer.set_model(model)

    listener = Listener(recognizer)
    listener.start()
    gtk.main()