/*
 * ===========================
 * VDK Visual Development Kit
 * Version 2.0.0
 * December 2000
 * ===========================
 *
 * Copyright (C) 1998,199,2000,2001 Mario Motta
 * Developed by Mario Motta <mmotta@guest.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#ifndef  _vdktextview_h
#define  _vdktextview_h
#include <vdk/widcontain.h>
#include <vdk/vdkprops.h>
#include <vdk/eventbox.h>
#define INSERT_MARK "insert"
class VDKTextView;
/*!
  \class VDKTextBuffer
  \brief Provides a gtk+ gtktextbuffer wrapper. 
  \par Programming notes:
  VDKTextBuffer can be shared by several VDKTextView.
  These objects aren't handled by garbage collection (aren't VDKObjects),
  however are reference conted so do not need to  be explicitely deleted 
  if assigned to a VDKTextView using either the appropriate VDKTextView 
  contructor or VDKText::Buffer() assignement method. 
  If constructed and never assigned to a VDKTextView they need to be 
  explicitely deleted or they will leak.
*/
#ifndef USE_SIGCPLUSPLUS
class VDKTextBuffer : public VDKNotCopyAble
#else
class VDKTextBuffer : public SigC::Object, public VDKNotCopyAble
#endif
{
  friend class VDKTextView;

private:
    unsigned int ref;
    void Ref();
    void Unref();
  
protected:

  GtkTextBuffer* buffer;
public:
  /*!
    Sets/gets text insertion position expressed in chars from text
    beginning. (counting from 0, gets -1 on failure)
   */
  VDKReadWriteValueProp<VDKTextBuffer,int>   Pointer;
  /*!
    Sets/gets text insertion position expressed in column offset
   */
  VDKReadWriteValueProp<VDKTextBuffer,int>   Column;
  /*!
    Sets/gets text insertion position expressed in lines
  */
  VDKReadWriteValueProp<VDKTextBuffer,int>   Line;
  /*
  Gets buffer length in chars
  */
  VDKReadOnlyValueProp<VDKTextBuffer,unsigned int> Length;
  /*!
    Sets/gets modified flag
  */
  VDKReadWriteValueProp<VDKTextBuffer,bool>   Changed;
  /*!
    Default constructor
  */
  VDKTextBuffer();
  /*!
    Default constructor
    \param filename builds a text buffer loading file <filename>
  */
  VDKTextBuffer(char* filename);
  /*!
    Destructor
  */
  virtual ~VDKTextBuffer();
  /*!
    Load text buffer with a file
    \param filename to load
  */
  bool LoadFromFile(const char* filename);
  /*!
  Save buffer to file
  */
  bool SaveToFile(const char* filename);
  /*!
    Returns underlying GtkTextBuffer
  */
  GtkTextBuffer* Buffer() { return buffer; }
  /*!
  Clears text buffer deleting all text.
  */
  void Clear();
  /*!
  Inserts text at cursor position.
  \param txt null terminating string to be inserted
  \param nchars how many chars have to be inserted (-1 means all)
  */
  void TextInsert(const char* txt, int nchar = -1); 
  /*!
  Returns a text segment
  \param start starting position
  \param end ending position, if -1 all text buffer from start to end will
  be returned.
  
  \par Tip
  Invisible chars won't be included.
  Returned address should be g_free()'d by user.
  */
  gchar* GetChars(int start, int end = -1);
  /*!
  Forward delete chars from insertion point
  */
  void  ForwardDelete(int nchars);  
  /*!
  Backward delete chars from insertion point
  */
  void  BackwardDelete(int nchars);
  /*!
  Undoes last operation
  */
    void Undo() { }
  /*!
  Gets line number at a buffer position
  \param offset buffer offset position (counting from 0)
  on error returns -1
  */
  int GetLineAtOffset(int offset);
  /*
  properties setting/getting functions
  */
  void SetPointer(int p);
  int GetPointer();
  void SetLine(int r);
  int GetLine();
  void SetColumn(int r);
  int GetColumn();
  unsigned int GetLength() 
  {
    return gtk_text_buffer_get_char_count(buffer);
  }
  bool GetChanged()
  {
    return gtk_text_buffer_get_modified(buffer);
  }
  void SetChanged(bool f)
  {
    gtk_text_buffer_set_modified(buffer,f); 
  }
};


/*!
  \class VDKTextView
  \brief Provides a gtk+ gtktextview wrapper.
  This widget works under a buffer-view model, text buffer take care of changes and text view of displaying.
  Any change to text buffer will reflect automagically into  text view.  
  This widget substitutes old VDKText used on vdkbuilder version <= 1.2.5 and now deprecated
  even if will remain in use on earlier versions of VDK 2.0 series.
 */
/* TextView border interested windows
  used by VDKTextView::TextBorder()
 */
#define TVB_ALL               0x0000
#define TVB_LEFT              0x0001
#define TVB_TOP               0x0002
#define TVB_RIGHT             0x0004
#define TVB_BOTTOM            0x0008
#define TVB_TYPEMASK          0x000F


class VDKTextView : public VDKObjectContainer
{
 protected:
  VDKTextBuffer* buffer;
  GtkWidget* view;
  void ConnectSignals();
  static void HandleRealize(GtkWidget*, gpointer);
  int left_border;
  public:
  /*!
    Constructor
    \param owner
    \param buffer NULL widget will make his own buffer, otherwise
    will share <buffer>
    \param left_border if greater than 0 a left window will be made
    with <left_border> size
    \par tip
    left_border should be greater than zero if user wants activate
    ShowLineNumbers property.
   */
  VDKTextView(VDKForm* owner, VDKTextBuffer* buffer = NULL,
            int left_border = 0);
  /*!
    Destructor
   */
  virtual ~VDKTextView();

  virtual void SetForeground(VDKRgb rgb, 
			     GtkStateType state = GTK_STATE_NORMAL);
  virtual void SetBackground(VDKRgb color, 
			     GtkStateType state = GTK_STATE_NORMAL);
  virtual void SetFont(VDKFont* font);
  /*!
  Sets/Gets text buffer.
  Several text views can share same text buffer:
  \code
  VDKTextBuffer* textbuff = new VDKTextBuffer();
  VDKTextView view = new VDKTextView(this,textbuff);
  VDKTextView view1 = new VDKTextView(this);  
  view1->Buffer(textbuff);
  \endcode
  */
  VDKTextBuffer* Buffer(VDKTextBuffer* buff = NULL);  
  /*!
  Sets borders around text window.
  \param size border size in pixels
  \param which side border should be set
  - TVB_LEFT
  - TVB_TOP
  - TVB_RIGHT
  - TVB_BOTTOM
  that can be ored togheter.
  - TVB_ALL (default) use this to set all sides border size.
  */
  void TextBorder(int size, int which = TVB_ALL);
  /*!
  Scrolls to a pointer pos or (default) to current
  pointer position, leaving <margin> pixels free
  */
  void ScrollToPos (int pointer = -1, int margin = 0);
  /*!
  Scrolls to a line,column leaving <margin> pixels free
  */
  void ScrollToLine(int line, int col, int margin = 0);
  /*!
    Sets/gets text insertion position expressed in chars from text
    beginning. (counting from 0, gets -1 on failure)
   */
  VDKReadWriteValueProp<VDKTextView,int>   Pointer;
  /*!
    Sets/gets text insertion position expressed in column offset
   */
  VDKReadWriteValueProp<VDKTextView,int>   Column;
  /*!
    Sets/gets text insertion position expressed in lines
   */
  VDKReadWriteValueProp<VDKTextView,int>   Line;
  /*
  Gets buffer length in chars
  */
  VDKReadOnlyValueProp<VDKTextView,unsigned int> Length;
  /*
  Sets/gets text view editable
  */
  VDKReadWriteValueProp<VDKTextView,bool> Editable;
  /*
  Sets/gets max undo (dummy for now)
  */
  VDKReadWriteValueProp<VDKTextView,unsigned int> MaxUndo;
  /*
  Sets/gets text line auto select (dummy for now)
  */
  VDKReadWriteValueProp<VDKTextView,bool> LineAutoSelect;
  /*
  Sets/gets text show line numbers (dummy for now)
  */
  VDKReadWriteValueProp<VDKTextView,bool> ShowLineNumbers;
  /*!
  Gets first visible line (lines are counted from 0)
  */
  VDKReadOnlyValueProp<VDKTextView,int> FirstVisibleLine;
  /*!
  Gets last visible line (lines are counted from 0)
  */
  VDKReadOnlyValueProp<VDKTextView,int> LastVisibleLine;
  /*!
    Sets/gets modified flag
  */
  VDKReadWriteValueProp<VDKTextView,bool>   Changed;

  /*!
    Load text buffer with a file
    \param filename
  */
  bool LoadFromFile(char* filename) 
  {
    return buffer->LoadFromFile(filename);
  }
  /*!
  Clears text buffer deleting all text.
  */
  void Clear() { buffer->Clear(); }
  /*!
  Returns a text segment
  \param start starting position
  \param end ending position, if -1 all text buffer from start to end will
  be returned.
  
  \par Tip
  Invisible chars won't be included.
  Returned address should be g_free()'d by user.
  */
  gchar* GetChars(int start = 0, int end = -1)
  {
    return buffer->GetChars(start,end);
  }
  /*!
  Save buffer to file
  */
  bool SaveToFile(char* filename) { return buffer->SaveToFile(filename); }
  /*!
  maintained for compatibility with old text widgets, but is dummy
  */
  void Thaw() {}
  /*!
  maintained for compatibility with old text widgets, but is dummy
  */
  void Freeze() 
    {
    }
    /*!
    Undoes last operation
    */
    void Undo() { buffer->Undo(); }
    /*!
    Inserts a new line at insertion point
    */
    void Eol() { TextInsert("\n"); }
    /*!
    Inserts text at cursor position.
    \param txt null terminating string to be inserted
    \param nchars how many chars have to be inserted (-1 means all)
    */
    void TextInsert(const char* txt, int nchar = -1)
       { 
           buffer->TextInsert(txt,nchar);
       }
  /*!
  Forward delete chars from insertion point
  */
  void  ForwardDelete(int nchars)
  { buffer->ForwardDelete(nchars); }
  /*!
  Backward delete chars from insertion point
  */
  void  BackwardDelete(int nchars)
    { buffer->BackwardDelete(nchars); }
  /*!
  Answer if a line is visible or not
  line are counted from 0
  \param line
  */
  bool IsLineVisible(int line)
    { 
      return (line >= FirstVisibleLine) && 
      (line <= LastVisibleLine);
    }
  /*!
  Gets line number at a buffer position
  \param offset buffer offset position (counting from 0)
  on error returns -1
  */
  int GetLineAtOffset(int offset)
  { return buffer->GetLineAtOffset(offset); }
  /*
  properties setting/getting functions
  */
  void SetPointer(int p) { buffer->SetPointer(p); }
  int GetPointer() { return buffer->GetPointer(); }
  void SetLine(int r) { buffer->SetLine(r); }
  int GetLine() { return buffer->GetLine(); }
  void SetColumn(int r) { buffer->SetColumn(r); }
  int GetColumn() { return buffer->GetColumn(); }
  unsigned int GetLength() { return buffer->GetLength(); }
  bool GetEditable() 
  { return gtk_text_view_get_editable (GTK_TEXT_VIEW(view));}
  void SetEditable(bool f)
  { gtk_text_view_set_editable (GTK_TEXT_VIEW(view),f);}
  void SetShowLineNumbers(bool f);
  int GetFirstVisibleLine();
  int GetLastVisibleLine();
  bool GetChanged() { return buffer->GetChanged(); }
  void SetChanged(bool f) { buffer->SetChanged(f); } 
};
#endif
