#include <stdio.h>
#include <string.h>
#include "lisptype.h"
#include "numbers.h"
#include "standard.h"
#include "errors.h"



//////////////////////////////////////////////////
///// bits_to_digits and digits_to_bits implementation
//////////////////////////////////////////////////

// lookup table for transforming the number of digits

const unsigned log2_table_size = 32;
// report the table size
unsigned log2_table_range()
{
  return log2_table_size;
}

#ifdef HAVE_MATH_H
#include <math.h>
// A lookup table of Ln(n)/Ln(2) for n = 1 .. 32.
// With this we don't have to use math.h if all we need is to convert the number of digits from one base to another. This is also faster.
// Generated by: PrintList(N(Ln(1 .. 32)/Ln(2)), ",") at precision 40
const double log2_table[log2_table_size] =
{
0.,
1.,
1.5849625007211561814537389439478165087598,
2.,
2.3219280948873623478703194294893901758648,
2.5849625007211561814537389439478165087598,
2.807354922057604107441969317231830808641,
3.,
3.1699250014423123629074778878956330175196,
3.3219280948873623478703194294893901758648,
3.4594316186372972561993630467257929587032,
3.5849625007211561814537389439478165087598,
3.7004397181410921603968126542566947336284,
3.807354922057604107441969317231830808641,
3.9068905956085185293240583734372066846246,
4.,
4.0874628412503394082540660108104043540112,
4.1699250014423123629074778878956330175196,
4.2479275134435854937935194229068344226935,
4.3219280948873623478703194294893901758648,
4.3923174227787602888957082611796473174008,
4.4594316186372972561993630467257929587032,
4.5235619560570128722941482441626688444988,
4.5849625007211561814537389439478165087598,
4.6438561897747246957406388589787803517296,
4.7004397181410921603968126542566947336284,
4.7548875021634685443612168318434495262794,
4.807354922057604107441969317231830808641,
4.8579809951275721207197733246279847624768,
4.9068905956085185293240583734372066846246,
4.9541963103868752088061235991755544235489,
5.
};

// table look-up of small integer logarithms, for converting the number of digits to binary and back
double log2_table_lookup(unsigned n)
{
    if (n<=log2_table_size && n>=BASE2)
      return log2_table[n-1];
    else
    {
      RaiseError("log2_table_lookup: error: invalid argument %d\n", n);
      return 0;
    }
}
// convert the number of digits in given base to the number of bits, and back.
// need to round the number of digits.
// to make sure that there is no hysteresis, we round upwards on digits_to_bits but round down on bits_to_digits
unsigned long digits_to_bits(unsigned long digits, unsigned base)
{
  return (unsigned long)ceil(double(digits)*log2_table_lookup(base));
}

unsigned long bits_to_digits(unsigned long bits, unsigned base)
{
  return (unsigned long)floor(double(bits)/log2_table_lookup(base));
}

#else  // if have no math.h
// on platforms without math.h we don't expect much serious math but we do need at least some precision in digit conversions.
// so we approximate logs by best rational fractions.
struct RationalFrac
{  // this represents p/q with integer p, q and q>0
  long p, q;
};

// need tables of both Ln(n)/Ln(2) and of Ln(2)/Ln(n).
// table of Ln(n)/Ln(2)
const RationalFrac log2_table_l[log2_table_size] =
{
{0, 1}, {1, 1}, {24727, 15601}, {2, 1}, {1493, 643}, {40328, 15601}, {1603, 571}, {3, 1}, {23673, 7468}, {2136, 643}, {3411, 986}, {55929, 15601}, {840, 227}, {2174, 571}, {13721, 3512}, {4, 1}, {1402, 343}, {31141, 7468}, {5637, 1327}, {2779, 643}, {3202, 729}, {4397, 986}, {13055, 2886}, {71530, 15601}, {2986, 643}, {1067, 227}, {22619, 4757}, {2745, 571}, {4139, 852}, {17233, 3512}, {157375, 31766}, {5, 1}
};
// table of Ln(2)/Ln(n)
const RationalFrac log2_table_linv[log2_table_size] =
{
{0, 1}, {1, 1}, {665, 1054}, {1, 2}, {4004, 9297}, {665, 1719}, {2393, 6718}, {1, 3}, {665, 2108}, {4004, 13301}, {1935, 6694}, {665, 2384}, {5231, 19357}, {2393, 9111}, {4049, 15819}, {1, 4}, {4036, 16497}, {665, 2773}, {1206, 5123}, {4004, 17305}, {1588, 6975}, {1935, 8629}, {3077, 13919}, {665, 3049}, {2002, 9297}, {5231, 24588}, {665, 3162}, {2393, 11504}, {4943, 24013}, {537, 2635}, {3515, 17414}, {1, 5}};


// need lookup of both Ln(n)/Ln(2) and of Ln(2)/Ln(n)
const RationalFrac& log2_table_lookup(const RationalFrac* table, unsigned n)
{
  if (n<=log2_table_size && n>=BASE2)
    return table[n-1];
  else
  {
    RaiseError("log2_table_fake_lookup: error: invalid argument %d\n", n);
    return table[0];
  }
};
const RationalFrac& log2_table_lookup_l(unsigned n)
{
  return log2_table_lookup(log2_table_l, n);
}
const RationalFrac& log2_table_lookup_linv(unsigned n)
{
  return log2_table_lookup(log2_table_linv, n);
}

// multiply aF by aX and round up to the nearest integer
unsigned long fake_multiply_ceil(const RationalFrac& aF, unsigned long aX)
{
  // compute aX * aF.p / aF.q carefully to avoid overflow and to not undershoot: (maximum number we need to support is p*q, not p*x)
  // result = p*Div(x,q) + Div(Mod(x, q)*p+q-1, q)
  return aF.p*(((unsigned long)aX)/((unsigned long)aF.q))
  + ((aX % ((unsigned long)aF.q))*aF.p+aF.q-1)/((unsigned long)aF.q);
}


// multiply aF by aX and round down to the nearest integer
unsigned long fake_multiply_floor(const RationalFrac& aF, unsigned long aX)
{
  // compute aX * aF.p / aF.q carefully to avoid overflow and to not undershoot: (maximum number we need to support is p*q, not p*x)
  // result = p*Div(x,q) + Div(Mod(x, q)*p, q)
  return aF.p*(((unsigned long)aX)/((unsigned long)aF.q))
  + ((aX % ((unsigned long)aF.q))*aF.p)/((unsigned long)aF.q);
}


// convert the number of digits in given base to the number of bits, and back
unsigned long digits_to_bits(unsigned long digits, unsigned base)
{
  return fake_multiply_ceil(log2_table_lookup_l(base), digits);
}

unsigned long bits_to_digits(unsigned long bits, unsigned base)
{
  return fake_multiply_floor(log2_table_lookup_linv(base), bits);
}

#endif // HAVE_MATH_H

//////////////////////////////////////////////////
///// End of bits_to_digits and digits_to_bits implementation
//////////////////////////////////////////////////
